/* otfdescrip.cc -- descriptions for OpenType tags
 *
 * Copyright (c) 2002-2023 Eddie Kohler
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <efont/otf.hh>

struct Mapping {
    uint32_t tag;
    const char *description;
};

#define T(str) ((str)[0]<<24 | (str)[1]<<16 | (str)[2]<<8 | (str)[3])

static const Mapping scripts[] = {
    { T("DFLT"), "Default" },
    { T("adlm"), "Adlam" },
    { T("aghb"), "Caucasian Albanian" },
    { T("ahom"), "Ahom" },
    { T("arab"), "Arabic" },
    { T("armi"), "Imperial Aramaic" },
    { T("armn"), "Armenian" },
    { T("avst"), "Avestan" },
    { T("bali"), "Balinese" },
    { T("bamu"), "Bamum" },
    { T("bass"), "Bassa Vah" },
    { T("batk"), "Batak" },
    { T("beng"), "Bengali" },
    { T("bhks"), "Bhaiksuki" },
    { T("bng2"), "Bengali v.2" },
    { T("bopo"), "Bopomofo" },
    { T("brah"), "Brahmi" },
    { T("brai"), "Braille" },
    { T("bugi"), "Buginese" },
    { T("buhd"), "Buhid" },
    { T("byzm"), "Byzantine Music" },
    { T("cakm"), "Chakma" },
    { T("cans"), "Canadian Syllabics" },
    { T("cari"), "Carian" },
    { T("cham"), "Cham" },
    { T("cher"), "Cherokee" },
    { T("chrs"), "Chorasmian" },
    { T("copt"), "Coptic" },
    { T("cpmn"), "Cypro-Minoan" },
    { T("cprt"), "Cypriot Syllabary" },
    { T("cyrl"), "Cyrillic" },
    { T("dev2"), "Devanagari v.2" },
    { T("deva"), "Devanagari" },
    { T("diak"), "Dives Akuru" },
    { T("dogr"), "Dogra" },
    { T("dsrt"), "Deseret" },
    { T("dupl"), "Duployan" },
    { T("egyp"), "Egyptian Hieroglyphs" },
    { T("elba"), "Elbasan" },
    { T("elym"), "Elymaic" },
    { T("ethi"), "Ethiopic" },
    { T("geor"), "Georgian" },
    { T("gjr2"), "Gujarati v.2" },
    { T("glag"), "Glagolitic" },
    { T("gong"), "Gunjala Gondi" },
    { T("gonm"), "Masaram Gondi" },
    { T("goth"), "Gothic" },
    { T("gran"), "Grantha" },
    { T("grek"), "Greek" },
    { T("gujr"), "Gujarati" },
    { T("gur2"), "Gurmukhi v.2" },
    { T("guru"), "Gurmukhi" },
    { T("hang"), "Hangul" },
    { T("hani"), "CJK Ideographic" },
    { T("hano"), "Hanunoo" },
    { T("hatr"), "Hatran" },
    { T("hebr"), "Hebrew" },
    { T("hluw"), "Anatolian Hieroglyphs" },
    { T("hmng"), "Pahawh Hmong" },
    { T("hmnp"), "Nyiakeng Puachue Hmong" },
    { T("hung"), "Old Hungarian" },
    { T("ital"), "Old Italic" },
    { T("jamo"), "Hangul Jamo" },    // not recommended -- prefer `hang`
    { T("java"), "Javanese" },
    { T("kali"), "Kayah Li" },
    { T("kana"), "Hiragana/Katakana" },
    { T("khar"), "Kharosthi" },
    { T("khmr"), "Khmer" },
    { T("khoj"), "Khojki" },
    { T("kits"), "Khitan Small Script" },
    { T("knd2"), "Kannada v.2" },
    { T("knda"), "Kannada" },
    { T("kthi"), "Kaithi" },
    { T("lana"), "Tai Tham (Lanna)" },
    { T("lao "), "Lao" },
    { T("latn"), "Latin" },
    { T("lepc"), "Lepcha" },
    { T("limb"), "Limbu" },
    { T("lina"), "Linear A" },
    { T("linb"), "Linear B" },
    { T("lisu"), "Lisu (Fraser)" },
    { T("lyci"), "Lycian" },
    { T("lydi"), "Lydian" },
    { T("mahj"), "Mahajani" },
    { T("maka"), "Makasar" },
    { T("mand"), "Mandaic, Mandaean" },
    { T("mani"), "Manichaean" },
    { T("marc"), "Marchen" },
    { T("math"), "Mathematical Alphanumeric Symbols" },
    { T("medf"), "Medefaidrin (Oberi Okaime, Oberi Ɔkaimɛ)" },
    { T("mend"), "Mende Kikakui" },
    { T("merc"), "Meroitic Cursive" },
    { T("mero"), "Meroitic Hieroglyphs" },
    { T("mlm2"), "Malayalam v.2" },
    { T("mlym"), "Malayalam" },
    { T("modi"), "Modi" },
    { T("mong"), "Mongolian" },
    { T("mroo"), "Mro" },
    { T("mtei"), "Meitei Mayek (Meithei, Meetei)" },
    { T("mult"), "Multani" },
    { T("musc"), "Musical Symbols" },
    { T("mym2"), "Myanmar v.2" },
    { T("mymr"), "Myanmar" },    // not recommended -- prefer `mym2`
    { T("nand"), "Nandinagari" },
    { T("narb"), "Old North Arabian" },
    { T("nbat"), "Nabataean" },
    { T("newa"), "Newa" },
    { T("nko "), "N'Ko" },
    { T("nshu"), "Nüshu" },
    { T("ogam"), "Ogham" },
    { T("olck"), "Ol Chiki" },
    { T("orkh"), "Old Turkic, Orkhon Runic" },
    { T("ory2"), "Odia v.2 (formerly Oriya v.2)" },
    { T("orya"), "Odia (formerly Oriya)" },
    { T("osge"), "Osage" },
    { T("osma"), "Osmanya" },
    { T("ougr"), "Old Uyghur" },
    { T("palm"), "Palmyrene" },
    { T("pauc"), "Pau Cin Hau" },
    { T("perm"), "Old Permic" },
    { T("phag"), "Phags-pa" },
    { T("phli"), "Inscriptional Pahlavi" },
    { T("phlp"), "Psalter Pahlavi" },
    { T("phnx"), "Phoenician" },
    { T("plrd"), "Miao" },
    { T("prti"), "Inscriptional Parthian" },
    { T("rjng"), "Rejang" },
    { T("rohg"), "Hanifi Rohingya" },
    { T("runr"), "Runic" },
    { T("samr"), "Samaritan" },
    { T("sarb"), "Old South Arabian" },
    { T("saur"), "Saurashtra" },
    { T("sgnw"), "Sign Writing" },
    { T("shaw"), "Shavian" },
    { T("shrd"), "Sharada" },
    { T("sidd"), "Siddham" },
    { T("sind"), "Khudawadi" },
    { T("sinh"), "Sinhala" },
    { T("sogd"), "Sogdian" },
    { T("sogo"), "Old Sogdian" },
    { T("sora"), "Sora Sompeng" },
    { T("soyo"), "Soyombo" },
    { T("sund"), "Sundanese" },
    { T("sylo"), "Syloti Nagri" },
    { T("syrc"), "Syriac" },
    { T("tagb"), "Tagbanwa" },
    { T("takr"), "Takri" },
    { T("tale"), "Tai Le" },
    { T("talu"), "New Tai Lue" },
    { T("taml"), "Tamil" },
    { T("tang"), "Tangut" },
    { T("tavt"), "Tai Viet" },
    { T("tel2"), "Telugu v.2" },
    { T("telu"), "Telugu" },
    { T("tfng"), "Tifinagh" },
    { T("tglg"), "Tagalog" },
    { T("thaa"), "Thaana" },
    { T("thai"), "Thai" },
    { T("tibt"), "Tibetan" },
    { T("tirh"), "Tirhuta" },
    { T("tml2"), "Tamil v.2" },
    { T("tnsa"), "Tangsa" },
    { T("toto"), "Toto" },
    { T("ugar"), "Ugaritic Cuneiform" },
    { T("vai "), "Vai" },
    { T("vith"), "Vithkuqi" },
    { T("wara"), "Warang Citi" },
    { T("wcho"), "Wancho" },
    { T("xpeo"), "Old Persian Cuneiform" },
    { T("xsux"), "Sumero-Akkadian Cuneiform" },
    { T("yezi"), "Yezidi" },
    { T("yi  "), "Yi" },
    { T("zanb"), "Zanabazar Square (Zanabazarin Dörböljin Useg, Xewtee Dörböljin Bicig, Horizontal Square Script)" }
};

static const Mapping languages[] = {
    { T("ABA "), "Abaza" },
    { T("ABK "), "Abkhazian" },
    { T("ACH "), "Acholi" },
    { T("ACR "), "Achi" },
    { T("ADY "), "Adyghe" },
    { T("AFK "), "Afrikaans" },
    { T("AFR "), "Afar" },
    { T("AGW "), "Agaw" },
    { T("AIO "), "Aiton" },
    { T("AKA "), "Akan" },
    { T("AKB "), "Batak Angkola" },
    { T("ALS "), "Alsatian" },
    { T("ALT "), "Altai" },
    { T("AMH "), "Amharic" },
    { T("ANG "), "Anglo-Saxon" },
    { T("APPH"), "Phonetic transcription—Americanist conventions" },
    { T("ARA "), "Arabic" },
    { T("ARG "), "Aragonese" },
    { T("ARI "), "Aari" },
    { T("ARK "), "Rakhine" },
    { T("ASM "), "Assamese" },
    { T("AST "), "Asturian" },
    { T("ATH "), "Athapaskan languages" },
    { T("AVN "), "Avatime" },
    { T("AVR "), "Avar" },
    { T("AWA "), "Awadhi" },
    { T("AYM "), "Aymara" },
    { T("AZB "), "Torki" },
    { T("AZE "), "Azerbaijani" },
    { T("BAD "), "Badaga" },
    { T("BAD0"), "Banda" },
    { T("BAG "), "Baghelkhandi" },
    { T("BAL "), "Balkar" },
    { T("BAN "), "Balinese" },
    { T("BAR "), "Bavarian" },
    { T("BAU "), "Baulé" },
    { T("BBC "), "Batak Toba" },
    { T("BBR "), "Berber" },
    { T("BCH "), "Bench" },
    { T("BCR "), "Bible Cree" },
    { T("BDY "), "Bandjalang" },
    { T("BEL "), "Belarussian" },
    { T("BEM "), "Bemba" },
    { T("BEN "), "Bengali" },
    { T("BGC "), "Haryanvi" },
    { T("BGQ "), "Bagri" },
    { T("BGR "), "Bulgarian" },
    { T("BHI "), "Bhili" },
    { T("BHO "), "Bhojpuri" },
    { T("BIK "), "Bikol" },
    { T("BIL "), "Bilen" },
    { T("BIS "), "Bislama" },
    { T("BJJ "), "Kanauji" },
    { T("BKF "), "Blackfoot" },
    { T("BLI "), "Baluchi" },
    { T("BLK "), "Pa’o Karen" },
    { T("BLN "), "Balante" },
    { T("BLT "), "Balti" },
    { T("BMB "), "Bambara (Bamanankan)" },
    { T("BML "), "Bamileke" },
    { T("BOS "), "Bosnian" },
    { T("BPY "), "Bishnupriya Manipuri" },
    { T("BRE "), "Breton" },
    { T("BRH "), "Brahui" },
    { T("BRI "), "Braj Bhasha" },
    { T("BRM "), "Burmese" },
    { T("BRX "), "Bodo" },
    { T("BSH "), "Bashkir" },
    { T("BSK "), "Burushaski" },
    { T("BTD "), "Batak Dairi (Pakpak)" },
    { T("BTI "), "Beti" },
    { T("BTK "), "Batak languages" },
    { T("BTM "), "Batak Mandailing" },
    { T("BTS "), "Batak Simalungun" },
    { T("BTX "), "Batak Karo" },
    { T("BTZ "), "Batak Alas-Kluet" },
    { T("BUG "), "Bugis" },
    { T("BYV "), "Medumba" },
    { T("CAK "), "Kaqchikel" },
    { T("CAT "), "Catalan" },
    { T("CBK "), "Zamboanga Chavacano" },
    { T("CCHN"), "Chinantec" },
    { T("CEB "), "Cebuano" },
    { T("CGG "), "Chiga" },
    { T("CHA "), "Chamorro" },
    { T("CHE "), "Chechen" },
    { T("CHG "), "Chaha Gurage" },
    { T("CHH "), "Chattisgarhi" },
    { T("CHI "), "Chichewa (Chewa, Nyanja)" },
    { T("CHK "), "Chukchi" },
    { T("CHK0"), "Chuukese" },
    { T("CHO "), "Choctaw" },
    { T("CHP "), "Chipewyan" },
    { T("CHR "), "Cherokee" },
    { T("CHU "), "Chuvash" },
    { T("CHY "), "Cheyenne" },
    { T("CJA "), "Western Cham" },
    { T("CJM "), "Eastern Cham" },
    { T("CMR "), "Comorian" },
    { T("COP "), "Coptic" },
    { T("COR "), "Cornish" },
    { T("COS "), "Corsican" },
    { T("CPP "), "Creoles" },
    { T("CRE "), "Cree" },
    { T("CRR "), "Carrier" },
    { T("CRT "), "Crimean Tatar" },
    { T("CSB "), "Kashubian" },
    { T("CSL "), "Church Slavonic" },
    { T("CSY "), "Czech" },
    { T("CTG "), "Chittagonian" },
    { T("CTT "), "Wayanad Chetti" },
    { T("CUK "), "San Blas Kuna" },
    { T("DAG "), "Dagbani" },
    { T("DAN "), "Danish" },
    { T("DAR "), "Dargwa" },
    { T("DAX "), "Dayi" },
    { T("DCR "), "Woods Cree" },
    { T("DEU "), "German" },
    { T("DGO "), "Dogri (individual language)" },
    { T("DGR "), "Dogri (macrolanguage)" },
    { T("DHG "), "Dhangu" },
    { T("DHV "), "Divehi (Dhivehi, Maldivian)" }, // deprecated
    { T("DIQ "), "Dimli" },
    { T("DIV "), "Divehi (Dhivehi, Maldivian)" },
    { T("DJR "), "Zarma" },
    { T("DJR0"), "Djambarrpuyngu" },
    { T("DNG "), "Dangme" },
    { T("DNJ "), "Dan" },
    { T("DNK "), "Dinka" },
    { T("DRI "), "Dari" },
    { T("DUJ "), "Dhuwal" },
    { T("DUN "), "Dungan" },
    { T("DZN "), "Dzongkha" },
    { T("EBI "), "Ebira" },
    { T("ECR "), "Eastern Cree" },
    { T("EDO "), "Edo" },
    { T("EFI "), "Efik" },
    { T("ELL "), "Greek" },
    { T("EMK "), "Eastern Maninkakan" },
    { T("ENG "), "English" },
    { T("ERZ "), "Erzya" },
    { T("ESP "), "Spanish" },
    { T("ESU "), "Central Yupik" },
    { T("ETI "), "Estonian" },
    { T("EUQ "), "Basque" },
    { T("EVK "), "Evenki" },
    { T("EVN "), "Even" },
    { T("EWE "), "Ewe" },
    { T("FAN "), "French Antillean" },
    { T("FAN0"), "Fang" },
    { T("FAR "), "Persian" },
    { T("FAT "), "Fanti" },
    { T("FIN "), "Finnish" },
    { T("FJI "), "Fijian" },
    { T("FLE "), "Dutch (Flemish)" },
    { T("FMP "), "Fe’fe’" },
    { T("FNE "), "Forest Enets" },
    { T("FON "), "Fon" },
    { T("FOS "), "Faroese" },
    { T("FRA "), "French" },
    { T("FRC "), "Cajun French" },
    { T("FRI "), "Frisian" },
    { T("FRL "), "Friulian" },
    { T("FRP "), "Arpitan" },
    { T("FTA "), "Futa" },
    { T("FUL "), "Fulah" },
    { T("FUV "), "Nigerian Fulfulde" },
    { T("GAD "), "Ga" },
    { T("GAE "), "Scottish Gaelic (Gaelic)" },
    { T("GAG "), "Gagauz" },
    { T("GAL "), "Galician" },
    { T("GAR "), "Garshuni" },
    { T("GAW "), "Garhwali" },
    { T("GEZ "), "Geez" },
    { T("GIH "), "Githabul" },
    { T("GIL "), "Gilyak" },
    { T("GIL0"), "Kiribati (Gilbertese)" },
    { T("GKP "), "Kpelle (Guinea)" },
    { T("GLK "), "Gilaki" },
    { T("GMZ "), "Gumuz" },
    { T("GNN "), "Gumatj" },
    { T("GOG "), "Gogo" },
    { T("GON "), "Gondi" },
    { T("GRN "), "Greenlandic" },
    { T("GRO "), "Garo" },
    { T("GUA "), "Guarani" },
    { T("GUC "), "Wayuu" },
    { T("GUF "), "Gupapuyngu" },
    { T("GUJ "), "Gujarati" },
    { T("GUZ "), "Gusii" },
    { T("HAI "), "Haitian (Haitian Creole)" },
    { T("HAI0"), "Haida" },
    { T("HAL "), "Halam (Falam Chin)" },
    { T("HAR "), "Harauti" },
    { T("HAU "), "Hausa" },
    { T("HAW "), "Hawaiian" },
    { T("HAY "), "Haya" },
    { T("HAZ "), "Hazaragi" },
    { T("HBN "), "Hammer-Banna" },
    { T("HEI "), "Heiltsuk" },
    { T("HER "), "Herero" },
    { T("HIL "), "Hiligaynon" },
    { T("HIN "), "Hindi" },
    { T("HMA "), "High Mari" },
    { T("HMD "), "A-Hmao" },
    { T("HMN "), "Hmong" },
    { T("HMO "), "Hiri Motu" },
    { T("HMZ "), "Hmong Shuat" },
    { T("HND "), "Hindko" },
    { T("HO  "), "Ho" },
    { T("HRI "), "Harari" },
    { T("HRV "), "Croatian" },
    { T("HUN "), "Hungarian" },
    { T("HYE "), "Armenian" },
    { T("HYE0"), "Armenian East" },
    { T("IBA "), "Iban" },
    { T("IBB "), "Ibibio" },
    { T("IBO "), "Igbo" },
    { T("IDO "), "Ido" },
    { T("IJO "), "Ijo languages" },
    { T("ILE "), "Interlingue" },
    { T("ILO "), "Ilokano" },
    { T("INA "), "Interlingua" },
    { T("IND "), "Indonesian" },
    { T("ING "), "Ingush" },
    { T("INU "), "Inuktitut" },
    { T("INUK"), "Nunavik Inuktitut" },
    { T("IPK "), "Inupiat" },
    { T("IPPH"), "Phonetic transcription—IPA conventions" },
    { T("IRI "), "Irish" },
    { T("IRT "), "Irish Traditional" },
    { T("IRU "), "Irula" },
    { T("ISL "), "Icelandic" },
    { T("ISM "), "Inari Sami" },
    { T("ITA "), "Italian" },
    { T("IWR "), "Hebrew" },
    { T("JAM "), "Jamaican Creole" },
    { T("JAN "), "Japanese" },
    { T("JAV "), "Javanese" },
    { T("JBO "), "Lojban" },
    { T("JCT "), "Krymchak" },
    { T("JII "), "Yiddish" },
    { T("JUD "), "Ladino" },
    { T("JUL "), "Jula" },
    { T("KAB "), "Kabardian" },
    { T("KAB0"), "Kabyle" },
    { T("KAC "), "Kachchi" },
    { T("KAL "), "Kalenjin" },
    { T("KAN "), "Kannada" },
    { T("KAR "), "Karachay" },
    { T("KAT "), "Georgian" },
    { T("KAW "), "Kawi (Old Javanese)" },
    { T("KAZ "), "Kazakh" },
    { T("KDE "), "Makonde" },
    { T("KEA "), "Kabuverdianu (Crioulo)" },
    { T("KEB "), "Kebena" },
    { T("KEK "), "Kekchi" },
    { T("KGE "), "Khutsuri Georgian" },
    { T("KHA "), "Khakass" },
    { T("KHK "), "Khanty-Kazim" },
    { T("KHM "), "Khmer" },
    { T("KHS "), "Khanty-Shurishkar" },
    { T("KHT "), "Khamti Shan" },
    { T("KHV "), "Khanty-Vakhi" },
    { T("KHW "), "Khowar" },
    { T("KIK "), "Kikuyu (Gikuyu)" },
    { T("KIR "), "Kirghiz (Kyrgyz)" },
    { T("KIS "), "Kisii" },
    { T("KIU "), "Kirmanjki" },
    { T("KJD "), "Southern Kiwai" },
    { T("KJP "), "Eastern Pwo Karen" },
    { T("KJZ "), "Bumthangkha" },
    { T("KKN "), "Kokni" },
    { T("KLM "), "Kalmyk" },
    { T("KMB "), "Kamba" },
    { T("KMN "), "Kumaoni" },
    { T("KMO "), "Komo" },
    { T("KMS "), "Komso" },
    { T("KMZ "), "Khorasani Turkic" },
    { T("KNR "), "Kanuri" },
    { T("KOD "), "Kodagu" },
    { T("KOH "), "Korean Old Hangul" },
    { T("KOK "), "Konkani" },
    { T("KOM "), "Komi" },
    { T("KON "), "Kikongo" },
    { T("KON0"), "Kongo" },
    { T("KOP "), "Komi-Permyak" },
    { T("KOR "), "Korean" },
    { T("KOS "), "Kosraean" },
    { T("KOZ "), "Komi-Zyrian" },
    { T("KPL "), "Kpelle" },
    { T("KRI "), "Krio" },
    { T("KRK "), "Karakalpak" },
    { T("KRL "), "Karelian" },
    { T("KRM "), "Karaim" },
    { T("KRN "), "Karen" },
    { T("KRT "), "Koorete" },
    { T("KSH "), "Kashmiri" },
    { T("KSH0"), "Ripuarian" },
    { T("KSI "), "Khasi" },
    { T("KSM "), "Kildin Sami" },
    { T("KSW "), "S’gaw Karen" },
    { T("KUA "), "Kuanyama" },
    { T("KUI "), "Kui" },
    { T("KUL "), "Kulvi" },
    { T("KUM "), "Kumyk" },
    { T("KUR "), "Kurdish" },
    { T("KUU "), "Kurukh" },
    { T("KUY "), "Kuy" },
    { T("KWK "), "Kwakʼwala" },
    { T("KYK "), "Koryak" },
    { T("KYU "), "Western Kayah" },
    { T("LAD "), "Ladin" },
    { T("LAH "), "Lahuli" },
    { T("LAK "), "Lak" },
    { T("LAM "), "Lambani" },
    { T("LAO "), "Lao" },
    { T("LAT "), "Latin" },
    { T("LAZ "), "Laz" },
    { T("LCR "), "L-Cree" },
    { T("LDK "), "Ladakhi" },
    { T("LEF "), "Lelemi" },
    { T("LEZ "), "Lezgi" },
    { T("LIJ "), "Ligurian" },
    { T("LIM "), "Limburgish" },
    { T("LIN "), "Lingala" },
    { T("LIS "), "Lisu" },
    { T("LJP "), "Lampung" },
    { T("LKI "), "Laki" },
    { T("LMA "), "Low Mari" },
    { T("LMB "), "Limbu" },
    { T("LMO "), "Lombard" },
    { T("LMW "), "Lomwe" },
    { T("LOM "), "Loma" },
    { T("LPO "), "Lipo" },
    { T("LRC "), "Luri" },
    { T("LSB "), "Lower Sorbian" },
    { T("LSM "), "Lule Sami" },
    { T("LTH "), "Lithuanian" },
    { T("LTZ "), "Luxembourgish" },
    { T("LUA "), "Luba-Lulua" },
    { T("LUB "), "Luba-Katanga" },
    { T("LUG "), "Ganda" },
    { T("LUH "), "Luyia" },
    { T("LUO "), "Luo" },
    { T("LVI "), "Latvian" },
    { T("MAD "), "Madura" },
    { T("MAG "), "Magahi" },
    { T("MAH "), "Marshallese" },
    { T("MAJ "), "Majang" },
    { T("MAK "), "Makhuwa" },
    { T("MAL "), "Malayalam" },
    { T("MAM "), "Mam" },
    { T("MAN "), "Mansi" },
    { T("MAP "), "Mapudungun" },
    { T("MAR "), "Marathi" },
    { T("MAW "), "Marwari" },
    { T("MBN "), "Mbundu" },
    { T("MBO "), "Mbo" },
    { T("MCH "), "Manchu" },
    { T("MCR "), "Moose Cree" },
    { T("MDE "), "Mende" },
    { T("MDR "), "Mandar" },
    { T("MEN "), "Me’en" },
    { T("MER "), "Meru" },
    { T("MFA "), "Pattani Malay" },
    { T("MFE "), "Morisyen" },
    { T("MIN "), "Minangkabau" },
    { T("MIZ "), "Mizo" },
    { T("MKD "), "Macedonian" },
    { T("MKR "), "Makasar" },
    { T("MKW "), "Kituba" },
    { T("MLE "), "Male" },
    { T("MLG "), "Malagasy" },
    { T("MLN "), "Malinke" },
    { T("MLR "), "Malayalam Reformed" },
    { T("MLY "), "Malay" },
    { T("MND "), "Mandinka" },
    { T("MNG "), "Mongolian" },
    { T("MNI "), "Manipuri" },
    { T("MNK "), "Maninka" },
    { T("MNX "), "Manx" },
    { T("MOH "), "Mohawk" },
    { T("MOK "), "Moksha" },
    { T("MOL "), "Moldavian" },
    { T("MON "), "Mon" },
    { T("MONT"), "Thailand Mon" },
    { T("MOR "), "Moroccan" },
    { T("MOS "), "Mossi" },
    { T("MRI "), "Maori" },
    { T("MTH "), "Maithili" },
    { T("MTS "), "Maltese" },
    { T("MUN "), "Mundari" },
    { T("MUS "), "Muscogee" },
    { T("MWL "), "Mirandese" },
    { T("MWW "), "Hmong Daw" },
    { T("MYN "), "Mayan" },
    { T("MZN "), "Mazanderani" },
    { T("NAG "), "Naga-Assamese" },
    { T("NAH "), "Nahuatl" },
    { T("NAN "), "Nanai" },
    { T("NAP "), "Neapolitan" },
    { T("NAS "), "Naskapi" },
    { T("NAU "), "Nauruan" },
    { T("NAV "), "Navajo" },
    { T("NCR "), "N-Cree" },
    { T("NDB "), "Ndebele" },
    { T("NDC "), "Ndau" },
    { T("NDG "), "Ndonga" },
    { T("NDS "), "Low Saxon" },
    { T("NEP "), "Nepali" },
    { T("NEW "), "Newari" },
    { T("NGA "), "Ngbaka" },
    { T("NGR "), "Nagari" },
    { T("NHC "), "Norway House Cree" },
    { T("NIS "), "Nisi" },
    { T("NIU "), "Niuean" },
    { T("NKL "), "Nyankole" },
    { T("NKO "), "N’Ko" },
    { T("NLD "), "Dutch" },
    { T("NOE "), "Nimadi" },
    { T("NOG "), "Nogai" },
    { T("NOR "), "Norwegian" },
    { T("NOV "), "Novial" },
    { T("NSM "), "Northern Sami" },
    { T("NSO "), "Northern Sotho" },
    { T("NTA "), "Northern Tai" },
    { T("NTO "), "Esperanto" },
    { T("NYM "), "Nyamwezi" },
    { T("NYN "), "Norwegian Nynorsk (Nynorsk, Norwegian)" },
    { T("NZA "), "Mbembe Tigon" },
    { T("OCI "), "Occitan" },
    { T("OCR "), "Oji-Cree" },
    { T("OJB "), "Ojibway" },
    { T("ORI "), "Odia (formerly Oriya)" },
    { T("ORO "), "Oromo" },
    { T("OSS "), "Ossetian" },
    { T("PAA "), "Palestinian Aramaic" },
    { T("PAG "), "Pangasinan" },
    { T("PAL "), "Pali" },
    { T("PAM "), "Pampangan" },
    { T("PAN "), "Punjabi" },
    { T("PAP "), "Palpa" },
    { T("PAP0"), "Papiamentu" },
    { T("PAS "), "Pashto" },
    { T("PAU "), "Palauan" },
    { T("PCC "), "Bouyei" },
    { T("PCD "), "Picard" },
    { T("PDC "), "Pennsylvania German" },
    { T("PGR "), "Polytonic Greek" },
    { T("PHK "), "Phake" },
    { T("PIH "), "Norfolk" },
    { T("PIL "), "Filipino" },
    { T("PLG "), "Palaung" },
    { T("PLK "), "Polish" },
    { T("PMS "), "Piemontese" },
    { T("PNB "), "Western Panjabi" },
    { T("POH "), "Pocomchi" },
    { T("PON "), "Pohnpeian" },
    { T("PRO "), "Provençal / Old Provençal" },
    { T("PTG "), "Portuguese" },
    { T("PWO "), "Western Pwo Karen" },
    { T("QIN "), "Chin" },
    { T("QUC "), "K’iche’" },
    { T("QUH "), "Quechua (Bolivia)" },
    { T("QUZ "), "Quechua" },
    { T("QVI "), "Quechua (Ecuador)" },
    { T("QWH "), "Quechua (Peru)" },
    { T("RAJ "), "Rajasthani" },
    { T("RAR "), "Rarotongan" },
    { T("RBU "), "Russian Buriat" },
    { T("RCR "), "R-Cree" },
    { T("REJ "), "Rejang" },
    { T("RIA "), "Riang" },
    { T("RHG "), "Rohingya" },
    { T("RIF "), "Tarifit" },
    { T("RIT "), "Ritarungo" },
    { T("RKW "), "Arakwal" },
    { T("RMS "), "Romansh" },
    { T("RMY "), "Vlax Romani" },
    { T("ROM "), "Romanian" },
    { T("ROY "), "Romany" },
    { T("RSY "), "Rusyn" },
    { T("RTM "), "Rotuman" },
    { T("RUA "), "Kinyarwanda" },
    { T("RUN "), "Rundi" },
    { T("RUP "), "Aromanian" },
    { T("RUS "), "Russian" },
    { T("SAD "), "Sadri" },
    { T("SAN "), "Sanskrit" },
    { T("SAS "), "Sasak" },
    { T("SAT "), "Santali" },
    { T("SAY "), "Sayisi" },
    { T("SCN "), "Sicilian" },
    { T("SCO "), "Scots" },
    { T("SCS "), "North Slavey" },
    { T("SEK "), "Sekota" },
    { T("SEL "), "Selkup" },
    { T("SFM "), "Small Flowery Miao" },
    { T("SGA "), "Old Irish" },
    { T("SGO "), "Sango" },
    { T("SGS "), "Samogitian" },
    { T("SHI "), "Tachelhit" },
    { T("SHN "), "Shan" },
    { T("SIB "), "Sibe" },
    { T("SID "), "Sidamo" },
    { T("SIG "), "Silte Gurage" },
    { T("SKS "), "Skolt Sami" },
    { T("SKY "), "Slovak" },
    { T("SLA "), "Slavey" },
    { T("SLV "), "Slovenian" },
    { T("SML "), "Somali" },
    { T("SMO "), "Samoan" },
    { T("SNA "), "Sena" },
    { T("SNA0"), "Shona" },
    { T("SND "), "Sindhi" },
    { T("SNH "), "Sinhala (Sinhalese)" },
    { T("SNK "), "Soninke" },
    { T("SOG "), "Sodo Gurage" },
    { T("SOP "), "Songe" },
    { T("SOT "), "Southern Sotho" },
    { T("SQI "), "Albanian" },
    { T("SRB "), "Serbian" },
    { T("SRD "), "Sardinian" },
    { T("SRK "), "Saraiki" },
    { T("SRR "), "Serer" },
    { T("SSL "), "South Slavey" },
    { T("SSM "), "Southern Sami" },
    { T("STQ "), "Saterland Frisian" },
    { T("SUK "), "Sukuma" },
    { T("SUN "), "Sundanese" },
    { T("SUR "), "Suri" },
    { T("SVA "), "Svan" },
    { T("SVE "), "Swedish" },
    { T("SWA "), "Swadaya Aramaic" },
    { T("SWK "), "Swahili" },
    { T("SWZ "), "Swati" },
    { T("SXT "), "Sutu" },
    { T("SXU "), "Upper Saxon" },
    { T("SYL "), "Sylheti" },
    { T("SYR "), "Syriac" },
    { T("SYRE"), "Syriac, Estrangela script-variant (equivalent to ISO 15924 'Syre')" },
    { T("SYRJ"), "Syriac, Western script-variant (equivalent to ISO 15924 'Syrj')" },
    { T("SYRN"), "Syriac, Eastern script-variant (equivalent to ISO 15924 'Syrn')" },
    { T("SZL "), "Silesian" },
    { T("TAB "), "Tabasaran" },
    { T("TAJ "), "Tajiki" },
    { T("TAM "), "Tamil" },
    { T("TAT "), "Tatar" },
    { T("TCR "), "TH-Cree" },
    { T("TDD "), "Dehong Dai" },
    { T("TEL "), "Telugu" },
    { T("TET "), "Tetum" },
    { T("TGL "), "Tagalog" },
    { T("TGN "), "Tongan" },
    { T("TGR "), "Tigre" },
    { T("TGY "), "Tigrinya" },
    { T("THA "), "Thai" },
    { T("THT "), "Tahitian" },
    { T("TIB "), "Tibetan" },
    { T("TIV "), "Tiv" },
    { T("TJL "), "Tai Laing" },
    { T("TKM "), "Turkmen" },
    { T("TLI "), "Tlingit" },
    { T("TMH "), "Tamashek" },
    { T("TMN "), "Temne" },
    { T("TNA "), "Tswana" },
    { T("TNE "), "Tundra Enets" },
    { T("TNG "), "Tonga" },
    { T("TOD "), "Todo" },
    { T("TOD0"), "Toma" },
    { T("TPI "), "Tok Pisin" },
    { T("TRK "), "Turkish" },
    { T("TSG "), "Tsonga" },
    { T("TSJ "), "Tshangla" },
    { T("TUA "), "Turoyo Aramaic" },
    { T("TUL "), "Tulu" },
    { T("TUM "), "Tumbuka" },
    { T("TUR "), "Turkish" },   // deprecated?
    { T("TUV "), "Tuvin" },
    { T("TVL "), "Tuvalu" },
    { T("TWI "), "Twi" },
    { T("TYZ "), "Tày" },
    { T("TZM "), "Tamazight" },
    { T("TZO "), "Tzotzil" },
    { T("UDM "), "Udmurt" },
    { T("UKR "), "Ukrainian" },
    { T("UMB "), "Umbundu" },
    { T("URD "), "Urdu" },
    { T("USB "), "Upper Sorbian" },
    { T("UYG "), "Uyghur" },
    { T("UZB "), "Uzbek" },
    { T("VEC "), "Venetian" },
    { T("VEN "), "Venda" },
    { T("VIT "), "Vietnamese" },
    { T("VOL "), "Volapük" },
    { T("VRO "), "Võro" },
    { T("WA  "), "Wa" },
    { T("WAG "), "Wagdi" },
    { T("WAR "), "Waray-Waray" },
    { T("WCI "), "Waci Gbe" },
    { T("WCR "), "West-Cree" },
    { T("WEL "), "Welsh" },
    { T("WLF "), "Wolof" },
    { T("WLN "), "Walloon" },
    { T("WTM "), "Mewati" },
    { T("XBD "), "Lü" },
    { T("XHS "), "Xhosa" },
    { T("XJB "), "Minjangbal" },
    { T("XKF "), "Khengkha" },
    { T("XOG "), "Soga" },
    { T("XPE "), "Kpelle (Liberia)" },
    { T("XUB "), "Bette Kuruma" },
    { T("XUJ "), "Jennu Kuruma" },
    { T("YAK "), "Sakha" },
    { T("YAO "), "Yao" },
    { T("YAP "), "Yapese" },
    { T("YBA "), "Yoruba" },
    { T("YCR "), "Y-Cree" },
    { T("YGP "), "Gepo" },
    { T("YIC "), "Yi Classic" },
    { T("YIM "), "Yi Modern" },
    { T("YNA "), "Aluo" },
    { T("YWQ "), "Wuding-Luquan Yi" },
    { T("ZEA "), "Zealandic" },
    { T("ZGH "), "Standard Moroccan Tamazight" },
    { T("ZHA "), "Zhuang" },
    { T("ZHH "), "Chinese, Traditional, Hong Kong SAR" },
    { T("ZHP "), "Chinese, Phonetic" },
    { T("ZHS "), "Chinese, Simplified" },
    { T("ZHT "), "Chinese, Traditional" },
    { T("ZHTM"), "Chinese, Traditional, Macao SAR" },
    { T("ZND "), "Zande" },
    { T("ZUL "), "Zulu" },
    { T("ZZA "), "Zazaki" }
};

static const Mapping features[] = {
    { T("aalt"), "Access All Alternates" },
    { T("abvf"), "Above-base Forms" },
    { T("abvm"), "Above-base Mark Positioning" },
    { T("abvs"), "Above-base Substitutions" },
    { T("afrc"), "Alternative Fractions" },
    { T("akhn"), "Akhand" },
    { T("blwf"), "Below-base Forms" },
    { T("blwm"), "Below-base Mark Positioning" },
    { T("blws"), "Below-base Substitutions" },
    { T("c2pc"), "Petite Capitals From Capitals" },
    { T("c2sc"), "Small Capitals From Capitals" },
    { T("calt"), "Contextual Alternates" },
    { T("case"), "Case-Sensitive Forms" },
    { T("ccmp"), "Glyph Composition/Decomposition" },
    { T("cfar"), "Conjunct Form After Ro" },
    { T("chws"), "Contextual Half-width Spacing" },
    { T("cjct"), "Conjunct Forms" },
    { T("clig"), "Contextual Ligatures" },
    { T("cpct"), "Centered CJK Punctuation" },
    { T("cpsp"), "Capital Spacing" },
    { T("cswh"), "Contextual Swash" },
    { T("curs"), "Cursive Positioning" },
    { T("cv01"), "Character Variants 1" },
    { T("cv02"), "Character Variants 2" },
    { T("cv03"), "Character Variants 3" },
    { T("cv04"), "Character Variants 4" },
    { T("cv05"), "Character Variants 5" },
    { T("cv06"), "Character Variants 6" },
    // ... up to cv99
    { T("dist"), "Distances" },
    { T("dlig"), "Discretionary Ligatures" },
    { T("dnom"), "Denominators" },
    { T("dtls"), "Dotless Forms" },
    { T("expt"), "Expert Forms" },
    { T("falt"), "Final Glyph on Line Alternates" },
    { T("fin2"), "Terminal Forms #2" },
    { T("fin3"), "Terminal Forms #3" },
    { T("fina"), "Terminal Forms" },
    { T("flac"), "Flattened Accent Forms" },
    { T("frac"), "Fractions" },
    { T("fwid"), "Full Widths" },
    { T("half"), "Half Forms" },
    { T("haln"), "Halant Forms" },
    { T("halt"), "Alternate Half Widths" },
    { T("hist"), "Historical Forms" },
    { T("hkna"), "Horizontal Kana Alternates" },
    { T("hlig"), "Historical Ligatures" },
    { T("hngl"), "Hangul" },
    { T("hojo"), "Hojo Kanji Forms (JIS X 0212-1990 Kanji Forms)" },
    { T("hwid"), "Half Widths" },
    { T("init"), "Initial Forms" },
    { T("isol"), "Isolated Forms" },
    { T("ital"), "Italics" },
    { T("jalt"), "Justification Alternates" },
    { T("jp04"), "JIS2004 Forms" },
    { T("jp78"), "JIS78 Forms" },
    { T("jp83"), "JIS83 Forms" },
    { T("jp90"), "JIS90 Forms" },
    { T("kern"), "Kerning" },
    { T("lfbd"), "Left Bounds" },
    { T("liga"), "Standard Ligatures" },
    { T("ljmo"), "Leading Jamo Forms" },
    { T("lnum"), "Lining Figures" },
    { T("locl"), "Localized Forms" },
    { T("ltra"), "Left-to-right alternates" },
    { T("ltrm"), "Left-to-right mirrored forms" },
    { T("mark"), "Mark Positioning" },
    { T("med2"), "Medial Forms #2" },
    { T("medi"), "Medial Forms" },
    { T("mgrk"), "Mathematical Greek" },
    { T("mkmk"), "Mark to Mark Positioning" },
    { T("mset"), "Mark Positioning via Substitution" },
    { T("nalt"), "Alternate Annotation Forms" },
    { T("nlck"), "NLC Kanji Forms" },
    { T("nukt"), "Nukta Forms" },
    { T("numr"), "Numerators" },
    { T("onum"), "Oldstyle Figures" },
    { T("opbd"), "Optical Bounds" },
    { T("ordn"), "Ordinals" },
    { T("ornm"), "Ornaments" },
    { T("palt"), "Proportional Alternate Widths" },
    { T("pcap"), "Petite Capitals" },
    { T("pkna"), "Proportional Kana" },
    { T("pnum"), "Proportional Figures" },
    { T("pref"), "Pre-Base Forms" },
    { T("pres"), "Pre-base Substitutions" },
    { T("pstf"), "Post-base Forms" },
    { T("psts"), "Post-base Substitutions" },
    { T("pwid"), "Proportional Widths" },
    { T("qwid"), "Quarter Widths" },
    { T("rand"), "Randomize" },
    { T("rclt"), "Required Contextual Alternates" },
    { T("rkrf"), "Rakar Forms" },
    { T("rlig"), "Required Ligatures" },
    { T("rphf"), "Reph Forms" },
    { T("rtbd"), "Right Bounds" },
    { T("rtla"), "Right-to-left Alternates" },
    { T("rtlm"), "Right-to-left Mirrored Forms" },
    { T("ruby"), "Ruby Notation Forms" },
    { T("salt"), "Stylistic Alternates" },
    { T("sinf"), "Scientific Inferiors" },
    { T("size"), "Optical Size" },
    { T("smcp"), "Small Capitals" },
    { T("smpl"), "Simplified Forms" },
    { T("ss01"), "Stylistic Set 1" },
    { T("ss02"), "Stylistic Set 2" },
    { T("ss03"), "Stylistic Set 3" },
    { T("ss04"), "Stylistic Set 4" },
    { T("ss05"), "Stylistic Set 5" },
    { T("ss06"), "Stylistic Set 6" },
    { T("ss07"), "Stylistic Set 7" },
    { T("ss08"), "Stylistic Set 8" },
    { T("ss09"), "Stylistic Set 9" },
    { T("ss10"), "Stylistic Set 10" },
    { T("ss11"), "Stylistic Set 11" },
    { T("ss12"), "Stylistic Set 12" },
    { T("ss13"), "Stylistic Set 13" },
    { T("ss14"), "Stylistic Set 14" },
    { T("ss15"), "Stylistic Set 15" },
    { T("ss16"), "Stylistic Set 16" },
    { T("ss17"), "Stylistic Set 17" },
    { T("ss18"), "Stylistic Set 18" },
    { T("ss19"), "Stylistic Set 19" },
    { T("ss20"), "Stylistic Set 20" },
    { T("ssty"), "Math Script Style Alternates" },
    { T("stch"), "Stretching Glyph Decomposition" },
    { T("subs"), "Subscript" },
    { T("sups"), "Superscript" },
    { T("swsh"), "Swash" },
    { T("titl"), "Titling" },
    { T("tjmo"), "Trailing Jamo Forms" },
    { T("tnam"), "Traditional Name Forms" },
    { T("tnum"), "Tabular Figures" },
    { T("trad"), "Traditional Forms" },
    { T("twid"), "Third Widths" },
    { T("unic"), "Unicase" },
    { T("valt"), "Alternate Vertical Metrics" },
    { T("vatu"), "Vattu Variants" },
    { T("vchw"), "Vertical Contextual Half-width Spacing" },
    { T("vert"), "Vertical Writing" },
    { T("vhal"), "Alternate Vertical Half Metrics" },
    { T("vjmo"), "Vowel Jamo Forms" },
    { T("vkna"), "Vertical Kana Alternates" },
    { T("vkrn"), "Vertical Kerning" },
    { T("vpal"), "Proportional Alternate Vertical Metrics" },
    { T("vrt2"), "Vertical Alternates and Rotation" },
    { T("zero"), "Slashed Zero" }
};

namespace Efont { namespace OpenType {

static const char *
find_description(uint32_t tag, const Mapping *maps, int n)
{
    // should use STL...
    int l = 0, r = n;
    while (l < r) {
        int m = l + (r - l) / 2;
        if (maps[m].tag < tag)
            l = m + 1;
        else if (maps[m].tag == tag)
            return maps[m].description;
        else
            r = m;
    }
    return 0;
}

const char *
Tag::script_description() const
{
    return find_description(value(), scripts, sizeof(scripts) / sizeof(Mapping));
}

const char *
Tag::language_description() const
{
    return find_description(value(), languages, sizeof(languages) / sizeof(Mapping));
}

const char *
Tag::feature_description() const
{
    return find_description(value(), features, sizeof(features) / sizeof(Mapping));
}

}}
