/*
 * NASPRO - The NASPRO Architecture for Sound PROcessing
 * Core library
 *
 * Copyright (C) 2007-2013 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

#define _POSIX_C_SOURCE	200809L

#include "internal.h"

#if defined(__APPLE__) && defined(__MACH__)
# include <mach/mach.h>
# include <mach/task.h>
# include <mach/semaphore.h>
#else
# include <semaphore.h>
#endif

_NACORE_DEF nacore_sem
nacore_sem_new(unsigned int value)
{
	nacore_sem ret;
#if defined(__APPLE__) && defined(__MACH__)
	kern_return_t err;

	ret = malloc(sizeof(semaphore_t));
#else
	ret = malloc(sizeof(sem_t));
#endif
	if (ret == NULL)
	  {
		errno = ENOMEM;
		return NULL;
	  }

#if defined(__APPLE__) && defined(__MACH__)
	err = semaphore_create(mach_task_self(), (semaphore_t *)ret,
			       SYNC_POLICY_FIFO, value);
	if (err != KERN_SUCCESS)
	  {
		free(ret);
		errno = (err == KERN_INVALID_ARGUMENT) ? EINVAL
						       : NACORE_EUNKNOWN;
		return NULL;
	  }
#else
	if (sem_init((sem_t *)ret, 0, value) != 0)
	  {
		free(ret);
		errno = EINVAL;
		return NULL;
	  }
#endif

	return ret;
}

_NACORE_DEF void
nacore_sem_free(nacore_sem sem)
{
#if defined(__APPLE__) && defined(__MACH__)
	semaphore_destroy(mach_task_self(), *((semaphore_t *)sem));
#else
	sem_destroy((sem_t *)sem);
#endif

	free(sem);
}

_NACORE_DEF int
nacore_sem_wait(nacore_sem sem)
{
#if defined(__APPLE__) && defined(__MACH__)
	return (semaphore_wait(*((semaphore_t *)sem)) != KERN_SUCCESS) ? EINTR
								       : 0;
#else
	return (sem_wait((sem_t *)sem) != 0) ? EINTR : 0;
#endif
}

_NACORE_DEF int
nacore_sem_trywait(nacore_sem sem)
{
#if defined(__APPLE__) && defined(__MACH__)
	mach_timespec_t t;
	kern_return _t err;

	t = { 0, 0 };
	err = semaphore_timedwait(*((semaphore_t *)sem)), t);
	return (err == KERN_SUCCESS)
	       ? 0 : ((err == KERN_OPERATION_TIMED_OUT) ? EAGAIN : EINTR);
#else
	return (sem_trywait((sem_t *)sem) != 0) ? errno : 0;
#endif
}

_NACORE_DEF int
nacore_sem_post(nacore_sem sem)
{
#if defined(__APPLE__) && defined(__MACH__)
	semaphore_signal((*((semaphore_t *)sem)));
#else
	sem_post((sem_t *)sem);
#endif

	return 0;
}
