/* Copyright (c) 2008-2025 the MRtrix3 contributors.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Covered Software is provided under this License on an "as is"
 * basis, without warranty of any kind, either expressed, implied, or
 * statutory, including, without limitation, warranties that the
 * Covered Software is free of defects, merchantable, fit for a
 * particular purpose or non-infringing.
 * See the Mozilla Public License v. 2.0 for more details.
 *
 * For more details, see http://www.mrtrix.org/.
 */

#include "file/dicom/element.h"

namespace MR {
  namespace File {
    namespace Dicom {

      std::unordered_map<uint32_t, const char*> Element::dict;


      // Note this implementation does not account for multiplicity
      // The invoking code is expected to have prior information as to how many
      // items are stored in any given tag.

      void Element::init_dict()
      {
        INFO ("initialising DICOM dictionary");

        /* the following was generated by running the following command on the dicom dictionary file:

           awk '{ print "{ 0x"$1$2"UL, "\"$3$4\" }," }' dict.txt
         */

        dict = {
          { 0x00020000UL, "ULFileMetaInformationGroupLength" },
          { 0x00020001UL, "OBFileMetaInformationVersion" },
          { 0x00020002UL, "UIMediaStorageSOPClassUID" },
          { 0x00020003UL, "UIMediaStorageSOPInstanceUID" },
          { 0x00020010UL, "UITransferSyntaxUID" },
          { 0x00020012UL, "UIImplementationClassUID" },
          { 0x00020013UL, "SHImplementationVersionName" },
          { 0x00020016UL, "AESourceApplicationEntityTitle" },
          { 0x00020100UL, "UIPrivateInformationCreatorUID" },
          { 0x00020102UL, "OBPrivateInformation" },
          { 0x00041130UL, "CSFileSetID" },
          { 0x00041141UL, "CSFileSetDescriptorFileID" },
          { 0x00041142UL, "CSSpecificCharacterSetOfFileSetDescriptorFile" },
          { 0x00041200UL, "ULOffsetOfTheFirstDirectoryRecordOfTheRootDirectoryEntity" },
          { 0x00041202UL, "ULOffsetOfTheLastDirectoryRecordOfTheRootDirectoryEntity" },
          { 0x00041212UL, "USFileSetConsistencyFlag" },
          { 0x00041220UL, "SQDirectoryRecordSequence" },
          { 0x00041400UL, "ULOffsetOfTheNextDirectoryRecord" },
          { 0x00041410UL, "USRecordInUseFlag" },
          { 0x00041420UL, "ULOffsetOfReferencedLowerLevelDirectoryEntity" },
          { 0x00041430UL, "CSDirectoryRecordType" },
          { 0x00041432UL, "UIPrivateRecordUID" },
          { 0x00041500UL, "CSReferencedFileID" },
          { 0x00041510UL, "UIReferencedSOPClassUIDInFile" },
          { 0x00041511UL, "UIReferencedSOPInstanceUIDInFile" },
          { 0x00041512UL, "UIReferencedTransferSyntaxUIDInFile" },
          { 0x0004151AUL, "UIReferencedRelatedGeneralSOPClassUIDInFile" },
          { 0x00080005UL, "CSSpecificCharacterSet" },
          { 0x00080006UL, "SQLanguageCodeSequence" },
          { 0x00080008UL, "CSImageType" },
          { 0x00080012UL, "DAInstanceCreationDate" },
          { 0x00080013UL, "TMInstanceCreationTime" },
          { 0x00080014UL, "UIInstanceCreatorUID" },
          { 0x00080016UL, "UISOPClassUID" },
          { 0x00080018UL, "UISOPInstanceUID" },
          { 0x0008001AUL, "UIRelatedGeneralSOPClassUID" },
          { 0x0008001BUL, "UIOriginalSpecializedSOPClassUID" },
          { 0x00080020UL, "DAStudyDate" },
          { 0x00080021UL, "DASeriesDate" },
          { 0x00080022UL, "DAAcquisitionDate" },
          { 0x00080023UL, "DAContentDate" },
          { 0x0008002AUL, "DTAcquisitionDateTime" },
          { 0x00080030UL, "TMStudyTime" },
          { 0x00080031UL, "TMSeriesTime" },
          { 0x00080032UL, "TMAcquisitionTime" },
          { 0x00080033UL, "TMContentTime" },
          { 0x00080050UL, "SHAccessionNumber" },
          { 0x00080051UL, "SQIssuerOfAccessionNumberSequence" },
          { 0x00080052UL, "CSQueryRetrieveLevel" },
          { 0x00080054UL, "AERetrieveAETitle" },
          { 0x00080056UL, "CSInstanceAvailability" },
          { 0x00080058UL, "UIFailedSOPInstanceUIDList" },
          { 0x00080060UL, "CSModality" },
          { 0x00080061UL, "CSModalitiesInStudy" },
          { 0x00080062UL, "UISOPClassesInStudy" },
          { 0x00080064UL, "CSConversionType" },
          { 0x00080068UL, "CSPresentationIntentType" },
          { 0x00080070UL, "LOManufacturer" },
          { 0x00080080UL, "LOInstitutionName" },
          { 0x00080081UL, "STInstitutionAddress" },
          { 0x00080082UL, "SQInstitutionCodeSequence" },
          { 0x00080090UL, "PNReferringPhysicianName" },
          { 0x00080092UL, "STReferringPhysicianAddress" },
          { 0x00080094UL, "SHReferringPhysicianTelephoneNumbers" },
          { 0x00080096UL, "SQReferringPhysicianIdentificationSequence" },
          { 0x00080100UL, "SHCodeValue" },
          { 0x00080102UL, "SHCodingSchemeDesignator" },
          { 0x00080103UL, "SHCodingSchemeVersion" },
          { 0x00080104UL, "LOCodeMeaning" },
          { 0x00080105UL, "CSMappingResource" },
          { 0x00080106UL, "DTContextGroupVersion" },
          { 0x00080107UL, "DTContextGroupLocalVersion" },
          { 0x0008010BUL, "CSContextGroupExtensionFlag" },
          { 0x0008010CUL, "UICodingSchemeUID" },
          { 0x0008010DUL, "UIContextGroupExtensionCreatorUID" },
          { 0x0008010FUL, "CSContextIdentifier" },
          { 0x00080110UL, "SQCodingSchemeIdentificationSequence" },
          { 0x00080112UL, "LOCodingSchemeRegistry" },
          { 0x00080114UL, "STCodingSchemeExternalID" },
          { 0x00080115UL, "STCodingSchemeName" },
          { 0x00080116UL, "STCodingSchemeResponsibleOrganization" },
          { 0x00080117UL, "UIContextUID" },
          { 0x00080201UL, "SHTimezoneOffsetFromUTC" },
          { 0x00081010UL, "SHStationName" },
          { 0x00081030UL, "LOStudyDescription" },
          { 0x00081032UL, "SQProcedureCodeSequence" },
          { 0x0008103EUL, "LOSeriesDescription" },
          { 0x0008103FUL, "SQSeriesDescriptionCodeSequence" },
          { 0x00081040UL, "LOInstitutionalDepartmentName" },
          { 0x00081048UL, "PNPhysiciansOfRecord" },
          { 0x00081049UL, "SQPhysiciansOfRecordIdentificationSequence" },
          { 0x00081050UL, "PNPerformingPhysicianName" },
          { 0x00081052UL, "SQPerformingPhysicianIdentificationSequence" },
          { 0x00081060UL, "PNNameOfPhysiciansReadingStudy" },
          { 0x00081062UL, "SQPhysiciansReadingStudyIdentificationSequence" },
          { 0x00081070UL, "PNOperatorsName" },
          { 0x00081072UL, "SQOperatorIdentificationSequence" },
          { 0x00081080UL, "LOAdmittingDiagnosesDescription" },
          { 0x00081084UL, "SQAdmittingDiagnosesCodeSequence" },
          { 0x00081090UL, "LOManufacturerModelName" },
          { 0x00081110UL, "SQReferencedStudySequence" },
          { 0x00081111UL, "SQReferencedPerformedProcedureStepSequence" },
          { 0x00081115UL, "SQReferencedSeriesSequence" },
          { 0x00081120UL, "SQReferencedPatientSequence" },
          { 0x00081125UL, "SQReferencedVisitSequence" },
          { 0x00081134UL, "SQReferencedStereometricInstanceSequence" },
          { 0x0008113AUL, "SQReferencedWaveformSequence" },
          { 0x00081140UL, "SQReferencedImageSequence" },
          { 0x0008114AUL, "SQReferencedInstanceSequence" },
          { 0x0008114BUL, "SQReferencedRealWorldValueMappingInstanceSequence" },
          { 0x00081150UL, "UIReferencedSOPClassUID" },
          { 0x00081155UL, "UIReferencedSOPInstanceUID" },
          { 0x0008115AUL, "UISOPClassesSupported" },
          { 0x00081160UL, "ISReferencedFrameNumber" },
          { 0x00081161UL, "ULSimpleFrameList" },
          { 0x00081162UL, "ULCalculatedFrameList" },
          { 0x00081163UL, "FDTimeRange" },
          { 0x00081164UL, "SQFrameExtractionSequence" },
          { 0x00081167UL, "UIMultiFrameSourceSOPInstanceUID" },
          { 0x00081195UL, "UITransactionUID" },
          { 0x00081197UL, "USFailureReason" },
          { 0x00081198UL, "SQFailedSOPSequence" },
          { 0x00081199UL, "SQReferencedSOPSequence" },
          { 0x00081200UL, "SQStudiesContainingOtherReferencedInstancesSequence" },
          { 0x00081250UL, "SQRelatedSeriesSequence" },
          { 0x00082111UL, "STDerivationDescription" },
          { 0x00082112UL, "SQSourceImageSequence" },
          { 0x00082120UL, "SHStageName" },
          { 0x00082122UL, "ISStageNumber" },
          { 0x00082124UL, "ISNumberOfStages" },
          { 0x00082127UL, "SHViewName" },
          { 0x00082128UL, "ISViewNumber" },
          { 0x00082129UL, "ISNumberOfEventTimers" },
          { 0x0008212AUL, "ISNumberOfViewsInStage" },
          { 0x00082130UL, "DSEventElapsedTimes" },
          { 0x00082132UL, "LOEventTimerNames" },
          { 0x00082133UL, "SQEventTimerSequence" },
          { 0x00082134UL, "FDEventTimeOffset" },
          { 0x00082135UL, "SQEventCodeSequence" },
          { 0x00082142UL, "ISStartTrim" },
          { 0x00082143UL, "ISStopTrim" },
          { 0x00082144UL, "ISRecommendedDisplayFrameRate" },
          { 0x00082218UL, "SQAnatomicRegionSequence" },
          { 0x00082220UL, "SQAnatomicRegionModifierSequence" },
          { 0x00082228UL, "SQPrimaryAnatomicStructureSequence" },
          { 0x00082229UL, "SQAnatomicStructureSpaceOrRegionSequence" },
          { 0x00082230UL, "SQPrimaryAnatomicStructureModifierSequence" },
          { 0x00083001UL, "SQAlternateRepresentationSequence" },
          { 0x00083010UL, "UIIrradiationEventUID" },
          { 0x00089007UL, "CSFrameType" },
          { 0x00089092UL, "SQReferencedImageEvidenceSequence" },
          { 0x00089121UL, "SQReferencedRawDataSequence" },
          { 0x00089123UL, "UICreatorVersionUID" },
          { 0x00089124UL, "SQDerivationImageSequence" },
          { 0x00089154UL, "SQSourceImageEvidenceSequence" },
          { 0x00089205UL, "CSPixelPresentation" },
          { 0x00089206UL, "CSVolumetricProperties" },
          { 0x00089207UL, "CSVolumeBasedCalculationTechnique" },
          { 0x00089208UL, "CSComplexImageComponent" },
          { 0x00089209UL, "CSAcquisitionContrast" },
          { 0x00089215UL, "SQDerivationCodeSequence" },
          { 0x00089237UL, "SQReferencedPresentationStateSequence" },
          { 0x00089410UL, "SQReferencedOtherPlaneSequence" },
          { 0x00089458UL, "SQFrameDisplaySequence" },
          { 0x00089459UL, "FLRecommendedDisplayFrameRateInFloat" },
          { 0x00089460UL, "CSSkipFrameRangeFlag" },
          { 0x00100010UL, "PNPatientName" },
          { 0x00100020UL, "LOPatientID" },
          { 0x00100021UL, "LOIssuerOfPatientID" },
          { 0x00100022UL, "CSTypeOfPatientID" },
          { 0x00100024UL, "SQIssuerOfPatientIDQualifiersSequence" },
          { 0x00100030UL, "DAPatientBirthDate" },
          { 0x00100032UL, "TMPatientBirthTime" },
          { 0x00100040UL, "CSPatientSex" },
          { 0x00100050UL, "SQPatientInsurancePlanCodeSequence" },
          { 0x00100101UL, "SQPatientPrimaryLanguageCodeSequence" },
          { 0x00100102UL, "SQPatientPrimaryLanguageModifierCodeSequence" },
          { 0x00101000UL, "LOOtherPatientIDs" },
          { 0x00101001UL, "PNOtherPatientNames" },
          { 0x00101002UL, "SQOtherPatientIDsSequence" },
          { 0x00101005UL, "PNPatientBirthName" },
          { 0x00101010UL, "ASPatientAge" },
          { 0x00101020UL, "DSPatientSize" },
          { 0x00101021UL, "SQPatientSizeCodeSequence" },
          { 0x00101030UL, "DSPatientWeight" },
          { 0x00101040UL, "LOPatientAddress" },
          { 0x00101060UL, "PNPatientMotherBirthName" },
          { 0x00101080UL, "LOMilitaryRank" },
          { 0x00101081UL, "LOBranchOfService" },
          { 0x00101090UL, "LOMedicalRecordLocator" },
          { 0x00102000UL, "LOMedicalAlerts" },
          { 0x00102110UL, "LOAllergies" },
          { 0x00102150UL, "LOCountryOfResidence" },
          { 0x00102152UL, "LORegionOfResidence" },
          { 0x00102154UL, "SHPatientTelephoneNumbers" },
          { 0x00102160UL, "SHEthnicGroup" },
          { 0x00102180UL, "SHOccupation" },
          { 0x001021A0UL, "CSSmokingStatus" },
          { 0x001021B0UL, "LTAdditionalPatientHistory" },
          { 0x001021C0UL, "USPregnancyStatus" },
          { 0x001021D0UL, "DALastMenstrualDate" },
          { 0x001021F0UL, "LOPatientReligiousPreference" },
          { 0x00102201UL, "LOPatientSpeciesDescription" },
          { 0x00102202UL, "SQPatientSpeciesCodeSequence" },
          { 0x00102203UL, "CSPatientSexNeutered" },
          { 0x00102210UL, "CSAnatomicalOrientationType" },
          { 0x00102292UL, "LOPatientBreedDescription" },
          { 0x00102293UL, "SQPatientBreedCodeSequence" },
          { 0x00102294UL, "SQBreedRegistrationSequence" },
          { 0x00102295UL, "LOBreedRegistrationNumber" },
          { 0x00102296UL, "SQBreedRegistryCodeSequence" },
          { 0x00102297UL, "PNResponsiblePerson" },
          { 0x00102298UL, "CSResponsiblePersonRole" },
          { 0x00102299UL, "LOResponsibleOrganization" },
          { 0x00104000UL, "LTPatientComments" },
          { 0x00109431UL, "FLExaminedBodyThickness" },
          { 0x00120010UL, "LOClinicalTrialSponsorName" },
          { 0x00120020UL, "LOClinicalTrialProtocolID" },
          { 0x00120021UL, "LOClinicalTrialProtocolName" },
          { 0x00120030UL, "LOClinicalTrialSiteID" },
          { 0x00120031UL, "LOClinicalTrialSiteName" },
          { 0x00120040UL, "LOClinicalTrialSubjectID" },
          { 0x00120042UL, "LOClinicalTrialSubjectReadingID" },
          { 0x00120050UL, "LOClinicalTrialTimePointID" },
          { 0x00120051UL, "STClinicalTrialTimePointDescription" },
          { 0x00120060UL, "LOClinicalTrialCoordinatingCenterName" },
          { 0x00120062UL, "CSPatientIdentityRemoved" },
          { 0x00120063UL, "LODeidentificationMethod" },
          { 0x00120064UL, "SQDeidentificationMethodCodeSequence" },
          { 0x00120071UL, "LOClinicalTrialSeriesID" },
          { 0x00120072UL, "LOClinicalTrialSeriesDescription" },
          { 0x00120081UL, "LOClinicalTrialProtocolEthicsCommitteeName" },
          { 0x00120082UL, "LOClinicalTrialProtocolEthicsCommitteeApprovalNumber" },
          { 0x00120083UL, "SQConsentForClinicalTrialUseSequence" },
          { 0x00120084UL, "CSDistributionType" },
          { 0x00120085UL, "CSConsentForDistributionFlag" },
          { 0x00140023UL, "STCADFileFormat" },
          { 0x00140024UL, "STComponentReferenceSystem" },
          { 0x00140025UL, "STComponentManufacturingProcedure" },
          { 0x00140028UL, "STComponentManufacturer" },
          { 0x00140030UL, "DSMaterialThickness" },
          { 0x00140032UL, "DSMaterialPipeDiameter" },
          { 0x00140034UL, "DSMaterialIsolationDiameter" },
          { 0x00140042UL, "STMaterialGrade" },
          { 0x00140044UL, "STMaterialPropertiesFileID" },
          { 0x00140045UL, "STMaterialPropertiesFileFormat" },
          { 0x00140046UL, "LTMaterialNotes" },
          { 0x00140050UL, "CSComponentShape" },
          { 0x00140052UL, "CSCurvatureType" },
          { 0x00140054UL, "DSOuterDiameter" },
          { 0x00140056UL, "DSInnerDiameter" },
          { 0x00141010UL, "STActualEnvironmentalConditions" },
          { 0x00141020UL, "DAExpiryDate" },
          { 0x00141040UL, "STEnvironmentalConditions" },
          { 0x00142002UL, "SQEvaluatorSequence" },
          { 0x00142004UL, "ISEvaluatorNumber" },
          { 0x00142006UL, "PNEvaluatorName" },
          { 0x00142008UL, "ISEvaluationAttempt" },
          { 0x00142012UL, "SQIndicationSequence" },
          { 0x00142014UL, "ISIndicationNumber" },
          { 0x00142016UL, "SHIndicationLabel" },
          { 0x00142018UL, "STIndicationDescription" },
          { 0x0014201AUL, "CSIndicationType" },
          { 0x0014201CUL, "CSIndicationDisposition" },
          { 0x0014201EUL, "SQIndicationROISequence" },
          { 0x00142030UL, "SQIndicationPhysicalPropertySequence" },
          { 0x00142032UL, "SHPropertyLabel" },
          { 0x00142202UL, "ISCoordinateSystemNumberOfAxes" },
          { 0x00142204UL, "SQCoordinateSystemAxesSequence" },
          { 0x00142206UL, "STCoordinateSystemAxisDescription" },
          { 0x00142208UL, "CSCoordinateSystemDataSetMapping" },
          { 0x0014220AUL, "ISCoordinateSystemAxisNumber" },
          { 0x0014220CUL, "CSCoordinateSystemAxisType" },
          { 0x0014220EUL, "CSCoordinateSystemAxisUnits" },
          { 0x00142210UL, "OBCoordinateSystemAxisValues" },
          { 0x00142220UL, "SQCoordinateSystemTransformSequence" },
          { 0x00142222UL, "STTransformDescription" },
          { 0x00142224UL, "ISTransformNumberOfAxes" },
          { 0x00142226UL, "ISTransformOrderOfAxes" },
          { 0x00142228UL, "CSTransformedAxisUnits" },
          { 0x0014222AUL, "DSCoordinateSystemTransformRotationAndScaleMatrix" },
          { 0x0014222CUL, "DSCoordinateSystemTransformTranslationMatrix" },
          { 0x00143011UL, "DSInternalDetectorFrameTime" },
          { 0x00143012UL, "DSNumberOfFramesIntegrated" },
          { 0x00143020UL, "SQDetectorTemperatureSequence" },
          { 0x00143022UL, "DSSensorName" },
          { 0x00143024UL, "DSHorizontalOffsetOfSensor" },
          { 0x00143026UL, "DSVerticalOffsetOfSensor" },
          { 0x00143028UL, "DSSensorTemperature" },
          { 0x00143040UL, "SQDarkCurrentSequence" },
          { 0x00143050UL, "OBDarkCurrentCounts" },
          { 0x00143060UL, "SQGainCorrectionReferenceSequence" },
          { 0x00143070UL, "OBAirCounts" },
          { 0x00143071UL, "DSKVUsedInGainCalibration" },
          { 0x00143072UL, "DSMAUsedInGainCalibration" },
          { 0x00143073UL, "DSNumberOfFramesUsedForIntegration" },
          { 0x00143074UL, "LOFilterMaterialUsedInGainCalibration" },
          { 0x00143075UL, "DSFilterThicknessUsedInGainCalibration" },
          { 0x00143076UL, "DADateOfGainCalibration" },
          { 0x00143077UL, "TMTimeOfGainCalibration" },
          { 0x00143080UL, "OBBadPixelImage" },
          { 0x00143099UL, "LTCalibrationNotes" },
          { 0x00144002UL, "SQPulserEquipmentSequence" },
          { 0x00144004UL, "CSPulserType" },
          { 0x00144006UL, "LTPulserNotes" },
          { 0x00144008UL, "SQReceiverEquipmentSequence" },
          { 0x0014400AUL, "CSAmplifierType" },
          { 0x0014400CUL, "LTReceiverNotes" },
          { 0x0014400EUL, "SQPreAmplifierEquipmentSequence" },
          { 0x0014400FUL, "LTPreAmplifierNotes" },
          { 0x00144010UL, "SQTransmitTransducerSequence" },
          { 0x00144011UL, "SQReceiveTransducerSequence" },
          { 0x00144012UL, "USNumberOfElements" },
          { 0x00144013UL, "CSElementShape" },
          { 0x00144014UL, "DSElementDimensionA" },
          { 0x00144015UL, "DSElementDimensionB" },
          { 0x00144016UL, "DSElementPitch" },
          { 0x00144017UL, "DSMeasuredBeamDimensionA" },
          { 0x00144018UL, "DSMeasuredBeamDimensionB" },
          { 0x00144019UL, "DSLocationOfMeasuredBeamDiameter" },
          { 0x0014401AUL, "DSNominalFrequency" },
          { 0x0014401BUL, "DSMeasuredCenterFrequency" },
          { 0x0014401CUL, "DSMeasuredBandwidth" },
          { 0x00144020UL, "SQPulserSettingsSequence" },
          { 0x00144022UL, "DSPulseWidth" },
          { 0x00144024UL, "DSExcitationFrequency" },
          { 0x00144026UL, "CSModulationType" },
          { 0x00144028UL, "DSDamping" },
          { 0x00144030UL, "SQReceiverSettingsSequence" },
          { 0x00144031UL, "DSAcquiredSoundpathLength" },
          { 0x00144032UL, "CSAcquisitionCompressionType" },
          { 0x00144033UL, "ISAcquisitionSampleSize" },
          { 0x00144034UL, "DSRectifierSmoothing" },
          { 0x00144035UL, "SQDACSequence" },
          { 0x00144036UL, "CSDACType" },
          { 0x00144038UL, "DSDACGainPoints" },
          { 0x0014403AUL, "DSDACTimePoints" },
          { 0x0014403CUL, "DSDACAmplitude" },
          { 0x00144040UL, "SQPreAmplifierSettingsSequence" },
          { 0x00144050UL, "SQTransmitTransducerSettingsSequence" },
          { 0x00144051UL, "SQReceiveTransducerSettingsSequence" },
          { 0x00144052UL, "DSIncidentAngle" },
          { 0x00144054UL, "STCouplingTechnique" },
          { 0x00144056UL, "STCouplingMedium" },
          { 0x00144057UL, "DSCouplingVelocity" },
          { 0x00144058UL, "DSCrystalCenterLocationX" },
          { 0x00144059UL, "DSCrystalCenterLocationZ" },
          { 0x0014405AUL, "DSSoundPathLength" },
          { 0x0014405CUL, "STDelayLawIdentifier" },
          { 0x00144060UL, "SQGateSettingsSequence" },
          { 0x00144062UL, "DSGateThreshold" },
          { 0x00144064UL, "DSVelocityOfSound" },
          { 0x00144070UL, "SQCalibrationSettingsSequence" },
          { 0x00144072UL, "STCalibrationProcedure" },
          { 0x00144074UL, "SHProcedureVersion" },
          { 0x00144076UL, "DAProcedureCreationDate" },
          { 0x00144078UL, "DAProcedureExpirationDate" },
          { 0x0014407AUL, "DAProcedureLastModifiedDate" },
          { 0x0014407CUL, "TMCalibrationTime" },
          { 0x0014407EUL, "DACalibrationDate" },
          { 0x00145002UL, "ISLINACEnergy" },
          { 0x00145004UL, "ISLINACOutput" },
          { 0x00180010UL, "LOContrastBolusAgent" },
          { 0x00180012UL, "SQContrastBolusAgentSequence" },
          { 0x00180014UL, "SQContrastBolusAdministrationRouteSequence" },
          { 0x00180015UL, "CSBodyPartExamined" },
          { 0x00180020UL, "CSScanningSequence" },
          { 0x00180021UL, "CSSequenceVariant" },
          { 0x00180022UL, "CSScanOptions" },
          { 0x00180023UL, "CSMRAcquisitionType" },
          { 0x00180024UL, "SHSequenceName" },
          { 0x00180025UL, "CSAngioFlag" },
          { 0x00180026UL, "SQInterventionDrugInformationSequence" },
          { 0x00180027UL, "TMInterventionDrugStopTime" },
          { 0x00180028UL, "DSInterventionDrugDose" },
          { 0x00180029UL, "SQInterventionDrugCodeSequence" },
          { 0x0018002AUL, "SQAdditionalDrugSequence" },
          { 0x00180031UL, "LORadiopharmaceutical" },
          { 0x00180034UL, "LOInterventionDrugName" },
          { 0x00180035UL, "TMInterventionDrugStartTime" },
          { 0x00180036UL, "SQInterventionSequence" },
          { 0x00180038UL, "CSInterventionStatus" },
          { 0x0018003AUL, "STInterventionDescription" },
          { 0x00180040UL, "ISCineRate" },
          { 0x00180042UL, "CSInitialCineRunState" },
          { 0x00180050UL, "DSSliceThickness" },
          { 0x00180060UL, "DSKVP" },
          { 0x00180070UL, "ISCountsAccumulated" },
          { 0x00180071UL, "CSAcquisitionTerminationCondition" },
          { 0x00180072UL, "DSEffectiveDuration" },
          { 0x00180073UL, "CSAcquisitionStartCondition" },
          { 0x00180074UL, "ISAcquisitionStartConditionData" },
          { 0x00180075UL, "ISAcquisitionTerminationConditionData" },
          { 0x00180080UL, "DSRepetitionTime" },
          { 0x00180081UL, "DSEchoTime" },
          { 0x00180082UL, "DSInversionTime" },
          { 0x00180083UL, "DSNumberOfAverages" },
          { 0x00180084UL, "DSImagingFrequency" },
          { 0x00180085UL, "SHImagedNucleus" },
          { 0x00180086UL, "ISEchoNumbers" },
          { 0x00180087UL, "DSMagneticFieldStrength" },
          { 0x00180088UL, "DSSpacingBetweenSlices" },
          { 0x00180089UL, "ISNumberOfPhaseEncodingSteps" },
          { 0x00180090UL, "DSDataCollectionDiameter" },
          { 0x00180091UL, "ISEchoTrainLength" },
          { 0x00180093UL, "DSPercentSampling" },
          { 0x00180094UL, "DSPercentPhaseFieldOfView" },
          { 0x00180095UL, "DSPixelBandwidth" },
          { 0x00181000UL, "LODeviceSerialNumber" },
          { 0x00181002UL, "UIDeviceUID" },
          { 0x00181003UL, "LODeviceID" },
          { 0x00181004UL, "LOPlateID" },
          { 0x00181005UL, "LOGeneratorID" },
          { 0x00181006UL, "LOGridID" },
          { 0x00181007UL, "LOCassetteID" },
          { 0x00181008UL, "LOGantryID" },
          { 0x00181010UL, "LOSecondaryCaptureDeviceID" },
          { 0x00181012UL, "DADateOfSecondaryCapture" },
          { 0x00181014UL, "TMTimeOfSecondaryCapture" },
          { 0x00181016UL, "LOSecondaryCaptureDeviceManufacturer" },
          { 0x00181018UL, "LOSecondaryCaptureDeviceManufacturerModelName" },
          { 0x00181019UL, "LOSecondaryCaptureDeviceSoftwareVersions" },
          { 0x00181020UL, "LOSoftwareVersions" },
          { 0x00181022UL, "SHVideoImageFormatAcquired" },
          { 0x00181023UL, "LODigitalImageFormatAcquired" },
          { 0x00181030UL, "LOProtocolName" },
          { 0x00181040UL, "LOContrastBolusRoute" },
          { 0x00181041UL, "DSContrastBolusVolume" },
          { 0x00181042UL, "TMContrastBolusStartTime" },
          { 0x00181043UL, "TMContrastBolusStopTime" },
          { 0x00181044UL, "DSContrastBolusTotalDose" },
          { 0x00181045UL, "ISSyringeCounts" },
          { 0x00181046UL, "DSContrastFlowRate" },
          { 0x00181047UL, "DSContrastFlowDuration" },
          { 0x00181048UL, "CSContrastBolusIngredient" },
          { 0x00181049UL, "DSContrastBolusIngredientConcentration" },
          { 0x00181050UL, "DSSpatialResolution" },
          { 0x00181060UL, "DSTriggerTime" },
          { 0x00181061UL, "LOTriggerSourceOrType" },
          { 0x00181062UL, "ISNominalInterval" },
          { 0x00181063UL, "DSFrameTime" },
          { 0x00181064UL, "LOCardiacFramingType" },
          { 0x00181065UL, "DSFrameTimeVector" },
          { 0x00181066UL, "DSFrameDelay" },
          { 0x00181067UL, "DSImageTriggerDelay" },
          { 0x00181068UL, "DSMultiplexGroupTimeOffset" },
          { 0x00181069UL, "DSTriggerTimeOffset" },
          { 0x0018106AUL, "CSSynchronizationTrigger" },
          { 0x0018106CUL, "USSynchronizationChannel" },
          { 0x0018106EUL, "ULTriggerSamplePosition" },
          { 0x00181070UL, "LORadiopharmaceuticalRoute" },
          { 0x00181071UL, "DSRadiopharmaceuticalVolume" },
          { 0x00181072UL, "TMRadiopharmaceuticalStartTime" },
          { 0x00181073UL, "TMRadiopharmaceuticalStopTime" },
          { 0x00181074UL, "DSRadionuclideTotalDose" },
          { 0x00181075UL, "DSRadionuclideHalfLife" },
          { 0x00181076UL, "DSRadionuclidePositronFraction" },
          { 0x00181077UL, "DSRadiopharmaceuticalSpecificActivity" },
          { 0x00181078UL, "DTRadiopharmaceuticalStartDateTime" },
          { 0x00181079UL, "DTRadiopharmaceuticalStopDateTime" },
          { 0x00181080UL, "CSBeatRejectionFlag" },
          { 0x00181081UL, "ISLowRRValue" },
          { 0x00181082UL, "ISHighRRValue" },
          { 0x00181083UL, "ISIntervalsAcquired" },
          { 0x00181084UL, "ISIntervalsRejected" },
          { 0x00181085UL, "LOPVCRejection" },
          { 0x00181086UL, "ISSkipBeats" },
          { 0x00181088UL, "ISHeartRate" },
          { 0x00181090UL, "ISCardiacNumberOfImages" },
          { 0x00181094UL, "ISTriggerWindow" },
          { 0x00181100UL, "DSReconstructionDiameter" },
          { 0x00181110UL, "DSDistanceSourceToDetector" },
          { 0x00181111UL, "DSDistanceSourceToPatient" },
          { 0x00181114UL, "DSEstimatedRadiographicMagnificationFactor" },
          { 0x00181120UL, "DSGantryDetectorTilt" },
          { 0x00181121UL, "DSGantryDetectorSlew" },
          { 0x00181130UL, "DSTableHeight" },
          { 0x00181131UL, "DSTableTraverse" },
          { 0x00181134UL, "CSTableMotion" },
          { 0x00181135UL, "DSTableVerticalIncrement" },
          { 0x00181136UL, "DSTableLateralIncrement" },
          { 0x00181137UL, "DSTableLongitudinalIncrement" },
          { 0x00181138UL, "DSTableAngle" },
          { 0x0018113AUL, "CSTableType" },
          { 0x00181140UL, "CSRotationDirection" },
          { 0x00181142UL, "DSRadialPosition" },
          { 0x00181143UL, "DSScanArc" },
          { 0x00181144UL, "DSAngularStep" },
          { 0x00181145UL, "DSCenterOfRotationOffset" },
          { 0x00181147UL, "CSFieldOfViewShape" },
          { 0x00181149UL, "ISFieldOfViewDimensions" },
          { 0x00181150UL, "ISExposureTime" },
          { 0x00181151UL, "ISXRayTubeCurrent" },
          { 0x00181152UL, "ISExposure" },
          { 0x00181153UL, "ISExposureInuAs" },
          { 0x00181154UL, "DSAveragePulseWidth" },
          { 0x00181155UL, "CSRadiationSetting" },
          { 0x00181156UL, "CSRectificationType" },
          { 0x0018115AUL, "CSRadiationMode" },
          { 0x0018115EUL, "DSImageAndFluoroscopyAreaDoseProduct" },
          { 0x00181160UL, "SHFilterType" },
          { 0x00181161UL, "LOTypeOfFilters" },
          { 0x00181162UL, "DSIntensifierSize" },
          { 0x00181164UL, "DSImagerPixelSpacing" },
          { 0x00181166UL, "CSGrid" },
          { 0x00181170UL, "ISGeneratorPower" },
          { 0x00181180UL, "SHCollimatorGridName" },
          { 0x00181181UL, "CSCollimatorType" },
          { 0x00181182UL, "ISFocalDistance" },
          { 0x00181183UL, "DSXFocusCenter" },
          { 0x00181184UL, "DSYFocusCenter" },
          { 0x00181190UL, "DSFocalSpots" },
          { 0x00181191UL, "CSAnodeTargetMaterial" },
          { 0x001811A0UL, "DSBodyPartThickness" },
          { 0x001811A2UL, "DSCompressionForce" },
          { 0x00181200UL, "DADateOfLastCalibration" },
          { 0x00181201UL, "TMTimeOfLastCalibration" },
          { 0x00181210UL, "SHConvolutionKernel" },
          { 0x00181242UL, "ISActualFrameDuration" },
          { 0x00181243UL, "ISCountRate" },
          { 0x00181244UL, "USPreferredPlaybackSequencing" },
          { 0x00181250UL, "SHReceiveCoilName" },
          { 0x00181251UL, "SHTransmitCoilName" },
          { 0x00181260UL, "SHPlateType" },
          { 0x00181261UL, "LOPhosphorType" },
          { 0x00181300UL, "DSScanVelocity" },
          { 0x00181301UL, "CSWholeBodyTechnique" },
          { 0x00181302UL, "ISScanLength" },
          { 0x00181310UL, "USAcquisitionMatrix" },
          { 0x00181312UL, "CSInPlanePhaseEncodingDirection" },
          { 0x00181314UL, "DSFlipAngle" },
          { 0x00181315UL, "CSVariableFlipAngleFlag" },
          { 0x00181316UL, "DSSAR" },
          { 0x00181318UL, "DSdBdt" },
          { 0x00181400UL, "LOAcquisitionDeviceProcessingDescription" },
          { 0x00181401UL, "LOAcquisitionDeviceProcessingCode" },
          { 0x00181402UL, "CSCassetteOrientation" },
          { 0x00181403UL, "CSCassetteSize" },
          { 0x00181404UL, "USExposuresOnPlate" },
          { 0x00181405UL, "ISRelativeXRayExposure" },
          { 0x00181411UL, "DSExposureIndex" },
          { 0x00181412UL, "DSTargetExposureIndex" },
          { 0x00181413UL, "DSDeviationIndex" },
          { 0x00181450UL, "DSColumnAngulation" },
          { 0x00181460UL, "DSTomoLayerHeight" },
          { 0x00181470UL, "DSTomoAngle" },
          { 0x00181480UL, "DSTomoTime" },
          { 0x00181490UL, "CSTomoType" },
          { 0x00181491UL, "CSTomoClass" },
          { 0x00181495UL, "ISNumberOfTomosynthesisSourceImages" },
          { 0x00181500UL, "CSPositionerMotion" },
          { 0x00181508UL, "CSPositionerType" },
          { 0x00181510UL, "DSPositionerPrimaryAngle" },
          { 0x00181511UL, "DSPositionerSecondaryAngle" },
          { 0x00181520UL, "DSPositionerPrimaryAngleIncrement" },
          { 0x00181521UL, "DSPositionerSecondaryAngleIncrement" },
          { 0x00181530UL, "DSDetectorPrimaryAngle" },
          { 0x00181531UL, "DSDetectorSecondaryAngle" },
          { 0x00181600UL, "CSShutterShape" },
          { 0x00181602UL, "ISShutterLeftVerticalEdge" },
          { 0x00181604UL, "ISShutterRightVerticalEdge" },
          { 0x00181606UL, "ISShutterUpperHorizontalEdge" },
          { 0x00181608UL, "ISShutterLowerHorizontalEdge" },
          { 0x00181610UL, "ISCenterOfCircularShutter" },
          { 0x00181612UL, "ISRadiusOfCircularShutter" },
          { 0x00181620UL, "ISVerticesOfThePolygonalShutter" },
          { 0x00181622UL, "USShutterPresentationValue" },
          { 0x00181623UL, "USShutterOverlayGroup" },
          { 0x00181624UL, "USShutterPresentationColorCIELabValue" },
          { 0x00181700UL, "CSCollimatorShape" },
          { 0x00181702UL, "ISCollimatorLeftVerticalEdge" },
          { 0x00181704UL, "ISCollimatorRightVerticalEdge" },
          { 0x00181706UL, "ISCollimatorUpperHorizontalEdge" },
          { 0x00181708UL, "ISCollimatorLowerHorizontalEdge" },
          { 0x00181710UL, "ISCenterOfCircularCollimator" },
          { 0x00181712UL, "ISRadiusOfCircularCollimator" },
          { 0x00181720UL, "ISVerticesOfThePolygonalCollimator" },
          { 0x00181800UL, "CSAcquisitionTimeSynchronized" },
          { 0x00181801UL, "SHTimeSource" },
          { 0x00181802UL, "CSTimeDistributionProtocol" },
          { 0x00181803UL, "LONTPSourceAddress" },
          { 0x00182001UL, "ISPageNumberVector" },
          { 0x00182002UL, "SHFrameLabelVector" },
          { 0x00182003UL, "DSFramePrimaryAngleVector" },
          { 0x00182004UL, "DSFrameSecondaryAngleVector" },
          { 0x00182005UL, "DSSliceLocationVector" },
          { 0x00182006UL, "SHDisplayWindowLabelVector" },
          { 0x00182010UL, "DSNominalScannedPixelSpacing" },
          { 0x00182020UL, "CSDigitizingDeviceTransportDirection" },
          { 0x00182030UL, "DSRotationOfScannedFilm" },
          { 0x00183100UL, "CSIVUSAcquisition" },
          { 0x00183101UL, "DSIVUSPullbackRate" },
          { 0x00183102UL, "DSIVUSGatedRate" },
          { 0x00183103UL, "ISIVUSPullbackStartFrameNumber" },
          { 0x00183104UL, "ISIVUSPullbackStopFrameNumber" },
          { 0x00183105UL, "ISLesionNumber" },
          { 0x00185000UL, "SHOutputPower" },
          { 0x00185010UL, "LOTransducerData" },
          { 0x00185012UL, "DSFocusDepth" },
          { 0x00185020UL, "LOProcessingFunction" },
          { 0x00185022UL, "DSMechanicalIndex" },
          { 0x00185024UL, "DSBoneThermalIndex" },
          { 0x00185026UL, "DSCranialThermalIndex" },
          { 0x00185027UL, "DSSoftTissueThermalIndex" },
          { 0x00185028UL, "DSSoftTissueFocusThermalIndex" },
          { 0x00185029UL, "DSSoftTissueSurfaceThermalIndex" },
          { 0x00185050UL, "ISDepthOfScanField" },
          { 0x00185100UL, "CSPatientPosition" },
          { 0x00185101UL, "CSViewPosition" },
          { 0x00185104UL, "SQProjectionEponymousNameCodeSequence" },
          { 0x00186000UL, "DSSensitivity" },
          { 0x00186011UL, "SQSequenceOfUltrasoundRegions" },
          { 0x00186012UL, "USRegionSpatialFormat" },
          { 0x00186014UL, "USRegionDataType" },
          { 0x00186016UL, "ULRegionFlags" },
          { 0x00186018UL, "ULRegionLocationMinX0" },
          { 0x0018601AUL, "ULRegionLocationMinY0" },
          { 0x0018601CUL, "ULRegionLocationMaxX1" },
          { 0x0018601EUL, "ULRegionLocationMaxY1" },
          { 0x00186020UL, "SLReferencePixelX0" },
          { 0x00186022UL, "SLReferencePixelY0" },
          { 0x00186024UL, "USPhysicalUnitsXDirection" },
          { 0x00186026UL, "USPhysicalUnitsYDirection" },
          { 0x00186028UL, "FDReferencePixelPhysicalValuX" },
          { 0x0018602AUL, "FDReferencePixelPhysicalValuY" },
          { 0x0018602CUL, "FDPhysicalDeltaX" },
          { 0x0018602EUL, "FDPhysicalDeltaY" },
          { 0x00186030UL, "ULTransducerFrequency" },
          { 0x00186031UL, "CSTransducerType" },
          { 0x00186032UL, "ULPulseRepetitionFrequency" },
          { 0x00186034UL, "FDDopplerCorrectionAngle" },
          { 0x00186036UL, "FDteeringAngle" },
          { 0x00186039UL, "SLDopplerSampleVolumeXPosiion" },
          { 0x0018603BUL, "SLDopplerSampleVolumeYPosiion" },
          { 0x0018603DUL, "SLTMLinePositionX0" },
          { 0x0018603FUL, "SLTMLinePositionY0" },
          { 0x00186041UL, "SLTMLinePositionX1" },
          { 0x00186043UL, "SLTMLinePositionY1" },
          { 0x00186044UL, "USPixelComponentOrganization" },
          { 0x00186046UL, "ULPixelComponentMask" },
          { 0x00186048UL, "ULPixelComponentRangeStart" },
          { 0x0018604AUL, "ULPixelComponentRangeStop" },
          { 0x0018604CUL, "USPixelComponentPhysicalUnits" },
          { 0x0018604EUL, "USPixelComponentDataType" },
          { 0x00186050UL, "ULNumberOfTableBreakPoints" },
          { 0x00186052UL, "ULTableOfXBreakPoints" },
          { 0x00186054UL, "FDTableOfYBreakPoints" },
          { 0x00186056UL, "ULNumberOfTableEntries" },
          { 0x00186058UL, "ULTableOfPixelValues" },
          { 0x0018605AUL, "FLTableOfParameterValues" },
          { 0x00186060UL, "FLRWaveTimeVector" },
          { 0x00187000UL, "CSDetectorConditionsNominalFlag" },
          { 0x00187001UL, "DSDetectorTemperature" },
          { 0x00187004UL, "CSDetectorType" },
          { 0x00187005UL, "CSDetectorConfiguration" },
          { 0x00187006UL, "LTDetectorDescription" },
          { 0x00187008UL, "LTDetectorMode" },
          { 0x0018700AUL, "SHDetectorID" },
          { 0x0018700CUL, "DADateOfLastDetectorCalibration" },
          { 0x0018700EUL, "TMTimeOfLastDetectorCalibration" },
          { 0x00187010UL, "ISExposuresOnDetectorSinceLastCalibration" },
          { 0x00187011UL, "ISExposuresOnDetectorSinceManufactured" },
          { 0x00187012UL, "DSDetectorTimeSinceLastExposure" },
          { 0x00187014UL, "DSDetectorActiveTime" },
          { 0x00187016UL, "DSDetectorActivationOffsetFromExposure" },
          { 0x0018701AUL, "DSDetectorBinning" },
          { 0x00187020UL, "DSDetectorElementPhysicalSize" },
          { 0x00187022UL, "DSDetectorElementSpacing" },
          { 0x00187024UL, "CSDetectorActiveShape" },
          { 0x00187026UL, "DSDetectorActiveDimensions" },
          { 0x00187028UL, "DSDetectorActiveOrigin" },
          { 0x0018702AUL, "LODetectorManufacturerName" },
          { 0x0018702BUL, "LODetectorManufacturerModelName" },
          { 0x00187030UL, "DSFieldOfViewOrigin" },
          { 0x00187032UL, "DSFieldOfViewRotation" },
          { 0x00187034UL, "CSFieldOfViewHorizontalFlip" },
          { 0x00187036UL, "FLPixelDataAreaOriginRelativeToFOV" },
          { 0x00187038UL, "FLPixelDataAreaRotationAngleRelativeToFOV" },
          { 0x00187040UL, "LTGridAbsorbingMaterial" },
          { 0x00187041UL, "LTGridSpacingMaterial" },
          { 0x00187042UL, "DSGridThickness" },
          { 0x00187044UL, "DSGridPitch" },
          { 0x00187046UL, "ISGridAspectRatio" },
          { 0x00187048UL, "DSGridPeriod" },
          { 0x0018704CUL, "DSGridFocalDistance" },
          { 0x00187050UL, "CSFilterMaterial" },
          { 0x00187052UL, "DSFilterThicknessMinimum" },
          { 0x00187054UL, "DSFilterThicknessMaximum" },
          { 0x00187056UL, "FLFilterBeamPathLengthMinimum" },
          { 0x00187058UL, "FLFilterBeamPathLengthMaximum" },
          { 0x00187060UL, "CSExposureControlMode" },
          { 0x00187062UL, "LTExposureControlModeDescription" },
          { 0x00187064UL, "CSExposureStatus" },
          { 0x00187065UL, "DSPhototimerSetting" },
          { 0x00188150UL, "DSExposureTimeInuS" },
          { 0x00188151UL, "DSXRayTubeCurrentInuA" },
          { 0x00189004UL, "CSContentQualification" },
          { 0x00189005UL, "SHPulseSequenceName" },
          { 0x00189006UL, "SQMRImagingModifierSequence" },
          { 0x00189008UL, "CSEchoPulseSequence" },
          { 0x00189009UL, "CSInversionRecovery" },
          { 0x00189010UL, "CSFlowCompensation" },
          { 0x00189011UL, "CSMultipleSpinEcho" },
          { 0x00189012UL, "CSMultiPlanarExcitation" },
          { 0x00189014UL, "CSPhaseContrast" },
          { 0x00189015UL, "CSTimeOfFlightContrast" },
          { 0x00189016UL, "CSSpoiling" },
          { 0x00189017UL, "CSSteadyStatePulseSequence" },
          { 0x00189018UL, "CSEchoPlanarPulseSequence" },
          { 0x00189019UL, "FDTagAngleFirstAxis" },
          { 0x00189020UL, "CSMagnetizationTransfer" },
          { 0x00189021UL, "CST2Preparation" },
          { 0x00189022UL, "CSBloodSignalNulling" },
          { 0x00189024UL, "CSSaturationRecovery" },
          { 0x00189025UL, "CSSpectrallySelectedSuppression" },
          { 0x00189026UL, "CSSpectrallySelectedExcitation" },
          { 0x00189027UL, "CSSpatialPresaturation" },
          { 0x00189028UL, "CSTagging" },
          { 0x00189029UL, "CSOversamplingPhase" },
          { 0x00189030UL, "FDTagSpacingFirstDimension" },
          { 0x00189032UL, "CSGeometryOfKSpaceTraversal" },
          { 0x00189033UL, "CSSegmentedKSpaceTraversal" },
          { 0x00189034UL, "CSRectilinearPhaseEncodeReordering" },
          { 0x00189035UL, "FDTagThickness" },
          { 0x00189036UL, "CSPartialFourierDirection" },
          { 0x00189037UL, "CSCardiacSynchronizationTechnique" },
          { 0x00189041UL, "LOReceiveCoilManufacturerName" },
          { 0x00189042UL, "SQMRReceiveCoilSequence" },
          { 0x00189043UL, "CSReceiveCoilType" },
          { 0x00189044UL, "CSQuadratureReceiveCoil" },
          { 0x00189045UL, "SQMultiCoilDefinitionSequence" },
          { 0x00189046UL, "LOMultiCoilConfiguration" },
          { 0x00189047UL, "SHMultiCoilElementName" },
          { 0x00189048UL, "CSMultiCoilElementUsed" },
          { 0x00189049UL, "SQMRTransmitCoilSequence" },
          { 0x00189050UL, "LOTransmitCoilManufacturerName" },
          { 0x00189051UL, "CSTransmitCoilType" },
          { 0x00189052UL, "FDSpectralWidth" },
          { 0x00189053UL, "FDChemicalShiftReference" },
          { 0x00189054UL, "CSVolumeLocalizationTechnique" },
          { 0x00189058UL, "USMRAcquisitionFrequencyEncodingSteps" },
          { 0x00189059UL, "CSDecoupling" },
          { 0x00189060UL, "CSDecoupledNucleus" },
          { 0x00189061UL, "FDDecouplingFrequency" },
          { 0x00189062UL, "CSDecouplingMethod" },
          { 0x00189063UL, "FDDecouplingChemicalShiftReference" },
          { 0x00189064UL, "CSKSpaceFiltering" },
          { 0x00189065UL, "CSTimeDomainFiltering" },
          { 0x00189066UL, "USNumberOfZeroFills" },
          { 0x00189067UL, "CSBaselineCorrection" },
          { 0x00189069UL, "FDParallelReductionFactorInPlane" },
          { 0x00189070UL, "FDCardiacRRIntervalSpecified" },
          { 0x00189073UL, "FDAcquisitionDuration" },
          { 0x00189074UL, "DTFrameAcquisitionDateTime" },
          { 0x00189075UL, "CSDiffusionDirectionality" },
          { 0x00189076UL, "SQDiffusionGradientDirectionSequence" },
          { 0x00189077UL, "CSParallelAcquisition" },
          { 0x00189078UL, "CSParallelAcquisitionTechnique" },
          { 0x00189079UL, "FDInversionTimes" },
          { 0x00189080UL, "STMetaboliteMapDescription" },
          { 0x00189081UL, "CSPartialFourier" },
          { 0x00189082UL, "FDEffectiveEchoTime" },
          { 0x00189083UL, "SQMetaboliteMapCodeSequence" },
          { 0x00189084UL, "SQChemicalShiftSequence" },
          { 0x00189085UL, "CSCardiacSignalSource" },
          { 0x00189087UL, "FDDiffusionBValue" },
          { 0x00189089UL, "FDDiffusionGradientOrientation" },
          { 0x00189090UL, "FDVelocityEncodingDirection" },
          { 0x00189091UL, "FDVelocityEncodingMinimumValue" },
          { 0x00189092UL, "SQVelocityEncodingAcquisitionSequence" },
          { 0x00189093UL, "USNumberOfKSpaceTrajectories" },
          { 0x00189094UL, "CSCoverageOfKSpace" },
          { 0x00189095UL, "ULSpectroscopyAcquisitionPhaseRows" },
          { 0x00189098UL, "FDTransmitterFrequency" },
          { 0x00189100UL, "CSResonantNucleus" },
          { 0x00189101UL, "CSFrequencyCorrection" },
          { 0x00189103UL, "SQMRSpectroscopyFOVGeometrySequence" },
          { 0x00189104UL, "FDSlabThickness" },
          { 0x00189105UL, "FDSlabOrientation" },
          { 0x00189106UL, "FDMidSlabPosition" },
          { 0x00189107UL, "SQMRSpatialSaturationSequence" },
          { 0x00189112UL, "SQMRTimingAndRelatedParametersSequence" },
          { 0x00189114UL, "SQMREchoSequence" },
          { 0x00189115UL, "SQMRModifierSequence" },
          { 0x00189117UL, "SQMRDiffusionSequence" },
          { 0x00189118UL, "SQCardiacSynchronizationSequence" },
          { 0x00189119UL, "SQMRAveragesSequence" },
          { 0x00189125UL, "SQMRFOVGeometrySequence" },
          { 0x00189126UL, "SQVolumeLocalizationSequence" },
          { 0x00189127UL, "ULSpectroscopyAcquisitionDataColumns" },
          { 0x00189147UL, "CSDiffusionAnisotropyType" },
          { 0x00189151UL, "DTFrameReferenceDateTime" },
          { 0x00189152UL, "SQMRMetaboliteMapSequence" },
          { 0x00189155UL, "FDParallelReductionFactorOutOfPlane" },
          { 0x00189159UL, "ULSpectroscopyAcquisitionOutOfPlanePhaseSteps" },
          { 0x00189168UL, "FDarallelReductionFactorSecndInPlane" },
          { 0x00189169UL, "CSCardiacBeatRejectionTechnique" },
          { 0x00189170UL, "CSRespiratoryMotionCompensationTechnique" },
          { 0x00189171UL, "CSRespiratorySignalSource" },
          { 0x00189172UL, "CSBulkMotionCompensationTechnique" },
          { 0x00189173UL, "CSBulkMotionSignalSource" },
          { 0x00189178UL, "CSOperatingMode" },
          { 0x00189179UL, "CSSpecificAbsorptionRateDefinition" },
          { 0x00189180UL, "CSGradientOutputType" },
          { 0x00189181UL, "FDSpecificAbsorptionRateValue" },
          { 0x00189182UL, "FDGradientOutput" },
          { 0x00189183UL, "CSFlowCompensationDirection" },
          { 0x00189184UL, "FDTaggingDelay" },
          { 0x00189185UL, "STRespiratoryMotionCompensationTechniqueDescription" },
          { 0x00189186UL, "SHRespiratorySignalSourceIDationLimitInHzrationLimitInHz" },
          { 0x00189197UL, "SQMRVelocityEncodingSequence" },
          { 0x00189198UL, "CSFirstOrderPhaseCorrection" },
          { 0x00189199UL, "CSWaterReferencedPhaseCorrection" },
          { 0x00189200UL, "CSMRSpectroscopyAcquisitionType" },
          { 0x00189214UL, "CSRespiratoryCyclePosition" },
          { 0x00189217UL, "FDVelocityEncodingMaximumValue" },
          { 0x00189218UL, "FDTagSpacingSecondDimension" },
          { 0x00189219UL, "SSTagAngleSecondAxis" },
          { 0x00189220UL, "FDFrameAcquisitionDuration" },
          { 0x00189226UL, "SQMRImageFrameTypeSequence" },
          { 0x00189227UL, "SQMRSpectroscopyFrameTypeSequence" },
          { 0x00189231UL, "USMRAcquisitionPhaseEncodingStepsInPlane" },
          { 0x00189232UL, "USMRAcquisitionPhaseEncodingStepsOutOfPlane" },
          { 0x00189234UL, "ULSpectroscopyAcquisitionPhaseColumns" },
          { 0x00189236UL, "CSCardiacCyclePosition" },
          { 0x00189239UL, "SQSpecificAbsorptionRateSequence" },
          { 0x00189240UL, "USRFEchoTrainLength" },
          { 0x00189241UL, "USGradientEchoTrainLength" },
          { 0x00189250UL, "CSArterialSpinLabelingContrast" },
          { 0x00189251UL, "SQMRArterialSpinLabelingSequence" },
          { 0x00189252UL, "LOASLTechniqueDescription" },
          { 0x00189253UL, "USASLSlabNumber" },
          { 0x00189254UL, "FDASLSlabThickness" },
          { 0x00189255UL, "FDASLSlabOrientation" },
          { 0x00189256UL, "FDASLMidSlabPosition" },
          { 0x00189257UL, "CSASLContext" },
          { 0x00189258UL, "ULASLPulseTrainDuration" },
          { 0x00189259UL, "CSASLCrusherFlag" },
          { 0x0018925AUL, "FDASLCrusherFlow" },
          { 0x0018925BUL, "LOASLCrusherDescription" },
          { 0x0018925CUL, "CSASLBolusCutoffFlag" },
          { 0x0018925DUL, "SQASLBolusCutoffTimingSequence" },
          { 0x0018925EUL, "LOASLBolusCutoffTechnique" },
          { 0x0018925FUL, "ULASLBolusCutoffDelayTime" },
          { 0x00189260UL, "SQASLSlabSequence" },
          { 0x00189295UL, "FDChemicalShiftMinimumIntegrationLimitInppm" },
          { 0x00189296UL, "FDChemicalShiftMaximumIntegrationLimitInppm" },
          { 0x00189301UL, "SQCTAcquisitionTypeSequence" },
          { 0x00189302UL, "CSAcquisitionType" },
          { 0x00189303UL, "FDTubeAngle" },
          { 0x00189304UL, "SQCTAcquisitionDetailsSequence" },
          { 0x00189305UL, "FDRevolutionTime" },
          { 0x00189306UL, "FDSingleCollimationWidth" },
          { 0x00189307UL, "FDTotalCollimationWidth" },
          { 0x00189308UL, "SQCTTableDynamicsSequence" },
          { 0x00189309UL, "FDTableSpeed" },
          { 0x00189310UL, "FDTableFeedPerRotation" },
          { 0x00189311UL, "FDSpiralPitchFactor" },
          { 0x00189312UL, "SQCTGeometrySequence" },
          { 0x00189313UL, "FDDataCollectionCenterPatient" },
          { 0x00189314UL, "SQCTReconstructionSequence" },
          { 0x00189315UL, "CSReconstructionAlgorithm" },
          { 0x00189316UL, "CSConvolutionKernelGroup" },
          { 0x00189317UL, "FDReconstructionFieldOfView" },
          { 0x00189318UL, "FDReconstructionTargetCenterPatient" },
          { 0x00189319UL, "FDReconstructionAngle" },
          { 0x00189320UL, "SHImageFilter" },
          { 0x00189321UL, "SQCTExposureSequence" },
          { 0x00189322UL, "FDReconstructionPixelSpacing" },
          { 0x00189323UL, "CSExposureModulationType" },
          { 0x00189324UL, "FDEstimatedDoseSaving" },
          { 0x00189325UL, "SQCTXRayDetailsSequence" },
          { 0x00189326UL, "SQCTPositionSequence" },
          { 0x00189327UL, "FDTablePosition" },
          { 0x00189328UL, "FDExposureTimeInms" },
          { 0x00189329UL, "SQCTImageFrameTypeSequence" },
          { 0x00189330UL, "FDXRayTubeCurrentInmA" },
          { 0x00189332UL, "FDExposureInmAs" },
          { 0x00189333UL, "CSConstantVolumeFlag" },
          { 0x00189334UL, "CSFluoroscopyFlag" },
          { 0x00189335UL, "FDDistanceSourceToDataCollectionCenter" },
          { 0x00189337UL, "USContrastBolusAgentNumber" },
          { 0x00189338UL, "SQContrastBolusIngredientCodeSequence" },
          { 0x00189340UL, "SQContrastAdministrationProfileSequence" },
          { 0x00189341UL, "SQContrastBolusUsageSequence" },
          { 0x00189342UL, "CSContrastBolusAgentAdministered" },
          { 0x00189343UL, "CSContrastBolusAgentDetected" },
          { 0x00189344UL, "CSContrastBolusAgentPhase" },
          { 0x00189345UL, "FDCTDIvol" },
          { 0x00189346UL, "SQCTDIPhantomTypeCodeSequence" },
          { 0x00189351UL, "FLCalciumScoringMassFactorPatient" },
          { 0x00189352UL, "FLCalciumScoringMassFactorDevice" },
          { 0x00189353UL, "FLEnergyWeightingFactor" },
          { 0x00189360UL, "SQCTAdditionalXRaySourceSequence" },
          { 0x00189401UL, "SQProjectionPixelCalibrationSequence" },
          { 0x00189402UL, "FLDistanceSourceToIsocenter" },
          { 0x00189403UL, "FLDistanceObjectToTableTop" },
          { 0x00189404UL, "FLObjectPixelSpacingInCenterOfBeam" },
          { 0x00189405UL, "SQPositionerPositionSequence" },
          { 0x00189406UL, "SQTablePositionSequence" },
          { 0x00189407UL, "SQCollimatorShapeSequence" },
          { 0x00189410UL, "CSPlanesInAcquisition" },
          { 0x00189412UL, "SQXAXRFFrameCharacteristicsSequence" },
          { 0x00189417UL, "SQFrameAcquisitionSequence" },
          { 0x00189420UL, "CSXRayReceptorType" },
          { 0x00189423UL, "LOAcquisitionProtocolName" },
          { 0x00189424UL, "LTAcquisitionProtocolDescription" },
          { 0x00189425UL, "CSContrastBolusIngredientOpaque" },
          { 0x00189426UL, "FLDistanceReceptorPlaneToDetectorHousing" },
          { 0x00189427UL, "CSIntensifierActiveShape" },
          { 0x00189428UL, "FLIntensifierActiveDimensions" },
          { 0x00189429UL, "FLPhysicalDetectorSize" },
          { 0x00189430UL, "FLPositionOfIsocenterProjection" },
          { 0x00189432UL, "SQFieldOfViewSequence" },
          { 0x00189433UL, "LOFieldOfViewDescription" },
          { 0x00189434UL, "SQExposureControlSensingRegionsSequence" },
          { 0x00189435UL, "CSExposureControlSensingRegionShape" },
          { 0x00189436UL, "SSExposureControlSensingRegionLeftVerticalEdge" },
          { 0x00189437UL, "SSExposureControlSensingRegionRightVerticalEdge" },
          { 0x00189438UL, "SSExposureControlSensingRegionUpperHorizontalEdge" },
          { 0x00189439UL, "SSExposureControlSensingRegionLowerHorizontalEdge" },
          { 0x00189440UL, "SSCenterOfCircularExposureControlSensingRegion" },
          { 0x00189441UL, "USRadiusOfCircularExposureControlSensingRegion" },
          { 0x00189442UL, "SSVerticesOfThePolygonalExposureControlSensingRegion" },
          { 0x00189447UL, "FLColumnAngulationPatient" },
          { 0x00189449UL, "FLBeamAngle" },
          { 0x00189451UL, "SQFrameDetectorParametersSequence" },
          { 0x00189452UL, "FLCalculatedAnatomyThickness" },
          { 0x00189455UL, "SQCalibrationSequence" },
          { 0x00189456UL, "SQObjectThicknessSequence" },
          { 0x00189457UL, "CSPlaneIdentification" },
          { 0x00189461UL, "FLFieldOfViewDimensionsInFloat" },
          { 0x00189462UL, "SQIsocenterReferenceSystemSequence" },
          { 0x00189463UL, "FLPositionerIsocenterPrimaryAngle" },
          { 0x00189464UL, "FLPositionerIsocenterSecondaryAngle" },
          { 0x00189465UL, "FLPositionerIsocenterDetectorRotationAngle" },
          { 0x00189466UL, "FLTableXPositionToIsocenter" },
          { 0x00189467UL, "FLTableYPositionToIsocenter" },
          { 0x00189468UL, "FLTableZPositionToIsocenter" },
          { 0x00189469UL, "FLTableHorizontalRotationAngle" },
          { 0x00189470UL, "FLTableHeadTiltAngle" },
          { 0x00189471UL, "FLTableCradleTiltAngle" },
          { 0x00189472UL, "SQFrameDisplayShutterSequence" },
          { 0x00189473UL, "FLAcquiredImageAreaDoseProduct" },
          { 0x00189474UL, "CSCArmPositionerTabletopRelationship" },
          { 0x00189476UL, "SQXRayGeometrySequence" },
          { 0x00189477UL, "SQIrradiationEventIdentificationSequence" },
          { 0x00189504UL, "SQXRay3DFrameTypeSequence" },
          { 0x00189506UL, "SQContributingSourcesSequence" },
          { 0x00189507UL, "SQXRay3DAcquisitionSequence" },
          { 0x00189508UL, "FLPrimaryPositionerScanArc" },
          { 0x00189509UL, "FLSecondaryPositionerScanArc" },
          { 0x00189510UL, "FLPrimaryPositionerScanStartAngle" },
          { 0x00189511UL, "FLSecondaryPositionerScanStartAngle" },
          { 0x00189514UL, "FLPrimaryPositionerIncrement" },
          { 0x00189515UL, "FLSecondaryPositionerIncrement" },
          { 0x00189516UL, "DTStartAcquisitionDateTime" },
          { 0x00189517UL, "DTEndAcquisitionDateTime" },
          { 0x00189524UL, "LOApplicationName" },
          { 0x00189525UL, "LOApplicationVersion" },
          { 0x00189526UL, "LOApplicationManufacturer" },
          { 0x00189527UL, "CSAlgorithmType" },
          { 0x00189528UL, "LOAlgorithmDescription" },
          { 0x00189530UL, "SQXRay3DReconstructionSequence" },
          { 0x00189531UL, "LOReconstructionDescription" },
          { 0x00189538UL, "SQPerProjectionAcquisitionSequence" },
          { 0x00189601UL, "SQDiffusionBMatrixSequence" },
          { 0x00189602UL, "FDDiffusionBValueXX" },
          { 0x00189603UL, "FDDiffusionBValueXY" },
          { 0x00189604UL, "FDDiffusionBValueXZ" },
          { 0x00189605UL, "FDDiffusionBValueYY" },
          { 0x00189606UL, "FDDiffusionBValueYZ" },
          { 0x00189607UL, "FDDiffusionBValueZZ" },
          { 0x00189701UL, "DTDecayCorrectionDateTime" },
          { 0x00189715UL, "FDStartDensityThreshold" },
          { 0x00189716UL, "FDStartRelativeDensityDifferenceThreshold" },
          { 0x00189717UL, "FDStartCardiacTriggerCountThreshold" },
          { 0x00189718UL, "FDStartRespiratoryTriggerCountThreshold" },
          { 0x00189719UL, "FDTerminationCountsThreshold" },
          { 0x00189720UL, "FDTerminationDensityThreshold" },
          { 0x00189721UL, "FDTerminationRelativeDensityThreshold" },
          { 0x00189722UL, "FDTerminationTimeThreshold" },
          { 0x00189723UL, "FDTerminationCardiacTriggerCountThreshold" },
          { 0x00189724UL, "FDTerminationRespiratoryTriggerCountThreshold" },
          { 0x00189725UL, "CSDetectorGeometry" },
          { 0x00189726UL, "FDTransverseDetectorSeparation" },
          { 0x00189727UL, "FDAxialDetectorDimension" },
          { 0x00189729UL, "USRadiopharmaceuticalAgentNumber" },
          { 0x00189732UL, "SQPETFrameAcquisitionSequence" },
          { 0x00189733UL, "SQPETDetectorMotionDetailsSequence" },
          { 0x00189734UL, "SQPETTableDynamicsSequence" },
          { 0x00189735UL, "SQPETPositionSequence" },
          { 0x00189736UL, "SQPETFrameCorrectionFactorsSequence" },
          { 0x00189737UL, "SQRadiopharmaceuticalUsageSequence" },
          { 0x00189738UL, "CSAttenuationCorrectionSource" },
          { 0x00189739UL, "USNumberOfIterations" },
          { 0x00189740UL, "USNumberOfSubsets" },
          { 0x00189749UL, "SQPETReconstructionSequence" },
          { 0x00189751UL, "SQPETFrameTypeSequence" },
          { 0x00189755UL, "CSTimeOfFlightInformationUsed" },
          { 0x00189756UL, "CSReconstructionType" },
          { 0x00189758UL, "CSDecayCorrected" },
          { 0x00189759UL, "CSAttenuationCorrected" },
          { 0x00189760UL, "CSScatterCorrected" },
          { 0x00189761UL, "CSDeadTimeCorrected" },
          { 0x00189762UL, "CSGantryMotionCorrected" },
          { 0x00189763UL, "CSPatientMotionCorrected" },
          { 0x00189764UL, "CSCountLossNormalizationCorrected" },
          { 0x00189765UL, "CSRandomsCorrected" },
          { 0x00189766UL, "CSNonUniformRadialSamplingCorrected" },
          { 0x00189767UL, "CSSensitivityCalibrated" },
          { 0x00189768UL, "CSDetectorNormalizationCorrection" },
          { 0x00189769UL, "CSIterativeReconstructionMethod" },
          { 0x00189770UL, "CSAttenuationCorrectionTemporalRelationship" },
          { 0x00189771UL, "SQPatientPhysiologicalStateSequence" },
          { 0x00189772UL, "SQPatientPhysiologicalStateCodeSequence" },
          { 0x00189801UL, "FDDepthsOfFocus" },
          { 0x00189803UL, "SQExcludedIntervalsSequence" },
          { 0x00189804UL, "DTExclusionStartDatetime" },
          { 0x00189805UL, "FDExclusionDuration" },
          { 0x00189806UL, "SQUSImageDescriptionSequence" },
          { 0x00189807UL, "SQImageDataTypeSequence" },
          { 0x00189808UL, "CSDataType" },
          { 0x00189809UL, "SQTransducerScanPatternCodeSequence" },
          { 0x0018980BUL, "CSAliasedDataType" },
          { 0x0018980CUL, "CSPositionMeasuringDeviceUsed" },
          { 0x0018980DUL, "SQTransducerGeometryCodeSequence" },
          { 0x0018980EUL, "SQTransducerBeamSteeringCodeSequence" },
          { 0x0018980FUL, "SQTransducerApplicationCodeSequence" },
          { 0x0018A001UL, "SQContributingEquipmentSequence" },
          { 0x0018A002UL, "DTContributionDateTime" },
          { 0x0018A003UL, "STContributionDescription" },
          { 0x0020000DUL, "UIStudyInstanceUID" },
          { 0x0020000EUL, "UISeriesInstanceUID" },
          { 0x00200010UL, "SHStudyID" },
          { 0x00200011UL, "ISSeriesNumber" },
          { 0x00200012UL, "ISAcquisitionNumber" },
          { 0x00200013UL, "ISInstanceNumber" },
          { 0x00200019UL, "ISItemNumber" },
          { 0x00200020UL, "CSPatientOrientation" },
          { 0x00200032UL, "DSImagePositionPatient" },
          { 0x00200037UL, "DSImageOrientationPatient" },
          { 0x00200052UL, "UIFrameOfReferenceUID" },
          { 0x00200060UL, "CSLaterality" },
          { 0x00200062UL, "CSImageLaterality" },
          { 0x00200100UL, "ISTemporalPositionIdentifier" },
          { 0x00200105UL, "ISNumberOfTemporalPositions" },
          { 0x00200110UL, "DSTemporalResolution" },
          { 0x00200200UL, "UISynchronizationFrameOfReferenceUID" },
          { 0x00200242UL, "UISOPInstanceUIDOfConcatenationSource" },
          { 0x00201002UL, "ISImagesInAcquisition" },
          { 0x00201040UL, "LOPositionReferenceIndicator" },
          { 0x00201041UL, "DSSliceLocation" },
          { 0x00201200UL, "ISNumberOfPatientRelatedStudies" },
          { 0x00201202UL, "ISNumberOfPatientRelatedSeries" },
          { 0x00201204UL, "ISNumberOfPatientRelatedInstances" },
          { 0x00201206UL, "ISNumberOfStudyRelatedSeries" },
          { 0x00201208UL, "ISNumberOfStudyRelatedInstances" },
          { 0x00201209UL, "ISNumberOfSeriesRelatedInstances" },
          { 0x00204000UL, "LTImageComments" },
          { 0x00209056UL, "SHStackID" },
          { 0x00209057UL, "ULInStackPositionNumber" },
          { 0x00209071UL, "SQFrameAnatomySequence" },
          { 0x00209072UL, "CSFrameLaterality" },
          { 0x00209111UL, "SQFrameContentSequence" },
          { 0x00209113UL, "SQPlanePositionSequence" },
          { 0x00209116UL, "SQPlaneOrientationSequence" },
          { 0x00209128UL, "ULTemporalPositionIndex" },
          { 0x00209153UL, "FDNominalCardiacTriggerDelayTime" },
          { 0x00209154UL, "FLNominalCardiacTriggerTimePriorToRPeak" },
          { 0x00209155UL, "FLActualCardiacTriggerTimePriorToRPeak" },
          { 0x00209156UL, "USFrameAcquisitionNumber" },
          { 0x00209157UL, "ULDimensionIndexValues" },
          { 0x00209158UL, "LTFrameComments" },
          { 0x00209161UL, "UIConcatenationUID" },
          { 0x00209162UL, "USInConcatenationNumber" },
          { 0x00209163UL, "USInConcatenationTotalNumber" },
          { 0x00209164UL, "UIDimensionOrganizationUID" },
          { 0x00209165UL, "ATDimensionIndexPointer" },
          { 0x00209167UL, "ATFunctionalGroupPointer" },
          { 0x00209213UL, "LODimensionIndexPrivateCreator" },
          { 0x00209221UL, "SQDimensionOrganizationSequence" },
          { 0x00209222UL, "SQDimensionIndexSequence" },
          { 0x00209228UL, "ULConcatenationFrameOffsetNumber" },
          { 0x00209238UL, "LOFunctionalGroupPrivateCreator" },
          { 0x00209241UL, "FLNominalPercentageOfCardiacPhase" },
          { 0x00209245UL, "FLNominalPercentageOfRespiratoryPhase" },
          { 0x00209246UL, "FLStartingRespiratoryAmplitude" },
          { 0x00209247UL, "CSStartingRespiratoryPhase" },
          { 0x00209248UL, "FLEndingRespiratoryAmplitude" },
          { 0x00209249UL, "CSEndingRespiratoryPhase" },
          { 0x00209250UL, "CSRespiratoryTriggerType" },
          { 0x00209251UL, "FDRRIntervalTimeNominal" },
          { 0x00209252UL, "FDActualCardiacTriggerDelayTime" },
          { 0x00209253UL, "SQRespiratorySynchronizationSequence" },
          { 0x00209254UL, "FDRespiratoryIntervalTime" },
          { 0x00209255UL, "FDNominalRespiratoryTriggerDelayTime" },
          { 0x00209256UL, "FDRespiratoryTriggerDelayThreshold" },
          { 0x00209257UL, "FDActualRespiratoryTriggerDelayTime" },
          { 0x00209301UL, "FDImagePositionVolume" },
          { 0x00209302UL, "FDImageOrientationVolume" },
          { 0x00209307UL, "CSUltrasoundAcquisitionGeometry" },
          { 0x00209308UL, "FDApexPosition" },
          { 0x00209309UL, "FDVolumeToTransducerMappingMatrix" },
          { 0x0020930AUL, "FDVolumeToTableMappingMatrix" },
          { 0x0020930CUL, "CSPatientFrameOfReferenceSource" },
          { 0x0020930DUL, "FDTemporalPositionTimeOffset" },
          { 0x0020930EUL, "SQPlanePositionVolumeSequence" },
          { 0x0020930FUL, "SQPlaneOrientationVolumeSequence" },
          { 0x00209310UL, "SQTemporalPositionSequence" },
          { 0x00209311UL, "CSDimensionOrganizationType" },
          { 0x00209312UL, "UIVolumeFrameOfReferenceUID" },
          { 0x00209313UL, "UITableFrameOfReferenceUID" },
          { 0x00209421UL, "LODimensionDescriptionLabel" },
          { 0x00209450UL, "SQPatientOrientationInFrameSequence" },
          { 0x00209453UL, "LOFrameLabel" },
          { 0x00209518UL, "USAcquisitionIndex" },
          { 0x00209529UL, "SQContributingSOPInstancesReferenceSequence" },
          { 0x00209536UL, "USReconstructionIndex" },
          { 0x00220001UL, "USLightPathFilterPassThroughWavelength" },
          { 0x00220002UL, "USLightPathFilterPassBand" },
          { 0x00220003UL, "USImagePathFilterPassThroughWavelength" },
          { 0x00220004UL, "USImagePathFilterPassBand" },
          { 0x00220005UL, "CSPatientEyeMovementCommanded" },
          { 0x00220006UL, "SQPatientEyeMovementCommandCodeSequence" },
          { 0x00220007UL, "FLSphericalLensPower" },
          { 0x00220008UL, "FLCylinderLensPower" },
          { 0x00220009UL, "FLCylinderAxis" },
          { 0x0022000AUL, "FLEmmetropicMagnification" },
          { 0x0022000BUL, "FLIntraOcularPressure" },
          { 0x0022000CUL, "FLHorizontalFieldOfView" },
          { 0x0022000DUL, "CSPupilDilated" },
          { 0x0022000EUL, "FLDegreeOfDilation" },
          { 0x00220010UL, "FLStereoBaselineAngle" },
          { 0x00220011UL, "FLStereoBaselineDisplacement" },
          { 0x00220012UL, "FLStereoHorizontalPixelOffset" },
          { 0x00220013UL, "FLStereoVerticalPixelOffset" },
          { 0x00220014UL, "FLStereoRotation" },
          { 0x00220015UL, "SQAcquisitionDeviceTypeCodeSequence" },
          { 0x00220016UL, "SQIlluminationTypeCodeSequence" },
          { 0x00220017UL, "SQLightPathFilterTypeStackCodeSequence" },
          { 0x00220018UL, "SQImagePathFilterTypeStackCodeSequence" },
          { 0x00220019UL, "SQLensesCodeSequence" },
          { 0x0022001AUL, "SQChannelDescriptionCodeSequence" },
          { 0x0022001BUL, "SQRefractiveStateSequence" },
          { 0x0022001CUL, "SQMydriaticAgentCodeSequence" },
          { 0x0022001DUL, "SQRelativeImagePositionCodeSequence" },
          { 0x0022001EUL, "FLCameraAngleOfView" },
          { 0x00220020UL, "SQStereoPairsSequence" },
          { 0x00220021UL, "SQLeftImageSequence" },
          { 0x00220022UL, "SQRightImageSequence" },
          { 0x00220030UL, "FLAxialLengthOfTheEye" },
          { 0x00220031UL, "SQOphthalmicFrameLocationSequence" },
          { 0x00220032UL, "FLReferenceCoordinates" },
          { 0x00220035UL, "FLDepthSpatialResolution" },
          { 0x00220036UL, "FLMaximumDepthDistortion" },
          { 0x00220037UL, "FLAlongScanSpatialResolution" },
          { 0x00220038UL, "FLMaximumAlongScanDistortion" },
          { 0x00220039UL, "CSOphthalmicImageOrientation" },
          { 0x00220041UL, "FLDepthOfTransverseImage" },
          { 0x00220042UL, "SQMydriaticAgentConcentrationUnitsSequence" },
          { 0x00220048UL, "FLAcrossScanSpatialResolution" },
          { 0x00220049UL, "FLMaximumAcrossScanDistortion" },
          { 0x0022004EUL, "DSMydriaticAgentConcentration" },
          { 0x00220055UL, "FLIlluminationWaveLength" },
          { 0x00220056UL, "FLIlluminationPower" },
          { 0x00220057UL, "FLIlluminationBandwidth" },
          { 0x00220058UL, "SQMydriaticAgentSequence" },
          { 0x00221007UL, "SQOphthalmicAxialMeasurementsRightEyeSequence" },
          { 0x00221008UL, "SQOphthalmicAxialMeasurementsLeftEyeSequence" },
          { 0x00221010UL, "CSOphthalmicAxialLengthMeasurementsType" },
          { 0x00221019UL, "FLOphthalmicAxialLength" },
          { 0x00221024UL, "SQLensStatusCodeSequence" },
          { 0x00221025UL, "SQVitreousStatusCodeSequence" },
          { 0x00221028UL, "SQIOLFormulaCodeSequence" },
          { 0x00221029UL, "LOIOLFormulaDetail" },
          { 0x00221033UL, "FLKeratometerIndex" },
          { 0x00221035UL, "SQSourceOfOphthalmicAxialLengthCodeSequence" },
          { 0x00221037UL, "FLTargetRefraction" },
          { 0x00221039UL, "CSRefractiveProcedureOccurred" },
          { 0x00221040UL, "SQRefractiveSurgeryTypeCodeSequence" },
          { 0x00221044UL, "SQOphthalmicUltrasoundAxialMeasurementsTypeCodeSequence" },
          { 0x00221050UL, "SQOphthalmicAxialLengthMeasurementsSequence" },
          { 0x00221053UL, "FLIOLPower" },
          { 0x00221054UL, "FLPredictedRefractiveError" },
          { 0x00221059UL, "FLOphthalmicAxialLengthVelocity" },
          { 0x00221065UL, "LOLensStatusDescription" },
          { 0x00221066UL, "LOVitreousStatusDescription" },
          { 0x00221090UL, "SQIOLPowerSequence" },
          { 0x00221092UL, "SQLensConstantSequence" },
          { 0x00221093UL, "LOIOLManufacturer" },
          { 0x00221094UL, "LOLensConstantDescription" },
          { 0x00221096UL, "SQKeratometryMeasurementTypeCodeSequence" },
          { 0x00221100UL, "SQReferencedOphthalmicAxialMeasurementsSequence" },
          { 0x00221101UL, "SQOphthalmicAxialLengthMeasurementsSegmentNameCodeSequence" },
          { 0x00221103UL, "SQRefractiveErrorBeforeRefractiveSurgeryCodeSequence" },
          { 0x00221121UL, "FLIOLPowerForExactEmmetropia" },
          { 0x00221122UL, "FLIOLPowerForExactTargetRefraction" },
          { 0x00221125UL, "SQAnteriorChamberDepthDefinitionCodeSequence" },
          { 0x00221130UL, "FLLensThickness" },
          { 0x00221131UL, "FLAnteriorChamberDepth" },
          { 0x00221132UL, "SQSourceOfLensThicknessDataCodeSequence" },
          { 0x00221133UL, "SQSourceOfAnteriorChamberDepthDataCodeSequence" },
          { 0x00221135UL, "SQSourceOfRefractiveErrorDataCodeSequence" },
          { 0x00221140UL, "CSOphthalmicAxialLengthMeasurementModified" },
          { 0x00221150UL, "SQOphthalmicAxialLengthDataSourceCodeSequence" },
          { 0x00221153UL, "SQOphthalmicAxialLengthAcquisitionMethodCodeSequence" },
          { 0x00221155UL, "FLSignalToNoiseRatio" },
          { 0x00221159UL, "LOOphthalmicAxialLengthDataSourceDescription" },
          { 0x00221210UL, "SQOphthalmicAxialLengthMeasurementsTotalLengthSequence" },
          { 0x00221211UL, "SQOphthalmicAxialLengthMeasurementsSegmentalLengthSequence" },
          { 0x00221212UL, "SQOphthalmicAxialLengthMeasurementsLengthSummationSequence" },
          { 0x00221220UL, "SQUltrasoundOphthalmicAxialLengthMeasurementsSequence" },
          { 0x00221225UL, "SQOpticalOphthalmicAxialLengthMeasurementsSequence" },
          { 0x00221230UL, "SQUltrasoundSelectedOphthalmicAxialLengthSequence" },
          { 0x00221250UL, "SQOphthalmicAxialLengthSelectionMethodCodeSequence" },
          { 0x00221255UL, "SQOpticalSelectedOphthalmicAxialLengthSequence" },
          { 0x00221257UL, "SQSelectedSegmentalOphthalmicAxialLengthSequence" },
          { 0x00221260UL, "SQSelectedTotalOphthalmicAxialLengthSequence" },
          { 0x00221262UL, "SQOphthalmicAxialLengthQualityMetricSequence" },
          { 0x00221273UL, "LOOphthalmicAxialLengthQualityMetricTypeDescription" },
          { 0x00221300UL, "SQIntraocularLensCalculationsRightEyeSequence" },
          { 0x00221310UL, "SQIntraocularLensCalculationsLeftEyeSequence" },
          { 0x00221330UL, "SQReferencedOphthalmicAxialLengthMeasurementQCImageSequence" },
          { 0x00240010UL, "FLVisualFieldHorizontalExtent" },
          { 0x00240011UL, "FLVisualFieldVerticalExtent" },
          { 0x00240012UL, "CSVisualFieldShape" },
          { 0x00240016UL, "SQScreeningTestModeCodeSequence" },
          { 0x00240018UL, "FLMaximumStimulusLuminance" },
          { 0x00240020UL, "FLBackgroundLuminance" },
          { 0x00240021UL, "SQStimulusColorCodeSequence" },
          { 0x00240024UL, "SQBackgroundIlluminationColorCodeSequence" },
          { 0x00240025UL, "FLStimulusArea" },
          { 0x00240028UL, "FLStimulusPresentationTime" },
          { 0x00240032UL, "SQFixationSequence" },
          { 0x00240033UL, "SQFixationMonitoringCodeSequence" },
          { 0x00240034UL, "SQVisualFieldCatchTrialSequence" },
          { 0x00240035UL, "USFixationCheckedQuantity" },
          { 0x00240036UL, "USPatientNotProperlyFixatedQuantity" },
          { 0x00240037UL, "CSPresentedVisualStimuliDataFlag" },
          { 0x00240038UL, "USNumberOfVisualStimuli" },
          { 0x00240039UL, "CSExcessiveFixationLossesDataFlag" },
          { 0x00240040UL, "CSExcessiveFixationLosses" },
          { 0x00240042UL, "USStimuliRetestingQuantity" },
          { 0x00240044UL, "LTCommentsOnPatientPerformanceOfVisualField" },
          { 0x00240045UL, "CSFalseNegativesEstimateFlag" },
          { 0x00240046UL, "FLFalseNegativesEstimate" },
          { 0x00240048UL, "USNegativeCatchTrialsQuantity" },
          { 0x00240050UL, "USFalseNegativesQuantity" },
          { 0x00240051UL, "CSExcessiveFalseNegativesDataFlag" },
          { 0x00240052UL, "CSExcessiveFalseNegatives" },
          { 0x00240053UL, "CSFalsePositivesEstimateFlag" },
          { 0x00240054UL, "FLFalsePositivesEstimate" },
          { 0x00240055UL, "CSCatchTrialsDataFlag" },
          { 0x00240056UL, "USPositiveCatchTrialsQuantity" },
          { 0x00240057UL, "CSTestPointNormalsDataFlag" },
          { 0x00240058UL, "SQTestPointNormalsSequence" },
          { 0x00240059UL, "CSGlobalDeviationProbabilityNormalsFlag" },
          { 0x00240060UL, "USFalsePositivesQuantity" },
          { 0x00240061UL, "CSExcessiveFalsePositivesDataFlag" },
          { 0x00240062UL, "CSExcessiveFalsePositives" },
          { 0x00240063UL, "CSVisualFieldTestNormalsFlag" },
          { 0x00240064UL, "SQResultsNormalsSequence" },
          { 0x00240065UL, "SQAgeCorrectedSensitivityDeviationAlgorithmSequence" },
          { 0x00240066UL, "FLGlobalDeviationFromNormal" },
          { 0x00240067UL, "SQGeneralizedDefectSensitivityDeviationAlgorithmSequence" },
          { 0x00240068UL, "FLLocalizedDeviationfromNormal" },
          { 0x00240069UL, "LOPatientReliabilityIndicator" },
          { 0x00240070UL, "FLVisualFieldMeanSensitivity" },
          { 0x00240071UL, "FLGlobalDeviationProbability" },
          { 0x00240072UL, "CSLocalDeviationProbabilityNormalsFlag" },
          { 0x00240073UL, "FLLocalizedDeviationProbability" },
          { 0x00240074UL, "CSShortTermFluctuationCalculated" },
          { 0x00240075UL, "FLShortTermFluctuation" },
          { 0x00240076UL, "CSShortTermFluctuationProbabilityCalculated" },
          { 0x00240077UL, "FLShortTermFluctuationProbability" },
          { 0x00240078UL, "CSCorrectedLocalizedDeviationFromNormalCalculated" },
          { 0x00240079UL, "FLCorrectedLocalizedDeviationFromNormal" },
          { 0x00240080UL, "CSCorrectedLocalizedDeviationFromNormalProbabilityCalculated" },
          { 0x00240081UL, "FLCorrectedLocalizedDeviationFromNormalProbability" },
          { 0x00240083UL, "SQGlobalDeviationProbabilitySequence" },
          { 0x00240085UL, "SQLocalizedDeviationProbabilitySequence" },
          { 0x00240086UL, "CSFovealSensitivityMeasured" },
          { 0x00240087UL, "FLFovealSensitivity" },
          { 0x00240088UL, "FLVisualFieldTestDuration" },
          { 0x00240089UL, "SQVisualFieldTestPointSequence" },
          { 0x00240090UL, "FLVisualFieldTestPointXCoordinate" },
          { 0x00240091UL, "FLVisualFieldTestPointYCoordinate" },
          { 0x00240092UL, "FLAgeCorrectedSensitivityDeviationValue" },
          { 0x00240093UL, "CSStimulusResults" },
          { 0x00240094UL, "FLSensitivityValue" },
          { 0x00240095UL, "CSRetestStimulusSeen" },
          { 0x00240096UL, "FLRetestSensitivityValue" },
          { 0x00240097UL, "SQVisualFieldTestPointNormalsSequence" },
          { 0x00240098UL, "FLQuantifiedDefect" },
          { 0x00240100UL, "FLAgeCorrectedSensitivityDeviationProbabilityValue" },
          { 0x00240102UL, "CSGeneralizedDefectCorrectedSensitivityDeviationFlag" },
          { 0x00240103UL, "FLGeneralizedDefectCorrectedSensitivityDeviationValue" },
          { 0x00240104UL, "FLGeneralizedDefectCorrectedSensitivityDeviationProbabilityValue" },
          { 0x00240105UL, "FLMinimumSensitivityValue" },
          { 0x00240106UL, "CSBlindSpotLocalized" },
          { 0x00240107UL, "FLBlindSpotXCoordinate" },
          { 0x00240108UL, "FLBlindSpotYCoordinate" },
          { 0x00240110UL, "SQVisualAcuityMeasurementSequence" },
          { 0x00240112UL, "SQRefractiveParametersUsedOnPatientSequence" },
          { 0x00240113UL, "CSMeasurementLaterality" },
          { 0x00240114UL, "SQOphthalmicPatientClinicalInformationLeftEyeSequence" },
          { 0x00240115UL, "SQOphthalmicPatientClinicalInformationRightEyeSequence" },
          { 0x00240117UL, "CSFovealPointNormativeDataFlag" },
          { 0x00240118UL, "FLFovealPointProbabilityValue" },
          { 0x00240120UL, "CSScreeningBaselineMeasured" },
          { 0x00240122UL, "SQScreeningBaselineMeasuredSequence" },
          { 0x00240124UL, "CSScreeningBaselineType" },
          { 0x00240126UL, "FLScreeningBaselineValue" },
          { 0x00240202UL, "LOAlgorithmSource" },
          { 0x00240306UL, "LODataSetName" },
          { 0x00240307UL, "LODataSetVersion" },
          { 0x00240308UL, "LODataSetSource" },
          { 0x00240309UL, "LODataSetDescription" },
          { 0x00240317UL, "SQVisualFieldTestReliabilityGlobalIndexSequence" },
          { 0x00240320UL, "SQVisualFieldGlobalResultsIndexSequence" },
          { 0x00240325UL, "SQDataObservationSequence" },
          { 0x00240338UL, "CSIndexNormalsFlag" },
          { 0x00240341UL, "FLIndexProbability" },
          { 0x00240344UL, "SQIndexProbabilitySequence" },
          { 0x00280002UL, "USSamplesPerPixel" },
          { 0x00280003UL, "USSamplesPerPixelUsed" },
          { 0x00280004UL, "CSPhotometricInterpretation" },
          { 0x00280006UL, "USPlanarConfiguration" },
          { 0x00280008UL, "ISNumberOfFrames" },
          { 0x00280009UL, "ATFrameIncrementPointer" },
          { 0x0028000AUL, "ATFrameDimensionPointer" },
          { 0x00280010UL, "USRows" },
          { 0x00280011UL, "USColumns" },
          { 0x00280014UL, "USUltrasoundColorDataPresent" },
          { 0x00280030UL, "DSPixelSpacing" },
          { 0x00280031UL, "DSZoomFactor" },
          { 0x00280032UL, "DSZoomCenter" },
          { 0x00280034UL, "ISPixelAspectRatio" },
          { 0x00280051UL, "CSCorrectedImage" },
          { 0x00280100UL, "USBitsAllocated" },
          { 0x00280101UL, "USBitsStored" },
          { 0x00280102UL, "USHighBit" },
          { 0x00280103UL, "USPixelRepresentation" },
          { 0x00280106UL, "USSmallestImagePixelValue" },
          { 0x00280107UL, "USLargestImagePixelValue" },
          { 0x00280108UL, "USSmallestPixelValueInSeries" },
          { 0x00280109UL, "USLargestPixelValueInSeries" },
          { 0x00280120UL, "USPixelPaddingValue" },
          { 0x00280121UL, "USPixelPaddingRangeLimit" },
          { 0x00280300UL, "CSQualityControlImage" },
          { 0x00280301UL, "CSBurnedInAnnotation" },
          { 0x00280302UL, "CSRecognizableVisualFeatures" },
          { 0x00280303UL, "CSLongitudinalTemporalInformationModified" },
          { 0x00280A02UL, "CSPixelSpacingCalibrationType" },
          { 0x00280A04UL, "LOPixelSpacingCalibrationDescription" },
          { 0x00281040UL, "CSPixelIntensityRelationship" },
          { 0x00281041UL, "SSPixelIntensityRelationshipSign" },
          { 0x00281050UL, "DSWindowCenter" },
          { 0x00281051UL, "DSWindowWidth" },
          { 0x00281052UL, "DSRescaleIntercept" },
          { 0x00281053UL, "DSRescaleSlope" },
          { 0x00281054UL, "LORescaleType" },
          { 0x00281055UL, "LOWindowCenterWidthExplanation" },
          { 0x00281056UL, "CSVOILUTFunction" },
          { 0x00281090UL, "CSRecommendedViewingMode" },
          { 0x00281101UL, "USRedPaletteColorLookupTableDescriptor" },
          { 0x00281102UL, "USGreenPaletteColorLookupTableDescriptor" },
          { 0x00281103UL, "USBluePaletteColorLookupTableDescriptor" },
          { 0x00281104UL, "USAlphaPaletteColorLookupTableDescriptor" },
          { 0x00281199UL, "UIPaletteColorLookupTableUID" },
          { 0x00281201UL, "OWRedPaletteColorLookupTableData" },
          { 0x00281202UL, "OWGreenPaletteColorLookupTableData" },
          { 0x00281203UL, "OWBluePaletteColorLookupTableData" },
          { 0x00281204UL, "OWAlphaPaletteColorLookupTableData" },
          { 0x00281221UL, "OWSegmentedRedPaletteColorLookupTableData" },
          { 0x00281222UL, "OWSegmentedGreenPaletteColorLookupTableData" },
          { 0x00281223UL, "OWSegmentedBluePaletteColorLookupTableData" },
          { 0x00281300UL, "CSBreastImplantPresent" },
          { 0x00281350UL, "CSPartialView" },
          { 0x00281351UL, "STPartialViewDescription" },
          { 0x00281352UL, "SQPartialViewCodeSequence" },
          { 0x0028135AUL, "CSSpatialLocationsPreserved" },
          { 0x00281401UL, "SQDataFrameAssignmentSequence" },
          { 0x00281402UL, "CSDataPathAssignment" },
          { 0x00281403UL, "USBitsMappedToColorLookupTable" },
          { 0x00281404UL, "SQBlendingLUT1Sequence" },
          { 0x00281405UL, "CSBlendingLUT1TransferFunction" },
          { 0x00281406UL, "FDBlendingWeightConstant" },
          { 0x00281407UL, "USBlendingLookupTableDescriptor" },
          { 0x00281408UL, "OWBlendingLookupTableData" },
          { 0x0028140BUL, "SQEnhancedPaletteColorLookupTableSequence" },
          { 0x0028140CUL, "SQBlendingLUT2Sequence" },
          { 0x0028140DUL, "CSBlendingLUT2TransferFunction" },
          { 0x0028140EUL, "CSDataPathID" },
          { 0x0028140FUL, "CSRGBLUTTransferFunction" },
          { 0x00281410UL, "CSAlphaLUTTransferFunction" },
          { 0x00282000UL, "OBICCProfile" },
          { 0x00282110UL, "CSLossyImageCompression" },
          { 0x00282112UL, "DSLossyImageCompressionRatio" },
          { 0x00282114UL, "CSLossyImageCompressionMethod" },
          { 0x00283000UL, "SQModalityLUTSequence" },
          { 0x00283002UL, "USLUTDescriptor" },
          { 0x00283003UL, "LOLUTExplanation" },
          { 0x00283004UL, "LOModalityLUTType" },
          { 0x00283006UL, "USLUTData" },
          { 0x00283010UL, "SQVOILUTSequence" },
          { 0x00283110UL, "SQSoftcopyVOILUTSequences" },
          { 0x00286010UL, "USRepresentativeFrameNumber" },
          { 0x00286020UL, "USFrameNumbersOfInterest" },
          { 0x00286022UL, "LOFrameOfInterestDescription" },
          { 0x00286023UL, "CSFrameOfInterestType" },
          { 0x00286040UL, "USRWavePointer" },
          { 0x00286100UL, "SQMaskSubtractionSequence" },
          { 0x00286101UL, "CSMaskOperation" },
          { 0x00286102UL, "USApplicableFrameRange" },
          { 0x00286110UL, "USMaskFrameNumbers" },
          { 0x00286112UL, "USContrastFrameAveraging" },
          { 0x00286114UL, "FLMaskSubPixelShift" },
          { 0x00286120UL, "SSTIDOffset" },
          { 0x00286190UL, "STMaskOperationExplanation" },
          { 0x00287FE0UL, "UTPixelDataProviderURL" },
          { 0x00289001UL, "ULDataPointRows" },
          { 0x00289002UL, "ULDataPointColumns" },
          { 0x00289003UL, "CSSignalDomainColumnsue" },
          { 0x00289108UL, "CSDataRepresentation" },
          { 0x00289110UL, "SQPixelMeasuresSequence" },
          { 0x00289132UL, "SQFrameVOILUTSequence" },
          { 0x00289145UL, "SQPixelValueTransformationSequence" },
          { 0x00289235UL, "CSSignalDomainRows" },
          { 0x00289411UL, "FLDisplayFilterPercentage" },
          { 0x00289415UL, "SQFramePixelShiftSequence" },
          { 0x00289416UL, "USSubtractionItemID" },
          { 0x00289422UL, "SQPixelIntensityRelationshipLUTSequence" },
          { 0x00289443UL, "SQFramePixelDataPropertiesSequence" },
          { 0x00289444UL, "CSGeometricalProperties" },
          { 0x00289445UL, "FLGeometricMaximumDistortion" },
          { 0x00289446UL, "CSImageProcessingApplied" },
          { 0x00289454UL, "CSMaskSelectionMode" },
          { 0x00289474UL, "CSLUTFunction" },
          { 0x00289478UL, "FLMaskVisibilityPercentage" },
          { 0x00289501UL, "SQPixelShiftSequence" },
          { 0x00289502UL, "SQRegionPixelShiftSequence" },
          { 0x00289503UL, "SSVerticesOfTheRegion" },
          { 0x00289505UL, "SQMultiFramePresentationSequence" },
          { 0x00289506UL, "USPixelShiftFrameRange" },
          { 0x00289507UL, "USLUTFrameRange" },
          { 0x00289520UL, "DSImageToEquipmentMappingMatrix" },
          { 0x00289537UL, "CSEquipmentCoordinateSystemIdentification" },
          { 0x00290010UL, "LOPrivateCreatorCode" },
          { 0x00290011UL, "LOAnotherPrivateCreatorCode" },
          { 0x00291008UL, "CSCSAImageHeaderType" },
          { 0x00291009UL, "LOCSAImageHeaderVersion" },
          { 0x00291010UL, "OBCSAImageHeaderInfo" },
          { 0x00291018UL, "CSCSASeriesHeaderType" },
          { 0x00291019UL, "LOCSASeriesHeaderVersion" },
          { 0x00291020UL, "OBCSASeriesHeaderInfo" },
          { 0x00291131UL, "LOPMTFInformation1" },
          { 0x00291132UL, "ULPMTFInformation2" },
          { 0x00291133UL, "ULPMTFInformation3" },
          { 0x00291134UL, "CSPMTFInformation4" },
          { 0x00321031UL, "SQRequestingPhysicianIdentificationSequence" },
          { 0x00321032UL, "PNRequestingPhysician" },
          { 0x00321033UL, "LORequestingService" },
          { 0x00321034UL, "SQRequestingServiceCodeSequence" },
          { 0x00321060UL, "LORequestedProcedureDescription" },
          { 0x00321064UL, "SQRequestedProcedureCodeSequence" },
          { 0x00321070UL, "LORequestedContrastAgent" },
          { 0x00380004UL, "SQReferencedPatientAliasSequence" },
          { 0x00380008UL, "CSVisitStatusID" },
          { 0x00380010UL, "LOAdmissionID" },
          { 0x00380014UL, "SQIssuerOfAdmissionIDSequence" },
          { 0x00380016UL, "LORouteOfAdmissionsesidence" },
          { 0x00380020UL, "DAAdmittingDate" },
          { 0x00380021UL, "TMAdmittingTime" },
          { 0x00380050UL, "LOSpecialNeeds" },
          { 0x00380060UL, "LOServiceEpisodeID" },
          { 0x00380062UL, "LOServiceEpisodeDescription" },
          { 0x00380064UL, "SQIssuerOfServiceEpisodeIDSequence" },
          { 0x00380100UL, "SQPertinentDocumentsSequence" },
          { 0x00380300UL, "LOCurrentPatientLocation" },
          { 0x00380400UL, "LOPatientInstitutionResidence" },
          { 0x00380500UL, "LOPatientState" },
          { 0x00380502UL, "SQPatientClinicalTrialParticipationSequence" },
          { 0x00384000UL, "LTVisitComments" },
          { 0x003A0004UL, "CSWaveformOriginality" },
          { 0x003A0005UL, "USNumberOfWaveformChannels" },
          { 0x003A0010UL, "ULNumberOfWaveformSamples" },
          { 0x003A001AUL, "DSSamplingFrequency" },
          { 0x003A0020UL, "SHMultiplexGroupLabel" },
          { 0x003A0200UL, "SQChannelDefinitionSequence" },
          { 0x003A0202UL, "ISWaveformChannelNumber" },
          { 0x003A0203UL, "SHChannelLabel" },
          { 0x003A0205UL, "CSChannelStatus" },
          { 0x003A0208UL, "SQChannelSourceSequence" },
          { 0x003A0209UL, "SQChannelSourceModifiersSequence" },
          { 0x003A020AUL, "SQSourceWaveformSequence" },
          { 0x003A020CUL, "LOChannelDerivationDescription" },
          { 0x003A0210UL, "DSChannelSensitivity" },
          { 0x003A0211UL, "SQChannelSensitivityUnitsSequence" },
          { 0x003A0212UL, "DSChannelSensitivityCorrectionFactor" },
          { 0x003A0213UL, "DSChannelBaseline" },
          { 0x003A0214UL, "DSChannelTimeSkew" },
          { 0x003A0215UL, "DSChannelSampleSkew" },
          { 0x003A0218UL, "DSChannelOffset" },
          { 0x003A021AUL, "USWaveformBitsStored" },
          { 0x003A0220UL, "DSFilterLowFrequency" },
          { 0x003A0221UL, "DSFilterHighFrequency" },
          { 0x003A0222UL, "DSNotchFilterFrequency" },
          { 0x003A0223UL, "DSNotchFilterBandwidth" },
          { 0x003A0230UL, "FLWaveformDataDisplayScale" },
          { 0x003A0231UL, "USWaveformDisplayBackgroundCIELabValue" },
          { 0x003A0240UL, "SQWaveformPresentationGroupSequence" },
          { 0x003A0241UL, "USPresentationGroupNumber" },
          { 0x003A0242UL, "SQChannelDisplaySequence" },
          { 0x003A0244UL, "USChannelRecommendedDisplayCIELabValue" },
          { 0x003A0245UL, "FLChannelPosition" },
          { 0x003A0246UL, "CSDisplayShadingFlag" },
          { 0x003A0247UL, "FLFractionalChannelDisplayScale" },
          { 0x003A0248UL, "FLAbsoluteChannelDisplayScale" },
          { 0x003A0300UL, "SQMultiplexedAudioChannelsDescriptionCodeSequence" },
          { 0x003A0301UL, "ISChannelIdentificationCode" },
          { 0x003A0302UL, "CSChannelMode" },
          { 0x00400001UL, "AEScheduledStationAETitle" },
          { 0x00400002UL, "DAScheduledProcedureStepStartDate" },
          { 0x00400003UL, "TMScheduledProcedureStepStartTime" },
          { 0x00400004UL, "DAScheduledProcedureStepEndDate" },
          { 0x00400005UL, "TMScheduledProcedureStepEndTime" },
          { 0x00400006UL, "PNScheduledPerformingPhysicianName" },
          { 0x00400007UL, "LOScheduledProcedureStepDescription" },
          { 0x00400008UL, "SQScheduledProtocolCodeSequence" },
          { 0x00400009UL, "SHScheduledProcedureStepID" },
          { 0x0040000AUL, "SQStageCodeSequence" },
          { 0x0040000BUL, "SQScheduledPerformingPhysicianIdentificationSequence" },
          { 0x00400010UL, "SHScheduledStationName" },
          { 0x00400011UL, "SHScheduledProcedureStepLocation" },
          { 0x00400012UL, "LOPreMedication" },
          { 0x00400020UL, "CSScheduledProcedureStepStatus" },
          { 0x00400026UL, "SQOrderPlacerIdentifierSequence" },
          { 0x00400027UL, "SQOrderFillerIdentifierSequence" },
          { 0x00400031UL, "UTLocalNamespaceEntityID" },
          { 0x00400032UL, "UTUniversalEntityID" },
          { 0x00400033UL, "CSUniversalEntityIDType" },
          { 0x00400035UL, "CSIdentifierTypeCode" },
          { 0x00400036UL, "SQAssigningFacilitySequence" },
          { 0x00400039UL, "SQAssigningJurisdictionCodeSequence" },
          { 0x0040003AUL, "SQAssigningAgencyOrDepartmentCodeSequence" },
          { 0x00400100UL, "SQScheduledProcedureStepSequence" },
          { 0x00400220UL, "SQReferencedNonImageCompositeSOPInstanceSequence" },
          { 0x00400241UL, "AEPerformedStationAETitle" },
          { 0x00400242UL, "SHPerformedStationName" },
          { 0x00400243UL, "SHPerformedLocation" },
          { 0x00400244UL, "DAPerformedProcedureStepStartDate" },
          { 0x00400245UL, "TMPerformedProcedureStepStartTime" },
          { 0x00400250UL, "DAPerformedProcedureStepEndDate" },
          { 0x00400251UL, "TMPerformedProcedureStepEndTime" },
          { 0x00400252UL, "CSPerformedProcedureStepStatus" },
          { 0x00400253UL, "SHPerformedProcedureStepID" },
          { 0x00400254UL, "LOPerformedProcedureStepDescription" },
          { 0x00400255UL, "LOPerformedProcedureTypeDescription" },
          { 0x00400260UL, "SQPerformedProtocolCodeSequence" },
          { 0x00400261UL, "CSPerformedProtocolType" },
          { 0x00400270UL, "SQScheduledStepAttributesSequence" },
          { 0x00400275UL, "SQRequestAttributesSequence" },
          { 0x00400280UL, "STCommentsOnThePerformedProcedureStep" },
          { 0x00400281UL, "SQPerformedProcedureStepDiscontinuationReasonCodeSequence" },
          { 0x00400293UL, "SQQuantitySequence" },
          { 0x00400294UL, "DSQuantity" },
          { 0x00400295UL, "SQMeasuringUnitsSequence" },
          { 0x00400296UL, "SQBillingItemSequence" },
          { 0x00400300UL, "USTotalTimeOfFluoroscopy" },
          { 0x00400301UL, "USTotalNumberOfExposures" },
          { 0x00400302UL, "USEntranceDose" },
          { 0x00400303UL, "USExposedArea" },
          { 0x00400306UL, "DSDistanceSourceToEntrance" },
          { 0x0040030EUL, "SQExposureDoseSequence" },
          { 0x00400310UL, "STCommentsOnRadiationDose" },
          { 0x00400312UL, "DSXRayOutput" },
          { 0x00400314UL, "DSHalfValueLayer" },
          { 0x00400316UL, "DSOrganDose" },
          { 0x00400318UL, "CSOrganExposed" },
          { 0x00400320UL, "SQBillingProcedureStepSequence" },
          { 0x00400321UL, "SQFilmConsumptionSequence" },
          { 0x00400324UL, "SQBillingSuppliesAndDevicesSequence" },
          { 0x00400340UL, "SQPerformedSeriesSequence" },
          { 0x00400400UL, "LTCommentsOnTheScheduledProcedureStep" },
          { 0x00400440UL, "SQProtocolContextSequence" },
          { 0x00400441UL, "SQContentItemModifierSequence" },
          { 0x00400500UL, "SQScheduledSpecimenSequence" },
          { 0x00400512UL, "LOContainerIdentifier" },
          { 0x00400513UL, "SQIssuerOfTheContainerIdentifierSequence" },
          { 0x00400515UL, "SQAlternateContainerIdentifierSequence" },
          { 0x00400518UL, "SQContainerTypeCodeSequence" },
          { 0x0040051AUL, "LOContainerDescription" },
          { 0x00400520UL, "SQContainerComponentSequence" },
          { 0x00400551UL, "LOSpecimenIdentifiernceTrial" },
          { 0x00400554UL, "UISpecimenUID" },
          { 0x00400555UL, "SQAcquisitionContextSequence" },
          { 0x00400556UL, "STAcquisitionContextDescription" },
          { 0x00400560UL, "SQSpecimenDescriptionSequence" },
          { 0x00400562UL, "SQIssuerOfTheSpecimenIdentifierSequence" },
          { 0x0040059AUL, "SQSpecimenTypeCodeSequence" },
          { 0x00400600UL, "LOSpecimenShortDescription" },
          { 0x00400602UL, "UTSpecimenDetailedDescription" },
          { 0x00400610UL, "SQSpecimenPreparationSequence" },
          { 0x00400612UL, "SQSpecimenPreparationStepContentItemSequence" },
          { 0x00400620UL, "SQSpecimenLocalizationContentItemSequence" },
          { 0x0040071AUL, "SQImageCenterPointCoordinatesSequence" },
          { 0x0040072AUL, "DSXOffsetInSlideCoordinateSystem" },
          { 0x0040073AUL, "DSYOffsetInSlideCoordinateSystem" },
          { 0x0040074AUL, "DSZOffsetInSlideCoordinateSystem" },
          { 0x004008EAUL, "SQMeasurementUnitsCodeSequence" },
          { 0x00401001UL, "SHRequestedProcedureID" },
          { 0x00401002UL, "LOReasonForTheRequestedProcedure" },
          { 0x00401003UL, "SHRequestedProcedurePriority" },
          { 0x00401004UL, "LOPatientTransportArrangements" },
          { 0x00401005UL, "LORequestedProcedureLocation" },
          { 0x00401008UL, "LOConfidentialityCode" },
          { 0x00401009UL, "SHReportingPriority" },
          { 0x0040100AUL, "SQReasonForRequestedProcedureCodeSequence" },
          { 0x00401010UL, "PNNamesOfIntendedRecipientsOfResults" },
          { 0x00401011UL, "SQIntendedRecipientsOfResultsIdentificationSequence" },
          { 0x00401012UL, "SQReasonForPerformedProcedureCodeSequence" },
          { 0x00401101UL, "SQPersonIdentificationCodeSequence" },
          { 0x00401102UL, "STPersonAddress" },
          { 0x00401103UL, "LOPersonTelephoneNumbers" },
          { 0x00401400UL, "LTRequestedProcedureComments" },
          { 0x00402004UL, "DAIssueDateOfImagingServiceRequest" },
          { 0x00402005UL, "TMIssueTimeOfImagingServiceRequest" },
          { 0x00402008UL, "PNOrderEnteredBy" },
          { 0x00402009UL, "SHOrderEntererLocation" },
          { 0x00402010UL, "SHOrderCallbackPhoneNumber" },
          { 0x00402016UL, "LOPlacerOrderNumberImagingServiceRequest" },
          { 0x00402017UL, "LOFillerOrderNumberImagingServiceRequest" },
          { 0x00402400UL, "LTImagingServiceRequestComments" },
          { 0x00403001UL, "LOConfidentialityConstraintOnPatientDataDescription" },
          { 0x00404001UL, "CSGeneralPurposeScheduledProcedureStepStatus" },
          { 0x00404002UL, "CSGeneralPurposePerformedProcedureStepStatus" },
          { 0x00404003UL, "CSGeneralPurposeScheduledProcedureStepPriority" },
          { 0x00404004UL, "SQScheduledProcessingApplicationsCodeSequence" },
          { 0x00404005UL, "DTScheduledProcedureStepStartDateTime" },
          { 0x00404006UL, "CSMultipleCopiesFlag" },
          { 0x00404007UL, "SQPerformedProcessingApplicationsCodeSequence" },
          { 0x00404009UL, "SQHumanPerformerCodeSequence" },
          { 0x00404010UL, "DTScheduledProcedureStepModificationDateTime" },
          { 0x00404011UL, "DTExpectedCompletionDateTime" },
          { 0x00404015UL, "SQResultingGeneralPurposePerformedProcedureStepsSequence" },
          { 0x00404016UL, "SQReferencedGeneralPurposeScheduledProcedureStepSequence" },
          { 0x00404018UL, "SQScheduledWorkitemCodeSequence" },
          { 0x00404019UL, "SQPerformedWorkitemCodeSequence" },
          { 0x00404020UL, "CSInputAvailabilityFlag" },
          { 0x00404021UL, "SQInputInformationSequence" },
          { 0x00404022UL, "SQRelevantInformationSequence" },
          { 0x00404023UL, "UIReferencedGeneralPurposeScheduledProcedureStepTransactionUID" },
          { 0x00404025UL, "SQScheduledStationNameCodeSequence" },
          { 0x00404026UL, "SQScheduledStationClassCodeSequence" },
          { 0x00404027UL, "SQScheduledStationGeographicLocationCodeSequence" },
          { 0x00404028UL, "SQPerformedStationNameCodeSequence" },
          { 0x00404029UL, "SQPerformedStationClassCodeSequence" },
          { 0x00404030UL, "SQPerformedStationGeographicLocationCodeSequence" },
          { 0x00404031UL, "SQRequestedSubsequentWorkitemCodeSequence" },
          { 0x00404032UL, "SQNonDICOMOutputCodeSequence" },
          { 0x00404033UL, "SQOutputInformationSequence" },
          { 0x00404034UL, "SQScheduledHumanPerformersSequence" },
          { 0x00404035UL, "SQActualHumanPerformersSequence" },
          { 0x00404036UL, "LOHumanPerformerOrganization" },
          { 0x00404037UL, "PNHumanPerformerName" },
          { 0x00404040UL, "CSRawDataHandling" },
          { 0x00404041UL, "CSInputReadinessState" },
          { 0x00404050UL, "DTPerformedProcedureStepStartDateTime" },
          { 0x00404051UL, "DTPerformedProcedureStepEndDateTime" },
          { 0x00404052UL, "DTProcedureStepCancellationDateTime" },
          { 0x00408302UL, "DSEntranceDoseInmGy" },
          { 0x00409094UL, "SQReferencedImageRealWorldValueMappingSequence" },
          { 0x00409096UL, "SQRealWorldValueMappingSequence" },
          { 0x00409098UL, "SQPixelValueMappingCodeSequence" },
          { 0x00409210UL, "SHLUTLabel" },
          { 0x00409211UL, "USRealWorldValueLastValueMapped" },
          { 0x00409212UL, "FDRealWorldValueLUTData" },
          { 0x00409216UL, "USRealWorldValueFirstValueMapped" },
          { 0x00409224UL, "FDRealWorldValueIntercept" },
          { 0x00409225UL, "FDRealWorldValueSlope" },
          { 0x0040A010UL, "CSRelationshipType" },
          { 0x0040A027UL, "LOVerifyingOrganizationentifierCodeSequenceTrial" },
          { 0x0040A030UL, "DTVerificationDateTime" },
          { 0x0040A032UL, "DTObservationDateTime" },
          { 0x0040A040UL, "CSValueType" },
          { 0x0040A043UL, "SQConceptNameCodeSequence" },
          { 0x0040A050UL, "CSContinuityOfContent" },
          { 0x0040A073UL, "SQVerifyingObserverSequence" },
          { 0x0040A075UL, "PNVerifyingObserverNameierCodeSequenceTrial" },
          { 0x0040A078UL, "SQAuthorObserverSequence" },
          { 0x0040A07AUL, "SQParticipantSequence" },
          { 0x0040A07CUL, "SQCustodialOrganizationSequence" },
          { 0x0040A080UL, "CSParticipationType" },
          { 0x0040A082UL, "DTParticipationDateTime" },
          { 0x0040A084UL, "CSObserverType" },
          { 0x0040A088UL, "SQVerifyingObserverIdentificationCodeSequence" },
          { 0x0040A0B0UL, "USReferencedWaveformChannels" },
          { 0x0040A120UL, "DTDateTime" },
          { 0x0040A121UL, "DADate" },
          { 0x0040A122UL, "TMTime" },
          { 0x0040A123UL, "PNPersonName" },
          { 0x0040A124UL, "UIUID" },
          { 0x0040A130UL, "CSTemporalRangeType" },
          { 0x0040A132UL, "ULReferencedSamplePositions" },
          { 0x0040A136UL, "USReferencedFrameNumbers" },
          { 0x0040A138UL, "DSReferencedTimeOffsets" },
          { 0x0040A13AUL, "DTReferencedDateTime" },
          { 0x0040A160UL, "UTTextValue" },
          { 0x0040A168UL, "SQConceptCodeSequence" },
          { 0x0040A170UL, "SQPurposeOfReferenceCodeSequence" },
          { 0x0040A180UL, "USAnnotationGroupNumber" },
          { 0x0040A195UL, "SQModifierCodeSequence" },
          { 0x0040A300UL, "SQMeasuredValueSequence" },
          { 0x0040A301UL, "SQNumericValueQualifierCodeSequence" },
          { 0x0040A30AUL, "DSNumericValue" },
          { 0x0040A360UL, "SQPredecessorDocumentsSequence" },
          { 0x0040A370UL, "SQReferencedRequestSequence" },
          { 0x0040A372UL, "SQPerformedProcedureCodeSequence" },
          { 0x0040A375UL, "SQCurrentRequestedProcedureEvidenceSequence" },
          { 0x0040A385UL, "SQPertinentOtherEvidenceSequence" },
          { 0x0040A390UL, "SQHL7StructuredDocumentReferenceSequencealeSequenceTrial" },
          { 0x0040A491UL, "CSCompletionFlag" },
          { 0x0040A492UL, "LOCompletionFlagDescription" },
          { 0x0040A493UL, "CSVerificationFlag" },
          { 0x0040A494UL, "CSArchiveRequested" },
          { 0x0040A496UL, "CSPreliminaryFlag" },
          { 0x0040A504UL, "SQContentTemplateSequence" },
          { 0x0040A525UL, "SQIdenticalDocumentsSequencelagTrial" },
          { 0x0040A730UL, "SQContentSequenceenceTrialall" },
          { 0x0040B020UL, "SQWaveformAnnotationSequence" },
          { 0x0040DB00UL, "CSTemplateIdentifierionUID" },
          { 0x0040DB73UL, "ULReferencedContentItemIdentifier" },
          { 0x0040E001UL, "STHL7InstanceIdentifier" },
          { 0x0040E004UL, "DTHL7DocumentEffectiveTime" },
          { 0x0040E006UL, "SQHL7DocumentTypeCodeSequence" },
          { 0x0040E008UL, "SQDocumentClassCodeSequence" },
          { 0x0040E010UL, "UTRetrieveURI" },
          { 0x0040E011UL, "UIRetrieveLocationUID" },
          { 0x0040E020UL, "CSTypeOfInstances" },
          { 0x0040E021UL, "SQDICOMRetrievalSequence" },
          { 0x0040E022UL, "SQDICOMMediaRetrievalSequence" },
          { 0x0040E023UL, "SQWADORetrievalSequence" },
          { 0x0040E024UL, "SQXDSRetrievalSequence" },
          { 0x0040E030UL, "UIRepositoryUniqueID" },
          { 0x0040E031UL, "UIHomeCommunityID" },
          { 0x00420010UL, "STDocumentTitle" },
          { 0x00420011UL, "OBEncapsulatedDocument" },
          { 0x00420012UL, "LOMIMETypeOfEncapsulatedDocument" },
          { 0x00420013UL, "SQSourceInstanceSequence" },
          { 0x00420014UL, "LOListOfMIMETypes" },
          { 0x00440001UL, "STProductPackageIdentifier" },
          { 0x00440002UL, "CSSubstanceAdministrationApproval" },
          { 0x00440003UL, "LTApprovalStatusFurtherDescription" },
          { 0x00440004UL, "DTApprovalStatusDateTime" },
          { 0x00440007UL, "SQProductTypeCodeSequence" },
          { 0x00440008UL, "LOProductName" },
          { 0x00440009UL, "LTProductDescription" },
          { 0x0044000AUL, "LOProductLotIdentifier" },
          { 0x0044000BUL, "DTProductExpirationDateTime" },
          { 0x00440010UL, "DTSubstanceAdministrationDateTime" },
          { 0x00440011UL, "LOSubstanceAdministrationNotes" },
          { 0x00440012UL, "LOSubstanceAdministrationDeviceID" },
          { 0x00440013UL, "SQProductParameterSequence" },
          { 0x00440019UL, "SQSubstanceAdministrationParameterSequence" },
          { 0x00460012UL, "LOLensDescription" },
          { 0x00460014UL, "SQRightLensSequence" },
          { 0x00460015UL, "SQLeftLensSequence" },
          { 0x00460016UL, "SQUnspecifiedLateralityLensSequence" },
          { 0x00460018UL, "SQCylinderSequence" },
          { 0x00460028UL, "SQPrismSequence" },
          { 0x00460030UL, "FDHorizontalPrismPower" },
          { 0x00460032UL, "CSHorizontalPrismBase" },
          { 0x00460034UL, "FDVerticalPrismPower" },
          { 0x00460036UL, "CSVerticalPrismBase" },
          { 0x00460038UL, "CSLensSegmentType" },
          { 0x00460040UL, "FDOpticalTransmittance" },
          { 0x00460042UL, "FDChannelWidth" },
          { 0x00460044UL, "FDPupilSize" },
          { 0x00460046UL, "FDCornealSize" },
          { 0x00460050UL, "SQAutorefractionRightEyeSequence" },
          { 0x00460052UL, "SQAutorefractionLeftEyeSequence" },
          { 0x00460060UL, "FDDistancePupillaryDistance" },
          { 0x00460062UL, "FDNearPupillaryDistance" },
          { 0x00460063UL, "FDIntermediatePupillaryDistance" },
          { 0x00460064UL, "FDOtherPupillaryDistance" },
          { 0x00460070UL, "SQKeratometryRightEyeSequence" },
          { 0x00460071UL, "SQKeratometryLeftEyeSequence" },
          { 0x00460074UL, "SQSteepKeratometricAxisSequence" },
          { 0x00460075UL, "FDRadiusOfCurvature" },
          { 0x00460076UL, "FDKeratometricPower" },
          { 0x00460077UL, "FDKeratometricAxis" },
          { 0x00460080UL, "SQFlatKeratometricAxisSequence" },
          { 0x00460092UL, "CSBackgroundColor" },
          { 0x00460094UL, "CSOptotype" },
          { 0x00460095UL, "CSOptotypePresentation" },
          { 0x00460097UL, "SQSubjectiveRefractionRightEyeSequence" },
          { 0x00460098UL, "SQSubjectiveRefractionLeftEyeSequence" },
          { 0x00460100UL, "SQAddNearSequence" },
          { 0x00460101UL, "SQAddIntermediateSequence" },
          { 0x00460102UL, "SQAddOtherSequence" },
          { 0x00460104UL, "FDAddPower" },
          { 0x00460106UL, "FDViewingDistance" },
          { 0x00460121UL, "SQVisualAcuityTypeCodeSequence" },
          { 0x00460122UL, "SQVisualAcuityRightEyeSequence" },
          { 0x00460123UL, "SQVisualAcuityLeftEyeSequence" },
          { 0x00460124UL, "SQVisualAcuityBothEyesOpenSequence" },
          { 0x00460125UL, "CSViewingDistanceType" },
          { 0x00460135UL, "SSVisualAcuityModifiers" },
          { 0x00460137UL, "FDDecimalVisualAcuity" },
          { 0x00460139UL, "LOOptotypeDetailedDefinition" },
          { 0x00460145UL, "SQReferencedRefractiveMeasurementsSequence" },
          { 0x00460146UL, "FDSpherePower" },
          { 0x00460147UL, "FDCylinderPower" },
          { 0x00480001UL, "FLImagedVolumeWidth" },
          { 0x00480002UL, "FLImagedVolumeHeight" },
          { 0x00480003UL, "FLImagedVolumeDepth" },
          { 0x00480006UL, "ULTotalPixelMatrixColumns" },
          { 0x00480007UL, "ULTotalPixelMatrixRows" },
          { 0x00480008UL, "SQTotalPixelMatrixOriginSequence" },
          { 0x00480010UL, "CSSpecimenLabelInImage" },
          { 0x00480011UL, "CSFocusMethod" },
          { 0x00480012UL, "CSExtendedDepthOfField" },
          { 0x00480013UL, "USNumberOfFocalPlanes" },
          { 0x00480014UL, "FLDistanceBetweenFocalPlanes" },
          { 0x00480015UL, "USRecommendedAbsentPixelCIELabValue" },
          { 0x00480100UL, "SQIlluminatorTypeCodeSequence" },
          { 0x00480102UL, "DSImageOrientationSlide" },
          { 0x00480105UL, "SQOpticalPathSequence" },
          { 0x00480106UL, "SHOpticalPathIdentifier" },
          { 0x00480107UL, "STOpticalPathDescription" },
          { 0x00480108UL, "SQIlluminationColorCodeSequence" },
          { 0x00480110UL, "SQSpecimenReferenceSequence" },
          { 0x00480111UL, "DSCondenserLensPower" },
          { 0x00480112UL, "DSObjectiveLensPower" },
          { 0x00480113UL, "DSObjectiveLensNumericalAperture" },
          { 0x00480120UL, "SQPaletteColorLookupTableSequence" },
          { 0x00480200UL, "SQReferencedImageNavigationSequence" },
          { 0x00480201UL, "USTopLeftHandCornerOfLocalizerArea" },
          { 0x00480202UL, "USBottomRightHandCornerOfLocalizerArea" },
          { 0x00480207UL, "SQOpticalPathIdentificationSequence" },
          { 0x0048021AUL, "SQPlanePositionSlideSequence" },
          { 0x0048021EUL, "SLRowPositionInTotalImagePixelMatrix" },
          { 0x0048021FUL, "SLColumnPositionInTotalImagePixelMatrix" },
          { 0x00480301UL, "CSPixelOriginInterpretation" },
          { 0x00500004UL, "CSCalibrationImage" },
          { 0x00500010UL, "SQDeviceSequence" },
          { 0x00500012UL, "SQContainerComponentTypeCodeSequence" },
          { 0x00500013UL, "FDContainerComponentThickness" },
          { 0x00500014UL, "DSDeviceLength" },
          { 0x00500015UL, "FDContainerComponentWidth" },
          { 0x00500016UL, "DSDeviceDiameter" },
          { 0x00500017UL, "CSDeviceDiameterUnits" },
          { 0x00500018UL, "DSDeviceVolume" },
          { 0x00500019UL, "DSInterMarkerDistance" },
          { 0x0050001AUL, "CSContainerComponentMaterial" },
          { 0x0050001BUL, "LOContainerComponentID" },
          { 0x0050001CUL, "FDContainerComponentLength" },
          { 0x0050001DUL, "FDContainerComponentDiameter" },
          { 0x0050001EUL, "LOContainerComponentDescription" },
          { 0x00500020UL, "LODeviceDescription" },
          { 0x00520001UL, "FLContrastBolusIngredientPercentByVolume" },
          { 0x00520002UL, "FDOCTFocalDistance" },
          { 0x00520003UL, "FDBeamSpotSize" },
          { 0x00520004UL, "FDEffectiveRefractiveIndex" },
          { 0x00520006UL, "CSOCTAcquisitionDomain" },
          { 0x00520007UL, "FDOCTOpticalCenterWavelength" },
          { 0x00520008UL, "FDAxialResolution" },
          { 0x00520009UL, "FDRangingDepth" },
          { 0x00520011UL, "FDALineRate" },
          { 0x00520012UL, "USALinesPerFrame" },
          { 0x00520013UL, "FDCatheterRotationalRate" },
          { 0x00520014UL, "FDALinePixelSpacing" },
          { 0x00520016UL, "SQModeOfPercutaneousAccessSequence" },
          { 0x00520025UL, "SQIntravascularOCTFrameTypeSequence" },
          { 0x00520026UL, "CSOCTZOffsetApplied" },
          { 0x00520027UL, "SQIntravascularFrameContentSequence" },
          { 0x00520028UL, "FDIntravascularLongitudinalDistance" },
          { 0x00520029UL, "SQIntravascularOCTFrameContentSequence" },
          { 0x00520030UL, "SSOCTZOffsetCorrection" },
          { 0x00520031UL, "CSCatheterDirectionOfRotation" },
          { 0x00520033UL, "FDSeamLineLocation" },
          { 0x00520034UL, "FDFirstALineLocation" },
          { 0x00520036UL, "USSeamLineIndex" },
          { 0x00520038UL, "USNumberOfPaddedAlines" },
          { 0x00520039UL, "CSInterpolationType" },
          { 0x0052003AUL, "CSRefractiveIndexApplied" },
          { 0x00540010UL, "USEnergyWindowVector" },
          { 0x00540011UL, "USNumberOfEnergyWindows" },
          { 0x00540012UL, "SQEnergyWindowInformationSequence" },
          { 0x00540013UL, "SQEnergyWindowRangeSequence" },
          { 0x00540014UL, "DSEnergyWindowLowerLimit" },
          { 0x00540015UL, "DSEnergyWindowUpperLimit" },
          { 0x00540016UL, "SQRadiopharmaceuticalInformationSequence" },
          { 0x00540017UL, "ISResidualSyringeCounts" },
          { 0x00540018UL, "SHEnergyWindowName" },
          { 0x00540020UL, "USDetectorVector" },
          { 0x00540021UL, "USNumberOfDetectors" },
          { 0x00540022UL, "SQDetectorInformationSequence" },
          { 0x00540030UL, "USPhaseVector" },
          { 0x00540031UL, "USNumberOfPhases" },
          { 0x00540032UL, "SQPhaseInformationSequence" },
          { 0x00540033UL, "USNumberOfFramesInPhase" },
          { 0x00540036UL, "ISPhaseDelay" },
          { 0x00540038UL, "ISPauseBetweenFrames" },
          { 0x00540039UL, "CSPhaseDescription" },
          { 0x00540050UL, "USRotationVector" },
          { 0x00540051UL, "USNumberOfRotations" },
          { 0x00540052UL, "SQRotationInformationSequence" },
          { 0x00540053UL, "USNumberOfFramesInRotation" },
          { 0x00540060UL, "USRRIntervalVector" },
          { 0x00540061UL, "USNumberOfRRIntervals" },
          { 0x00540062UL, "SQGatedInformationSequence" },
          { 0x00540063UL, "SQDataInformationSequence" },
          { 0x00540070UL, "USTimeSlotVector" },
          { 0x00540071UL, "USNumberOfTimeSlots" },
          { 0x00540072UL, "SQTimeSlotInformationSequence" },
          { 0x00540073UL, "DSTimeSlotTime" },
          { 0x00540080UL, "USSliceVector" },
          { 0x00540081UL, "USNumberOfSlices" },
          { 0x00540090UL, "USAngularViewVector" },
          { 0x00540100UL, "USTimeSliceVector" },
          { 0x00540101UL, "USNumberOfTimeSlices" },
          { 0x00540200UL, "DSStartAngle" },
          { 0x00540202UL, "CSTypeOfDetectorMotion" },
          { 0x00540210UL, "ISTriggerVector" },
          { 0x00540211UL, "USNumberOfTriggersInPhase" },
          { 0x00540220UL, "SQViewCodeSequence" },
          { 0x00540222UL, "SQViewModifierCodeSequence" },
          { 0x00540300UL, "SQRadionuclideCodeSequence" },
          { 0x00540302UL, "SQAdministrationRouteCodeSequence" },
          { 0x00540304UL, "SQRadiopharmaceuticalCodeSequence" },
          { 0x00540306UL, "SQCalibrationDataSequence" },
          { 0x00540308UL, "USEnergyWindowNumber" },
          { 0x00540400UL, "SHImageID" },
          { 0x00540410UL, "SQPatientOrientationCodeSequence" },
          { 0x00540412UL, "SQPatientOrientationModifierCodeSequence" },
          { 0x00540414UL, "SQPatientGantryRelationshipCodeSequence" },
          { 0x00540500UL, "CSSliceProgressionDirection" },
          { 0x00541000UL, "CSSeriesType" },
          { 0x00541001UL, "CSUnits" },
          { 0x00541002UL, "CSCountsSource" },
          { 0x00541004UL, "CSReprojectionMethod" },
          { 0x00541006UL, "CSSUVType" },
          { 0x00541100UL, "CSRandomsCorrectionMethod" },
          { 0x00541101UL, "LOAttenuationCorrectionMethod" },
          { 0x00541102UL, "CSDecayCorrection" },
          { 0x00541103UL, "LOReconstructionMethod" },
          { 0x00541104UL, "LODetectorLinesOfResponseUsed" },
          { 0x00541105UL, "LOScatterCorrectionMethod" },
          { 0x00541200UL, "DSAxialAcceptance" },
          { 0x00541201UL, "ISAxialMash" },
          { 0x00541202UL, "ISTransverseMash" },
          { 0x00541203UL, "DSDetectorElementSize" },
          { 0x00541210UL, "DSCoincidenceWindowWidth" },
          { 0x00541220UL, "CSSecondaryCountsType" },
          { 0x00541300UL, "DSFrameReferenceTime" },
          { 0x00541310UL, "ISPrimaryPromptsCountsAccumulated" },
          { 0x00541311UL, "ISSecondaryCountsAccumulated" },
          { 0x00541320UL, "DSSliceSensitivityFactor" },
          { 0x00541321UL, "DSDecayFactor" },
          { 0x00541322UL, "DSDoseCalibrationFactor" },
          { 0x00541323UL, "DSScatterFractionFactor" },
          { 0x00541324UL, "DSDeadTimeFactor" },
          { 0x00541330UL, "USImageIndex" },
          { 0x00603000UL, "SQHistogramSequence" },
          { 0x00603002UL, "USHistogramNumberOfBins" },
          { 0x00603004UL, "USHistogramFirstBinValue" },
          { 0x00603006UL, "USHistogramLastBinValue" },
          { 0x00603008UL, "USHistogramBinWidth" },
          { 0x00603010UL, "LOHistogramExplanation" },
          { 0x00603020UL, "ULHistogramData" },
          { 0x00620001UL, "CSSegmentationType" },
          { 0x00620002UL, "SQSegmentSequence" },
          { 0x00620003UL, "SQSegmentedPropertyCategoryCodeSequence" },
          { 0x00620004UL, "USSegmentNumber" },
          { 0x00620005UL, "LOSegmentLabel" },
          { 0x00620006UL, "STSegmentDescription" },
          { 0x00620008UL, "CSSegmentAlgorithmType" },
          { 0x00620009UL, "LOSegmentAlgorithmName" },
          { 0x0062000AUL, "SQSegmentIdentificationSequence" },
          { 0x0062000BUL, "USReferencedSegmentNumber" },
          { 0x0062000CUL, "USRecommendedDisplayGrayscaleValue" },
          { 0x0062000DUL, "USRecommendedDisplayCIELabValue" },
          { 0x0062000EUL, "USMaximumFractionalValue" },
          { 0x0062000FUL, "SQSegmentedPropertyTypeCodeSequence" },
          { 0x00620010UL, "CSSegmentationFractionalType" },
          { 0x00640002UL, "SQDeformableRegistrationSequence" },
          { 0x00640003UL, "UISourceFrameOfReferenceUID" },
          { 0x00640005UL, "SQDeformableRegistrationGridSequence" },
          { 0x00640007UL, "ULGridDimensions" },
          { 0x00640008UL, "FDGridResolution" },
          { 0x00640009UL, "OFVectorGridData" },
          { 0x0064000FUL, "SQPreDeformationMatrixRegistrationSequence" },
          { 0x00640010UL, "SQPostDeformationMatrixRegistrationSequence" },
          { 0x00660001UL, "ULNumberOfSurfaces" },
          { 0x00660002UL, "SQSurfaceSequence" },
          { 0x00660003UL, "ULSurfaceNumber" },
          { 0x00660004UL, "LTSurfaceComments" },
          { 0x00660009UL, "CSSurfaceProcessing" },
          { 0x0066000AUL, "FLSurfaceProcessingRatio" },
          { 0x0066000BUL, "LOSurfaceProcessingDescription" },
          { 0x0066000CUL, "FLRecommendedPresentationOpacity" },
          { 0x0066000DUL, "CSRecommendedPresentationType" },
          { 0x0066000EUL, "CSFiniteVolume" },
          { 0x00660010UL, "CSManifold" },
          { 0x00660011UL, "SQSurfacePointsSequence" },
          { 0x00660012UL, "SQSurfacePointsNormalsSequence" },
          { 0x00660013UL, "SQSurfaceMeshPrimitivesSequence" },
          { 0x00660015UL, "ULNumberOfSurfacePoints" },
          { 0x00660016UL, "OFPointCoordinatesData" },
          { 0x00660017UL, "FLPointPositionAccuracy" },
          { 0x00660018UL, "FLMeanPointDistance" },
          { 0x00660019UL, "FLMaximumPointDistance" },
          { 0x0066001AUL, "FLPointsBoundingBoxCoordinates" },
          { 0x0066001BUL, "FLAxisOfRotation" },
          { 0x0066001CUL, "FLCenterOfRotation" },
          { 0x0066001EUL, "ULNumberOfVectors" },
          { 0x0066001FUL, "USVectorDimensionality" },
          { 0x00660020UL, "FLVectorAccuracy" },
          { 0x00660021UL, "OFVectorCoordinateData" },
          { 0x00660023UL, "OWTrianglePointIndexList" },
          { 0x00660024UL, "OWEdgePointIndexList" },
          { 0x00660025UL, "OWVertexPointIndexList" },
          { 0x00660026UL, "SQTriangleStripSequence" },
          { 0x00660027UL, "SQTriangleFanSequence" },
          { 0x00660028UL, "SQLineSequence" },
          { 0x00660029UL, "OWPrimitivePointIndexList" },
          { 0x0066002AUL, "ULSurfaceCount" },
          { 0x0066002BUL, "SQReferencedSurfaceSequence" },
          { 0x0066002CUL, "ULReferencedSurfaceNumber" },
          { 0x0066002DUL, "SQSegmentSurfaceGenerationAlgorithmIdentificationSequence" },
          { 0x0066002EUL, "SQSegmentSurfaceSourceInstanceSequence" },
          { 0x0066002FUL, "SQAlgorithmFamilyCodeSequence" },
          { 0x00660030UL, "SQAlgorithmNameCodeSequence" },
          { 0x00660031UL, "LOAlgorithmVersion" },
          { 0x00660032UL, "LTAlgorithmParameters" },
          { 0x00660034UL, "SQFacetSequence" },
          { 0x00660035UL, "SQSurfaceProcessingAlgorithmIdentificationSequence" },
          { 0x00660036UL, "LOAlgorithmName" },
          { 0x00686210UL, "LOImplantSize" },
          { 0x00686221UL, "LOImplantTemplateVersion" },
          { 0x00686222UL, "SQReplacedImplantTemplateSequence" },
          { 0x00686223UL, "CSImplantType" },
          { 0x00686224UL, "SQDerivationImplantTemplateSequence" },
          { 0x00686225UL, "SQOriginalImplantTemplateSequence" },
          { 0x00686226UL, "DTEffectiveDateTime" },
          { 0x00686230UL, "SQImplantTargetAnatomySequence" },
          { 0x00686260UL, "SQInformationFromManufacturerSequence" },
          { 0x00686265UL, "SQNotificationFromManufacturerSequencerSequence" },
          { 0x00686270UL, "DTInformationIssueDateTime" },
          { 0x00686280UL, "STInformationSummary" },
          { 0x006862A0UL, "SQImplantRegulatoryDisapprovalCodeSequence" },
          { 0x006862A5UL, "FDOverallTemplateSpatialTolerance" },
          { 0x006862C0UL, "SQHPGLDocumentSequence" },
          { 0x006862D0UL, "USHPGLDocumentID" },
          { 0x006862D5UL, "LOHPGLDocumentLabel" },
          { 0x006862E0UL, "SQViewOrientationCodeSequence" },
          { 0x006862F0UL, "FDViewOrientationModifier" },
          { 0x006862F2UL, "FDHPGLDocumentScaling" },
          { 0x00686300UL, "OBHPGLDocument" },
          { 0x00686310UL, "USHPGLContourPenNumber" },
          { 0x00686320UL, "SQHPGLPenSequence" },
          { 0x00686330UL, "USHPGLPenNumber" },
          { 0x00686340UL, "LOHPGLPenLabel" },
          { 0x00686345UL, "STHPGLPenDescription" },
          { 0x00686346UL, "FDRecommendedRotationPoint" },
          { 0x00686347UL, "FDBoundingRectangle" },
          { 0x00686350UL, "USImplantTemplate3DModelSurfaceNumber" },
          { 0x00686360UL, "SQSurfaceModelDescriptionSequence" },
          { 0x00686380UL, "LOSurfaceModelLabel" },
          { 0x00686390UL, "FDSurfaceModelScalingFactor" },
          { 0x006863A0UL, "SQMaterialsCodeSequence" },
          { 0x006863A4UL, "SQCoatingMaterialsCodeSequence" },
          { 0x006863A8UL, "SQImplantTypeCodeSequence" },
          { 0x006863ACUL, "SQFixationMethodCodeSequence" },
          { 0x006863B0UL, "SQMatingFeatureSetsSequence" },
          { 0x006863C0UL, "USMatingFeatureSetID" },
          { 0x006863D0UL, "LOMatingFeatureSetLabel" },
          { 0x006863E0UL, "SQMatingFeatureSequence" },
          { 0x006863F0UL, "USMatingFeatureID" },
          { 0x00686400UL, "SQMatingFeatureDegreeOfFreedomSequence" },
          { 0x00686410UL, "USDegreeOfFreedomID" },
          { 0x00686420UL, "CSDegreeOfFreedomType" },
          { 0x00686430UL, "SQTwoDMatingFeatureCoordinatesSequence" },
          { 0x00686440UL, "USReferencedHPGLDocumentID" },
          { 0x00686450UL, "FDTwoDMatingPoint" },
          { 0x00686460UL, "FDTwoDMatingAxes" },
          { 0x00686470UL, "SQTwoDDegreeOfFreedomSequence" },
          { 0x00686490UL, "FDThreeDDegreeOfFreedomAxis" },
          { 0x006864A0UL, "FDRangeOfFreedom" },
          { 0x006864C0UL, "FDThreeDMatingPoint" },
          { 0x006864D0UL, "FDThreeDMatingAxes" },
          { 0x006864F0UL, "FDTwoDDegreeOfFreedomAxis" },
          { 0x00686500UL, "SQPlanningLandmarkPointSequence" },
          { 0x00686510UL, "SQPlanningLandmarkLineSequence" },
          { 0x00686520UL, "SQPlanningLandmarkPlaneSequence" },
          { 0x00686530UL, "USPlanningLandmarkID" },
          { 0x00686540UL, "LOPlanningLandmarkDescription" },
          { 0x00686545UL, "SQPlanningLandmarkIdentificationCodeSequence" },
          { 0x00686550UL, "SQTwoDPointCoordinatesSequence" },
          { 0x00686560UL, "FDTwoDPointCoordinates" },
          { 0x00686590UL, "FDThreeDPointCoordinates" },
          { 0x006865A0UL, "SQTwoDLineCoordinatesSequence" },
          { 0x006865B0UL, "FDTwoDLineCoordinates" },
          { 0x006865D0UL, "FDThreeDLineCoordinates" },
          { 0x006865E0UL, "SQTwoDPlaneCoordinatesSequence" },
          { 0x006865F0UL, "FDTwoDPlaneIntersection" },
          { 0x00686610UL, "FDThreeDPlaneOrigin" },
          { 0x00686620UL, "FDThreeDPlaneNormal" },
          { 0x00700001UL, "SQGraphicAnnotationSequence" },
          { 0x00700002UL, "CSGraphicLayer" },
          { 0x00700003UL, "CSBoundingBoxAnnotationUnits" },
          { 0x00700004UL, "CSAnchorPointAnnotationUnits" },
          { 0x00700005UL, "CSGraphicAnnotationUnits" },
          { 0x00700006UL, "STUnformattedTextValue" },
          { 0x00700008UL, "SQTextObjectSequence" },
          { 0x00700009UL, "SQGraphicObjectSequence" },
          { 0x00700010UL, "FLBoundingBoxTopLeftHandCorner" },
          { 0x00700011UL, "FLBoundingBoxBottomRightHandCorner" },
          { 0x00700012UL, "CSBoundingBoxTextHorizontalJustification" },
          { 0x00700014UL, "FLAnchorPoint" },
          { 0x00700015UL, "CSAnchorPointVisibility" },
          { 0x00700020UL, "USGraphicDimensions" },
          { 0x00700021UL, "USNumberOfGraphicPoints" },
          { 0x00700022UL, "FLGraphicData" },
          { 0x00700023UL, "CSGraphicType" },
          { 0x00700024UL, "CSGraphicFilled" },
          { 0x00700041UL, "CSImageHorizontalFlip" },
          { 0x00700042UL, "USImageRotationCornerTrialandCornerTrial" },
          { 0x00700052UL, "SLDisplayedAreaTopLeftHandCorner" },
          { 0x00700053UL, "SLDisplayedAreaBottomRightHandCorner" },
          { 0x0070005AUL, "SQDisplayedAreaSelectionSequence" },
          { 0x00700060UL, "SQGraphicLayerSequence" },
          { 0x00700062UL, "ISGraphicLayerOrder" },
          { 0x00700066UL, "USGraphicLayerRecommendedDisplayGrayscaleValueDisplayRGBValue" },
          { 0x00700068UL, "LOGraphicLayerDescription" },
          { 0x00700080UL, "CSContentLabel" },
          { 0x00700081UL, "LOContentDescription" },
          { 0x00700082UL, "DAPresentationCreationDate" },
          { 0x00700083UL, "TMPresentationCreationTime" },
          { 0x00700084UL, "PNContentCreatorName" },
          { 0x00700086UL, "SQContentCreatorIdentificationCodeSequence" },
          { 0x00700087UL, "SQAlternateContentDescriptionSequence" },
          { 0x00700100UL, "CSPresentationSizeMode" },
          { 0x00700101UL, "DSPresentationPixelSpacing" },
          { 0x00700102UL, "ISPresentationPixelAspectRatio" },
          { 0x00700103UL, "FLPresentationPixelMagnificationRatio" },
          { 0x00700207UL, "LOGraphicGroupLabel" },
          { 0x00700208UL, "STGraphicGroupDescription" },
          { 0x00700209UL, "SQCompoundGraphicSequence" },
          { 0x00700226UL, "ULCompoundGraphicInstanceID" },
          { 0x00700227UL, "LOFontName" },
          { 0x00700228UL, "CSFontNameType" },
          { 0x00700229UL, "LOCSSFontName" },
          { 0x00700230UL, "FDRotationAngle" },
          { 0x00700231UL, "SQTextStyleSequence" },
          { 0x00700232UL, "SQLineStyleSequence" },
          { 0x00700233UL, "SQFillStyleSequence" },
          { 0x00700234UL, "SQGraphicGroupSequence" },
          { 0x00700241UL, "USTextColorCIELabValue" },
          { 0x00700242UL, "CSHorizontalAlignment" },
          { 0x00700243UL, "CSVerticalAlignment" },
          { 0x00700244UL, "CSShadowStyle" },
          { 0x00700245UL, "FLShadowOffsetX" },
          { 0x00700246UL, "FLShadowOffsetY" },
          { 0x00700247UL, "USShadowColorCIELabValue" },
          { 0x00700248UL, "CSUnderlined" },
          { 0x00700249UL, "CSBold" },
          { 0x00700250UL, "CSItalic" },
          { 0x00700251UL, "USPatternOnColorCIELabValue" },
          { 0x00700252UL, "USPatternOffColorCIELabValue" },
          { 0x00700253UL, "FLLineThickness" },
          { 0x00700254UL, "CSLineDashingStyle" },
          { 0x00700255UL, "ULLinePattern" },
          { 0x00700256UL, "OBFillPattern" },
          { 0x00700257UL, "CSFillMode" },
          { 0x00700258UL, "FLShadowOpacity" },
          { 0x00700261UL, "FLGapLength" },
          { 0x00700262UL, "FLDiameterOfVisibility" },
          { 0x00700273UL, "FLRotationPoint" },
          { 0x00700274UL, "CSTickAlignment" },
          { 0x00700278UL, "CSShowTickLabel" },
          { 0x00700279UL, "CSTickLabelAlignment" },
          { 0x00700282UL, "CSCompoundGraphicUnits" },
          { 0x00700284UL, "FLPatternOnOpacity" },
          { 0x00700285UL, "FLPatternOffOpacity" },
          { 0x00700287UL, "SQMajorTicksSequence" },
          { 0x00700288UL, "FLTickPosition" },
          { 0x00700289UL, "SHTickLabel" },
          { 0x00700294UL, "CSCompoundGraphicType" },
          { 0x00700295UL, "ULGraphicGroupID" },
          { 0x00700306UL, "CSShapeType" },
          { 0x00700308UL, "SQRegistrationSequence" },
          { 0x00700309UL, "SQMatrixRegistrationSequence" },
          { 0x0070030AUL, "SQMatrixSequence" },
          { 0x0070030CUL, "CSFrameOfReferenceTransformationMatrixType" },
          { 0x0070030DUL, "SQRegistrationTypeCodeSequence" },
          { 0x0070030FUL, "STFiducialDescription" },
          { 0x00700310UL, "SHFiducialIdentifier" },
          { 0x00700311UL, "SQFiducialIdentifierCodeSequence" },
          { 0x00700312UL, "FDContourUncertaintyRadius" },
          { 0x00700314UL, "SQUsedFiducialsSequence" },
          { 0x00700318UL, "SQGraphicCoordinatesDataSequence" },
          { 0x0070031AUL, "UIFiducialUID" },
          { 0x0070031CUL, "SQFiducialSetSequence" },
          { 0x0070031EUL, "SQFiducialSequence" },
          { 0x00700401UL, "USGraphicLayerRecommendedDisplayCIELabValue" },
          { 0x00700402UL, "SQBlendingSequence" },
          { 0x00700403UL, "FLRelativeOpacity" },
          { 0x00700404UL, "SQReferencedSpatialRegistrationSequence" },
          { 0x00700405UL, "CSBlendingPosition" },
          { 0x00720002UL, "SHHangingProtocolName" },
          { 0x00720004UL, "LOHangingProtocolDescription" },
          { 0x00720006UL, "CSHangingProtocolLevel" },
          { 0x00720008UL, "LOHangingProtocolCreator" },
          { 0x0072000AUL, "DTHangingProtocolCreationDateTime" },
          { 0x0072000CUL, "SQHangingProtocolDefinitionSequence" },
          { 0x0072000EUL, "SQHangingProtocolUserIdentificationCodeSequence" },
          { 0x00720010UL, "LOHangingProtocolUserGroupName" },
          { 0x00720012UL, "SQSourceHangingProtocolSequence" },
          { 0x00720014UL, "USNumberOfPriorsReferenced" },
          { 0x00720020UL, "SQImageSetsSequence" },
          { 0x00720022UL, "SQImageSetSelectorSequence" },
          { 0x00720024UL, "CSImageSetSelectorUsageFlag" },
          { 0x00720026UL, "ATSelectorAttribute" },
          { 0x00720028UL, "USSelectorValueNumber" },
          { 0x00720030UL, "SQTimeBasedImageSetsSequence" },
          { 0x00720032UL, "USImageSetNumber" },
          { 0x00720034UL, "CSImageSetSelectorCategory" },
          { 0x00720038UL, "USRelativeTime" },
          { 0x0072003AUL, "CSRelativeTimeUnits" },
          { 0x0072003CUL, "SSAbstractPriorValue" },
          { 0x0072003EUL, "SQAbstractPriorCodeSequence" },
          { 0x00720040UL, "LOImageSetLabel" },
          { 0x00720050UL, "CSSelectorAttributeVR" },
          { 0x00720052UL, "ATSelectorSequencePointer" },
          { 0x00720054UL, "LOSelectorSequencePointerPrivateCreator" },
          { 0x00720056UL, "LOSelectorAttributePrivateCreator" },
          { 0x00720060UL, "ATSelectorATValue" },
          { 0x00720062UL, "CSSelectorCSValue" },
          { 0x00720064UL, "ISSelectorISValue" },
          { 0x00720066UL, "LOSelectorLOValue" },
          { 0x00720068UL, "LTSelectorLTValue" },
          { 0x0072006AUL, "PNSelectorPNValue" },
          { 0x0072006CUL, "SHSelectorSHValue" },
          { 0x0072006EUL, "STSelectorSTValue" },
          { 0x00720070UL, "UTSelectorUTValue" },
          { 0x00720072UL, "DSSelectorDSValue" },
          { 0x00720074UL, "FDSelectorFDValue" },
          { 0x00720076UL, "FLSelectorFLValue" },
          { 0x00720078UL, "ULSelectorULValue" },
          { 0x0072007AUL, "USSelectorUSValue" },
          { 0x0072007CUL, "SLSelectorSLValue" },
          { 0x0072007EUL, "SSSelectorSSValue" },
          { 0x00720080UL, "SQSelectorCodeSequenceValue" },
          { 0x00720100UL, "USNumberOfScreens" },
          { 0x00720102UL, "SQNominalScreenDefinitionSequence" },
          { 0x00720104UL, "USNumberOfVerticalPixels" },
          { 0x00720106UL, "USNumberOfHorizontalPixels" },
          { 0x00720108UL, "FDDisplayEnvironmentSpatialPosition" },
          { 0x0072010AUL, "USScreenMinimumGrayscaleBitDepth" },
          { 0x0072010CUL, "USScreenMinimumColorBitDepth" },
          { 0x0072010EUL, "USApplicationMaximumRepaintTime" },
          { 0x00720200UL, "SQDisplaySetsSequence" },
          { 0x00720202UL, "USDisplaySetNumber" },
          { 0x00720203UL, "LODisplaySetLabel" },
          { 0x00720204UL, "USDisplaySetPresentationGroup" },
          { 0x00720206UL, "LODisplaySetPresentationGroupDescription" },
          { 0x00720208UL, "CSPartialDataDisplayHandling" },
          { 0x00720210UL, "SQSynchronizedScrollingSequence" },
          { 0x00720212UL, "USDisplaySetScrollingGroup" },
          { 0x00720214UL, "SQNavigationIndicatorSequence" },
          { 0x00720216UL, "USNavigationDisplaySet" },
          { 0x00720218UL, "USReferenceDisplaySets" },
          { 0x00720300UL, "SQImageBoxesSequence" },
          { 0x00720302UL, "USImageBoxNumber" },
          { 0x00720304UL, "CSImageBoxLayoutType" },
          { 0x00720306UL, "USImageBoxTileHorizontalDimension" },
          { 0x00720308UL, "USImageBoxTileVerticalDimension" },
          { 0x00720310UL, "CSImageBoxScrollDirection" },
          { 0x00720312UL, "CSImageBoxSmallScrollType" },
          { 0x00720314UL, "USImageBoxSmallScrollAmount" },
          { 0x00720316UL, "CSImageBoxLargeScrollType" },
          { 0x00720318UL, "USImageBoxLargeScrollAmount" },
          { 0x00720320UL, "USImageBoxOverlapPriority" },
          { 0x00720330UL, "FDCineRelativeToRealTime" },
          { 0x00720400UL, "SQFilterOperationsSequence" },
          { 0x00720402UL, "CSFilterByCategory" },
          { 0x00720404UL, "CSFilterByAttributePresence" },
          { 0x00720406UL, "CSFilterByOperator" },
          { 0x00720420UL, "USStructuredDisplayBackgroundCIELabValue" },
          { 0x00720421UL, "USEmptyImageBoxCIELabValue" },
          { 0x00720422UL, "SQStructuredDisplayImageBoxSequence" },
          { 0x00720424UL, "SQStructuredDisplayTextBoxSequence" },
          { 0x00720427UL, "SQReferencedFirstFrameSequence" },
          { 0x00720430UL, "SQImageBoxSynchronizationSequence" },
          { 0x00720432UL, "USSynchronizedImageBoxList" },
          { 0x00720434UL, "CSTypeOfSynchronization" },
          { 0x00720500UL, "CSBlendingOperationType" },
          { 0x00720510UL, "CSReformattingOperationType" },
          { 0x00720512UL, "FDReformattingThickness" },
          { 0x00720514UL, "FDReformattingInterval" },
          { 0x00720516UL, "CSReformattingOperationInitialViewDirection" },
          { 0x00720520UL, "CSThreeDRenderingType" },
          { 0x00720600UL, "SQSortingOperationsSequence" },
          { 0x00720602UL, "CSSortByCategory" },
          { 0x00720604UL, "CSSortingDirection" },
          { 0x00720700UL, "CSDisplaySetPatientOrientation" },
          { 0x00720702UL, "CSVOIType" },
          { 0x00720704UL, "CSPseudoColorType" },
          { 0x00720705UL, "SQPseudoColorPaletteInstanceReferenceSequence" },
          { 0x00720706UL, "CSShowGrayscaleInverted" },
          { 0x00720710UL, "CSShowImageTrueSizeFlag" },
          { 0x00720712UL, "CSShowGraphicAnnotationFlag" },
          { 0x00720714UL, "CSShowPatientDemographicsFlag" },
          { 0x00720716UL, "CSShowAcquisitionTechniquesFlag" },
          { 0x00720717UL, "CSDisplaySetHorizontalJustification" },
          { 0x00720718UL, "CSDisplaySetVerticalJustification" },
          { 0x00740120UL, "FDContinuationStartMeterset" },
          { 0x00740121UL, "FDContinuationEndMeterset" },
          { 0x00741000UL, "CSProcedureStepState" },
          { 0x00741002UL, "SQProcedureStepProgressInformationSequence" },
          { 0x00741004UL, "DSProcedureStepProgress" },
          { 0x00741006UL, "STProcedureStepProgressDescription" },
          { 0x00741008UL, "SQProcedureStepCommunicationsURISequence" },
          { 0x0074100aUL, "STContactURI" },
          { 0x0074100cUL, "LOContactDisplayName" },
          { 0x0074100eUL, "SQProcedureStepDiscontinuationReasonCodeSequence" },
          { 0x00741020UL, "SQBeamTaskSequence" },
          { 0x00741022UL, "CSBeamTaskType" },
          { 0x00741026UL, "FDTableTopVerticalAdjustedPosition" },
          { 0x00741027UL, "FDTableTopLongitudinalAdjustedPosition" },
          { 0x00741028UL, "FDTableTopLateralAdjustedPosition" },
          { 0x0074102AUL, "FDPatientSupportAdjustedAngle" },
          { 0x0074102BUL, "FDTableTopEccentricAdjustedAngle" },
          { 0x0074102CUL, "FDTableTopPitchAdjustedAngle" },
          { 0x0074102DUL, "FDTableTopRollAdjustedAngle" },
          { 0x00741030UL, "SQDeliveryVerificationImageSequence" },
          { 0x00741032UL, "CSVerificationImageTiming" },
          { 0x00741034UL, "CSDoubleExposureFlag" },
          { 0x00741036UL, "CSDoubleExposureOrdering" },
          { 0x00741040UL, "RelatedReferenceRTImageSequenc" },
          { 0x00741042UL, "SQGeneralMachineVerificationSequence" },
          { 0x00741044UL, "SQConventionalMachineVerificationSequence" },
          { 0x00741046UL, "SQIonMachineVerificationSequence" },
          { 0x00741048UL, "SQFailedAttributesSequence" },
          { 0x0074104AUL, "SQOverriddenAttributesSequence" },
          { 0x0074104CUL, "SQConventionalControlPointVerificationSequence" },
          { 0x0074104EUL, "SQIonControlPointVerificationSequence" },
          { 0x00741050UL, "SQAttributeOccurrenceSequence" },
          { 0x00741052UL, "ATAttributeOccurrencePointer" },
          { 0x00741054UL, "ULAttributeItemSelector" },
          { 0x00741056UL, "LOAttributeOccurrencePrivateCreator" },
          { 0x00741057UL, "ISSelectorSequencePointerItems" },
          { 0x00741200UL, "CSScheduledProcedureStepPriority" },
          { 0x00741202UL, "LOWorklistLabel" },
          { 0x00741204UL, "LOProcedureStepLabel" },
          { 0x00741210UL, "SQScheduledProcessingParametersSequence" },
          { 0x00741212UL, "SQPerformedProcessingParametersSequence" },
          { 0x00741216UL, "SQUnifiedProcedureStepPerformedProcedureSequence" },
          { 0x00741224UL, "SQReplacedProcedureStepSequence" },
          { 0x00741230UL, "LODeletionLock" },
          { 0x00741234UL, "AEReceivingAE" },
          { 0x00741236UL, "AERequestingAE" },
          { 0x00741238UL, "LTReasonForCancellation" },
          { 0x00741242UL, "CSSCPStatus" },
          { 0x00741244UL, "CSSubscriptionListStatus" },
          { 0x00741246UL, "CSUnifiedProcedureStepListStatus" },
          { 0x00741324UL, "ULBeamOrderIndex" },
          { 0x00741338UL, "FDDoubleExposureMeterset" },
          { 0x0074133AUL, "FDDoubleExposureFieldDelta" },
          { 0x00760001UL, "LOImplantAssemblyTemplateName" },
          { 0x00760003UL, "LOImplantAssemblyTemplateIssuer" },
          { 0x00760006UL, "LOImplantAssemblyTemplateVersion" },
          { 0x00760008UL, "SQReplacedImplantAssemblyTemplateSequence" },
          { 0x0076000AUL, "CSImplantAssemblyTemplateType" },
          { 0x0076000CUL, "SQOriginalImplantAssemblyTemplateSequence" },
          { 0x0076000EUL, "SQDerivationImplantAssemblyTemplateSequence" },
          { 0x00760010UL, "SQImplantAssemblyTemplateTargetAnatomySequence" },
          { 0x00760020UL, "SQProcedureTypeCodeSequence" },
          { 0x00760030UL, "LOSurgicalTechnique" },
          { 0x00760032UL, "SQComponentTypesSequence" },
          { 0x00760034UL, "CSComponentTypeCodeSequence" },
          { 0x00760036UL, "CSExclusiveComponentType" },
          { 0x00760038UL, "CSMandatoryComponentType" },
          { 0x00760040UL, "SQComponentSequence" },
          { 0x00760055UL, "USComponentID" },
          { 0x00760060UL, "SQComponentAssemblySequence" },
          { 0x00760070UL, "USComponent1ReferencedID" },
          { 0x00760080UL, "USComponent1ReferencedMatingFeatureSetID" },
          { 0x00760090UL, "USComponent1ReferencedMatingFeatureID" },
          { 0x007600A0UL, "USComponent2ReferencedID" },
          { 0x007600B0UL, "USComponent2ReferencedMatingFeatureSetID" },
          { 0x007600C0UL, "USComponent2ReferencedMatingFeatureID" },
          { 0x00780001UL, "LOImplantTemplateGroupName" },
          { 0x00780010UL, "STImplantTemplateGroupDescription" },
          { 0x00780020UL, "LOImplantTemplateGroupIssuer" },
          { 0x00780024UL, "LOImplantTemplateGroupVersion" },
          { 0x00780026UL, "SQReplacedImplantTemplateGroupSequence" },
          { 0x00780028UL, "SQImplantTemplateGroupTargetAnatomySequence" },
          { 0x0078002AUL, "SQImplantTemplateGroupMembersSequence" },
          { 0x0078002EUL, "USImplantTemplateGroupMemberID" },
          { 0x00780050UL, "FDThreeDImplantTemplateGroupMemberMatchingPoint" },
          { 0x00780060UL, "FDThreeDImplantTemplateGroupMemberMatchingAxes" },
          { 0x00780070UL, "SQImplantTemplateGroupMemberMatching2DCoordinatesSequence" },
          { 0x00780090UL, "FDTwoDImplantTemplateGroupMemberMatchingPoint" },
          { 0x007800A0UL, "FDTwoDImplantTemplateGroupMemberMatchingAxes" },
          { 0x007800B0UL, "SQImplantTemplateGroupVariationDimensionSequence" },
          { 0x007800B2UL, "LOImplantTemplateGroupVariationDimensionName" },
          { 0x007800B4UL, "SQImplantTemplateGroupVariationDimensionRankSequence" },
          { 0x007800B6UL, "USReferencedImplantTemplateGroupMemberID" },
          { 0x007800B8UL, "USImplantTemplateGroupVariationDimensionRank" },
          { 0x00880130UL, "SHStorageMediaFileSetID" },
          { 0x00880140UL, "UIStorageMediaFileSetUID" },
          { 0x00880200UL, "SQIconImageSequence" },
          { 0x01000410UL, "CSSOPInstanceStatus" },
          { 0x01000420UL, "DTSOPAuthorizationDateTime" },
          { 0x01000424UL, "LTSOPAuthorizationComment" },
          { 0x01000426UL, "LOAuthorizationEquipmentCertificationNumber" },
          { 0x04000005UL, "USMACIDNumber" },
          { 0x04000010UL, "UIMACCalculationTransferSyntaxUID" },
          { 0x04000015UL, "CSMACAlgorithm" },
          { 0x04000020UL, "ATDataElementsSigned" },
          { 0x04000100UL, "UIDigitalSignatureUID" },
          { 0x04000105UL, "DTDigitalSignatureDateTime" },
          { 0x04000110UL, "CSCertificateType" },
          { 0x04000115UL, "OBCertificateOfSigner" },
          { 0x04000120UL, "OBSignature" },
          { 0x04000305UL, "CSCertifiedTimestampType" },
          { 0x04000310UL, "OBCertifiedTimestamp" },
          { 0x04000401UL, "SQDigitalSignaturePurposeCodeSequence" },
          { 0x04000402UL, "SQReferencedDigitalSignatureSequence" },
          { 0x04000403UL, "SQReferencedSOPInstanceMACSequence" },
          { 0x04000404UL, "OBMAC" },
          { 0x04000500UL, "SQEncryptedAttributesSequence" },
          { 0x04000510UL, "UIEncryptedContentTransferSyntaxUID" },
          { 0x04000520UL, "OBEncryptedContent" },
          { 0x04000550UL, "SQModifiedAttributesSequence" },
          { 0x04000561UL, "SQOriginalAttributesSequence" },
          { 0x04000562UL, "DTAttributeModificationDateTime" },
          { 0x04000563UL, "LOModifyingSystem" },
          { 0x04000564UL, "LOSourceOfPreviousValues" },
          { 0x04000565UL, "CSReasonForTheAttributeModification" },
          { 0x20000010UL, "ISNumberOfCopies" },
          { 0x2000001EUL, "SQPrinterConfigurationSequence" },
          { 0x20000020UL, "CSPrintPriority" },
          { 0x20000030UL, "CSMediumType" },
          { 0x20000040UL, "CSFilmDestination" },
          { 0x20000050UL, "LOFilmSessionLabel" },
          { 0x20000060UL, "ISMemoryAllocation" },
          { 0x20000061UL, "ISMaximumMemoryAllocationag" },
          { 0x200000A0UL, "USMemoryBitDepth" },
          { 0x200000A1UL, "USPrintingBitDepth" },
          { 0x200000A2UL, "SQMediaInstalledSequence" },
          { 0x200000A4UL, "SQOtherMediaAvailableSequence" },
          { 0x200000A8UL, "SQSupportedImageDisplayFormatsSequence" },
          { 0x20000500UL, "SQReferencedFilmBoxSequenceence" },
          { 0x20100010UL, "STImageDisplayFormat" },
          { 0x20100030UL, "CSAnnotationDisplayFormatID" },
          { 0x20100040UL, "CSFilmOrientation" },
          { 0x20100050UL, "CSFilmSizeID" },
          { 0x20100052UL, "CSPrinterResolutionID" },
          { 0x20100054UL, "CSDefaultPrinterResolutionID" },
          { 0x20100060UL, "CSMagnificationType" },
          { 0x20100080UL, "CSSmoothingType" },
          { 0x201000A6UL, "CSDefaultMagnificationType" },
          { 0x201000A7UL, "CSOtherMagnificationTypesAvailable" },
          { 0x201000A8UL, "CSDefaultSmoothingType" },
          { 0x201000A9UL, "CSOtherSmoothingTypesAvailable" },
          { 0x20100100UL, "CSBorderDensity" },
          { 0x20100110UL, "CSEmptyImageDensity" },
          { 0x20100120UL, "USMinDensity" },
          { 0x20100130UL, "USMaxDensity" },
          { 0x20100140UL, "CSTrim" },
          { 0x20100150UL, "STConfigurationInformation" },
          { 0x20100152UL, "LTConfigurationInformationDescription" },
          { 0x20100154UL, "ISMaximumCollatedFilms" },
          { 0x2010015EUL, "USIllumination" },
          { 0x20100160UL, "USReflectedAmbientLight" },
          { 0x20100376UL, "DSPrinterPixelSpacing" },
          { 0x20100500UL, "SQReferencedFilmSessionSequence" },
          { 0x20100510UL, "SQReferencedImageBoxSequence" },
          { 0x20100520UL, "SQReferencedBasicAnnotationBoxSequence" },
          { 0x20200010UL, "USImageBoxPosition" },
          { 0x20200020UL, "CSPolarity" },
          { 0x20200030UL, "DSRequestedImageSize" },
          { 0x20200040UL, "CSRequestedDecimateCropBehavior" },
          { 0x20200050UL, "CSRequestedResolutionID" },
          { 0x202000A0UL, "CSRequestedImageSizeFlag" },
          { 0x202000A2UL, "CSDecimateCropResult" },
          { 0x20200110UL, "SQBasicGrayscaleImageSequence" },
          { 0x20200111UL, "SQBasicColorImageSequencexSequenceence" },
          { 0x20300010UL, "USAnnotationPosition" },
          { 0x20300020UL, "LOTextString" },
          { 0x20500010UL, "SQPresentationLUTSequence" },
          { 0x20500020UL, "CSPresentationLUTShape" },
          { 0x20500500UL, "SQReferencedPresentationLUTSequence" },
          { 0x21000020UL, "CSExecutionStatus" },
          { 0x21000030UL, "CSExecutionStatusInfo" },
          { 0x21000040UL, "DACreationDate" },
          { 0x21000050UL, "TMCreationTime" },
          { 0x21000070UL, "AEOriginator" },
          { 0x21000160UL, "SHOwnerID" },
          { 0x21000170UL, "ISNumberOfFilmsePullStoredPrint" },
          { 0x21100010UL, "CSPrinterStatus" },
          { 0x21100020UL, "CSPrinterStatusInfo" },
          { 0x21100030UL, "LOPrinterNamesSequence" },
          { 0x22000001UL, "CSLabelUsingInformationExtractedFromInstances" },
          { 0x22000002UL, "UTLabelText" },
          { 0x22000003UL, "CSLabelStyleSelection" },
          { 0x22000004UL, "LTMediaDisposition" },
          { 0x22000005UL, "LTBarcodeValue" },
          { 0x22000006UL, "CSBarcodeSymbology" },
          { 0x22000007UL, "CSAllowMediaSplitting" },
          { 0x22000008UL, "CSIncludeNonDICOMObjects" },
          { 0x22000009UL, "CSIncludeDisplayApplication" },
          { 0x2200000AUL, "CSPreserveCompositeInstancesAfterMediaCreation" },
          { 0x2200000BUL, "USTotalNumberOfPiecesOfMediaCreated" },
          { 0x2200000CUL, "LORequestedMediaApplicationProfile" },
          { 0x2200000DUL, "SQReferencedStorageMediaSequence" },
          { 0x2200000EUL, "ATFailureAttributes" },
          { 0x2200000FUL, "CSAllowLossyCompression" },
          { 0x22000020UL, "CSRequestPriority" },
          { 0x30020002UL, "SHRTImageLabel" },
          { 0x30020003UL, "LORTImageName" },
          { 0x30020004UL, "STRTImageDescription" },
          { 0x3002000AUL, "CSReportedValuesOrigin" },
          { 0x3002000CUL, "CSRTImagePlane" },
          { 0x3002000DUL, "DSXRayImageReceptorTranslation" },
          { 0x3002000EUL, "DSXRayImageReceptorAngle" },
          { 0x30020010UL, "DSRTImageOrientation" },
          { 0x30020011UL, "DSImagePlanePixelSpacing" },
          { 0x30020012UL, "DSRTImagePosition" },
          { 0x30020020UL, "SHRadiationMachineName" },
          { 0x30020022UL, "DSRadiationMachineSAD" },
          { 0x30020024UL, "DSRadiationMachineSSD" },
          { 0x30020026UL, "DSRTImageSID" },
          { 0x30020028UL, "DSSourceToReferenceObjectDistance" },
          { 0x30020029UL, "ISFractionNumber" },
          { 0x30020030UL, "SQExposureSequence" },
          { 0x30020032UL, "DSMetersetExposure" },
          { 0x30020034UL, "DSDiaphragmPosition" },
          { 0x30020040UL, "SQFluenceMapSequence" },
          { 0x30020041UL, "CSFluenceDataSource" },
          { 0x30020042UL, "DSFluenceDataScalePrimaryFluenceModeSeque" },
          { 0x30020050UL, "nceSQ" },
          { 0x30020051UL, "CSFluenceMode" },
          { 0x30020052UL, "SHFluenceModeID" },
          { 0x30040001UL, "CSDVHType" },
          { 0x30040002UL, "CSDoseUnits" },
          { 0x30040004UL, "CSDoseType" },
          { 0x30040006UL, "LODoseComment" },
          { 0x30040008UL, "DSNormalizationPoint" },
          { 0x3004000AUL, "CSDoseSummationType" },
          { 0x3004000CUL, "DSGridFrameOffsetVector" },
          { 0x3004000EUL, "DSDoseGridScaling" },
          { 0x30040010UL, "SQRTDoseROISequence" },
          { 0x30040012UL, "DSDoseValue" },
          { 0x30040014UL, "CSTissueHeterogeneityCorrection" },
          { 0x30040040UL, "DSDVHNormalizationPoint" },
          { 0x30040042UL, "DSDVHNormalizationDoseValue" },
          { 0x30040050UL, "SQDVHSequence" },
          { 0x30040052UL, "DSDVHDoseScaling" },
          { 0x30040054UL, "CSDVHVolumeUnits" },
          { 0x30040056UL, "ISDVHNumberOfBins" },
          { 0x30040058UL, "DSDVHData" },
          { 0x30040060UL, "SQDVHReferencedROISequence" },
          { 0x30040062UL, "CSDVHROIContributionType" },
          { 0x30040070UL, "DSDVHMinimumDose" },
          { 0x30040072UL, "DSDVHMaximumDose" },
          { 0x30040074UL, "DSDVHMeanDose" },
          { 0x30060002UL, "SHStructureSetLabel" },
          { 0x30060004UL, "LOStructureSetName" },
          { 0x30060006UL, "STStructureSetDescription" },
          { 0x30060008UL, "DAStructureSetDate" },
          { 0x30060009UL, "TMStructureSetTime" },
          { 0x30060010UL, "SQReferencedFrameOfReferenceSequence" },
          { 0x30060012UL, "SQRTReferencedStudySequence" },
          { 0x30060014UL, "SQRTReferencedSeriesSequence" },
          { 0x30060016UL, "SQContourImageSequence" },
          { 0x30060020UL, "SQStructureSetROISequence" },
          { 0x30060022UL, "ISROINumber" },
          { 0x30060024UL, "UIReferencedFrameOfReferenceUID" },
          { 0x30060026UL, "LOROIName" },
          { 0x30060028UL, "STROIDescription" },
          { 0x3006002AUL, "ISROIDisplayColor" },
          { 0x3006002CUL, "DSROIVolume" },
          { 0x30060030UL, "SQRTRelatedROISequence" },
          { 0x30060033UL, "CSRTROIRelationship" },
          { 0x30060036UL, "CSROIGenerationAlgorithm" },
          { 0x30060038UL, "LOROIGenerationDescription" },
          { 0x30060039UL, "SQROIContourSequence" },
          { 0x30060040UL, "SQContourSequence" },
          { 0x30060042UL, "CSContourGeometricType" },
          { 0x30060044UL, "DSContourSlabThickness" },
          { 0x30060045UL, "DSContourOffsetVector" },
          { 0x30060046UL, "ISNumberOfContourPoints" },
          { 0x30060048UL, "ISContourNumber" },
          { 0x30060049UL, "ISAttachedContours" },
          { 0x30060050UL, "DSContourData" },
          { 0x30060080UL, "SQRTROIObservationsSequence" },
          { 0x30060082UL, "ISObservationNumber" },
          { 0x30060084UL, "ISReferencedROINumber" },
          { 0x30060085UL, "SHROIObservationLabel" },
          { 0x30060086UL, "SQRTROIIdentificationCodeSequence" },
          { 0x30060088UL, "STROIObservationDescription" },
          { 0x300600A0UL, "SQRelatedRTROIObservationsSequence" },
          { 0x300600A4UL, "CSRTROIInterpretedType" },
          { 0x300600A6UL, "PNROIInterpreter" },
          { 0x300600B0UL, "SQROIPhysicalPropertiesSequence" },
          { 0x300600B2UL, "CSROIPhysicalProperty" },
          { 0x300600B4UL, "DSROIPhysicalPropertyValue" },
          { 0x300600B6UL, "SQROIElementalCompositionSequence" },
          { 0x300600B7UL, "USROIElementalCompositionAtomicNumber" },
          { 0x300600B8UL, "FLROIElementalCompositionAtomicMassFraction" },
          { 0x300600C0UL, "SQFrameOfReferenceRelationshipSequence" },
          { 0x300600C2UL, "UIRelatedFrameOfReferenceUID" },
          { 0x300600C4UL, "CSFrameOfReferenceTransformationType" },
          { 0x300600C6UL, "DSFrameOfReferenceTransformationMatrix" },
          { 0x300600C8UL, "LOFrameOfReferenceTransformationComment" },
          { 0x30080010UL, "SQMeasuredDoseReferenceSequence" },
          { 0x30080012UL, "STMeasuredDoseDescription" },
          { 0x30080014UL, "CSMeasuredDoseType" },
          { 0x30080016UL, "DSMeasuredDoseValue" },
          { 0x30080020UL, "SQTreatmentSessionBeamSequence" },
          { 0x30080022UL, "ISCurrentFractionNumber" },
          { 0x30080024UL, "DATreatmentControlPointDate" },
          { 0x30080025UL, "TMTreatmentControlPointTime" },
          { 0x3008002AUL, "CSTreatmentTerminationStatus" },
          { 0x3008002BUL, "SHTreatmentTerminationCode" },
          { 0x3008002CUL, "CSTreatmentVerificationStatus" },
          { 0x30080030UL, "SQReferencedTreatmentRecordSequence" },
          { 0x30080032UL, "DSSpecifiedPrimaryMeterset" },
          { 0x30080033UL, "DSSpecifiedSecondaryMeterset" },
          { 0x30080036UL, "DSDeliveredPrimaryMeterset" },
          { 0x30080037UL, "DSDeliveredSecondaryMeterset" },
          { 0x3008003AUL, "DSSpecifiedTreatmentTime" },
          { 0x3008003BUL, "DSDeliveredTreatmentTime" },
          { 0x30080040UL, "SQControlPointDeliverySequence" },
          { 0x30080042UL, "DSSpecifiedMeterset" },
          { 0x30080044UL, "DSDeliveredMeterset" },
          { 0x30080048UL, "DSDoseRateDelivered" },
          { 0x30080050UL, "SQTreatmentSummaryCalculatedDoseReferenceSequence" },
          { 0x30080052UL, "DSCumulativeDosetoDoseReference" },
          { 0x30080054UL, "DAFirstTreatmentDate" },
          { 0x30080056UL, "DAMostRecentTreatmentDate" },
          { 0x3008005AUL, "ISNumberofFractionsDelivered" },
          { 0x30080060UL, "SQOverrideSequence" },
          { 0x30080062UL, "ATOverrideParameterPointer" },
          { 0x30080064UL, "ISMeasuredDoseReferenceNumber" },
          { 0x30080066UL, "STOverrideReason" },
          { 0x30080070UL, "SQCalculatedDoseReferenceSequence" },
          { 0x30080072UL, "ISCalculatedDoseReferenceNumber" },
          { 0x30080074UL, "STCalculatedDoseReferenceDescription" },
          { 0x30080076UL, "DSCalculatedDoseReferenceDoseValue" },
          { 0x30080078UL, "DSStartMeterset" },
          { 0x3008007AUL, "DSEndMeterset" },
          { 0x30080080UL, "SQReferencedMeasuredDoseReferenceSequence" },
          { 0x30080082UL, "ISReferencedMeasuredDoseReferenceNumber" },
          { 0x30080090UL, "SQReferencedCalculatedDoseReferenceSequence" },
          { 0x30080092UL, "ISReferencedCalculatedDoseReferenceNumber" },
          { 0x300800A0UL, "SQBeamLimitingDeviceLeafPairsSequence" },
          { 0x300800B0UL, "SQRecordedWedgeSequence" },
          { 0x300800C0UL, "SQRecordedCompensatorSequence" },
          { 0x300800D0UL, "SQRecordedBlockSequence" },
          { 0x300800E0UL, "SQTreatmentSummaryMeasuredDoseReferenceSequence" },
          { 0x30080100UL, "SQRecordedSourceSequence" },
          { 0x30080105UL, "LOSourceSerialNumber" },
          { 0x30080110UL, "SQTreatmentSessionApplicationSetupSequence" },
          { 0x30080116UL, "CSApplicationSetupCheck" },
          { 0x30080120UL, "SQRecordedBrachyAccessoryDeviceSequence" },
          { 0x30080122UL, "ISReferencedBrachyAccessoryDeviceNumber" },
          { 0x30080130UL, "SQRecordedChannelSequence" },
          { 0x30080132UL, "DSSpecifiedChannelTotalTime" },
          { 0x30080134UL, "DSDeliveredChannelTotalTime" },
          { 0x30080136UL, "ISSpecifiedNumberofPulses" },
          { 0x30080138UL, "ISDeliveredNumberofPulses" },
          { 0x3008013AUL, "DSSpecifiedPulseRepetitionInterval" },
          { 0x3008013CUL, "DSDeliveredPulseRepetitionInterval" },
          { 0x30080140UL, "SQRecordedSourceApplicatorSequence" },
          { 0x30080142UL, "ISReferencedSourceApplicatorNumber" },
          { 0x30080150UL, "SQRecordedChannelShieldSequence" },
          { 0x30080152UL, "ISReferencedChannelShieldNumber" },
          { 0x30080160UL, "SQBrachyControlPointDeliveredSequence" },
          { 0x30080162UL, "DASafePositionExitDate" },
          { 0x30080164UL, "TMSafePositionExitTime" },
          { 0x30080166UL, "DASafePositionReturnDate" },
          { 0x30080168UL, "TMSafePositionReturnTime" },
          { 0x30080200UL, "CSCurrentTreatmentStatus" },
          { 0x30080202UL, "STTreatmentStatusComment" },
          { 0x30080220UL, "SQFractionGroupSummarySequence" },
          { 0x30080223UL, "ISReferencedFractionNumber" },
          { 0x30080224UL, "CSFractionGroupType" },
          { 0x30080230UL, "CSBeamStopperPosition" },
          { 0x30080240UL, "SQFractionStatusSummarySequence" },
          { 0x30080250UL, "DATreatmentDate" },
          { 0x30080251UL, "TMTreatmentTime" },
          { 0x300A0002UL, "SHRTPlanLabel" },
          { 0x300A0003UL, "LORTPlanName" },
          { 0x300A0004UL, "STRTPlanDescription" },
          { 0x300A0006UL, "DARTPlanDate" },
          { 0x300A0007UL, "TMRTPlanTime" },
          { 0x300A0009UL, "LOTreatmentProtocols" },
          { 0x300A000AUL, "CSPlanIntent" },
          { 0x300A000BUL, "LOTreatmentSites" },
          { 0x300A000CUL, "CSRTPlanGeometry" },
          { 0x300A000EUL, "STPrescriptionDescription" },
          { 0x300A0010UL, "SQDoseReferenceSequence" },
          { 0x300A0012UL, "ISDoseReferenceNumber" },
          { 0x300A0013UL, "UIDoseReferenceUID" },
          { 0x300A0014UL, "CSDoseReferenceStructureType" },
          { 0x300A0015UL, "CSNominalBeamEnergyUnit" },
          { 0x300A0016UL, "LODoseReferenceDescription" },
          { 0x300A0018UL, "DSDoseReferencePointCoordinates" },
          { 0x300A001AUL, "DSNominalPriorDose" },
          { 0x300A0020UL, "CSDoseReferenceType" },
          { 0x300A0021UL, "DSConstraintWeight" },
          { 0x300A0022UL, "DSDeliveryWarningDose" },
          { 0x300A0023UL, "DSDeliveryMaximumDose" },
          { 0x300A0025UL, "DSTargetMinimumDose" },
          { 0x300A0026UL, "DSTargetPrescriptionDose" },
          { 0x300A0027UL, "DSTargetMaximumDose" },
          { 0x300A0028UL, "DSTargetUnderdoseVolumeFraction" },
          { 0x300A002AUL, "DSOrganAtRiskFullVolumeDose" },
          { 0x300A002BUL, "DSOrganAtRiskLimitDose" },
          { 0x300A002CUL, "DSOrganAtRiskMaximumDose" },
          { 0x300A002DUL, "DSOrganAtRiskOverdoseVolumeFraction" },
          { 0x300A0040UL, "SQToleranceTableSequence" },
          { 0x300A0042UL, "ISToleranceTableNumber" },
          { 0x300A0043UL, "SHToleranceTableLabel" },
          { 0x300A0044UL, "DSGantryAngleTolerance" },
          { 0x300A0046UL, "DSBeamLimitingDeviceAngleTolerance" },
          { 0x300A0048UL, "SQBeamLimitingDeviceToleranceSequence" },
          { 0x300A004AUL, "DSBeamLimitingDevicePositionTolerance" },
          { 0x300A004BUL, "FLSnoutPositionTolerance" },
          { 0x300A004CUL, "DSPatientSupportAngleTolerance" },
          { 0x300A004EUL, "DSTableTopEccentricAngleTolerance" },
          { 0x300A004FUL, "FLTableTopPitchAngleTolerance" },
          { 0x300A0050UL, "FLTableTopRollAngleTolerance" },
          { 0x300A0051UL, "DSTableTopVerticalPositionTolerance" },
          { 0x300A0052UL, "DSTableTopLongitudinalPositionTolerance" },
          { 0x300A0053UL, "DSTableTopLateralPositionTolerance" },
          { 0x300A0055UL, "CSRTPlanRelationship" },
          { 0x300A0070UL, "SQFractionGroupSequence" },
          { 0x300A0071UL, "ISFractionGroupNumber" },
          { 0x300A0072UL, "LOFractionGroupDescription" },
          { 0x300A0078UL, "ISNumberOfFractionsPlanned" },
          { 0x300A0079UL, "ISNumberOfFractionPatternDigitsPerDay" },
          { 0x300A007AUL, "ISRepeatFractionCycleLength" },
          { 0x300A007BUL, "LTFractionPattern" },
          { 0x300A0080UL, "ISNumberOfBeams" },
          { 0x300A0082UL, "DSBeamDoseSpecificationPoint" },
          { 0x300A0084UL, "DSBeamDose" },
          { 0x300A0086UL, "DSBeamMeterset" },
          { 0x300A0088UL, "FLBeamDosePointDepth" },
          { 0x300A0089UL, "FLBeamDosePointEquivalentDepth" },
          { 0x300A008AUL, "FLBeamDosePointSSD" },
          { 0x300A00A0UL, "ISNumberOfBrachyApplicationSetups" },
          { 0x300A00A2UL, "DSBrachyApplicationSetupDoseSpecificationPoint" },
          { 0x300A00A4UL, "DSBrachyApplicationSetupDose" },
          { 0x300A00B0UL, "SQBeamSequence" },
          { 0x300A00B2UL, "SHTreatmentMachineName" },
          { 0x300A00B3UL, "CSPrimaryDosimeterUnit" },
          { 0x300A00B4UL, "DSSourceAxisDistance" },
          { 0x300A00B6UL, "SQBeamLimitingDeviceSequence" },
          { 0x300A00B8UL, "CSRTBeamLimitingDeviceType" },
          { 0x300A00BAUL, "DSSourceToBeamLimitingDeviceDistance" },
          { 0x300A00BBUL, "FLIsocenterToBeamLimitingDeviceDistance" },
          { 0x300A00BCUL, "ISNumberOfLeafJawPairs" },
          { 0x300A00BEUL, "DSLeafPositionBoundaries" },
          { 0x300A00C0UL, "ISBeamNumber" },
          { 0x300A00C2UL, "LOBeamName" },
          { 0x300A00C3UL, "STBeamDescription" },
          { 0x300A00C4UL, "CSBeamType" },
          { 0x300A00C6UL, "CSRadiationType" },
          { 0x300A00C7UL, "CSHighDoseTechniqueType" },
          { 0x300A00C8UL, "ISReferenceImageNumber" },
          { 0x300A00CAUL, "SQPlannedVerificationImageSequence" },
          { 0x300A00CCUL, "LOImagingDeviceSpecificAcquisitionParameters" },
          { 0x300A00CEUL, "CSTreatmentDeliveryType" },
          { 0x300A00D0UL, "ISNumberOfWedges" },
          { 0x300A00D1UL, "SQWedgeSequence" },
          { 0x300A00D2UL, "ISWedgeNumber" },
          { 0x300A00D3UL, "CSWedgeType" },
          { 0x300A00D4UL, "SHWedgeID" },
          { 0x300A00D5UL, "ISWedgeAngle" },
          { 0x300A00D6UL, "DSWedgeFactor" },
          { 0x300A00D7UL, "FLTotalWedgeTrayWaterEquivalentThickness" },
          { 0x300A00D8UL, "DSWedgeOrientation" },
          { 0x300A00D9UL, "FLIsocenterToWedgeTrayDistance" },
          { 0x300A00DAUL, "DSSourceToWedgeTrayDistance" },
          { 0x300A00DBUL, "FLWedgeThinEdgePosition" },
          { 0x300A00DCUL, "SHBolusID" },
          { 0x300A00DDUL, "STBolusDescription" },
          { 0x300A00E0UL, "ISNumberOfCompensators" },
          { 0x300A00E1UL, "SHMaterialID" },
          { 0x300A00E2UL, "DSTotalCompensatorTrayFactor" },
          { 0x300A00E3UL, "SQCompensatorSequence" },
          { 0x300A00E4UL, "ISCompensatorNumber" },
          { 0x300A00E5UL, "SHCompensatorID" },
          { 0x300A00E6UL, "DSSourceToCompensatorTrayDistance" },
          { 0x300A00E7UL, "ISCompensatorRows" },
          { 0x300A00E8UL, "ISCompensatorColumns" },
          { 0x300A00E9UL, "DSCompensatorPixelSpacing" },
          { 0x300A00EAUL, "DSCompensatorPosition" },
          { 0x300A00EBUL, "DSCompensatorTransmissionData" },
          { 0x300A00ECUL, "DSCompensatorThicknessData" },
          { 0x300A00EDUL, "ISNumberOfBoli" },
          { 0x300A00EEUL, "CSCompensatorType" },
          { 0x300A00F0UL, "ISNumberOfBlocks" },
          { 0x300A00F2UL, "DSTotalBlockTrayFactor" },
          { 0x300A00F3UL, "FLTotalBlockTrayWaterEquivalentThickness" },
          { 0x300A00F4UL, "SQBlockSequence" },
          { 0x300A00F5UL, "SHBlockTrayID" },
          { 0x300A00F6UL, "DSSourceToBlockTrayDistance" },
          { 0x300A00F7UL, "FLIsocenterToBlockTrayDistance" },
          { 0x300A00F8UL, "CSBlockType" },
          { 0x300A00F9UL, "LOAccessoryCode" },
          { 0x300A00FAUL, "CSBlockDivergence" },
          { 0x300A00FBUL, "CSBlockMountingPosition" },
          { 0x300A00FCUL, "ISBlockNumber" },
          { 0x300A00FEUL, "LOBlockName" },
          { 0x300A0100UL, "DSBlockThickness" },
          { 0x300A0102UL, "DSBlockTransmission" },
          { 0x300A0104UL, "ISBlockNumberOfPoints" },
          { 0x300A0106UL, "DSBlockData" },
          { 0x300A0107UL, "SQApplicatorSequence" },
          { 0x300A0108UL, "SHApplicatorID" },
          { 0x300A0109UL, "CSApplicatorType" },
          { 0x300A010AUL, "LOApplicatorDescription" },
          { 0x300A010CUL, "DSCumulativeDoseReferenceCoefficient" },
          { 0x300A010EUL, "DSFinalCumulativeMetersetWeight" },
          { 0x300A0110UL, "ISNumberOfControlPoints" },
          { 0x300A0111UL, "SQControlPointSequence" },
          { 0x300A0112UL, "ISControlPointIndex" },
          { 0x300A0114UL, "DSNominalBeamEnergy" },
          { 0x300A0115UL, "DSDoseRateSet" },
          { 0x300A0116UL, "SQWedgePositionSequence" },
          { 0x300A0118UL, "CSWedgePosition" },
          { 0x300A011AUL, "SQBeamLimitingDevicePositionSequence" },
          { 0x300A011CUL, "DSLeafJawPositions" },
          { 0x300A011EUL, "DSGantryAngle" },
          { 0x300A011FUL, "CSGantryRotationDirection" },
          { 0x300A0120UL, "DSBeamLimitingDeviceAngle" },
          { 0x300A0121UL, "CSBeamLimitingDeviceRotationDirection" },
          { 0x300A0122UL, "DSPatientSupportAngle" },
          { 0x300A0123UL, "CSPatientSupportRotationDirection" },
          { 0x300A0124UL, "DSTableTopEccentricAxisDistance" },
          { 0x300A0125UL, "DSTableTopEccentricAngle" },
          { 0x300A0126UL, "CSTableTopEccentricRotationDirection" },
          { 0x300A0128UL, "DSTableTopVerticalPosition" },
          { 0x300A0129UL, "DSTableTopLongitudinalPosition" },
          { 0x300A012AUL, "DSTableTopLateralPosition" },
          { 0x300A012CUL, "DSIsocenterPosition" },
          { 0x300A012EUL, "DSSurfaceEntryPoint" },
          { 0x300A0130UL, "DSSourceToSurfaceDistance" },
          { 0x300A0134UL, "DSCumulativeMetersetWeight" },
          { 0x300A0140UL, "FLTableTopPitchAngle" },
          { 0x300A0142UL, "CSTableTopPitchRotationDirection" },
          { 0x300A0144UL, "FLTableTopRollAngle" },
          { 0x300A0146UL, "CSTableTopRollRotationDirection" },
          { 0x300A0148UL, "FLHeadFixationAngle" },
          { 0x300A014AUL, "FLGantryPitchAngle" },
          { 0x300A014CUL, "CSGantryPitchRotationDirection" },
          { 0x300A014EUL, "FLGantryPitchAngleTolerance" },
          { 0x300A0180UL, "SQPatientSetupSequence" },
          { 0x300A0182UL, "ISPatientSetupNumber" },
          { 0x300A0183UL, "LOPatientSetupLabel" },
          { 0x300A0184UL, "LOPatientAdditionalPosition" },
          { 0x300A0190UL, "SQFixationDeviceSequence" },
          { 0x300A0192UL, "CSFixationDeviceType" },
          { 0x300A0194UL, "SHFixationDeviceLabel" },
          { 0x300A0196UL, "STFixationDeviceDescription" },
          { 0x300A0198UL, "SHFixationDevicePosition" },
          { 0x300A0199UL, "FLFixationDevicePitchAngle" },
          { 0x300A019AUL, "FLFixationDeviceRollAngle" },
          { 0x300A01A0UL, "SQShieldingDeviceSequence" },
          { 0x300A01A2UL, "CSShieldingDeviceType" },
          { 0x300A01A4UL, "SHShieldingDeviceLabel" },
          { 0x300A01A6UL, "STShieldingDeviceDescription" },
          { 0x300A01A8UL, "SHShieldingDevicePosition" },
          { 0x300A01B0UL, "CSSetupTechnique" },
          { 0x300A01B2UL, "STSetupTechniqueDescription" },
          { 0x300A01B4UL, "SQSetupDeviceSequence" },
          { 0x300A01B6UL, "CSSetupDeviceType" },
          { 0x300A01B8UL, "SHSetupDeviceLabel" },
          { 0x300A01BAUL, "STSetupDeviceDescription" },
          { 0x300A01BCUL, "DSSetupDeviceParameter" },
          { 0x300A01D0UL, "STSetupReferenceDescription" },
          { 0x300A01D2UL, "DSTableTopVerticalSetupDisplacement" },
          { 0x300A01D4UL, "DSTableTopLongitudinalSetupDisplacement" },
          { 0x300A01D6UL, "DSTableTopLateralSetupDisplacement" },
          { 0x300A0200UL, "CSBrachyTreatmentTechnique" },
          { 0x300A0202UL, "CSBrachyTreatmentType" },
          { 0x300A0206UL, "SQTreatmentMachineSequence" },
          { 0x300A0210UL, "SQSourceSequence" },
          { 0x300A0212UL, "ISSourceNumber" },
          { 0x300A0214UL, "CSSourceType" },
          { 0x300A0216UL, "LOSourceManufacturer" },
          { 0x300A0218UL, "DSActiveSourceDiameter" },
          { 0x300A021AUL, "DSActiveSourceLength" },
          { 0x300A0222UL, "DSSourceEncapsulationNominalThickness" },
          { 0x300A0224UL, "DSSourceEncapsulationNominalTransmission" },
          { 0x300A0226UL, "LOSourceIsotopeName" },
          { 0x300A0228UL, "DSSourceIsotopeHalfLife" },
          { 0x300A0229UL, "CSSourceStrengthUnits" },
          { 0x300A022AUL, "DSReferenceAirKermaRate" },
          { 0x300A022BUL, "DSSourceStrength" },
          { 0x300A022CUL, "DASourceStrengthReferenceDate" },
          { 0x300A022EUL, "TMSourceStrengthReferenceTime" },
          { 0x300A0230UL, "SQApplicationSetupSequence" },
          { 0x300A0232UL, "CSApplicationSetupType" },
          { 0x300A0234UL, "ISApplicationSetupNumber" },
          { 0x300A0236UL, "LOApplicationSetupName" },
          { 0x300A0238UL, "LOApplicationSetupManufacturer" },
          { 0x300A0240UL, "ISTemplateNumber" },
          { 0x300A0242UL, "SHTemplateType" },
          { 0x300A0244UL, "LOTemplateName" },
          { 0x300A0250UL, "DSTotalReferenceAirKerma" },
          { 0x300A0260UL, "SQBrachyAccessoryDeviceSequence" },
          { 0x300A0262UL, "ISBrachyAccessoryDeviceNumber" },
          { 0x300A0263UL, "SHBrachyAccessoryDeviceID" },
          { 0x300A0264UL, "CSBrachyAccessoryDeviceType" },
          { 0x300A0266UL, "LOBrachyAccessoryDeviceName" },
          { 0x300A026AUL, "DSBrachyAccessoryDeviceNominalThickness" },
          { 0x300A026CUL, "DSBrachyAccessoryDeviceNominalTransmission" },
          { 0x300A0280UL, "SQChannelSequence" },
          { 0x300A0282UL, "ISChannelNumber" },
          { 0x300A0284UL, "DSChannelLength" },
          { 0x300A0286UL, "DSChannelTotalTime" },
          { 0x300A0288UL, "CSSourceMovementType" },
          { 0x300A028AUL, "ISNumberOfPulses" },
          { 0x300A028CUL, "DSPulseRepetitionInterval" },
          { 0x300A0290UL, "ISSourceApplicatorNumber" },
          { 0x300A0291UL, "SHSourceApplicatorID" },
          { 0x300A0292UL, "CSSourceApplicatorType" },
          { 0x300A0294UL, "LOSourceApplicatorName" },
          { 0x300A0296UL, "DSSourceApplicatorLength" },
          { 0x300A0298UL, "LOSourceApplicatorManufacturer" },
          { 0x300A029CUL, "DSSourceApplicatorWallNominalThickness" },
          { 0x300A029EUL, "DSSourceApplicatorWallNominalTransmission" },
          { 0x300A02A0UL, "DSSourceApplicatorStepSize" },
          { 0x300A02A2UL, "ISTransferTubeNumber" },
          { 0x300A02A4UL, "DSTransferTubeLength" },
          { 0x300A02B0UL, "SQChannelShieldSequence" },
          { 0x300A02B2UL, "ISChannelShieldNumber" },
          { 0x300A02B3UL, "SHChannelShieldID" },
          { 0x300A02B4UL, "LOChannelShieldName" },
          { 0x300A02B8UL, "DSChannelShieldNominalThickness" },
          { 0x300A02BAUL, "DSChannelShieldNominalTransmission" },
          { 0x300A02C8UL, "DSFinalCumulativeTimeWeight" },
          { 0x300A02D0UL, "SQBrachyControlPointSequence" },
          { 0x300A02D2UL, "DSControlPointRelativePosition" },
          { 0x300A02D4UL, "DSControlPoint3DPosition" },
          { 0x300A02D6UL, "DSCumulativeTimeWeight" },
          { 0x300A02E0UL, "CSCompensatorDivergence" },
          { 0x300A02E1UL, "CSCompensatorMountingPosition" },
          { 0x300A02E2UL, "DSSourceToCompensatorDistance" },
          { 0x300A02E3UL, "FLTotalCompensatorTrayWaterEquivalentThicknessrEquivalentThickness" },
          { 0x300A02E4UL, "FLIsocenterToCompensatorTrayDistance" },
          { 0x300A02E5UL, "FLCompensatorColumnOffset" },
          { 0x300A02E6UL, "FLIsocenterToCompensatorDistances" },
          { 0x300A02E7UL, "FLCompensatorRelativeStoppingPowerRatio" },
          { 0x300A02E8UL, "FLCompensatorMillingToolDiameter" },
          { 0x300A02EAUL, "SQIonRangeCompensatorSequence" },
          { 0x300A02EBUL, "LTCompensatorDescription" },
          { 0x300A0302UL, "ISRadiationMassNumber" },
          { 0x300A0304UL, "ISRadiationAtomicNumber" },
          { 0x300A0306UL, "SSRadiationChargeState" },
          { 0x300A0308UL, "CSScanMode" },
          { 0x300A030AUL, "FLVirtualSourceAxisDistances" },
          { 0x300A030CUL, "SQSnoutSequence" },
          { 0x300A030DUL, "FLSnoutPosition" },
          { 0x300A030FUL, "SHSnoutID" },
          { 0x300A0312UL, "ISNumberOfRangeShifters" },
          { 0x300A0314UL, "SQRangeShifterSequence" },
          { 0x300A0316UL, "ISRangeShifterNumber" },
          { 0x300A0318UL, "SHRangeShifterID" },
          { 0x300A0320UL, "CSRangeShifterType" },
          { 0x300A0322UL, "LORangeShifterDescription" },
          { 0x300A0330UL, "ISNumberOfLateralSpreadingDevices" },
          { 0x300A0332UL, "SQLateralSpreadingDeviceSequence" },
          { 0x300A0334UL, "ISLateralSpreadingDeviceNumber" },
          { 0x300A0336UL, "SHLateralSpreadingDeviceID" },
          { 0x300A0338UL, "CSLateralSpreadingDeviceType" },
          { 0x300A033AUL, "LOLateralSpreadingDeviceDescription" },
          { 0x300A033CUL, "FLLateralSpreadingDeviceWaterEquivalentThickness" },
          { 0x300A0340UL, "ISNumberOfRangeModulators" },
          { 0x300A0342UL, "SQRangeModulatorSequence" },
          { 0x300A0344UL, "ISRangeModulatorNumber" },
          { 0x300A0346UL, "SHRangeModulatorID" },
          { 0x300A0348UL, "CSRangeModulatorType" },
          { 0x300A034AUL, "LORangeModulatorDescription" },
          { 0x300A034CUL, "SHBeamCurrentModulationID" },
          { 0x300A0350UL, "CSPatientSupportType" },
          { 0x300A0352UL, "SHPatientSupportID" },
          { 0x300A0354UL, "LOPatientSupportAccessoryCode" },
          { 0x300A0356UL, "FLFixationLightAzimuthalAngle" },
          { 0x300A0358UL, "FLFixationLightPolarAngle" },
          { 0x300A035AUL, "FLMetersetRate" },
          { 0x300A0360UL, "SQRangeShifterSettingsSequence" },
          { 0x300A0362UL, "LORangeShifterSetting" },
          { 0x300A0364UL, "FLIsocenterToRangeShifterDistance" },
          { 0x300A0366UL, "FLRangeShifterWaterEquivalentThickness" },
          { 0x300A0370UL, "SQLateralSpreadingDeviceSettingsSequence" },
          { 0x300A0372UL, "LOLateralSpreadingDeviceSetting" },
          { 0x300A0374UL, "FLIsocenterToLateralSpreadingDeviceDistance" },
          { 0x300A0380UL, "SQRangeModulatorSettingsSequence" },
          { 0x300A0382UL, "FLRangeModulatorGatingStartValue" },
          { 0x300A0384UL, "FLRangeModulatorGatingStopValue" },
          { 0x300A0386UL, "FLRangeModulatorGatingStartWaterEquivalentThickness" },
          { 0x300A0388UL, "FLRangeModulatorGatingStopWaterEquivalentThickness" },
          { 0x300A038AUL, "FLIsocenterToRangeModulatorDistance" },
          { 0x300A0390UL, "SHScanSpotTuneID" },
          { 0x300A0392UL, "ISNumberOfScanSpotPositions" },
          { 0x300A0394UL, "FLScanSpotPositionMap" },
          { 0x300A0396UL, "FLScanSpotMetersetWeights" },
          { 0x300A0398UL, "FLScanningSpotSize" },
          { 0x300A039AUL, "ISNumberOfPaintings" },
          { 0x300A03A0UL, "SQIonToleranceTableSequence" },
          { 0x300A03A2UL, "SQIonBeamSequence" },
          { 0x300A03A4UL, "SQIonBeamLimitingDeviceSequence" },
          { 0x300A03A6UL, "SQIonBlockSequence" },
          { 0x300A03A8UL, "SQIonControlPointSequence" },
          { 0x300A03AAUL, "SQIonWedgeSequence" },
          { 0x300A03ACUL, "SQIonWedgePositionSequence" },
          { 0x300A0401UL, "SQReferencedSetupImageSequence" },
          { 0x300A0402UL, "STSetupImageComment" },
          { 0x300A0410UL, "SQMotionSynchronizationSequence" },
          { 0x300A0412UL, "FLControlPointOrientation" },
          { 0x300A0420UL, "SQGeneralAccessorySequence" },
          { 0x300A0421UL, "SHGeneralAccessoryID" },
          { 0x300A0422UL, "STGeneralAccessoryDescription" },
          { 0x300A0423UL, "CSGeneralAccessoryType" },
          { 0x300A0424UL, "ISGeneralAccessoryNumber" },
          { 0x300A0431UL, "SQApplicatorGeometrySequence" },
          { 0x300A0432UL, "CSApplicatorApertureShape" },
          { 0x300A0433UL, "FLApplicatorOpening" },
          { 0x300A0434UL, "FLApplicatorOpeningX" },
          { 0x300A0435UL, "FLApplicatorOpeningY" },
          { 0x300A0436UL, "FLSourceToApplicatorMountingPositionDistance" },
          { 0x300C0002UL, "SQReferencedRTPlanSequence" },
          { 0x300C0004UL, "SQReferencedBeamSequence" },
          { 0x300C0006UL, "ISReferencedBeamNumber" },
          { 0x300C0007UL, "ISReferencedReferenceImageNumber" },
          { 0x300C0008UL, "DSStartCumulativeMetersetWeight" },
          { 0x300C0009UL, "DSEndCumulativeMetersetWeight" },
          { 0x300C000AUL, "SQReferencedBrachyApplicationSetupSequence" },
          { 0x300C000CUL, "ISReferencedBrachyApplicationSetupNumber" },
          { 0x300C000EUL, "ISReferencedSourceNumber" },
          { 0x300C0020UL, "SQReferencedFractionGroupSequence" },
          { 0x300C0022UL, "ISReferencedFractionGroupNumber" },
          { 0x300C0040UL, "SQReferencedVerificationImageSequence" },
          { 0x300C0042UL, "SQReferencedReferenceImageSequence" },
          { 0x300C0050UL, "SQReferencedDoseReferenceSequence" },
          { 0x300C0051UL, "ISReferencedDoseReferenceNumber" },
          { 0x300C0055UL, "SQBrachyReferencedDoseReferenceSequence" },
          { 0x300C0060UL, "SQReferencedStructureSetSequence" },
          { 0x300C006AUL, "ISReferencedPatientSetupNumber" },
          { 0x300C0080UL, "SQReferencedDoseSequence" },
          { 0x300C00A0UL, "ISReferencedToleranceTableNumber" },
          { 0x300C00B0UL, "SQReferencedBolusSequence" },
          { 0x300C00C0UL, "ISReferencedWedgeNumber" },
          { 0x300C00D0UL, "ISReferencedCompensatorNumber" },
          { 0x300C00E0UL, "ISReferencedBlockNumber" },
          { 0x300C00F0UL, "ISReferencedControlPointIndex" },
          { 0x300C00F2UL, "SQReferencedControlPointSequence" },
          { 0x300C00F4UL, "ISReferencedStartControlPointIndex" },
          { 0x300C00F6UL, "ISReferencedStopControlPointIndex" },
          { 0x300C0100UL, "ISReferencedRangeShifterNumber" },
          { 0x300C0102UL, "ISReferencedLateralSpreadingDeviceNumber" },
          { 0x300C0104UL, "ISReferencedRangeModulatorNumber" },
          { 0x300E0002UL, "CSApprovalStatus" },
          { 0x300E0004UL, "DAReviewDate" },
          { 0x300E0005UL, "TMReviewTime" },
          { 0x300E0008UL, "PNReviewerName" },
          { 0x40100001UL, "CSLowEnergyDetectors" },
          { 0x40100002UL, "CSHighEnergyDetectors" },
          { 0x40100004UL, "SQDetectorGeometrySequence" },
          { 0x40101001UL, "SQThreatROIVoxelSequence" },
          { 0x40101004UL, "FLThreatROIBase" },
          { 0x40101005UL, "FLThreatROIExtents" },
          { 0x40101006UL, "OBThreatROIBitmap" },
          { 0x40101007UL, "SHRouteSegmentID" },
          { 0x40101008UL, "CSGantryType" },
          { 0x40101009UL, "CSOOIOwnerType" },
          { 0x4010100AUL, "SQRouteSegmentSequence" },
          { 0x40101010UL, "USPotentialThreatObjectID" },
          { 0x40101011UL, "SQThreatSequence" },
          { 0x40101012UL, "CSThreatCategory" },
          { 0x40101013UL, "LTThreatCategoryDescription" },
          { 0x40101014UL, "CSATDAbilityAssessment" },
          { 0x40101015UL, "CSATDAssessmentFlag" },
          { 0x40101016UL, "FLATDAssessmentProbability" },
          { 0x40101017UL, "FLMass" },
          { 0x40101018UL, "FLDensity" },
          { 0x40101019UL, "FLZEffective" },
          { 0x4010101AUL, "SHBoardingPassID" },
          { 0x4010101BUL, "FLCenterOfMass" },
          { 0x4010101CUL, "FLCenterOfPTO" },
          { 0x4010101DUL, "FLBoundingPolygon" },
          { 0x4010101EUL, "SHRouteSegmentStartLocationID" },
          { 0x4010101FUL, "SHRouteSegmentEndLocationID" },
          { 0x40101020UL, "CSRouteSegmentLocationIDType" },
          { 0x40101021UL, "CSAbortReason" },
          { 0x40101023UL, "FLVolumeOfPTO" },
          { 0x40101024UL, "CSAbortFlag" },
          { 0x40101025UL, "DTRouteSegmentStartTime" },
          { 0x40101026UL, "DTRouteSegmentEndTime" },
          { 0x40101027UL, "CSTDRType" },
          { 0x40101028UL, "CSInternationalRouteSegment" },
          { 0x40101029UL, "LOThreatDetectionAlgorithmandVersion" },
          { 0x4010102AUL, "SHAssignedLocation" },
          { 0x4010102BUL, "DTAlarmDecisionTime" },
          { 0x40101031UL, "CSAlarmDecision" },
          { 0x40101033UL, "USNumberOfTotalObjects" },
          { 0x40101034UL, "USNumberOfAlarmObjects" },
          { 0x40101037UL, "SQPTORepresentationSequence" },
          { 0x40101038UL, "SQATDAssessmentSequence" },
          { 0x40101039UL, "CSTIPType" },
          { 0x4010103AUL, "CSDICOSVersion" },
          { 0x40101041UL, "DTOOIOwnerCreationTime" },
          { 0x40101042UL, "CSOOIType" },
          { 0x40101043UL, "FLOOISize" },
          { 0x40101044UL, "CSAcquisitionStatus" },
          { 0x40101045UL, "SQBasisMaterialsCodeSequence" },
          { 0x40101046UL, "CSPhantomType" },
          { 0x40101047UL, "SQOOIOwnerSequence" },
          { 0x40101048UL, "CSScanType" },
          { 0x40101051UL, "LOItineraryID" },
          { 0x40101052UL, "SHItineraryIDType" },
          { 0x40101053UL, "LOItineraryIDAssigningAuthority" },
          { 0x40101054UL, "SHRouteID" },
          { 0x40101055UL, "SHRouteIDAssigningAuthority" },
          { 0x40101056UL, "CSInboundArrivalType" },
          { 0x40101058UL, "SHCarrierID" },
          { 0x40101059UL, "CSCarrierIDAssigningAuthority" },
          { 0x40101060UL, "FLSourceOrientation" },
          { 0x40101061UL, "FLSourcePosition" },
          { 0x40101062UL, "FLBeltHeight" },
          { 0x40101064UL, "SQAlgorithmRoutingCodeSequence" },
          { 0x40101067UL, "CSTransportClassification" },
          { 0x40101068UL, "LTOOITypeDescriptor" },
          { 0x40101069UL, "FLTotalProcessingTime" },
          { 0x4010106CUL, "OBDetectorCalibrationData" },
          { 0x4FFE0001UL, "SQMACParametersSequence" },
          { 0x52009229UL, "SQSharedFunctionalGroupsSequence" },
          { 0x52009230UL, "SQPerFrameFunctionalGroupsSequence" },
          { 0x54000100UL, "SQWaveformSequence" },
          { 0x54000110UL, "OBChannelMinimumValue" },
          { 0x54000112UL, "OBChannelMaximumValue" },
          { 0x54001004UL, "USWaveformBitsAllocated" },
          { 0x54001006UL, "CSWaveformSampleInterpretation" },
          { 0x5400100AUL, "OBWaveformPaddingValue" },
          { 0x54001010UL, "OBWaveformData" },
          { 0x56000010UL, "OFFirstOrderPhaseCorrectionAngle" },
          { 0x56000020UL, "OFSpectroscopyData" },
          { 0x7FE00010UL, "OWPixelData" },
          { 0x7FE11010UL, "OBCSAData" },
          { 0xFFFAFFFAUL, "SQDigitalSignaturesSequence" },
          { 0xFFFCFFFCUL, "OBDataSetTrailingPadding" },
          { 0xFFFEE000UL, "UNItem" },
          { 0xFFFEE00DUL, "UNItemDelimitationItem" },
          { 0xFFFEE0DDUL, "UNSequenceDelimitationItem" },

          // Philips private fields
          // Added based on Gianlorenzo Fagiolo's custom dicom.dic
          // Augmented with data from Xiangrui Li, author of MatLab DICOM to NIfTI converter
          { 0x00080000UL, "ULPhilips_0008_0000" },
          { 0x00100000UL, "ULPhilips_0010_0000" },
          { 0x00180000UL, "ULPhilips_0018_0000" },
          { 0x00200000UL, "ULPhilips_0020_0000" },
          { 0x00280000UL, "ULPhilips_0028_0000" },
          { 0x00320000UL, "ULPhilips_0032_0000" },
          { 0x00400000UL, "ULPhilips_0040_0000" },
          { 0x20010000UL, "ULPhilips_2001_0000" },
          { 0x20010010UL, "LOPhilips_2001_0010" },
          { 0x20010090UL, "LOPhilips_2001_0090" },
          { 0x20011001UL, "FLChemicalShift" },
          { 0x20011002UL, "ISChemicalShiftNumberMR" },
          { 0x20011003UL, "FLB_factor" },
          { 0x20011004UL, "CSDiffusionDirection" },
          { 0x20011005UL, "SSGraphicAnnotationParentID" },
          { 0x20011006UL, "CSImageEnhanced" },
          { 0x20011007UL, "CSImageTypeEDES" },
          { 0x20011008UL, "ISPhaseNumber" },
          { 0x20011009UL, "FLImagePrepulseDelay" },
          { 0x2001100AUL, "ISSliceNumberMR" },
          { 0x2001100BUL, "CSSliceOrientation" },
          { 0x2001100CUL, "CSArrhythmiaRejection" },
          { 0x2001100EUL, "CSCardiacCycled" },
          { 0x2001100FUL, "SSCardiacGateWidth" },
          { 0x20011010UL, "CSCardiacSync" },
          { 0x20011011UL, "FLDiffusionEchoTime" },
          { 0x20011012UL, "CSDynamicSeries" },
          { 0x20011013UL, "SLEPIFactor" },
          { 0x20011014UL, "SLNumberOfEchoes" },
          { 0x20011015UL, "SSNumberOfLocations" },
          { 0x20011016UL, "SSNumberOfPCDirections" },
          { 0x20011017UL, "SLNumberOfPhasesMR" },
          { 0x20011018UL, "SLLocationsInAcquisition" },
          { 0x20011019UL, "CSPartialMatrixScanned" },
          { 0x2001101AUL, "FLPCVelocity" },
          { 0x2001101BUL, "FLPrepulseDelay" },
          { 0x2001101CUL, "CSPrepulseType" },
          { 0x2001101DUL, "ISReconstructionNumberMR" },
          { 0x2001101EUL, "CSReformatAccuracyMR" },
          { 0x2001101FUL, "CSRespirationSync" },
          { 0x20011020UL, "LOScanningSequence" },
          { 0x20011021UL, "CSSPIR" },
          { 0x20011022UL, "FLWaterFatShift" },
          { 0x20011023UL, "DSMRSeriesFlipAngle" },
          { 0x20011024UL, "CSSeriesIsInteractive" },
          { 0x20011025UL, "SHEchoTimeDisplay" },
          { 0x20011026UL, "CSPresentationStateSubtractionActive" },
          { 0x2001102DUL, "SSStackNumberOfSlices" },
          { 0x20011032UL, "FLStackRadialAngle" },
          { 0x20011033UL, "CSStackRadialAxis" },
          { 0x20011035UL, "SSStackSliceNumber" },
          { 0x20011036UL, "CSStackType" },
          { 0x2001103DUL, "ULContourFillColor" },
          { 0x2001103FUL, "CSDisplayedAreaZoomInterpolationMeth" },
          { 0x2001104BUL, "CSInterpolationMethod" },
          { 0x2001104CUL, "CSPolyLineBeginPointStyle" },
          { 0x2001104DUL, "CSPolyLineEndPointStype" },
          { 0x2001104EUL, "CSWindowSmoothingTaste" },
          { 0x20011051UL, "ISOverlayPlaneID" },
          { 0x20011052UL, "UIImagePresentationStateUID" },
          { 0x20011053UL, "CSWindowInvert" },
          { 0x20011054UL, "FLContourFillTransparency" },
          { 0x20011055UL, "ULGraphicLineColor" },
          { 0x20011056UL, "CSGraphicType" },
          { 0x2001105AUL, "STGraphicAnnotationModel" },
          { 0x2001105DUL, "STMeasurementTextUnits" },
          { 0x2001105EUL, "STMeasurementTextType" },
          { 0x2001105FUL, "SQStack" },
          { 0x20011060UL, "SLNumberOfStacks" },
          { 0x20011061UL, "CSSeriesTransmitted" },
          { 0x20011062UL, "CSSeriesCommitted" },
          { 0x20011063UL, "CSExaminationSource" },
          { 0x20011064UL, "SHTextType" },
          { 0x20011065UL, "SQOverlayPlane" },
          { 0x20011067UL, "CSLinearPresentationGLTrafoShapeSub" },
          { 0x20011068UL, "SQLinearModalityGLTrafo" },
          { 0x20011069UL, "SQDisplayShutter" },
          { 0x2001106AUL, "SQSpatialTransformation" },
          { 0x2001106DUL, "LOTextFontSpatialTransformation" },
          { 0x2001106EUL, "SHSeriesType" },
          { 0x20011071UL, "CSGraphicConstraint" },
          { 0x20011072UL, "ISEllipsDisplShutOtherAxScndEndPnt" },
          { 0x20011076UL, "UINumberOfFramesMR" },
          { 0x20011077UL, "CSGLTrafoType" },
          { 0x2001107AUL, "FLWindowRoundingFactor" },
          { 0x2001107BUL, "ISMRSeriesAcquisitionNumber" },
          { 0x2001107CUL, "USFrameNumber" },
          { 0x20011080UL, "LOTextAnchorPointAlignment" },
          { 0x20011081UL, "ISNumberOfDynamicScans" },
          { 0x20011082UL, "ISMRSeriesEchoTrainLength" },
          { 0x20011083UL, "DSMRSeriesImagingFrequency" },
          { 0x20011084UL, "DSMRSeriesInversionTime" },
          { 0x20011085UL, "DSMRSeriesMagneticFieldStrength" },
          { 0x20011086UL, "ISMRSeriesNrOfPhaseEncodingSteps" },
          { 0x20011087UL, "SHMRSeriesNucleus" },
          { 0x20011088UL, "DSMRSeriesNumberOfAverages" },
          { 0x20011089UL, "DSMRSeriesPercentPhaseFieldOfView" },
          { 0x2001108AUL, "DSMRSeriesPercentSampling" },
          { 0x2001108BUL, "SHMRSeriesTransmittingCoil" },
          { 0x20011090UL, "LOTextForegroundColor" },
          { 0x20011091UL, "LOTextBackgroundColor" },
          { 0x20011092UL, "LOTextShadowColor" },
          { 0x20011093UL, "LOTextStyle" },
          { 0x2001109BUL, "ULGraphicNumber" },
          { 0x2001109CUL, "LOGraphicAnnotationLabel" },
          { 0x2001109FUL, "USPixelProcessingKernelSize" },
          { 0x200110A3UL, "ULTextColorForeground" },
          { 0x200110A4UL, "ULTextColorBackground" },
          { 0x200110A5UL, "ULTextColorShadow" },
          { 0x200110C8UL, "LOExamCardName" },
          { 0x200110CCUL, "STSeriesDerivationDescription" },
          { 0x200110F1UL, "FLProspectiveMotionCorrection" },
          { 0x200110F2UL, "FLRetrospectiveMotionCorrection" },
          { 0x20019000UL, "SQPhilips_2001_9000" },
          { 0x20050000UL, "ULPhilips_2005_0000" },
          { 0x20050010UL, "LOPhilips_2005_0010" },
          { 0x20050011UL, "LOPhilips_2005_0011" },
          { 0x20050012UL, "LOPhilips_2005_0012" },
          { 0x20050013UL, "LOPhilips_2005_0013" },
          { 0x20050014UL, "LOPhilips_2005_0014" },
          { 0x20051000UL, "FLMRImageAngulationAP" },
          { 0x20051001UL, "FLMRImageAngulationFH" },
          { 0x20051002UL, "FLMRImageAngulationRL" },
          { 0x20051004UL, "CSMRImageDisplayOrientation" },
          { 0x20051008UL, "FLMRImageOffCentreAP" },
          { 0x20051009UL, "FLMRImageOffCentreFH" },
          { 0x2005100AUL, "FLMRImageOffCentreRL" },
          { 0x2005100BUL, "FLMRMaxFP" },
          { 0x2005100CUL, "FLMRMinFP" },
          { 0x2005100DUL, "FLMRScaleIntercept" },
          { 0x2005100EUL, "FLMRScaleSlope" },
          { 0x2005100FUL, "DSWindowCenterOriginal" },
          { 0x20051010UL, "DSWindowWidthOriginal" },
          { 0x20051011UL, "CSMRImageTypeMR" },
          { 0x20051012UL, "CSMRCardiacGating" },
          { 0x20051013UL, "CSMRSeriesDevelopmentMode" },
          { 0x20051014UL, "CSMRSeriesDiffusion" },
          { 0x20051015UL, "CSMRFatSaturationTechnique" },
          { 0x20051016UL, "CSMRFlowCompensation" },
          { 0x20051017UL, "CSMRFourierInterpolation" },
          { 0x20051018UL, "LOMRHardcopyProtocol" },
          { 0x20051019UL, "CSMRInverseReconstructed" },
          { 0x2005101AUL, "SSMRLabelSyntax" },
          { 0x2005101BUL, "CSMRMagnetiPrepared" },
          { 0x2005101CUL, "CSMRMagnetTransferConst" },
          { 0x2005101DUL, "SSMRMeasurementScanResolution" },
          { 0x2005101EUL, "SHMIPProtocol" },
          { 0x2005101FUL, "SHMPRProtocol" },
          { 0x20051020UL, "SLNumberOfChemicalShift" },
          { 0x20051021UL, "SSMRNumberOfMixes" },
          { 0x20051022UL, "ISMRNumberOfReferences" },
          { 0x20051023UL, "SSMRNumberOfSlabs" },
          { 0x20051025UL, "SSMRNumberOfVolumes" },
          { 0x20051026UL, "CSMROverSampleingPhase" },
          { 0x20051027UL, "CSMRPackageMode" },
          { 0x20051028UL, "CSMRPartialFourierFrequency" },
          { 0x20051029UL, "CSMRPartialFourierPhase" },
          { 0x2005102AUL, "ISMRPatientReferenceID" },
          { 0x2005102BUL, "SSMRPercentScanComplete" },
          { 0x2005102CUL, "CSMRPhaseEncodedRecording" },
          { 0x2005102DUL, "ISMRPlanScanSurveyNumberOfImages" },
          { 0x2005102EUL, "CSMRPPGPPUGating" },
          { 0x2005102FUL, "CSMRSpatialPresaturation" },
          { 0x20051030UL, "FLMRSeriesRepetitionTime" },
          { 0x20051031UL, "CSMRRespiratoryGating" },
          { 0x20051032UL, "CSPhilips_2005_1032" },
          { 0x20051033UL, "FLMRSeriesScanDuration" },
          { 0x20051034UL, "CSMRSegmentedKSpace" },
          { 0x20051035UL, "CSMRSeriesDataType" },
          { 0x20051036UL, "CSMRSeriesIsCardiac" },
          { 0x20051037UL, "CSMRSeriesIsSpectro" },
          { 0x20051038UL, "CSMRSpoiled" },
          { 0x20051039UL, "CSMRSteadyState" },
          { 0x2005103AUL, "SHMRSubAnatomy" },
          { 0x2005103BUL, "CSMRTimeReversedSteadyState" },
          { 0x2005103CUL, "CSMRSeriesTone" },
          { 0x2005103DUL, "SSMRNumberOfRRIntervalRanges" },
          { 0x2005103EUL, "SLMRRRIntervalsDistribution" },
          { 0x2005103FUL, "SLMRPlanScanAcquisitionNo" },
          { 0x20051040UL, "SLMRChemicalShiftNo" },
          { 0x20051041UL, "SLMRPlanScanDynamicScanNo" },
          { 0x20051042UL, "SLMRPlanScanSurveyEchoNo" },
          { 0x20051043UL, "CSMRPlanScanImageType" },
          { 0x20051044UL, "SLMRPlanScanPhaseNo" },
          { 0x20051045UL, "SLMRPlanScanReconstructionNo" },
          { 0x20051046UL, "CSMRPlanScanScanningSequence" },
          { 0x20051047UL, "SLMRPlanScanSliceNo" },
          { 0x20051048UL, "ISMRReferenceAcquisitionNo" },
          { 0x20051049UL, "ISMRReferenceChemicalShiftNo" },
          { 0x2005104AUL, "ISMRReferenceDynamicScanNo" },
          { 0x2005104BUL, "ISMRReferenceEchoNo" },
          { 0x2005104CUL, "CSMRReferenceEntity" },
          { 0x2005104DUL, "CSMRReferenceImageType" },
          { 0x2005104EUL, "FLMRSlabFovRL" },
          { 0x2005104FUL, "FLMRSlabOffcentreAP" },
          { 0x20051050UL, "FLMRSlabOffcentreFH" },
          { 0x20051051UL, "FLMRSlabOffcentreRL" },
          { 0x20051052UL, "CSMRSlabType" },
          { 0x20051053UL, "CSMRSlabViewAxis" },
          { 0x20051054UL, "FLMRVolumeAngulationAP" },
          { 0x20051055UL, "FLMRVolumeAngulationFH" },
          { 0x20051056UL, "FLMRVolumeAngulationRL" },
          { 0x20051057UL, "FLMRVolumeFovAP" },
          { 0x20051058UL, "FLMRVolumeFovFH" },
          { 0x20051059UL, "FLMRVolumeFovRL" },
          { 0x2005105AUL, "FLMRVolumeOffcentreAP" },
          { 0x2005105BUL, "FLMRVolumeOffcentreFH" },
          { 0x2005105CUL, "FLMRVolumeOffcentreRL" },
          { 0x2005105DUL, "CSMRVolumeType" },
          { 0x2005105EUL, "CSMRVolumeViewAxis" },
          { 0x2005105FUL, "CSMRStudyOrigin" },
          { 0x20051060UL, "ISMRStudySequenceNumber" },
          { 0x20051061UL, "CSMRImagePrepulseType" },
          { 0x20051063UL, "SSMRfMRIStatusIndication" },
          { 0x20051064UL, "ISMRReferencePhaseNo" },
          { 0x20051065UL, "ISMRReferenceReconstructionNo" },
          { 0x20051066UL, "CSMRReferenceScanningSequence" },
          { 0x20051067UL, "ISMRReferenceSliceNo" },
          { 0x20051068UL, "CSMRReferenceType" },
          { 0x20051069UL, "FLMRSlabAngulationAP" },
          { 0x2005106AUL, "FLMRSlabAngulationFH" },
          { 0x2005106BUL, "FLMRSlabAngulationRL" },
          { 0x2005106CUL, "FLMRSlabFovAP" },
          { 0x2005106DUL, "FLMRSlabFovFH" },
          { 0x2005106EUL, "CSMRImageScanningSequencePrivate" },
          { 0x2005106FUL, "CSMRAcquisitionType" },
          { 0x20051070UL, "LOMRSeriesHardcopyProtocolEV" },
          { 0x20051071UL, "FLMRStackAngulationAP" },
          { 0x20051072UL, "FLMRStackAngulationFH" },
          { 0x20051073UL, "FLMRStackAngulationRL" },
          { 0x20051074UL, "FLMRStackFovAP" },
          { 0x20051075UL, "FLMRStackFovFH" },
          { 0x20051076UL, "FLMRStackFovRL" },
          { 0x20051078UL, "FLMRStackOffCentreAP" },
          { 0x20051079UL, "FLMRStackOffCentreFH" },
          { 0x2005107AUL, "FLMRStackOffCentreRL" },
          { 0x2005107BUL, "CSMRStackPreparationDirection" },
          { 0x2005107EUL, "FLMRStackSliceDistance" },
          { 0x20051080UL, "SQSeriesPlanScan" },
          { 0x20051081UL, "CSMRStackViewAxis" },
          { 0x20051083UL, "SQSeriesSlab" },
          { 0x20051084UL, "SQSeriesReference" },
          { 0x20051085UL, "SQSeriesVolume" },
          { 0x20051086UL, "SSMRNumberOfGeometry" },
          { 0x20051087UL, "SLMRNumberOfGeometrySlices" },
          { 0x20051088UL, "FLMRGeomAngulationAP" },
          { 0x20051089UL, "FLMRGeomAngulationFH" },
          { 0x2005108AUL, "FLMRGeomAngulationRL" },
          { 0x2005108BUL, "FLMRGeomFOVAP" },
          { 0x2005108CUL, "FLMRGeomFOVFH" },
          { 0x2005108DUL, "FLMRGeomFOVRL" },
          { 0x2005108EUL, "FLMRGeomOffCentreAP" },
          { 0x2005108FUL, "FLMRGeomOffCentreFH" },
          { 0x20051090UL, "FLMRGeomOffCentreRL" },
          { 0x20051091UL, "CSMRGeomPreparationDirect" },
          { 0x20051092UL, "FLMRGeomRadialAngle" },
          { 0x20051093UL, "CSMRGeomRadialAxis" },
          { 0x20051094UL, "FLMRGeomSliceDistance" },
          { 0x20051095UL, "SLMRGeomSliceNumber" },
          { 0x20051096UL, "CSMRGeomType" },
          { 0x20051097UL, "CSMRGeomViewAxis" },
          { 0x20051098UL, "CSMRGeomColour" },
          { 0x20051099UL, "CSMRGeomApplicationType" },
          { 0x2005109AUL, "SLMRGeomId" },
          { 0x2005109BUL, "SHMRGeomApplicationName" },
          { 0x2005109CUL, "SHMRGeomLabelName" },
          { 0x2005109DUL, "CSMRGeomLineStyle" },
          { 0x2005109EUL, "SQSeriesGeom" },
          { 0x2005109FUL, "CSMRSeriesSpectralSelectiveExcitationPulse" },
          { 0x200510A0UL, "FLMRImageDynamicScanBeginTime" },
          { 0x200510A1UL, "CSSyncraScanType" },
          { 0x200510A2UL, "CSMRIsCOCA" },
          { 0x200510A3UL, "ISMRStackCoilID" },
          { 0x200510A4UL, "ISMRStackCBBCoil1" },
          { 0x200510A5UL, "ISMRStackCBBCoil2" },
          { 0x200510A6UL, "ISMRStackChannelCombi" },
          { 0x200510A7UL, "CSMRStackCoilConn" },
          { 0x200510A8UL, "DSMRPrivateInversionTime" },
          { 0x200510A9UL, "CSMRSeriesGeometryCorrection" },
          { 0x200510B0UL, "FLDiffusionDirectionX" },
          { 0x200510B1UL, "FLDiffusionDirectionY" },
          { 0x200510B2UL, "FLDiffusionDirectionZ" },
          { 0x200510C0UL, "CSSequenceVariant" },
          { 0x20051134UL, "LTSeriesTransactionUID" },
          { 0x20051199UL, "ULMRNumberOfRequestExcerpts" },
          { 0x20051200UL, "ULMRNumberOfSOPCommon" },
          { 0x20051201UL, "ULMRNoOfFilmConsumption" },
          { 0x20051213UL, "ULMRNumberOfCodes" },
          { 0x20051234UL, "SLPhilips_2005_1234" },
          { 0x20051243UL, "SSMRNoDateOfLastCalibration" },
          { 0x20051244UL, "SSMRNoTimeOfLastCalibration" },
          { 0x20051245UL, "SSMRNrOfSoftwareVersion" },
          { 0x20051247UL, "SSMRNrOfPatientOtherNames" },
          { 0x20051248UL, "SSMRNrOfReqRecipeOfResults" },
          { 0x20051249UL, "SSMRNrOfSeriesOperatorsName" },
          { 0x20051250UL, "SSMRNrOfSeriesPerfPhysiName" },
          { 0x20051251UL, "SSMRNrOfStudyAdmittingDiagnosticDescr" },
          { 0x20051252UL, "SSMRNrOfStudyPatientContrastAllergies" },
          { 0x20051253UL, "SSMRNrOfStudyPatientMedicalAlerts" },
          { 0x20051254UL, "SSMRNrOfStudyPhysiciansOfRecord" },
          { 0x20051255UL, "SSMRNrOfStudyPhysiReadingStudy" },
          { 0x20051256UL, "SSMRNrSCSoftwareVersions" },
          { 0x20051257UL, "SSMRNrRunningAttributes" },
          { 0x20051281UL, "UIDefaultImageUID" },
          { 0x20051282UL, "CSRunningAttributes" },
          { 0x20051300UL, "SSMRSpectromExtraNumber" },
          { 0x20051301UL, "SSMRSpectrumKxCoordinate" },
          { 0x20051302UL, "SSMRSpectrumKyCoordinate" },
          { 0x20051303UL, "SSMRSpectrumLocationNumber" },
          { 0x20051304UL, "SSMRSpectrumMixNumber" },
          { 0x20051305UL, "SSMRSpectrumXCoordinate" },
          { 0x20051306UL, "SSMRSpectrumYCoordinate" },
          { 0x20051307UL, "FLMRSpectrumDCLevel" },
          { 0x20051308UL, "FLMRSpectrumNoiseLevel" },
          { 0x20051309UL, "FLMRSpectrumBeginTime" },
          { 0x20051310UL, "FLMRSpectrumEchoTime" },
          { 0x20051312UL, "FLSpectrumNumber" },
          { 0x20051313UL, "SSMRSpectrumNumber" },
          { 0x20051314UL, "SSMRSpectrumNumberOfAverages" },
          { 0x20051315UL, "SSMRSpectrumNumberOfSamples" },
          { 0x20051316UL, "SSMRSpectrumScanSequenceNumber" },
          { 0x20051317UL, "SSMRSpectrumNumberOfPeaks" },
          { 0x20051318UL, "SQMRSpectrumPeak" },
          { 0x20051319UL, "FLMRSpectrumPeakIntensity" },
          { 0x20051320UL, "LOMRSpectrumPeakLabel" },
          { 0x20051321UL, "FLMRSpectrumPeakPhase" },
          { 0x20051322UL, "FLMRSpectrumPeakPosition" },
          { 0x20051323UL, "CSMRSpectrumPeakType" },
          { 0x20051324UL, "FLMRSpectrumPeakWidth" },
          { 0x20051325UL, "CSMRSpectroSIB0Correction" },
          { 0x20051326UL, "FLMRSpectroB0EchoTopPosition" },
          { 0x20051327UL, "CSMRSpectroComplexComponent" },
          { 0x20051328UL, "CSMRSpectroDataOrigin" },
          { 0x20051329UL, "FLMRSpectroEchoTopPosition" },
          { 0x20051330UL, "CSMRInPlaneTransforms" },
          { 0x20051331UL, "SSMRNumberOfSpectraAcquired" },
          { 0x20051333UL, "FLMRPhaseEncodingEchoTopPositions" },
          { 0x20051334UL, "CSMRPhysicalQuantityForChemicalShift" },
          { 0x20051335UL, "CSMRPhysicalQuantitySpatial" },
          { 0x20051336UL, "FLMRReferenceFrequency" },
          { 0x20051337UL, "FLMRSampleOffset" },
          { 0x20051338UL, "FLMRSamplePitch" },
          { 0x20051339UL, "SSMRSearchIntervalForPeaks" },
          { 0x20051340UL, "CSMRSignalDomainForChemicalShift" },
          { 0x20051341UL, "CSMRSignalDomainSpatial" },
          { 0x20051342UL, "CSMRSignalType" },
          { 0x20051343UL, "CSMRSpectroAdditionalRotations" },
          { 0x20051344UL, "SSMRSpectroDisplayRanges" },
          { 0x20051345UL, "CSMRSpectroEchoAcquisition" },
          { 0x20051346UL, "CSMRSpectroFrequencyUnit" },
          { 0x20051347UL, "FLMRSpectroGamma" },
          { 0x20051348UL, "CSMRSpectroHiddenLineRemoval" },
          { 0x20051349UL, "FLMRSpectroHorizontalShift" },
          { 0x20051350UL, "FLMRSpectroHorizontalWindow" },
          { 0x20051351UL, "SSMRSpectroNumberOfDisplayRanges" },
          { 0x20051352UL, "SSMRSpectroNumberOfEchoPulses" },
          { 0x20051353UL, "LOMRSpectroProcessingHistory" },
          { 0x20051354UL, "CSMRSpectroScanType" },
          { 0x20051355UL, "FLMRSpectroSICSIntervals" },
          { 0x20051356UL, "CSMRSpectroSIMode" },
          { 0x20051357UL, "SSMRSpectroSpectralBW" },
          { 0x20051358UL, "LOMRSpectroTitleLine" },
          { 0x20051359UL, "FLMRSpectroTurboEchoSpacing" },
          { 0x20051360UL, "FLMRSpectroVerticalShift" },
          { 0x20051361UL, "FLMRSpectroVerticalWindow" },
          { 0x20051362UL, "FLMRSpectroOffset" },
          { 0x20051363UL, "FLMRSpectrumPitch" },
          { 0x20051364UL, "CSMRVolumeSelection" },
          { 0x20051370UL, "SSMRNoMixesSpectro" },
          { 0x20051371UL, "SQMRSeriesSPMix" },
          { 0x20051372UL, "SSSPMixTResolution" },
          { 0x20051373UL, "SSSPMixKXResolution" },
          { 0x20051374UL, "SSSPMixKYResolution" },
          { 0x20051375UL, "SSSPMixFResolution" },
          { 0x20051376UL, "SSSPMixXResolution" },
          { 0x20051377UL, "SSSPMixYResolution" },
          { 0x20051378UL, "SSSPMixNoSpectraIntended" },
          { 0x20051379UL, "SSSPMixNoAverages" },
          { 0x20051380UL, "SLMRSeriesNrOfMFImageObjects" },
          { 0x20051381UL, "ISMRScanoGramSurveyNumberOfImages" },
          { 0x20051382UL, "ULMRNumberOfProcedureCodes" },
          { 0x20051383UL, "CSSortAttributes" },
          { 0x20051384UL, "SSMRNrSortAttributes" },
          { 0x20051385UL, "CSImageDisplayOrientation" },
          { 0x20051386UL, "CSInsetScanogram" },
          { 0x20051387UL, "SSMRDisplayLayoutNrColumns" },
          { 0x20051388UL, "SSMRDisplayLayoutNrRows" },
          { 0x20051389UL, "SQViewingProtocol" },
          { 0x20051390UL, "CSMRStackCoilFunction" },
          { 0x20051391UL, "PNPhilips_2005_1391" },
          { 0x20051392UL, "ISMRGeolinkID" },
          { 0x20051393UL, "ISMRStationNo" },
          { 0x20051394UL, "CSProcessingHistory" },
          { 0x20051395UL, "UIViewProcedureString" },
          { 0x20051396UL, "CSMRFlowImagesPresent" },
          { 0x20051397UL, "LOAnatomicRegCodeValue" },
          { 0x20051398UL, "CSMRMobiviewEnabled" },
          { 0x20051399UL, "CSMRIViewBoldEnabled" },
          { 0x20051400UL, "CSMRVolumeViewEnabled" },
          { 0x20051401UL, "ULMRNumberOfStudyReference" },
          { 0x20051402UL, "SQSPSCode" },
          { 0x20051403UL, "ULMRNumberOfSPSCodes" },
          { 0x20051406UL, "SSPhilips_2005_1406" },
          { 0x20051407UL, "SSMRNrOfSpecificCharacterSet" },
          { 0x20051409UL, "DSRescaleInterceptOriginal" },
          { 0x2005140AUL, "DSRescaleSlopeOriginal" },
          { 0x2005140BUL, "LORescaleTypeOriginal" },
          { 0x2005140EUL, "SQPrivateSharedSq" },
          { 0x2005140FUL, "SQPrivatePerFrameSq" },
          { 0x20051410UL, "ISMFConvTreatSpectorMixNo" },
          { 0x20051411UL, "UIMFPrivateReferencedSOPInstanceUID" },
          { 0x20051412UL, "ISMRImageDiffBValueNumber" },
          { 0x20051413UL, "ISMRImageGradientOrientationNumber" },
          { 0x20051414UL, "SLMRSeriesNrOfDiffBValues" },
          { 0x20051415UL, "SLMRSeriesNrOfDiffGradOrients" },
          { 0x20051416UL, "CSRSeriesPlanMode" },
          { 0x20051417UL, "FDB_matrix" },
          { 0x20051418UL, "CSPrivOperatingModeType" },
          { 0x20051419UL, "CSPrivOperatingMode" },
          { 0x2005141AUL, "CSMRFatSaturationTechnique" },
          { 0x2005141BUL, "ISMRVersionNumberDeletedImages" },
          { 0x2005141CUL, "ISMRVersionNumberDeletedSpectra" },
          { 0x2005141DUL, "ISMRVersionNumberDeletedBlobsets" },
          { 0x2005141EUL, "ULLUT1Offset" },
          { 0x2005141FUL, "ULLUT1Range" },
          { 0x20051420UL, "ULLUT1BeginColor" },
          { 0x20051421UL, "ULLUT1EndColor" },
          { 0x20051422UL, "ULLUT2Offset" },
          { 0x20051423UL, "ULLUT2Range" },
          { 0x20051424UL, "ULLUT2BeginColor" },
          { 0x20051425UL, "ULLUT2EndColor" },
          { 0x20051426UL, "CSViewingHardcopyOnly" },
          { 0x20051427UL, "SQPrivateEMR" },
          { 0x20051428UL, "SLMRSeriesNrOfLabelTypes" },
          { 0x20051429UL, "CSMRImageLabelType" },
          { 0x2005142AUL, "CSExamPrintStatus" },
          { 0x2005142BUL, "CSExamExportStatus" },
          { 0x2005142CUL, "CSExamStorageCommitStatus" },
          { 0x2005142DUL, "CSExamMediaWriteStatus" },
          { 0x2005142EUL, "FLMRSeriesDBdt" },
          { 0x2005142FUL, "FLMRSeriesProtonSAR" },
          { 0x20051430UL, "FLMRSeriesNonProtonSAR" },
          { 0x20051431UL, "FLMRSeriesLocalSAR" },
          { 0x20051432UL, "CSMRSeriesSafetyOverrideMode" },
          { 0x20051433UL, "DTEVDVDJobInParamsDatetime" },
          { 0x20051434UL, "DTDVDJobInParamsVolumeLabel" },
          { 0x20051435UL, "CSSpectroExamcard" },
          { 0x20051436UL, "UIMRRefSeriesInstanceUID" },
          { 0x20051437UL, "CSColorLUTType" },
          { 0x2005143AUL, "LTDataDictionaryContentsVersion" },
          { 0x2005143BUL, "CSMRIsCoilSurvey" },
          { 0x2005143CUL, "FLMRStackTablePosLong" },
          { 0x2005143DUL, "FLMRStackTablePosLat" },
          { 0x2005143EUL, "FLMRStackPosteriorCoilPos" },
          { 0x2005143FUL, "CSAIMDLimitsApplied" },
          { 0x20051440UL, "FLAIMDHeadSARLimit" },
          { 0x20051441UL, "FLAIMDWholeBodySARLimit" },
          { 0x20051442UL, "FLAIMDB1RMSLimit" },
          { 0x20051443UL, "FLAIMDdbDtLimit" },
          { 0x20051444UL, "ISTFEFactor" },
          { 0x20051445UL, "CSAttenuationCorrection" },
          { 0x20051446UL, "FLFWHMShim" },
          { 0x20051447UL, "FLPowerOptimization" },
          { 0x20051448UL, "FLCoilQ" },
          { 0x20051449UL, "FLReceiverGain" },
          { 0x2005144AUL, "FLDataWindowDuration" },
          { 0x2005144BUL, "FLMixingTime" },
          { 0x2005144CUL, "FLFirstEchoTime" },
          { 0x2005144DUL, "CSIsB0Series" },
          { 0x2005144EUL, "CSIsB1Series" },
          { 0x2005144FUL, "CSVolumeSelect" },
          { 0x20051450UL, "SSMRNrOfPatientOtherIDs" },
          { 0x20051451UL, "ISPrivateSeriesNumber" },
          { 0x20051452UL, "UIPrivateSeriesInstanceUID" },
          { 0x20051453UL, "CSSplitSeriesJobParams" },
          { 0x20051454UL, "SSPreferredDimensionForSplitting" },
          { 0x20051455UL, "FDImageVelocityEncodingDirection" },
          { 0x20051456UL, "SSContrastBolusNoInjections" },
          { 0x20051457UL, "LTContrastBolusAgentCode" },
          { 0x20051458UL, "LTContrastBolusAdminRouteCode" },
          { 0x20051459UL, "DSContrastBolusVolume" },
          { 0x2005145AUL, "DSContrastBolusIngredientConcentration" },
          { 0x2005145BUL, "ISContrastBolusDynamicNuumber" },
          { 0x2005145CUL, "SQSeriesBolusContrast" },
          { 0x2005145DUL, "ISContrastBolusID" },
          { 0x20051460UL, "CSLUTToRGBJobParams" },
          { 0x20500000UL, "ULPhilips_2050_0000" },
          { 0x20500020UL, "CSPresentationLUTShape" },
          { 0x52000000UL, "ULPhilips_5200_0000" },

          // GE private tags:
          { 0x001910BBUL, "DSGEGradDirX" },
          { 0x001910BCUL, "DSGEGradDirY" },
          { 0x001910BDUL, "DSGEGradDirZ" },
          { 0x00431039UL, "ISGEMS_PARMS_01" },

          // Siemens private tags:
          { 0x0019100CUL, "DSSiemensBValue" },
          { 0x0019100EUL, "FDSiemensDWDirection" },
          { 0x00291010UL, "UNSiemensCSA_1010" },
          { 0x00291020UL, "UNSiemensCSA_1020" },
          { 0x00291110UL, "UNSiemensCSA_1110" },
          { 0x00291120UL, "UNSiemensCSA_1120" },
          { 0x00291210UL, "UNSiemensCSA_1210" },
          { 0x00291220UL, "UNSiemensCSA_1220" }
        };
      }

    }
  }
}

