/******************************************************************************
 * This file is part of MPlayer Audio Visualization.                          *
 *                                                                            *
 *                                                                            *
 *  MPlayer Audio Visualization is free software; you can redistribute it     *
 *  and/or modify it under the terms of the GNU General Public License as     *
 *  published by the Free Software Foundation; either version 2 of the        *
 *  License, or (at your option) any later version.                           *
 *                                                                            *
 *  MPlayer Audio Visualization is distributed in the hope that it will be    *
 *  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General *
 *  Public License for more details.                                          *
 *                                                                            * 
 *  You should have received a copy of the GNU General Public License         *
 *  along with MPlayer Audio Visualization; if not, write to the Free         *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA         *
 *  02111-1307  USA                                                           *
 ******************************************************************************/
#ifndef _H_MSGLIST_
#define _H_MSGLIST_

#include <SDL.h>
#include <SDL/SDL_ttf.h>
#include <stdlib.h>
#include <string.h>
#include <mpav.h>

typedef struct msglist_s
{
  struct msglist_s *next;  /**< next in list */
  TTF_Font *font;          /**< font to use */
  int x;                   /**< x coordinate */
  int y;                   /**< y coordinate */
  int shx;                 /**< shadow offset in x */
  int shy;                 /**< shadow offset in y */
  char *msg;               /**< message */
  Uint32 fgcolor;          /**< foreground color */
  Uint32 shcolor;          /**< shadow color */
  int shadow;              /**< use shadow? */
} msglist_t;

#define msgnode_t msglist_t /**< alias to turn code more legible */

/**
 * add the node to the end of the list
 *
 * @param list the list you want the node added
 * @param node the node you want to add
 * @return 0 if ok, -1 if list is Null
 */
int msglist_addnode( msglist_t **list, msgnode_t *node );

/**
 * remove the given node from list and return it
 *
 * @param list the list you want the node removed
 * @param node the node you want to remove
 * @return the removed node or NULL if it doesn't exists in list
 */
msgnode_t *msglist_delnode( msglist_t **list, msgnode_t *node );

/**
 * prints the entire list 
 *
 * @param the list to be printed
 */
void msglist_print( msglist_t *list );

/**
 * create node
 *
 * @param msg the message string
 * @param fontfile the font file (must be the entire path)
 * @param ptsize the font size
 * @param x the x coordinate
 * @param y the y coordinate
 * @param fgcolor the color in 0xRRGGBB (ie: 0xffffff is white)
 * @param shadow if should use shadow or not
 * @param shx if using shadow, the x offset
 * @param shy if using shadow, the y offset
 * @param shcolor if using shadow, it's color
 * @return the created node
 */
msgnode_t *msglist_createnode( char *msg,
			       char *fontfile, int ptsize,
			       int x, int y, Uint32 fgcolor,
			       int shadow, int shx, int shy, Uint32 shcolor
			       );

/**
 * create node without shadow (just a wrapper to 
 * msglist_create())
 *
 * @param msg the message string
 * @param fontfile the font file (must be the entire path)
 * @param ptsize the font size
 * @param x the x coordinate
 * @param y the y coordinate
 * @param fgcolor the color in 0xRRGGBB (ie: 0xffffff is white)
 * @return the created node
 *
 * @see msglist_create()
 */ 
msgnode_t *msglist_createnode_nsh( char *msg,
				   char *fontfile, int ptsize,
				   int x, int y, Uint32 fgcolor			       
				   );

/**
 * destroy node
 *
 * @param the node to be destroyed. 
 * @warning will be free()'ed
 */
int msglist_destroynode( msgnode_t **node );

#endif // _H_MSGLIST_
