/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef HISTORYITEM_H
#define HISTORYITEM_H

// Qt stuffs
#include <QString>
#include <QVariant>

// CamiTK stuffs
#include <Component.h>

namespace camitk {

  /**
   * @class HistoryItem
   * @brief HistoryItem class describes the entry of an action used in a pipeline, in the history.
   * 
   * In CamiTK, every action is stored in a history. Each history entry is an instance of @class{HistoryItem} 
   * Each history item provides information about the processed action :
   * - the name of the action processed
   * - the properties (name, values)
   * - its input components
   * - its output components
   * History item are mainly useful for getting back to a previous state, undoing an action or for scripting, storing in a file a pipeline of actions.
   * 
   * The history is stored in the Application class and uses the "construct on first use" idiom/design-pattern (with singletons).
   * It therefore avoids the infamous "static initialization order fiasco",
   * see http://www.parashift.com/c++-faq/ctors.html
   */
class CAMITK_API HistoryItem
{
private:
  
  /// The name of the action in the pipeline
  QString name;
  /// The properties stored for this action
  QMap<QByteArray, QVariant> properties;
  /// The input components for this action
  ComponentList inputComponents;
  /// The outpu components for this action
  ComponentList outputComponents;

public:
  /// Empty constructor
  HistoryItem();
  
  /// Default constructor
  /// @param name
  /// name of the action stored in the history
  HistoryItem(QString name);
  
  /// Virtual destructor
  virtual ~HistoryItem();
  
  /// Add a property of the corresponding action to the history item
  /// @param name Name of the property to addProperty
  /// @param value Value of the property to add (no type specified)
  void addProperty(QByteArray name, QVariant value);
  
  /// Set the input components of the history item action
  /// @param inputComponents The list of input components
  void setInputComponents(ComponentList inputComponents);
  
  /// Set the output components of the historu item action
  /// @param outputComponents The list of output components
  void setOutputComponents(ComponentList outputComponents);
  
};

}

#endif // HISTORYITEM_H
