; APT (Automated Program Transformations) Library
;
; Copyright (C) 2021 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "APT")

(include-book "std/util/define" :dir :system)
(include-book "kestrel/axe/rewriter-basic" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define solve-call-axe-rewriter ((matrix pseudo-termp)
                                 (method-rules symbol-listp)
                                 ctx
                                 state)
  :returns (mv erp (rewritten-term "A @(tsee pseudo-termp).") state)
  :mode :program
  :parents (solve-implementation)
  :short "Call the Axe rewriter on the matrix of @('old')."
  :long
  (xdoc::topstring
   (xdoc::p
    "Including the file @('axe/rewriter-basic.lisp')
     provides an Axe function @('acl2::simp-term-basic'),
     which is generated by a macro (so it cannot be directly seen in any file).
     This is the function that we call to rewrite a term.")
   (xdoc::p
    "This Axe rewriter function takes the following inputs:")
   (xdoc::ul
    (xdoc::li
     "The term to rewrite. Here we pass the matrix of @('old').")
    (xdoc::li
     "Assumptions under which the rewriting must occur.
      Here we pass @('nil'), as we rewrite unconditionally.")
    (xdoc::li
     "A rule alist, which can be constructed by calling the Axe function
      @('acl2::make-rule-alist') on a list of symbol and an ACL2 world.
      We call this function here,
      passing the method rules supplied by the user.")
    (xdoc::li
     "An alist of interpreted functions, which we set to @('nil') here.")
    (xdoc::li
     "A list of monitored symbols, which we set to @('nil') here.")
    (xdoc::li
     "A memoization flag, which we set to @('nil') here.")
    (xdoc::li
     "A hit count flag, which we set to @('nil') here.")
    (xdoc::li
     "An ACL2 world, which is straightforward."))
   (xdoc::p
    "This Axe rewrite function returns the following outputs:")
   (xdoc::ul
    (xdoc::li
     "An error flag. If non-@('nil'), we fail.")
    (xdoc::li
     "A DAG, which represents the rewritten term in DAG form."))
   (xdoc::p
    "We use the Axe function @('acl2::dag-to-term')
     to turn the DAG into a term."))
  (b* ((wrld (w state))
       ((mv erp rule-alist) (acl2::make-rule-alist method-rules wrld))
       ((when erp) (er-soft+ ctx t nil
                             "Failed to make Axe rules from the rules ~x0."
                             method-rules))
       ((mv erp term) (acl2::simp-term-basic matrix
                                             nil ; assumptions
                                             rule-alist
                                             nil ; interpreted-function-alist
                                             nil ; monitored-symbols
                                             nil ; memoizep
                                             nil ; count-hitsp
                                             wrld))
       ((when erp) (er-soft+ ctx t nil
                             "The Axe rewiter failed on the term ~x0, ~
                              with error ~x1 and result ~x2."
                             matrix erp term)))
    (value term)))
