; SV - Symbolic Vector Hardware Analysis Framework
; Copyright (C) 2014-2015 Centaur Technology
;
; Contact:
;   Centaur Technology Formal Verification Group
;   7600-C N. Capital of Texas Highway, Suite 300, Austin, TX 78731, USA.
;   http://www.centtech.com/
;
; License: (An MIT/X11-style license)
;
;   Permission is hereby granted, free of charge, to any person obtaining a
;   copy of this software and associated documentation files (the "Software"),
;   to deal in the Software without restriction, including without limitation
;   the rights to use, copy, modify, merge, publish, distribute, sublicense,
;   and/or sell copies of the Software, and to permit persons to whom the
;   Software is furnished to do so, subject to the following conditions:
;
;   The above copyright notice and this permission notice shall be included in
;   all copies or substantial portions of the Software.
;
;   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
;   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
;   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
;   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
;   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
;   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
;   DEALINGS IN THE SOFTWARE.
;
; Original authors: Sol Swords <sswords@centtech.com>

(in-package "SV")

;; (include-book "debug")
(include-book "eval-phases")
(include-book "probe")
(include-book "fsm-obj")
(include-book "../svex/rewrite")
(include-book "../mods/moddb")
(include-book "../mods/path-string")
(include-book "std/strings/hexify" :dir :system)
(include-book "expand")
(include-book "../svex/env-ops")

;; (include-book "std/stobjs/updater-independence" :dir :system)
;; This tool operates from an initialized DEBUGDATA stobj, as generated by
;; svtv-debug-core, or (without dumping a vcd) by svtv-debug-init followed by
;; svtv-debug-set-svtv.

;; It takes an input alist and expands it to an environment for each phase and
;; initial state so that we can evaluate a signal at a given phase.

;; Then we start from some given signal (an output/internal name or path and
;; phase).  We can evaluate that signal using svex-eval-svtv-phases.



(define svex-alistlist-eval ((x svex-alistlist-p)
                             (env svex-env-p))
  :returns (envs svex-envlist-p)
  (if (atom x)
      nil
    (cons (svex-alist-eval (car x) env)
          (svex-alistlist-eval (cdr x) env))))

(define make-fast-alists (x)
  :enabled t
  (mbe :logic x
       :exec (if (atom x)
                 x
               (cons (make-fast-alist (car x))
                     (make-fast-alists (cdr x))))))



;; There are basically three types of signals/svex variables used here:
;;  - If V is a key of nextstates/delays, then it is a previous state signal.
;;  - If V is a key of updates/assigns, then it is an internal signal.
;;  - If V appears in an expression (value) of updates or nextstates but is not
;;  a key of either one, then it is a primary input.

;;    A previous state signal cannot also be an internal signal (a key of
;;    nextstates cannot also be a key of updates).  But each V that is a
;;    previous state/key of updates is also a key of delays, and the
;;    corresponding value in delays is an internal signal or primary input that
;;    is also the nextstate of V.

;; The keys of nextstates should be the same as those of delays.  The keys of
;; updates should be the same as those of assigns.


;; We want to support exploring the design by walking through following
;; dependencies back to drivers. Generally we'll have a stack starting from
;; some signal and going back though its drivers (across phases).

;; The current position will be tracked as a variable, phase, right-shift, and
;; mask.  The mask is relative to the right-shift, so the caremask for the
;; whole variable is mask << right-shift.

;; We'll find dependencies of a variable modulo the caremask, and the resulting
;; dependency variables will be reported along with their caremasks.
;; Internally we'll use right-shifts and masks, but when we print things out to
;; the user we'll want to translate back to relative indices.  E.g. if a
;; variable is declared as x[10:2] and we have a rightshift of 3 and mask
;; #b101, we'll show the range as x[7:5].



(define svtv-chase-eval ((var svar-p)
                         (phase integerp)
                         &key
                         ((updates svex-alist-p) 'updates)
                         ((evaldata svtv-evaldata-p) 'evaldata))
  :returns (val 4vec-p)
  (b* ((phase (lifix phase))
       (var (svar-fix var))
       ((when (< phase 0))
        (svex-env-lookup var (svtv-evaldata->initst evaldata)))
       (svex (svex-fastlookup var updates))
       ((when svex)
        (svex-eval-svtv-phases svex phase evaldata)))
    (svex-eval-svtv-phases (svex-var var) phase evaldata)))


(define svtv-chase-evallist ((vars svarlist-p)
                             (phase integerp)
                             &key
                             ((updates svex-alist-p) 'updates)
                             ((evaldata svtv-evaldata-p) 'evaldata))
  :returns (vals 4veclist-p)
  (if (atom vars)
      nil
    (cons (svtv-chase-eval (car vars) phase)
          (svtv-chase-evallist (cdr vars) phase)))
  ///
  (defret len-of-<fn>
    (equal (len vals) (len vars))))
       


(define svex-mask-alist-to-4vmask-alist ((x svex-mask-alist-p))
  :returns (new-x 4vmask-alist-p)
  (if (atom x)
      nil
    (if (and (mbt (and (consp (car x)) (svex-p (caar x))))
             (svex-case (caar x) :var)
             (not (sparseint-equal (cdar x) 0)))
        (cons (cons (svex-var->name (caar x))
                    (sparseint-fix (cdar x)))
              (svex-mask-alist-to-4vmask-alist (cdr x)))
      (svex-mask-alist-to-4vmask-alist (cdr x)))))


(local (defthm svarlist-p-alist-keys-when-4vmask-alist-p
         (implies (4vmask-alist-p x)
                  (svarlist-p (alist-keys x)))
         :hints(("Goal" :in-theory (enable alist-keys)))))


;; (local
;;  (defsection svarlist-addr-p-of-svexlist-compute-masks
;;    (defret member-vars-of-svex-args-apply-masks
;;      (implies (and (not (member v (svexlist-vars args)))
;;                    (not (member v (svexlist-vars (svex-mask-alist-keys mask-al)))))
;;               (not (member v (svexlist-vars (svex-mask-alist-keys mask-al1)))))
;;      :hints(("Goal" :in-theory (enable svex-args-apply-masks)))
;;      :fn svex-args-apply-masks)

;;    (defret member-vars-of-svexlist-compute-masks
;;      (implies (and (not (member v (svexlist-vars x)))
;;                    (not (member v (svexlist-vars (svex-mask-alist-keys mask-al)))))
;;               (not (member v (svexlist-vars (svex-mask-alist-keys mask-al1)))))
;;      :hints(("Goal" :in-theory (enable svexlist-compute-masks)))
;;      :fn svexlist-compute-masks)

;;    (defret member-alist-keys-of-svex-mask-alist-to-4vmask-alist
;;      (implies (not (member v (svexlist-vars (svex-mask-alist-keys x))))
;;               (not (member v (alist-keys new-x))))
;;      :hints(("Goal" :in-theory (enable svex-mask-alist-to-4vmask-alist
;;                                        svex-mask-alist-keys
;;                                        alist-keys)))
;;      :fn svex-mask-alist-to-4vmask-alist)))


(local (in-theory (disable fast-alist-clean)))

(local
 (encapsulate nil
   (defthm svex-mask-alist-p-of-fast-alist-fork
     (implies (and (svex-mask-alist-p x)
                   (svex-mask-alist-p y))
              (svex-mask-alist-p (fast-alist-fork x y))))

   (local (defthm cdr-last-when-svex-mask-alist-p
            (implies (svex-mask-alist-p x)
                     (equal (cdr (last x)) nil))))



   (defthm svex-mask-alist-p-of-fast-alist-clean
     (implies (svex-mask-alist-p x)
              (svex-mask-alist-p (fast-alist-clean x)))
     :hints(("Goal" :in-theory (enable fast-alist-clean svex-mask-alist-p))))))



(define svexlist-compute-masks-with-eval ((x svexlist-p)
                                          (mask-al svex-mask-alist-p)
                                          (env svex-env-p))
  :returns (mask-al1 svex-mask-alist-p)
  (b* (((when (atom x))
        (mbe :logic (svex-mask-alist-fix mask-al)
             :exec mask-al))
       (first (car x))
       ((when (not (eq (svex-kind first) :call)))
        (svexlist-compute-masks-with-eval (cdr x) mask-al env))
       (mask (svex-mask-lookup first mask-al))
       ((when (sparseint-equal mask 0))
        (svexlist-compute-masks-with-eval (cdr x) mask-al env))
       (args (svex-call->args first))
       (argvals (4veclist-quote (svexlist-eval args env)))
       (argmasks (svex-argmasks mask
                                (svex-call->fn first)
                                argvals))
       (mask-al (svex-args-apply-masks args argmasks mask-al)))
    (svexlist-compute-masks-with-eval (cdr x) mask-al env)))

(define svtv-chase-expr-deps ((expr svex-p)
                              (phase natp)
                              (rsh natp)
                              (mask 4vmask-p)
                              (smartp)
                              &key
                              ((updates svex-alist-p) 'updates)
                              ((evaldata svtv-evaldata-p) 'evaldata))
  :returns (deps svex-mask-alist-p)
  (b* (((mv toposort al) (svex-toposort expr nil nil))
       (- (fast-alist-free al))
       (start-mask-al (svex-mask-acons expr (sparseint-ash mask rsh) nil))
       ((unless smartp)
        (fast-alist-free
         (fast-alist-clean
          (svexlist-compute-masks toposort start-mask-al))))
       (vars (svex-collect-vars expr))
       (env (make-fast-alist (pairlis$ vars (svtv-chase-evallist vars phase))))
       (mask-al (fast-alist-free
                 (fast-alist-clean
                  (svexlist-compute-masks-with-eval
                   toposort (svex-mask-acons expr (sparseint-ash mask rsh) nil)
                   env)))))
    (fast-alist-free env)
    mask-al))
       

(defprod chase-position
  ((path path-p)
   ;; Integerp largely because it's occasionally convenient to have a phase of -1 to denote initial states.
   (phase integerp :rule-classes :type-prescription)
   (rsh natp :rule-classes :type-prescription)
   (mask 4vmask-p)))


(deflist chase-stack :elt-type chase-position :true-listp t)



(local (defun svtv-chase-data-renaming (field-names)
         (b* (((when (atom field-names)) nil)
              (field (car field-names))
              (new-field (intern$ (cat "SVTV-CHASE-DATA->" (symbol-name field)) "SV"))
              (update (intern$ (cat "UPDATE-" (symbol-name field)) "SV"))
              (new-update (intern$ (cat "SET-SVTV-CHASE-DATA->" (symbol-name field)) "SV"))
              (pred (intern$ (cat (symbol-name field) "P") "SV"))
              (new-pred (intern$ (cat "SVTV-CHASE-DATA->" (symbol-name field) "P") "SV")))
           (cons (list field new-field)
                 (cons (list update new-update)
                       (cons (list pred new-pred)
                             (svtv-chase-data-renaming (cdr field-names))))))))

(make-event
 (b* ((fields
       `((stack :type (satisfies chase-stack-p) :initially nil)
         (sigtype :type symbol)
         (vars :type (satisfies 4vmask-alist-p))
         (expr :type (satisfies svex-p) :initially ,(svex-x))
         ;; (new-phase :type (integer 0 *) :initially 0)
         (evaldata :type (satisfies svtv-evaldata-p) :initially ,(make-svtv-evaldata))
         (smartp :initially t)
         (updates :type (satisfies svex-alist-p))
         (delays :type (satisfies svar-map-p))
         (assigns :type (satisfies svex-alist-p))
         (modidx :type (integer 0 *) :initially 0)
         (probes :type (satisfies svtv-probealist-p))
         (namemap :type (satisfies svtv-name-lhs-map-p))))
      (field-names (strip-cars fields))
      (renaming (svtv-chase-data-renaming field-names))
      ;; (fns (append '(debugdatap create-debugdata)
      ;;              (acl2::strip-cadrs renaming)))
      (make-binder (std::da-make-binder 'svtv-chase-data field-names)))
   
   `(progn
      (defstobj svtv-chase-data
        ,@fields
        :renaming ,renaming)
      (in-theory (disable create-svtv-chase-data svtv-chase-datap))
      ,make-binder)))



(define svtv-chase-deps ((var svar-p)
                         (phase integerp)
                         (rsh natp)
                         (mask 4vmask-p)
                         &key
                         (svtv-chase-data 'svtv-chase-data))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  :returns (mv (type symbolp :rule-classes :type-prescription)
               (vars 4vmask-alist-p)
               (expr svex-p))
  (b* (((svtv-chase-data svtv-chase-data))
       (phase (lifix phase))
       (var (svar-fix var))
       (type
        (b* (((when (< phase 0))
              :initst)
             (svex (svex-fastlookup var svtv-chase-data.updates))
             ((when svex)
              :update)
             (prev-var-look (hons-get var (svar-map-fix svtv-chase-data.delays)))
             ((when prev-var-look)
              :prevst))
          :input))

       ((when (or (eq type :input)
                  (eq type :initst)))
        (mv type nil (svex-var var)))

       ((when (eq type :prevst))
        (mv type (list (cons var (sparseint-ash mask rsh)))
            (svex-var var)))

       (expr (svex-fastlookup var svtv-chase-data.assigns))

       ((unless expr)
        (mv :error nil (svex-x)))

       (mask-al (svtv-chase-expr-deps expr phase rsh mask svtv-chase-data.smartp
                                      :updates svtv-chase-data.updates
                                      :evaldata svtv-chase-data.evaldata))
       
       (vars (svex-mask-alist-to-4vmask-alist mask-al)))
    (mv type vars expr))
  ///
  ;; (local (defthm svar-addr-p-lookup-in-svar-map
  ;;          (implies (And (svarlist-addr-p (svar-map-vars x))
  ;;                        (hons-assoc-equal k (svar-map-fix x)))
  ;;                   (svar-addr-p (cdr (hons-assoc-equal k (svar-map-fix x)))))
  ;;          :hints(("Goal" :in-theory (e/d (svar-map-vars svar-map-fix)
  ;;                                         (hons-assoc-equal-of-svar-map-fix))
  ;;                  :induct (svar-map-vars x)))
  ;;          :rule-classes
  ;;          ((:rewrite :corollary
  ;;            (implies (And (svarlist-addr-p (svar-map-vars x))
  ;;                          (svar-p k)
  ;;                          (hons-assoc-equal k x))
  ;;                   (svar-addr-p (cdr (hons-assoc-equal k x))))))))

  ;; (local (defthm member-svex-mask-alist-keys-of-fast-alist-fork
  ;;          (implies (and (not (member v (svexlist-vars (svex-mask-alist-keys x))))
  ;;                        (not (member v (svexlist-vars (svex-mask-alist-keys y)))))
  ;;                   (not (member v (svexlist-vars (svex-mask-alist-keys (fast-alist-fork x y))))))
  ;;          :hints(("Goal" :in-theory (enable svex-mask-alist-keys)))))

  ;; (local (defthm svex-mask-alist-keys-of-atom
  ;;          (implies (atom x)
  ;;                   (equal (svex-mask-alist-keys x) nil))
  ;;          :hints(("Goal" :in-theory (enable svex-mask-alist-keys)))))

  ;; (defret svarlist-addr-p-of-<fn>
  ;;   (implies (and (svar-addr-p var)
  ;;                 (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
  ;;                 (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata))))
  ;;            (svarlist-addr-p (alist-keys vars)))
  ;;   :hints(("Goal" :in-theory (enable alist-keys))))

  

  ;; (defret svarlist-addr-p-expr-of-<fn>
  ;;   (implies (and (svar-addr-p var)
  ;;                 (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
  ;;                 (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata))))
  ;;            (svarlist-addr-p (svex-vars expr)))
  ;;   :hints(("Goal" :in-theory (enable alist-keys))))
  )


(define svtv-chase-normalize-mask ((rsh natp) (mask 4vmask-p))
  :returns (mv (new-rsh natp :rule-classes :type-prescription)
               (new-mask 4vmask-p))
  (b* ((mask-rsh (sparseint-trailing-0-count mask))
       (norm-mask (sparseint-ash mask (- mask-rsh))))
    (mv (+ mask-rsh (lnfix rsh)) norm-mask))
  ///
  (local (include-book "centaur/bitops/ihsext-basics" :dir :system))
  (local (include-book "arithmetic/top" :dir :system))

  (local (defthm ash-of-trailing-0-count
           (equal (ash (logtail (bitops::trailing-0-count x) x)
                       (bitops::trailing-0-count x))
                  (ifix x))
           :hints(("Goal" :in-theory (enable* bitops::trailing-0-count
                                              bitops::ihsext-recursive-redefs)))))

  (defret <fn>-correct
    (equal (ash (sparseint-val new-mask) new-rsh)
           (ash (sparseint-val mask) (nfix rsh)))
    :hints (("goal" :in-theory (disable bitops::ash-of-ash)
             :use ((:instance bitops::ash-of-ash
                    (x (LOGTAIL (BITOPS::TRAILING-0-COUNT (SPARSEINT-VAL MASK))
                                (SPARSEINT-VAL MASK)))
                    (sh1 (BITOPS::TRAILING-0-COUNT (SPARSEINT-VAL MASK)))
                    (sh2 (nfix rsh))))))))

(define svtv-chase-var-name/range ((var svar-p)
                                   (rsh natp)
                                   (mask 4vmask-p)
                                   (modidx natp)
                                   &key
                                   ((moddb moddb-ok) 'moddb))
  :returns (name-range-msg)
  :guard (< modidx (moddb->nmods moddb))
  (b* (((svar var))
       ((mv rsh mask) (svtv-chase-normalize-mask rsh mask))
       (maskwidth (and (not (sparseint-< mask 0))
                       (sparseint-length mask)))
       ((unless (address-p var.name))
        (b* (((when maskwidth)
              (msg "~x0[~x1:~x2]" var.name (1- maskwidth) rsh)))
          (cw! "Warning: For non-address variable ~x0, caremask was ~
                unbounded!~%" var)
          (msg "~x0[??:~x1]" var.name rsh)))
       (path (address->path var.name))
       (name (path->string-top path))
       ((mv err wire & &) (moddb-path->wireidx/decl path modidx moddb))
       ((when err)
        (cw! "[ERROR finding wire ~s0]: ~@1~%" (path->string-top path) err)
        (b* (((when maskwidth)
              (msg "~s0[~x1:~x2]" name (1- maskwidth) rsh)))
          (cw! "Warning: For unrecognized wire ~s0, caremask was unbounded!~%" name)
          (msg "~s0[??:~x1]" (path->string-top path) rsh)))
       ((wire wire))
       (width (if maskwidth
                  (min (max 1 maskwidth) wire.width)
                wire.width))
       (lsb (if wire.revp
                (- (+ wire.low-idx (- wire.width 1)) rsh)
              (+ rsh wire.low-idx)))
       (msb (if wire.revp
                (- lsb (- width 1))
              (+ lsb (- width 1)))))
    (msg "~s0[~x1:~x2]" name msb lsb)))


(define svtv-chase-print-signal ((index acl2::maybe-natp)
                                 (var svar-p)
                                 (rsh natp)
                                 (mask 4vmask-p)
                                 (val 4vec-p "Value for the whole variable.")
                                 (modidx natp)
                                 &key
                                 ((moddb moddb-ok) 'moddb))
  :guard (and ;; (svar-addr-p var)
              (< modidx (moddb->nmods moddb)))
  :prepwork ((local (in-theory (e/d (svar-addr-p)
                                    (str::hexify max)))))
  (b* ((name/range (svtv-chase-var-name/range var rsh mask modidx))
       (mask (sparseint-val mask))
       (masked-val (4vec-bitand (4vec-shift-core (- (lnfix rsh)) val)
                                (2vec mask)))
       (delay (svar->delay var))
       (delay-msg (if (eql delay 0)
                      ""
                    (msg " (delay ~x0)" delay))))
  (if (2vec-p masked-val)
      (b* ((val (str::hexify (2vec->val masked-val)))
           (mask (str::hexify mask))
           (vl (length val))
           (ml (length mask))
           (maxl (max vl ml))
           (pad-v (- maxl vl))
           (pad-m (- maxl ml)))
        
        (cw! "~@0 ~@1~@2: ~t3~_4~s5~%"
             (if index (msg "~x0." index) "")
             name/range
             delay-msg
             30
             pad-v
             val)
        (cw! " caremask:~t0~_1~s2~%"
             30
             pad-m
             mask))
    (b* (((4vec val) masked-val)
         (upper (str::hexify val.upper))
         (lower (str::hexify val.lower))
         (xmask (str::hexify (logxor val.upper val.lower)))
         (mask  (str::hexify mask))
         ;; padding for right-aligning the three values
         (ul (length upper)) (ll (length lower)) (xl (length xmask)) (ml (length mask))
         (maxl (max ml (max xl (max ul ll))))
         (pad-u (- maxl ul))
         (pad-l (- maxl ll))
         (pad-x (- maxl xl))
         (pad-m (- maxl ml)))
      (cw! "~@0 ~@1~@2: ~t3(  ~_4~s5~%"
           (if index (msg "~x0." index) "")
           name/range
           delay-msg
           30 pad-u upper)
      (cw! "~t0 . ~_1~s2 )~%" 30 pad-l lower)
      (cw! " non-Boolean portion: ~t0   ~_1~s2~%" 30 pad-x xmask)
      (cw! " caremask:            ~t0   ~_1~s2~%" 30 pad-m mask)))))

(local (defthm len-equal-0
         (equal (equal (len x) 0)
                (not (consp x)))))

(define svtv-chase-print-signals ((index natp)
                                  (vars 4vmask-alist-p)
                                  (vals 4veclist-p)
                                  (modidx natp)
                                  &key
                                  ((moddb moddb-ok) 'moddb))
  :guard (and (eql (len vars) (len vals))
              ;; (svarlist-addr-p (alist-keys vars))
              (< modidx (moddb->nmods moddb)))
  :guard-hints (("goal" :in-theory (enable alist-keys)))
  (if (atom vars)
      nil
    (if (mbt (consp (car vars)))
        (prog2$ (svtv-chase-print-signal (lnfix index)
                                         (caar vars)
                                         0 (cdar vars)
                                         (car vals)
                                         modidx)
                (svtv-chase-print-signals (1+ (lnfix index)) (cdr vars) (cdr vals) modidx))
      (svtv-chase-print-signals (1+ (lnfix index)) (cdr vars) (cdr vals) modidx))))

      
                     
         
(local (defthm len-alist-keys-when-4vmask-alist-p
         (implies (4vmask-alist-p x)
                  (equal (len (alist-keys x))
                         (len x)))
         :hints(("Goal" :in-theory (enable alist-keys)))))

(define svtv-chase-normalize-var/phase ((var svar-p)
                                        (phase integerp))
  :returns (mv (new-var svar-p)
               (new-phase integerp :rule-classes :type-prescription))
  (b* (((svar var))
       ((when (eql 0 var.delay))
        (mv (svar-fix var) (lifix phase))))
    (mv (change-svar var :delay 0) (- (lifix phase) var.delay)))
  ///
  ;; (defret svar-addr-p-of-<fn>
  ;;   (implies (svar-addr-p var)
  ;;            (svar-addr-p new-var))
  ;;   :hints(("Goal" :in-theory (enable svar-addr-p))))
  )
    
(local (in-theory (disable nth update-nth)))


(define svtv-chase-signal ((var svar-p)
                           (phase integerp)
                           (rsh natp)
                           (mask 4vmask-p)
                           &key
                           (svtv-chase-data 'svtv-chase-data)
                           ((moddb moddb-ok) 'moddb))
  :guard (and ;; (svar-addr-p var)
              ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap))))
  :returns (mv (type symbolp :rule-classes :type-prescription)
               (vars 4vmask-alist-p)
               (expr svex-p))
  (b* (((svtv-chase-data svtv-chase-data))
       ((mv var phase) (svtv-chase-normalize-var/phase var phase))
       (modidx svtv-chase-data.modidx)
       (updates svtv-chase-data.updates)
       (evaldata svtv-chase-data.evaldata)
       ((4vec val) (svtv-chase-eval var phase))
       (- (svtv-chase-print-signal nil var rsh mask val modidx)
          (cw! "(Phase ~x0.)~%" phase))
       ((mv type vars expr)
        (svtv-chase-deps var phase rsh mask)))
    (b* (((when (eq type :error))
          (cw! "Error! Somehow this signal wasn't what we expected.~%"))
         ((when (eq type :input))
          (cw! "Primary input.~%"))
         ((when (eq type :initst))
          (cw! "Initial state.~%"))
         (vals (svtv-chase-evallist (alist-keys vars) phase))
         ((when (eq type :prevst))
          (cw! "Previous state var.~%")
          (svtv-chase-print-signals 0 vars vals modidx)))
      (cw! "Internal signal; dependencies:~%")
      (svtv-chase-print-signals 0 vars vals modidx))
    (mv type vars expr))
  ///
  ;; (defret svarlist-addr-p-of-<fn>
  ;;   (implies (and (svar-addr-p var)
  ;;                 (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
  ;;                 (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata))))
  ;;            (svarlist-addr-p (alist-keys vars)))
  ;;   :hints(("Goal" :in-theory (enable alist-keys))))

  

  ;; (defret svarlist-addr-p-expr-of-<fn>
  ;;   (implies (and (svar-addr-p var)
  ;;                 (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
  ;;                 (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata))))
  ;;            (svarlist-addr-p (svex-vars expr)))
  ;;   :hints(("Goal" :in-theory (enable alist-keys))))
  )

;; (define svar-addr-p! (x)
;;   :enabled t
;;   (and (svar-p x)
;;        (svar-addr-p x)))

;; (define 4vmask-alist-addr-p! (x)
;;   :enabled t
;;   (and (4vmask-alist-p x)
;;        (svarlist-addr-p (alist-keys x))))

;; (define svex-addr-p! (x)
;;   :enabled t
;;   (and (svex-p x)
;;        (svarlist-addr-p (svex-vars x))))


(define svtv-chase-signal-data ((pos chase-position-p)
                                &key
                                ((moddb moddb-ok) 'moddb)
                                (svtv-chase-data 'svtv-chase-data))
  :guard (and ;; (chase-position-addr-p pos)
              ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :guard-hints (("goal" :in-theory (enable svtv-chase-datap
                                           ;; chase-position-addr-p
                                           ;; chase-stack-addr-p
                                           )))
  :returns (new-svtv-chase-data)
  (b* (((chase-position pos))
       ((mv type vars expr)
        (svtv-chase-signal (make-svar :name (make-address :path pos.path))
                           pos.phase pos.rsh pos.mask))
       ((when (eq type :error))
        (cw! "[Error -- discrepancy between stored updates and assignments!]~%")
        svtv-chase-data)
       (svtv-chase-data (set-svtv-chase-data->stack (cons pos (svtv-chase-data->stack svtv-chase-data))
                                            svtv-chase-data))
       (svtv-chase-data (set-svtv-chase-data->sigtype type svtv-chase-data))
       (svtv-chase-data (set-svtv-chase-data->vars vars svtv-chase-data))
       (svtv-chase-data (set-svtv-chase-data->expr expr svtv-chase-data))
       ;; (svtv-chase-data (set-svtv-chase-data->new-phase new-phase svtv-chase-data))
       )
    svtv-chase-data)
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))


(define svtv-chase-range ((pos chase-position-p)
                          (msb integerp)
                          (lsb integerp)
                          &key
                          ((moddb moddb-ok) 'moddb)
                          (svtv-chase-data 'svtv-chase-data))
  :guard (and ;; (chase-position-addr-p pos)
              ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable ;; chase-position-addr-p
                                   ;; svar-addr-p
                                   svtv-chase-datap))))
  :returns (new-svtv-chase-data)
  :prepwork ((local (in-theory (disable logmask not))))
  (b* ((modidx (svtv-chase-data->modidx svtv-chase-data))
       ((chase-position pos))
       ;; (path (address->path (svar->name pos.var)))
       ((mv err wire & &) (moddb-path->wireidx/decl pos.path modidx moddb))
       ((when err)
        (cw! "[ERROR finding wire ~s0]: ~@1~%" (path->string-top pos.path) err)
        svtv-chase-data)
       ((wire wire))
       (msb (lifix msb))
       (lsb (lifix lsb))
       (wire-lsb (if wire.revp
                     (+ wire.low-idx (- wire.width 1))
                   wire.low-idx))
       (wire-msb (if wire.revp
                     wire.low-idx
                   (+ wire.low-idx (- wire.width 1))))
       ((unless (if wire.revp
                    (and (<= wire-msb msb) (<= msb lsb) (<= lsb wire-lsb))
                  (and (<= wire-lsb lsb) (<= lsb msb) (<= msb wire-msb))))
        (cw! "Bad range for ~s0: declared range is [~x1:~x2]~%"
             (path->string-top pos.path) wire-msb wire-lsb)
        svtv-chase-data)
       (width (if wire.revp
                  (+ 1 (- lsb msb))
                (+ 1 (- msb lsb))))
       (rsh (if wire.revp (- wire-lsb lsb) (- lsb wire-lsb)))
       (mask (logmask width))
       (new-pos (change-chase-position pos
                                       :rsh rsh
                                       :mask (int-to-sparseint mask))))
    (svtv-chase-signal-data new-pos))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))

;; (local
;;  (defsection lhs-addr-p-of-svtv-wire->lhs

;;    (defret lhs-addr-p-of-svtv-1wire->lhs
;;      (implies (svarlist-addr-p (aliases-vars aliases))
;;               (svarlist-addr-p (lhs-vars lhs)))
;;      :hints(("Goal" :in-theory (enable svtv-1wire->lhs
;;                                        )))
;;      :fn svtv-1wire->lhs)

;;    (local (defthm member-lhs-vars-of-append
;;             (implies (and (not (member v (lhs-vars a)))
;;                           (not (member v (lhs-vars b))))
;;                      (not (member v (lhs-vars (append a b )))))
;;             :hints(("Goal" :in-theory (enable lhs-vars)))))

;;    (defret lhs-addr-p-of-svtv-concat->lhs
;;      (implies (svarlist-addr-p (aliases-vars aliases))
;;               (svarlist-addr-p (lhs-vars lhs)))
;;      :hints(("Goal" :in-theory (enable svtv-concat->lhs
;;                                        )))
;;      :fn svtv-concat->lhs)

;;    (defret lhs-addr-p-of-svtv-wire->lhs
;;      (implies (svarlist-addr-p (aliases-vars aliases))
;;               (svarlist-addr-p (lhs-vars lhs)))
;;      :hints(("Goal" :in-theory (enable svtv-wire->lhs)))
;;      :fn svtv-wire->lhs)))


;; (local (include-book "centaur/bitops/ihsext-basics" :dir :system
(local (in-theory (disable logmask)))

(define svtv-chase-goto-lhs ((lhs lhs-p)
                             (phase natp)
                             (debug-source-obj)
                             &key
                             ((moddb moddb-ok) 'moddb)
                             (svtv-chase-data 'svtv-chase-data))
  :guard (and ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              ;; (svarlist-addr-p (aliases-vars aliases))
              )
  :guard-hints (("goal" :in-theory (e/d (svtv-mod-alias-guard
                                           ;; chase-position-addr-p
                                           svtv-chase-datap)
                                        (logmask))
                 :do-not-induct t))
  ;; :prepwork ((local (defthm svar-addr-p-of-lhatom-var->name
  ;;                     (implies (and (lhatom-case x :var)
  ;;                                   (svarlist-addr-p (lhatom-vars x)))
  ;;                              (svar-addr-p (lhatom-var->name x)))
  ;;                     :hints(("Goal" :in-theory (enable lhatom-vars)))))
  ;;            (local (defthm member-vars-of-lhrange->atom
  ;;                     (implies (and (not (member v (lhs-vars x)))
  ;;                                   (consp x))
  ;;                              (not (member v (lhatom-vars (lhrange->atom (car x))))))
  ;;                     :hints(("Goal" :in-theory (enable lhs-vars)))))
  ;;            (local (in-theory (disable lhs-vars-when-consp))))
  :guard-debug t
  :returns (new-svtv-chase-data)
  (b* (((when (atom lhs))
        (cw! "Error interpreting name: ~x0~%" debug-source-obj)
        svtv-chase-data)
       ((when (consp (cdr lhs)))
        (cw! "Error interpreting name: ~x0 was a concatenation~%" debug-source-obj)
        svtv-chase-data)
       ((lhrange lhrange) (car lhs))
       ((unless (lhatom-case lhrange.atom :var))
        (cw! "Error interpreting name: ~x0 had no variable component~%" debug-source-obj)
        svtv-chase-data)
       ((lhatom-var lhrange.atom))
       ((svar lhrange.atom.name))
       ((unless (address-p lhrange.atom.name.name))
        (cw! "Error interpreting name: ~x0 produced a variable that was not an address~%" debug-source-obj)
        svtv-chase-data)
       (pos (make-chase-position :path (address->path lhrange.atom.name.name)
                                 :phase phase
                                 :rsh lhrange.atom.rsh
                                 :mask (int-to-sparseint (logmask lhrange.w)))))
    (svtv-chase-signal-data pos))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))
       

(define svtv-chase-goto ((str stringp)
                         (phase natp)
                         &key
                         ((moddb moddb-ok) 'moddb)
                         (aliases 'aliases)
                         (svtv-chase-data 'svtv-chase-data))
  :guard (and (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (<= (moddb-mod-totalwires (svtv-chase-data->modidx svtv-chase-data) moddb)
                  (aliass-length aliases)))
  :guard-hints (("goal" :in-theory (e/d (svtv-mod-alias-guard
                                           ;; chase-position-addr-p
                                           svtv-chase-datap)
                                        (logmask))
                 :do-not-induct t))
  :guard-debug t
  :returns (new-svtv-chase-data)
  (b* (((mv err lhs) (svtv-wire->lhs str (svtv-chase-data->modidx svtv-chase-data) moddb aliases))
       ((when err)
        (cw! "Error interpreting name: ~s0~%" str)
        svtv-chase-data))
    (svtv-chase-goto-lhs lhs phase str))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))

(define svtv-chase-goto-output ((name)
                                &key
                                ((moddb moddb-ok) 'moddb)
                                (svtv-chase-data 'svtv-chase-data))
  :guard (and (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :guard-hints (("goal" :in-theory (e/d (svtv-mod-alias-guard
                                           ;; chase-position-addr-p
                                           svtv-chase-datap)
                                        (logmask))
                 :do-not-induct t))
  :guard-debug t
  :returns (new-svtv-chase-data)
  (b* (((svtv-chase-data svtv-chase-data))
       (probe? (hons-assoc-equal name svtv-chase-data.probes))
       ((unless probe?)
        (cw! "Error: no output named ~x0~%" name)
        svtv-chase-data)
       ((svtv-probe probe) (cdr probe?))
       (lhs? (hons-assoc-equal probe.signal svtv-chase-data.namemap))
       ((unless lhs?)
        (cw! "Error: found output named ~x0 pointing to signal ~x1 but no such entry in namemap~%"
             name probe.signal)
        svtv-chase-data)
       (lhs (cdr lhs?)))
    (svtv-chase-goto-lhs lhs probe.time name))
  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))
       

(define svtv-chase-print (&key
                          ((moddb moddb-ok) 'moddb)
                          (svtv-chase-data 'svtv-chase-data))
  :guard (and (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb)))
  :returns new-svtv-chase-data
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t))
                ;; (and stable-under-simplificationp
                ;;      '(:in-theory (enable chase-position-addr-p)))
                )
  (b* ((stack (svtv-chase-data->stack svtv-chase-data))
       ((unless (consp stack))
        (cw! "Empty stack! Use (G \"path\" phase) to choose a signal, ? for more options.~%")
        svtv-chase-data)
       (pos (car stack))
       (svtv-chase-data (set-svtv-chase-data->stack (cdr stack) svtv-chase-data))
       (svtv-chase-data (svtv-chase-signal-data pos)))
    svtv-chase-data)
  ///
  (defmacro svtv-chase-print! (&rest args)
    `(b* ((svtv-chase-data (svtv-chase-print . ,args)))
       (mv nil svtv-chase-data state)))

  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))

(defconst *chase-usage*
  "
What you can enter at the SVTV-CHASE prompt:

 ?                  prints this help message

 X                  Exit the chase read-eval-print loop.

 P                  prints the current state, including the next signal choices

 (G \"path\" phase) Go to the signal named by the given path at the given phase

 (O name)           Go to the signal/phase corresponding to the named pipeline output.

 (R MSB LSB)        Select the given MSB:LSB range of the current signal

 Natural number     Select the given choice of next signal
 B                  Go back to the previous signal on the stack.

 EXPR               Print the assignment for the current signal.
 (EXPR N)           Print the assignment expression, limiting nesting depth to N.

 SMARTP             Toggle data-aware dependency reduction feature
                    (reduces the number of irrelevant signals listed).
                    On by default.

 (EV form)          Evaluates form using simple-translate-and-eval 
                    and prints the result.  You need to set up an attachment
                    to do this, which you can do by running
                    (sv::setup-ev-for-chase) in the ACL2 loop.  You can undo this
                    with (sv::unsetup-ev-for-chase).
")

(defmacro setup-ev-for-chase ()
  '(progn
     (defttag trans)
     (defattach (simple-translate-and-eval-logic acl2::simple-translate-and-eval-cmp)
       :skip-checks t)))

(defmacro unsetup-ev-for-chase ()
  '(progn
     (defttag nil)
     (defattach (simple-translate-and-eval-logic nil))))

;; (local (defthm chase-position-addr-p-car-when-chase-stack-addr-p
;;          (implies (and (svtv-chase-data->stack-addr-p x)
;;                        (consp x))
;;                   (svtv-chase-data->position-addr-p (car x)))
;;          :hints(("Goal" :in-theory (enable chase-stack-addr-p)))))

;; (local (defthm chase-stack-addr-p-cdr-when-chase-stack-addr-p
;;          (implies (svtv-chase-data->stack-addr-p x)
;;                   (svtv-chase-data->stack-addr-p (cdr x)))
;;          :hints(("Goal" :in-theory (enable chase-stack-addr-p)))))


(local (defthm nth-when-4vmask-alist-p
         (implies (and (4vmask-alist-p x)
                       (< (nfix n) (len x)))
                  (and (consp (nth n x))
                       (svar-p (car (nth n x)))
                       (sparseint-p (cdr (nth n x)))))
         :hints(("Goal" :in-theory (enable nth)))))

;; (local (defthm nth-svar-addr-p-when-4vmask-alist-p
;;          (implies (and (4vmask-alist-p x)
;;                        (svarlist-addr-p (alist-keys x))
;;                        (< (nfix n) (len x)))
;;                   (svar-addr-p (car (nth n x))))
;;          :hints(("Goal" :in-theory (enable nth alist-keys)))))

(local (in-theory (disable read-object
                           open-input-channel-p1
                           member)))

(include-book "std/io/file-measure" :dir :system)
(local (include-book "std/io/open-channels" :dir :system))
(local (in-theory (disable file-measure)))

(verify-termination evisc-tuple)
(verify-guards evisc-tuple)


(encapsulate
  (((simple-translate-and-eval-logic
     * * * * * * state * * *) => (mv * *)
    :formals (x alist ok-stobj-names msg ctx wrld state aok safe-mode gc-off)
    :guard t))
  (set-ignore-ok t)
  (set-irrelevant-formals-ok t)
  (local (defun simple-translate-and-eval-logic (x alist ok-stobj-names msg ctx wrld state aok safe-mode gc-off)
           (declare (xargs :stobjs state))
           (mv nil nil))))


(local (in-theory (disable w)))

(local (defthm w-of-read-object
         (equal (w (mv-nth 2 (read-object channel state)))
                (w state))
         :hints(("Goal" :in-theory (enable w read-object)))))

(define svtv-chase-rep (&key
                        ((moddb moddb-ok) 'moddb)
                        (svtv-chase-data 'svtv-chase-data)
                        (aliases 'aliases)
                        (state 'state))
  :guard (and (open-input-channel-p *standard-oi* :object state)
              ;; (svarlist-addr-p (svex-alist-vars (debugdata->override-assigns debugdata)))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (<= (moddb-mod-totalwires (svtv-chase-data->modidx svtv-chase-data) moddb)
                  (aliass-length aliases))
              ;; (svarlist-addr-p (aliases-vars aliases))
              )
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)
                ;; (and stable-under-simplificationp
                ;;      '(:in-theory (enable chase-position-addr-p)))
                ))
  :returns (mv exitp new-svtv-chase-data new-state)
  (b* ((- (cw! "SVTV-CHASE > "))
       ((mv err obj state) (read-object *standard-oi* state))
       ((when err)
        (mv t svtv-chase-data state))
       ((when (natp obj))
        (b* ((vars (svtv-chase-data->vars svtv-chase-data))
             (stack (svtv-chase-data->stack svtv-chase-data))
             ((unless (consp stack))
              (cw! "Empty stack! Use (G \"path\" phase) to choose a signal, ? for more options.~%")
              (mv nil svtv-chase-data state))
             ((chase-position pos) (car stack))
             ((unless (< obj (len vars)))
              (cw "Out of range! Enter P to print current state, ? for more options.~%")
              (mv nil svtv-chase-data state))
             ((cons new-var new-mask) (nth obj vars))
             ((mv rsh mask) (svtv-chase-normalize-mask 0 new-mask))
             ((mv new-var new-phase) (svtv-chase-normalize-var/phase new-var pos.phase))
             (name (svar->name new-var))
             ((unless (address-p name))
              (cw! "The chosen signal isn't an address, so it must be an ~
                    auxiliary variable supporting an override.~%Enter P to ~
                    print current state, ? for more options.~%")
              (mv nil svtv-chase-data state))
             (svtv-chase-data (svtv-chase-signal-data
                               (make-chase-position
                                :path (address->path name)
                                :phase new-phase
                                :rsh rsh :mask mask))))
          (mv nil svtv-chase-data state)))
       ((when (symbolp obj))
        (b* ((objname (symbol-name obj))
             ((when (equal objname "?"))
              (cw! *chase-usage*)
              (mv nil svtv-chase-data state))
             ((when (equal objname "P"))
              (svtv-chase-print!))
             ((when (equal objname "EXPR"))
              (cw! "~x0~%" (svtv-chase-data->expr svtv-chase-data))
              (mv nil svtv-chase-data state))
             ;; Too bad, walkabout isn't in logic mode
             ;; ((when (equal objname "WALK"))
             ;;  (b* (((mv ?err ?val state) (acl2::walkabout (chase-expr svtv-chase-data) state)))
             ;;    (mv nil svtv-chase-data state)))
             ((when (equal objname "B"))
              (b* ((stack (svtv-chase-data->stack svtv-chase-data))
                   ((unless (and (consp stack)
                                 (consp (cdr stack))))
                    (cw! "At end of stack!~%")
                    (mv nil svtv-chase-data state))
                   (svtv-chase-data (set-svtv-chase-data->stack (cdr stack) svtv-chase-data)))
                (svtv-chase-print!)))
             ((when (equal objname "X"))
              (mv t svtv-chase-data state))
             ((when (equal objname "SMARTP"))
              (b* ((smartp (svtv-chase-data->smartp svtv-chase-data))
                   (new-smartp (not smartp))
                   (svtv-chase-data (set-svtv-chase-data->smartp new-smartp svtv-chase-data)))
                (cw! "Turned data-aware dependency reduction ~s0.~%"
                     (if new-smartp "on" "off"))
                (svtv-chase-print!))))
          (cw! "Error -- unrecognized directive: ~x0~%Type ? for allowed commands.~%" obj)
          (mv nil svtv-chase-data state)))
       ((when (and (consp obj)
                   (symbolp (car obj))))
        (b* ((objname (symbol-name (car obj)))
             (args (cdr obj))
             ((when (equal objname "R"))
              (b* (((unless (and (consp args)
                                 (integerp (car args))
                                 (consp (cdr args))
                                 (integerp (cadr args))
                                 (not (cddr args))))
                    (cw! "R directive must be of the form (R MSB LSB) where MSB and LSB are integers.  ? for more options.~%")
                    (mv nil svtv-chase-data state))
                   (stack (svtv-chase-data->stack svtv-chase-data))
                   ((unless (consp stack))
                    (cw! "Empty stack! Use (G \"path\") to choose a signal, ? for more options.~%")
                    (mv nil svtv-chase-data state))
                   (pos (car stack))
                   (svtv-chase-data (svtv-chase-range pos (car args) (cadr args))))
                (mv nil svtv-chase-data state)))
             ((when (equal objname "G"))
              (b* (((unless (and (consp args)
                                 (stringp (car args))
                                 (consp (cdr args))
                                 (natp (cadr args))
                                 (not (cddr args))))
                    (cw! "G directive must be of the form (G \"path\" phase) ~
                          where the first argument is a string and the second ~
                          is a natural number.~%")
                    (mv nil svtv-chase-data state))
                   (svtv-chase-data (svtv-chase-goto (car args) (cadr args))))
                (mv nil svtv-chase-data state)))
             ((when (equal objname "O"))
              (b* (((unless (and (consp args)
                                 (not (cdr args))))
                    (cw! "O directive must be of the form (O name).~%")
                    (mv nil svtv-chase-data state))
                   (svtv-chase-data (svtv-chase-goto-output (car args))))
                (mv nil svtv-chase-data state)))
             ((when (equal objname "EXPR"))
              (b* (((unless (and (consp args)
                                 (acl2::maybe-natp (car args))
                                 (not (cdr args))))
                    (cw! "EXPR directive must be of the form (EXPR depth).~%")
                    (mv nil svtv-chase-data state)))
                (acl2::fmt-to-comment-window! "~x0~%"
                                              `((#\0 . ,(svtv-chase-data->expr svtv-chase-data)))
                                              0 (evisc-tuple (car args) nil nil nil) nil)
                (mv nil svtv-chase-data state)))
             ((when (equal objname "EV"))
              (b* (((unless (and (consp args)
                                 (not (cdr args))))
                    (cw! "EV directive must be of the form (EV term).~%")
                    (mv nil svtv-chase-data state))
                   (attachment (fgetprop 'simple-translate-and-eval-logic 'acl2::attachment nil (w state)))
                   ((unless (and attachment
                                 (alistp attachment)
                                 (eq (cdr (assoc-eq 'simple-translate-and-eval-logic attachment))
                                     'acl2::simple-translate-and-eval-cmp)))
                    (cw! "In order to use EV you must set ~x0 as the ~
                          attachment for ~x1, as in the following ~
                          form:~%~x2~%Note that to (mostly) undo this you may ~
                          do:~%~x3~%"
                         'acl2::simple-translate-and-eval-cmp
                         'simple-translate-and-eval-logic
                         '(setup-ev-for-chase)
                         '(unsetup-ev-for-chase))
                    (mv nil svtv-chase-data state))
                   ((mv err term-dot-val)
                    (simple-translate-and-eval-logic (car args) nil nil "The argument to EV"
                                                     'svtv-chase-rep (w state) state t nil nil))
                   ((when (or err (not (consp term-dot-val))))
                    (cw! "Failed to evaluate: ~@0~%" term-dot-val)
                    (mv nil svtv-chase-data state)))
                (cw! "~x0~%" (cdr term-dot-val))
                (mv nil svtv-chase-data state))))
          (cw! "Error -- unrecognized directive: ~x0~%Type ? for allowed commands.~%" obj)
          (mv nil svtv-chase-data state))))
    (cw! "Error -- unrecognized directive: ~x0~%Type ? for allowed commands.~%" obj)
    (mv nil svtv-chase-data state))
  ///
  (defret file-measure-of-svtv-chase-rep-weak
    (<= (file-measure *standard-oi* new-state)
        (file-measure *standard-oi* state))
    :rule-classes :linear)

  (defret file-measure-of-svtv-chase-rep-strong
    (implies (not exitp)
             (< (file-measure *standard-oi* new-state)
                (file-measure *standard-oi* state)))
    :rule-classes :linear)

  (defret open-input-channel-p1-of-<fn>
    (implies (open-input-channel-p1 *standard-oi* :object state)
             (open-input-channel-p1 *standard-oi* :object new-state)))
  
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->smartp*
                                               *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))
    :hints(("Goal" :in-theory (enable member-equal)))))

  

(define svtv-chase-repl (&key
                         ((moddb moddb-ok) 'moddb)
                         (aliases 'aliases)
                         (svtv-chase-data 'svtv-chase-data)
                         (state 'state))
  :guard (and (open-input-channel-p *standard-oi* :object state)
              ;; (svarlist-addr-p (svexlist-collect-vars (svex-alist-vals (debugdata->override-assigns debugdata))))
              ;; (svarlist-addr-p (svar-map-vars (debugdata->delays debugdata)))
              (< (svtv-chase-data->modidx svtv-chase-data) (moddb->nmods moddb))
              (<= (moddb-mod-totalwires (svtv-chase-data->modidx svtv-chase-data) moddb)
                  (aliass-length aliases))
              ;; (svarlist-addr-p (aliases-vars aliases))
              )
  :guard-hints ((and stable-under-simplificationp
                     '(:in-theory (enable svtv-chase-datap)
                       :do-not-induct t)))
  :returns (mv new-svtv-chase-data new-state)
  :measure (file-measure *standard-oi* state)
  :parents (svtv-chase)
  :short "Re-enter the @(see svtv-chase) read-eval-print loop, with no change to the environment or SVTV."
  (b* (((mv exitp svtv-chase-data state) (svtv-chase-rep))
       ((when exitp)
        (cw! "~%Exiting SVTV-CHASE.  You may execute ~x0 to re-enter or ~x1 ~
              to change the simulation inputs.~%"
             '(svtv-chase-repl) '(svtv-chase-update env))
        (mv svtv-chase-data state)))
    (svtv-chase-repl))

  ///
  (defret nth-of-<fn>
    (implies (not (member-equal (nfix n) (list *svtv-chase-data->smartp*
                                               *svtv-chase-data->stack*
                                               *svtv-chase-data->sigtype*
                                               *svtv-chase-data->vars*
                                               *svtv-chase-data->expr*)))
             (equal (nth n new-svtv-chase-data)
                    (nth n svtv-chase-data)))))

