/*
 * ssexec_reload.c
 *
 * Copyright (c) 2018-2025 Eric Vidal <eric@obarun.org>
 *
 * All rights reserved.
 *
 * This file is part of Obarun. It is subject to the license terms in
 * the LICENSE file found in the top-level directory of this
 * distribution.
 * This file may not be copied, modified, propagated, or distributed
 * except according to the terms contained in the LICENSE file./
 */

#include <stdint.h>
#include <errno.h>

#include <oblibs/log.h>
#include <oblibs/types.h>

#include <skalibs/sgetopt.h>

#include <66/ssexec.h>
#include <66/config.h>
#include <66/graph.h>
#include <66/svc.h>
#include <66/service.h>
#include <66/enum_parser.h>

int ssexec_reload(int argc, char const *const *argv, ssexec_t *info)
{
    log_flow() ;

    int r, nargc = 0 ;
    char const *nargv[argc] ;
    uint8_t siglen = 2 ;
    vertex_t *c, *tmp ;
    service_graph_t graph = GRAPH_SERVICE_ZERO ;
    uint32_t flag = GRAPH_WANT_SUPERVISED|GRAPH_WANT_DEPENDS, nservice = 0 ;

    unsigned int m = 0 ;

    {
        subgetopt l = SUBGETOPT_ZERO ;

        for (;;) {

            int opt = subgetopt_r(argc, argv, OPTS_SUBSTART, &l) ;
            if (opt == -1) break ;

            switch (opt) {

                case 'h' :

                    info_help(info->help, info->usage) ;
                    return 0 ;

                case 'P' :

                    FLAGS_CLEAR(flag, GRAPH_WANT_DEPENDS) ;
                    siglen++ ;
                    break ;

                default :

                    log_usage(info->usage, "\n", info->help) ;
            }
        }
        argc -= l.ind ; argv += l.ind ;
    }

    if (argc < 1)
        log_usage(info->usage, "\n", info->help) ;

    if ((svc_scandir_ok(info->scandir.s)) !=  1 )
        log_diesys(LOG_EXIT_SYS,"scandir: ", info->scandir.s, " is not running") ;

    if (!graph_new(&graph, (uint32_t)SS_MAX_SERVICE))
        log_dieusys(LOG_EXIT_SYS, "allocate the service graph") ;

    nservice = service_graph_build_arguments(&graph, argv, argc, info, flag) ;

    if (!nservice) {
        if (errno == EINVAL)
            log_dieusys(LOG_EXIT_SYS, "build the graph") ;
        log_warn_return(LOG_EXIT_ZERO, "service selection is not supervised -- try to start it first") ;
    }

    char *sig[siglen] ;
    if (siglen > 2) {

        sig[0] = "-P" ;
        sig[1] = "-H" ;
        sig[2] = 0 ;

    } else {

        sig[0] = "-H" ;
        sig[1] = 0 ;
    }

    r = svc_send_wait(argv, argc, sig, siglen, info) ;
    if (r) {
        service_graph_destroy(&graph) ;
        return r ;
    }

    /** s6-supervise do not deal with oneshot service:
     * The previous send command will bring it down but
     * s6-supervise will not bring it up automatically.
     * Well, do it manually */

    HASH_ITER(hh, graph.g.vertexes, c, tmp) {

        struct resolve_hash_s *h = NULL ;
        h = hash_search(&graph.hres, c->name) ;
        if (h == NULL)
            log_dieusys(LOG_EXIT_SYS, "find service: ", c->name, " -- please make a bug report") ;

        if (h->res.type == E_PARSER_TYPE_ONESHOT) {
            nargc++ ;
            nargv[m++] = c->name ;
        }

    }

    if (nargc) {

        nargv[m] = 0 ;
        int verbo = VERBOSITY ;
        VERBOSITY = 0 ;
        char *nsig[siglen + 1] ;

        if (siglen > 2) {

            nsig[0] = "-P" ;
            nsig[1] = "-wU" ;
            nsig[2] = "-u" ;
            nsig[3] = 0 ;

        } else {

            nsig[0] = "-wU" ;
            nsig[1] = "-u" ;
            nsig[2] = 0 ;
        }
        siglen++ ;
        r = svc_send_wait(nargv, nargc, nsig, siglen, info) ;
        VERBOSITY = verbo ;
    }

    service_graph_destroy(&graph) ;

    return r ;
}
