# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent zephir pour la surveillance de services via tcpcheck
"""

from os import system
from twisted.internet.utils import getProcessOutput

from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager.util import status_to_img, boolean_to_onoff, log
from zephir.monitor.agentmanager import status

TCPCHECK = '/usr/bin/tcpcheck'

class SSHService(Agent):
    """agent spécifique à ssh. Utilise netstats pour éviter les logs
    d'avertissement générés par l'utilisation de tcpcheck.
    """

    def __init__(self, name,
                 tcp_services, # dict {"host:port": "comment"}
                 **params):
        Agent.__init__(self, name, **params)
        self.tcp_services = tcp_services
        self.table = TableData([
            ('status', "état", {'align':'center'}, status_to_img),
            ('description', "Description", {'align':'left'}, None),
            ('host', "Hôte", {'align':'left'}, None),
            ('port', "Port", {'align':'left'}, None) ])
        self.data = [self.table]


    def measure(self):
        tcpcheck = getProcessOutput('/bin/netstat',
                                    args = ['-ntlp'],
                                    env = {'LC_ALL': 'C'})
        tcpcheck.addCallback(self.measure_process)
        tcpcheck.addErrback(self.error)
        return tcpcheck

    def measure_process(self, tcpcheck_result):
        tcpcheck_result = tcpcheck_result.decode()
        status = boolean_to_onoff(False)
        lines = tcpcheck_result.splitlines()
        try:
            hostport = list(self.tcp_services.keys())[0]
            host = hostport[:hostport.rindex(':')]
            port = hostport[hostport.rindex(':')+1:]
        except:
            log.msg('SSH Agent, erreur de détermination du port (utilisation de 22 par défaut)')
            host = 'localhost'
            port = '22'
        for line in lines:
            if 'init' in line:
                bool_status = ':%s' % port in line and 'LISTEN' in line
                status = boolean_to_onoff(bool_status)
                if bool_status:
                    # status ok : on sort pour éviter les fausses alertes
                    # si une deuxième instance tourne sur un autre port
                    break
        mesures=[{ 'description': self.tcp_services[hostport],
                   'host': host,
                   'port': port,
                   'status': status }]
        self.measure_data['status'] = status
        return {'services': mesures}

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['services']

    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None:
            for service in self.last_measure.value['services']:
                if service['status'] != 'On':
                    return status.Error()
        else:
            # pas de mesure connue
            return status.Unknown()
        return status.OK()

    def error(self, result):
        self.status = status.Error('Erreur')

class TCPServices(Agent):

    def __init__(self, name,
                 tcp_services, # dict {"host:port": "comment"}
                 container_ip='127.0.0.1',
                 **params):
        Agent.__init__(self, name, **params)
        self.tcp_services = tcp_services
        self.container_ip = container_ip
        self.table = TableData([
            ('status', "état", {'align':'center'}, status_to_img),
            ('description', "Description", {'align':'left'}, None),
            ('host', "Hôte", {'align':'left'}, None),
            ('port', "Port", {'align':'left'}, None) ])
        self.data = [self.table]


    def measure(self):
        if self.container_ip == '127.0.0.1':
            cmd = TCPCHECK
            args = []
        else:
            cmd = '/usr/bin/ssh'
            args = ['-q', '-o', 'LogLevel=ERROR', '-o', 'StrictHostKeyChecking=no', 'root@%s' % self.container_ip, TCPCHECK]
        args.extend(['3'] + list(self.tcp_services.keys()))
        tcpcheck = getProcessOutput(cmd,
                                    args = args,
                                    env = {'LC_ALL': 'C'})
        tcpcheck.addCallback(self.measure_process)
        tcpcheck.addErrback(self.error)
        return tcpcheck

    def measure_process(self, tcpcheck_result):
        tcpcheck_result = tcpcheck_result.decode()
        lines = tcpcheck_result.splitlines()
        hostports_status = [l.split(' ', 1) for l in lines]
        for hostport, status in hostports_status:
            status = boolean_to_onoff(status.endswith('alive'))
            mesures=[{ 'description': self.tcp_services[hostport],
                       'host': hostport.split(':')[0],
                       'port': hostport.split(':')[1],
                       'status': status }]
            self.measure_data['status'] = status
        return {'services': mesures}

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['services']

    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None and self.last_measure.value is not None:
            for service in self.last_measure.value['services']:
                if service['status'] != 'On':
                    return status.Error()
        else:
            # pas de mesure connue
            return status.Unknown()
        return status.OK()

    def error(self, result):
        self.status = status.Error('Erreur')


class PIDService(Agent):

    def __init__(self, name,
                 tcp_services, # dict {"daemon": "comment"}
                 **params):
        Agent.__init__(self, name, **params)
        self.tcp_services = tcp_services
        self.table = TableData([
            ('status', "état", {'align':'center'}, status_to_img),
            ('description', "Description", {'align':'left'}, None),
            ('host', "Hôte", {'align':'left'}, None),
            ('port', "Port", {'align':'left'}, None) ])
        self.data = [self.table]
        self.cmd = '/bin/pidof "%s" >/dev/null'

    def measure(self):
        res = system(self.cmd % list(self.tcp_services.keys())[0])
        status = boolean_to_onoff(res==0)
        mesures=[{ 'description': list(self.tcp_services.values())[0],
                   'host': 'localhost',
                   'port': '',
                   'status': status }]
        self.measure_data['status'] = status
        return {'services': mesures}

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['services']

    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None:
            for service in self.last_measure.value['services']:
                if service['status'] != 'On':
                    return status.Error()
        else:
            # pas de mesure connue
            return status.Unknown()
        return status.OK()

class PgrepService(PIDService):

    def __init__(self, name,
                 tcp_services, # dict {"daemon": "comment"}
                 **params):
        PIDService.__init__(self, name, tcp_services, **params)
        self.cmd = '/usr/bin/pgrep "%s" >/dev/null'

class VPNService(Agent):

    def __init__(self, name,
                 tcp_services, # dict {"host:port": "comment"}
                 **params):
        Agent.__init__(self, name, **params)
        self.tcp_services = tcp_services
        self.table = TableData([
            ('status', "Etat", {'align':'center'}, status_to_img),
            ('description', "Description", {'align':'left'}, None),
            ('host', "Hôte", {'align':'left'}, None),
            ('port', "Port", {'align':'left'}, None) ])
        self.data = [self.table]


    def measure(self):
        cmd = "/usr/sbin/ipsec"
        vpncheck = getProcessOutput(cmd,
                                    args=['status'],
                                    env = {'LC_ALL': 'C'})
        vpncheck.addCallbacks(self.measure_vpn, self.service_error)
        return vpncheck

    def service_error(self,data):
        """status a remonté une erreur
        """
        log.msg("erreur status : %s" % str(data))
        status = 'Off'
        mesures=[{ 'description': "Service Réseau Virtuel Privé (IpSec)",
                   'host': 'localhost',
                   'port': '',
                   'status': status }]
        return {'services': mesures}

    def measure_vpn(self, data):
        status = 'Off'
        # si not running apparait dans les premières lignes de la sortie, ipsec est arreté
        data = data.split('\n')
        end = 5
        if len(data) < 5:
            end = len(data)
        first_lines = "".join(data[0:end])
        if first_lines.count('not running') == 0:
            status = 'On'

        mesures=[{ 'description': "Service Réseau Virtuel Privé (IpSec)",
                   'host': 'localhost',
                   'port': '',
                   'status': status }]
        return {'services': mesures}

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['services']

    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None:
            for service in self.last_measure.value['services']:
                if service['status'] != 'On':
                    return status.Error()
        else:
            # pas de mesure connue
            return status.Unknown()
        return status.OK()


class StatusService(PIDService):

    def __init__(self, name,
                 tcp_services, # dict {"daemon": "comment"}
                 **params):
        PIDService.__init__(self, name, tcp_services, **params)
        self.cmd = 'service %s status >/dev/null 2>&1'
