# -*- coding: utf-8 -*-
#
##########################################################################
# eoleauth - authclient.py: utilies for eoleauth client applications
# Copyright © 2013 Pôle de compétences EOLE <eole@ac-dijon.fr>
#
# License CeCILL:
#  * in french: http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
#  * in english http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
##########################################################################
"""Module d'authentification d'eole-aaa
"""
import urllib.request, urllib.parse, urllib.error, shutil, os
from functools import wraps
from flask import request, session, redirect, current_app, url_for, flash
from eoleflask.util import make_error_response, get_proxy_url

from eoleauthlib.plugins import PLUGINS
from eoleauthlib.client import UnauthorizedError
from eoleauthlib.i18n import _

# directory for local session storage
local_dir = '/root/.eoleauth'

def get_active_client(reset=False):
    """Defines plugin to use for authentication.
    @arg reset: if True, always destroy previous session

    client selection scheme:
    - default client uses PAM local auth.
    - if EOLEAUTH_PLUGIN is defined in current_app.config, cheks that
      check_authsource function of the plugin returns True before using it.
    """
    # default clients : PAMClient (or CASClient if CAS_URL defined)
    conf_client = 'PAMClient'
    client = PLUGINS[conf_client]
    if current_app.config.get('EOLEAUTH_PLUGIN', ''):
        try:
            app_plugin = current_app.config.get('EOLEAUTH_PLUGIN')
            assert app_plugin in PLUGINS
            conf_client = PLUGINS.get(app_plugin)
            # check if authentication source is available
            if conf_client().check_authsource():
                client = conf_client
            else:
                current_app.logger.error(_("authentication source not available for plugin {0}").format(app_plugin))
        except AssertionError:
            current_app.logger.error(_("unknown authentication plugin : {0}").format(app_plugin))
    mode = current_app.config.get('EOLEAUTH_MODE', 'GLOBAL').upper()
    return client(active=not reset, mode=mode)

def gen_redis_interface(local_dir, session_prefix, managed_app=current_app):
    import redis
    r_client = redis.Redis(unix_socket_path='/var/run/redis/redis.sock')
    server_info = r_client.info()
    # redis available, store sessions in local database (socket mode)
    from eoleauthlib.redissession import RedisSessionInterface
    session_interf = RedisSessionInterface(prefix="{0}:".format(session_prefix))
    if os.path.exists(local_dir):
        try:
            shutil.rmtree(local_dir)
        except:
            managed_app.logger.warning(_('eoleauth: could not purge previous session storage ({0})').format(local_dir))
    return session_interf

def gen_file_interface(local_dir, session_prefix, managed_app=current_app):
    from eoleauthlib.cachedsession import ManagedSessionInterface, CachingSessionManager, FileBackedSessionManager
    import datetime
    session_storage = FileBackedSessionManager('/root/.eoleauth', managed_app.config['SECRET_KEY'], "{0}:".format(session_prefix))
    session_cache = CachingSessionManager(session_storage, 1000)
    session_interf = ManagedSessionInterface(session_cache, ['/static'], datetime.timedelta(seconds=10))
    return session_interf

def _init_session_interface():
    """returns current session manager used by eoleauth
    - by default, store sessions in local Redis server
    - if Redis not configured in socket mode, use local storage in root/.eoleauth
      (for example : server freshly installed and not yet configured)
    local storage will be deleted once redis mode is available
    """
    # trick from https://stackoverflow.com/questions/73570041/flask-deprecated-before-first-request-how-to-update
    # Don't do the "remove" solution: it modifies the loop we're in!
    if hasattr(current_app, 'app_has_run_before'):
        return
    # Set the attribute so this code isn't called again
    setattr(current_app, 'app_has_run_before', True)
    if hasattr(current_app, 'eoleauth_session_prefix'):
        # session management is already initialized
        return current_app.eoleauth_session_prefix
    if current_app.config.get('EOLEAUTH_MODE', 'GLOBAL').upper() == 'LOCAL':
        session_prefix = current_app.name
    else:
        session_prefix='eoleauth'
    # defines session mapper attribute if defined in auth plugin
    current_app.eoleauth_map_attr = getattr(get_active_client(), 'map_attr', None)
    # used to check if application is already configured
    current_app.eoleauth_session_prefix = session_prefix
    current_app.config["SESSION_COOKIE_NAME"] = "{0}_session".format(session_prefix)
    current_app.logger.info('session/cookie prefix set to <{0}>'.format(session_prefix))
    # initializes session interface according to application configuration
    try:
        # try to use Redis if session storage set to 'REDIS' or not defined
        assert current_app.config.get('EOLEAUTH_STORAGE', 'REDIS') == 'REDIS'
        session_interf = gen_redis_interface(local_dir, session_prefix)
        current_app.logger.info(_('Session storage set to Redis'))
    except:
        # redis unavailable or not configured to listen on socket, use file storage instead (1000 sessions cached in memory)
        session_interf = gen_file_interface(local_dir, session_prefix)
        current_app.logger.info(_('Session storage set to local (Redis not configured properly or disabled in app config)'))
    current_app.session_interface = session_interf

def init_authentication(managed_app):
    """
    initializes session management and login/logout url for application managed_app
    root_url : specifies an entry point url to redirect to after logout
    """
    @managed_app.route("/login", methods=["GET", "POST"])
    def login():
        """Checks authentication and send to authentication process if needed
        """
        try:
            client = get_active_client()
            return client.authenticate()
        except Exception as e:
            current_app.logger.error(_('Internal error during Authentication : {0}').format(e), exc_info=True)
            flash(_('Erreur : {0}').format(e) , 'error')
            mode = current_app.config.get('EOLEAUTH_MODE', 'GLOBAL').upper()
            app_url = get_proxy_url(request, request.url)
            if mode == 'LOCAL':
                return redirect(get_proxy_url(request, url_for('login', app_url=app_url, _external=True)))
            else:
                server_url = urllib.parse.urlparse(app_url)
                server_url = get_proxy_url(request, '{0}://{1}'.format(server_url.scheme, server_url.netloc))
                return redirect('{0}/eoleauth/login?app_url={1}'.format(server_url, urllib.parse.quote(app_url, safe="")))

    @managed_app.route('/logout')
    def logout():
        """ends user session and display logged out message
        if return_url is passed in request args, redirect there
        (typically, return to application start page)
        """
        if 'username' in session:
            # remove the username from the session if it's there
            user = get_active_client()
            return user.logout()
        # déconnexion ok, on redirige sur l'url demandée ou on affiche un message
        if request.args.get('return_url',''):
            return redirect(request.args.get('return_url'))
        return _('You are disconnected')

    # setting temporarily session interface to local storage
    try:
        # Use Redis by default if redis_server properly configured
        assert managed_app.config.get('EOLEAUTH_STORAGE', 'REDIS') == 'REDIS'
        session_interf = gen_redis_interface(local_dir, managed_app.name, managed_app)
    except:
        session_interf = gen_file_interface(local_dir, managed_app.name, managed_app)
    managed_app.session_interface = session_interf
    # This ensures that eoleflask configuration is loaded before setting session
    # interface and cookie name for the application (not avalaible when app initializes)
    managed_app.before_request(_init_session_interface)

def login_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        mode = current_app.config.get('EOLEAUTH_MODE', 'GLOBAL').upper()
        current_app.logger.debug('In authentication decorator for {0}'.format(current_app.name))
        current_app.logger.debug('SESSION:  {0}'.format(session))
        current_app.logger.debug('  mode: {0}, current_url: {1}'.format(mode, request.url))
        try:
            assert session and session.get("username", None)
            # always check for local user restrictions even if global mode set
            if current_app.config.get('ALLOWED_USERS', []):
                try:
                    # use app.flash to display message on login page if failed ?
                    get_active_client().check_allowed_users(session)
                except UnauthorizedError as e:
                    return make_error_response(str(e), 401)
        except Exception as e:
            # no valid session or invalid user, redirect to eoleauth login page with current
            # request url passed as app_url
            app_url = get_proxy_url(request, request.url)
            if mode == 'LOCAL':
                return redirect(get_proxy_url(request, url_for('login', app_url=app_url, _external=True)))
            else:
                server_url = urllib.parse.urlparse(app_url)
                server_url = get_proxy_url(request, '{0}://{1}'.format(server_url.scheme, server_url.netloc))
                app_prefix = current_app.config.get('PREFIX', 'false')
                if app_prefix != 'false':
                    instance_url = '{0}/{1}'.format(server_url, app_prefix)
                    if app_url.startswith('{0}{1}'.format(instance_url, current_app.config.get('MOUNT_POINT', ''))):
                        # on conserve le prefixe si il est utilisé (ex: derrière reverse-proxy
                        server_url = instance_url
                return redirect('{0}/eoleauth/login?app_url={1}'.format(server_url, urllib.parse.quote(app_url,safe="")))
        return f(*args, **kwargs)
    return decorated_function
