from flask import Blueprint, request, current_app, session, json
from xmlrpc.client import ServerProxy
from flask.views import MethodView
from eoleauthlib.authclient import login_required
from eop.utils import secure_arg

documentsapi = Blueprint('documentsapi', __name__, url_prefix = '/api/documents/')

class DocumentsAPI(MethodView):
    decorators = [ login_required]

    def __init__(self):
        self.username = session["username"]
        self.proxy = ServerProxy(current_app.config['CONTROLEVNC_URL'])
        self.secret = current_app.config['SECRET_KEY']

    def get(self, todel=None):
        """Returns a list of all the documents of username :
           {"eop": {"groupe": "c31", "state": "ren", "in_perso": "true", "eleve_only": "true"},
            "yo": {"groupe": "c33", "state": "dist", "in_perso": true, "eleve_only": true},
            "test_mail": {"groupe": "c34", "state": "dist", "in_perso": true, "eleve_only": true}}
           If todel is specified then returns a list of document name that can be deleted :
           ["eop", "test_mail"]
           Parameters needed :
            in this object : self.secret, self.username
        """
        if self.secret is None:
            #FIXME
            return json.dumps('Error: You need a secret to talk to backend')
        if self.username is None:
            current_app.logger.warning('Unbound use of the API, no session username defined!.')
            return json.dumps('Error: You need a username to talk to backend')
        username = secure_arg(self.username)
        if todel is not None:
            return json.dumps(self.proxy.remote_get_dev_todel(self.secret, username))
        return json.dumps(self.proxy.remote_get_devoirs(self.secret, username))

    def post(self):
        """Distribute a document
           Parameters needed :
            in this object : self.secret, self.username
            in the post request :
                {"groupe":"c31", "nomdev":"eop", "eleve_only":"true", "in_perso":"true"}
        """
        #//return json.dumps([20, ["c31e1", "c31e3", "c31e2"], "toto"])
        #//return json.dumps("Error : test d'un message d'erreur")

        if self.username is None:
            current_app.logger.warning('Unbound use of the API, no session username defined!.')
            return json.dumps('Error: You need a username to talk to backend')

        if None in request.form:
            current_app.logger.warning('Unbound use of the API by user %s.' %self.username)
            return json.dumps('Error in args')
        response = request.form

        username = secure_arg(self.username)
        devname = secure_arg(response['nomdev'])

        docs = json.loads(self.get())
        if type(docs) == dict:
            if devname in docs:
                return json.dumps("Error : this document already exists")

        eleve_only = response['eleve_only'] == "true"
        in_perso = response['in_perso'] == "true"

        return json.dumps(self.proxy.remote_distribute(
            self.secret,
            username,
            response['groupe'],
            devname,
            eleve_only,
            in_perso
            ))

    def delete(self, devname):
        """Delete the datas of a document
           Parameters needed :
            in this object : self.secret, self.username
        """
        if self.secret is None:
            return json.dumps('Error: You need a secret key to talk to backend')
        if self.username is None:
            return json.dumps('Error: You need a username to talk to backend')
        if devname is None:
            return json.dumps('Error: You need a devname to talk to backend')
        username = secure_arg(self.username)
        devname = secure_arg(devname)
        return json.dumps(self.proxy.remote_supprimer(self.secret, username, devname))

    def put(self, devname, correction=None):
        """Pick up a document if correction is None else correct it
           Parameters needed :
            in this object : self.secret, self.username
        """
        if self.secret is None:
            return json.dumps('Error: You need a secret key to talk to backend')
        if self.username is None:
            return json.dumps('Error: You need a username to talk to backend')
        if devname is None:
            return json.dumps('Error: You need a devname to talk to backend')
        username = secure_arg(self.username)
        devname = secure_arg(devname)
        if correction is not None:
            return json.dumps(self.proxy.remote_rendre(self.secret, username, devname))
        return json.dumps(self.proxy.remote_ramasse(self.secret, username, devname))

    @classmethod
    def register(cls, mod):
        f = cls.as_view("documents_api")
        mod.add_url_rule('', view_func=f, methods=["POST"])
        mod.add_url_rule('<devname>', view_func=f, methods=['PUT', 'DELETE'])
        mod.add_url_rule('<devname>/<correction>', view_func=f, methods=['PUT'])
        mod.add_url_rule('', view_func=f, methods=['GET'])
        mod.add_url_rule('<todel>', view_func=f, methods=['GET'])

DocumentsAPI.register(documentsapi)
