#-*-coding:utf-8-*-
"""
    MétaClasse de la classe Action
    Enregistre l'ensemble des actions dans un dico
    Mappe les méthodes execute_
"""
from .typedobjects import TypedObject
from .execute_wrapper import check_signature
REGISTERED_ACTIONS = {}

class ActionMetaclasse(type):
    """
        métaclasse d'enregistrement automatique des actions
    """
    def __new__(mcs, name, bases, classdict):
        """
            Au moment de l'instanciation de la classe :
            - on récupère le nom de l'action et son instance et on l'enregistre
              dans un dictionnaire
        """
        # récupérons le nom de l'action
        actionname = classdict.get('name')
        # Récupération des méthodes execute
        execute_funcs = dict()
        for key, value in list(classdict.items()):
            if key.startswith('execute'):
                execute_funcs[key] = value

        # liste des attributs de paramètres de la classe d'action
        accepted = [attrname for attrname, attrvalue in list(classdict.items())
                    if isinstance(attrvalue, TypedObject)]

        # liste d'attributs hérités de la classe Action
        # on exclue les attributs spéciaux,
        # qui commencent et finissent par '__'
        if len(bases) == 1:
            for attrname, attrvalue in list(bases[0].__dict__.items()):
                if not attrname.startswith('__') and \
                   not attrname.endswith('__'):
                    if isinstance(attrvalue, TypedObject):
                        accepted.append(attrname)
        # On écrase les méthodes execute en intercalant check_signature
        for key, execute_func in list(execute_funcs.items()):
            classdict[key] = check_signature(execute_func, accepted)

        # creation de la classe action
        # uniquement si aucune autre action a le mme nom
        if actionname not in REGISTERED_ACTIONS:
            cls = type.__new__(mcs, name, bases, classdict)
        else:
            raise KeyError("an action named %s exists already" % actionname)
        if actionname:
            REGISTERED_ACTIONS[actionname] = cls
        return cls

    def get_data(mcs):
        """
            Renvoie des données complémentaires pour les attributs typés
            voir TypedObject pour plus de détails
        """
        data = {}
        for attrname, attrvalue in list(mcs.__dict__.items()):
            if isinstance(attrvalue, TypedObject):
                if hasattr(attrvalue,'get_data'):
                    data[attrname] = attrvalue.get_data()
        return data

    def get_params(mcs):
        """
            Renvoie la liste des attributs typés de la classe d'action
        """
        params = []
        for attrname, attrvalue in list(mcs.__dict__.items()):
            if isinstance(attrvalue, TypedObject):
                data = []
                if hasattr(attrvalue,'get_data'):
                    data = attrvalue.get_data()
                params.append((attrname, attrvalue.doc,
                               attrvalue.__class__.__name__ ,
                               attrvalue.default, data))
        return params
