# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action Status
#
# Page d'accueil d'un serveur de commande
#
###########################################################################

""" Actions affichant les différents statuts du serveur """
import xmlrpc.client, socket
from os import stat
from os.path import isfile

from zephir.monitor.agentmanager import config as config_agent

from pyeole.bareos import bareos_rapport_load, BAREOS_RAPPORT_UNKNOWN, BAREOS_RAPPORT_OK, BAREOS_RAPPORT_ERR

from ead2.config.config import BAREOS_SD_ACTIVATE, BAREOS_DIR_ACTIVATE, \
    BAREOS_SD_INSTALLED, BLACKLIST_INSTALLED
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions.tools import get_request
from ead2.backend.actions.lib.widgets import main as M, ajax
from ead2.backend.config.config import AGENT_PORT, AGENT_PROTOCOL
from ead2.backend.config.filenames import bareos, maj, index
from ead2.frontend.web.config import TEST_SOCKET_TIMEOUT
import importlib

try:
    from scribe.importation.config import RAPPORTFILE
    IMPORTATION = True
except BaseException:
    RAPPORTFILE = ''
    IMPORTATION = False

def is_empty(filename):
    """
    vérifie que le fichier existe et qu'il n'est pas vide
    """
    return not isfile(filename) or stat(filename).st_size == 0


class Status(Action):
    """ Action de presentation du statut du serveur
        page d'accueil du serveur
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'main_status'
    libelle = 'Accueil'
    category = ""
    description = "Le statut actuel de votre serveur"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'aff_maj', 'aff_save',
                         'aff_blacklist', 'aff_extract',
                         'aff_account_list', 'url'])

    def execute(self):
        """ Renvoie les données pour l'affichage
            1 - Renvoie les données pour la mise en page des résumés de
                rapports de sauvegarde, mise à jour, mise à jour de blacklist
            2 - Renvoie les données pour l'affichage complet de rapport
        """
        params, self.server_nb = get_request(self.request)
        result = {}
        toexec = ajax.call(self.server_nb,
                           "rapport_maj",
                           only=False,
                           container="maj_container")
        # présence de eole-bareos
        if BAREOS_SD_INSTALLED:
            toexec += ajax.call(self.server_nb,
                                "rapport_sav",
                                only=False,
                                container="bareos_container")
        # présence de eole-proxy
        if BLACKLIST_INSTALLED:
            toexec += ajax.call(self.server_nb,
                                "rapport_blacklist",
                                only=False,
                                container="blackslit_container")
        # présence des librairies scribe
        if IMPORTATION:
            toexec += ajax.call(self.server_nb,
                                "rapport_extraction",
                                only=False,
                                container="extraction_container")
        result.update(self.get_services_status())
        result['toexec'] = toexec
        return self.send_all(result, template='index', templates=['main'])

    def get_agent_status(self):
        """
            récupère le status des agents
        """
        serv = xmlrpc.client.ServerProxy('%s://localhost:%s/xmlrpc/' % (
                                        AGENT_PROTOCOL, AGENT_PORT))
        socket.setdefaulttimeout(TEST_SOCKET_TIMEOUT)
        status = serv.status_for_agents()
        agents_menu = serv.agents_menu()
        return status, agents_menu

    def get_services_status(self):
        """ renvoie les données pour la mise en page des status des services
            - les statuts des services
            - la description des boutons de connexion
        """
        result = {}
        if AGENT_PORT is not None:
            try:
                status, agents_menu = self.get_agent_status()
            except:
                # z_stats arrêté ou timeout
                return dict(services={}, serv_error="Erreur : agents Zéphir non disponibles")
            menu_services = {}
            for group, services in list(agents_menu.items()):
                options = [[service[0],
                            service[1]]for service in services]
                group_status = 'Unknown'
                for service in options:
                    service_status = status.get(service[0], None)
                    if service_status:
                        datas = {}
                        datas['message'] = service_status.get('message',
                                                     "")
                        datas['level'] = service_status.get('level',
                                                  '')
                        # calcul de l'état du groupe d'agents
                        if datas['level'] == 'OK':
                            if group_status == 'Unknown':
                                group_status = 'OK'
                        elif datas['level'] == 'Warn':
                            if group_status != 'NO':
                                group_status = 'Warn'
                        elif datas['level'] == 'Error':
                            group_status = 'NO'
                        importlib.reload(config_agent)
                        client_url = self.user_description['ip']
                        url = '%s://%s:%s/agents/%s/%s/' % (AGENT_PROTOCOL,
                                                       client_url,
                                                       AGENT_PORT,
                                                       config_agent.id_serveur,
                                                       service[0])
                        if 'admin' in self.user_description['role']:
                            datas['href'] = ajax.popup(url, "Statistiques de surveillance", 800, 600)
                        else:
                            datas['href'] = ''
                        service.append(datas)
                    else:
                        services.remove(service)
                menu_services[group] = {'main_status':group_status,
                                        'options':options}
            return dict(services=menu_services)
        else:
            #pas de menu "Services"
            return {}

class Rapport(Action):
    """
        Action génrique de renvoie de rapport (sauvegarde, maj, blacklist...)
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'rapport'
    libelle = 'Rapport'
    category = None
    description = "Action générique"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'aff', 'url'])
    ## configuration
    filename = "/tmp/void"
    html_container = ""
    texte1 = 'Dernier rapport : '
    default_texte = 'Aucun rapport'
    title = "Rapport générique"
    template = "rapport"

    def execute(self):
        """
        """
        params, self.server_nb = get_request(self.request)
        if params.get('aff', None):
            if params.get('aff', ['false'])[0] == 'true':
                datas = self._get_rapport(True)
            else:
                datas = dict(btn=self._get_btn(False))
                datas['rapport'] = []
            return self.send_frag(datas,
                                  template=self.template,
                                  templates=["main"])
        result = dict(titre=self.title)
        result.update(self._get_rapport())
        result.update(dict(html_container=self.html_container))
        return self.send_all(result,
                             template=self.template,
                             templates=['main'])

    def _get_rapport(self, aff=False):
        """
            gestion de l'affichage d'un rapport
        """
        if is_empty(self.filename):
            return dict(rapport_resume=[self.texte1,
                                        self.default_texte],
                        diode="",
                        btn="")
        elif aff:
            return self._get_rapport_complet()
        else:
            return self._get_rapport_resume()
        return dict()

    def _get_rapport_complet(self):
        """
            renvoie le rapport complet
        """
        fic = open(self.filename, 'r')
        rapport = fic.readlines()
        fic.close()
        return dict(rapport=rapport,
                    btn=self._get_btn(True))

    def _get_rapport_resume(self):
        """
            renvoie la première ligne du rapport
        """
        fic = open(self.filename, 'r')
        rapport = fic.readlines()
        fic.close()
        diode = self._get_diode(rapport)
        if len(rapport) >= 1:
            rapport = rapport[0]
        return dict(rapport_resume=[self.texte1, rapport],
                    btn=self._get_btn(False),
                    diode=diode)

    def _get_diode(self, rapport):
        """
            renvoie les descriptions pour les diodes de statut
        """
        return ""

    def _get_btn(self, aff=False):
        """
            renvoie la description pour la mise en forme des boutons
        """
        if aff:
            href = ajax.call(self.server_nb,
                             self.name,
                             container=self.html_container,
                             aff='false')
            href += ajax.top()
            return M.Bouton(href=href,
                            icone='/image/reduce.jpg',
                            libelle='Masquer le rapport',
                            _class='btn')
        else:
            href = ajax.call(self.server_nb,
                             self.name,
                             container=self.html_container,
                             aff='true')
            return M.Bouton(href=href,
                            icone='/image/increase.jpg',
                            libelle='Afficher le rapport',
                            _class='btn')

class RapportBareos(Rapport):
    """
        Rapport de sauvegarde
    """
    name = 'rapport_sav'
    libelle = "Rapport de sauvegarde"
    description  = "Rapport de sauvegarde"
    ## configuration
    filename = bareos['rapport']
    rapport_resume = bareos['rapport_simple']
    html_container = "save_report_container"
    texte1 = 'Dernière sauvegarde : '
    default_texte = 'Aucune sauvegarde'
    title = "Sauvegarde"

    def _get_rapport(self, aff=False):
        """
            renvoie le rapport
            aff:rapport complet (True) / rapport résumé (False)
        """
        if not BAREOS_SD_ACTIVATE and not BAREOS_DIR_ACTIVATE:
            return dict(rapport_resume=[self.texte1,
   '<i>Les sauvegardes sont désactivées dans le dictionnaire</i>'],
                        btn='',
                        diode='')

        if is_empty(self.rapport_resume):
            return dict(rapport_resume=[self.texte1, self.default_texte],
                        diode="",
                        btn="")
        if aff:
            return self._get_rapport_complet()
        else:
            print("appel au résumé")
            return self._get_rapport_resume()
        return dict()

    def _get_rapport_resume(self):
        """
            renvoie un rapport résumé
        """
        rapport = bareos_rapport_load()
        resultpre = bareos_rapport_load('cronpre')
        resultcat = bareos_rapport_load('catalogue')
        diode = {BAREOS_RAPPORT_UNKNOWN: '', BAREOS_RAPPORT_OK: "OK", BAREOS_RAPPORT_ERR: "No"}.get(rapport[0])
        resume = [self.texte1, rapport[1]]
        for i in [ resultpre, resultcat ]:
            if i[0] == BAREOS_RAPPORT_ERR:
                resume = [self.texte1, i[1]]
                diode = "No"
        return dict(rapport_resume=resume,
                    btn=self._get_btn(),
                    diode=diode)

class RapportMaj(Rapport):
    """
        Rapport de mise à jour
    """
    name = 'rapport_maj'
    libelle = "Rapport de mise à jour"
    description  = "Rapport de mise à jour"
    ## configuration
    filename = maj['rapport']
    html_container = "maj_report_container"
    texte1 = 'Dernière mise à jour : '
    default_texte = 'Aucune mise à jour'
    title = "Mise à jour"

    def _get_diode(self, rapport):
        for line in rapport:
            if "Mise à jour OK" in line or "Update successful" in line:
                return "OK"
        return "No"

class RapportBlacklist(Rapport):
    """
        Rapport de mise à jour de listes de sites interdits
    """
    name = 'rapport_blacklist'
    libelle = "Mise à jour des listes de sites interdits"
    description  = "Rapport de mise à jour des listes de sites interdits"
    ## configuration
    filename = index['blacklist']
    html_container = "blacklist_report_container"
    texte1 = 'Dernière mise à jour de la liste de sites interdits : '
    default_texte = 'Aucune mise à jour'
    title = "Liste de sites interdits"


class RapportExtraction(Rapport):
    """
        Rapport d'importation de comptes utilisateurs
    """
    name = 'rapport_extraction'
    libelle = "Rapport d'extraction"
    description  = "Rapport d'extraction"
    filename = RAPPORTFILE
    html_container = "extraction_report_container"
    texte1 = 'Dernière importation : '
    default_texte = 'Aucune importation'
    title = 'Importation'

    def _get_rapport_complet(self):
        """
            rapport complet
        """
        fic = open(self.filename, 'r')
        rapport = fic.readlines()
        fic.close()
        return dict(rapport=rapport,
                    btn=self._get_btn(True))

    def _get_rapport_resume(self):
        """
            résumé (1ère ligne du rapport)
        """
        fic = open(self.filename, 'r')
        rapport = fic.readlines()
        fic.close()
        diode = self._get_diode(rapport)
        if len(rapport) >= 1:
            rapport = rapport[0]
        return dict(rapport_resume=[self.texte1, rapport],
                    btn=self._get_btn(False),
                    diode=diode)

