# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Redémarre et stoppe des services depuis une liste simplifiée
#
###########################################################################
"""
Redémarrage et arrêt des services
"""
from twisted.python import log
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.backend.config.config import EAD_SERVICES, NON_STOP_SERVICES
from ead2.backend.actions.lib import services_tools as ss_tools
from ead2.backend.actions.lib.widgets import main as M, form as F, ajax

used_templates = ['main', 'form', 'listlist', 'checklist', 'formselect']

def format_label(service, container):
    """
    formatage de l'affichage d'un service
    """
    if service in EAD_SERVICES or service in NON_STOP_SERVICES:
        libelle = "%s* (%s)" % (service, container)
    else:
        libelle = "%s (%s)" % (service, container)
    return libelle

class SimpleServices(Action):
    """ page de redémarrage simplifiée de services """
    user_description = Dict(default={}, doc="Description de l'exécutant",
                            keys=['name', 'role', 'ip'])
    name = "asimple_services"
    libelle = "Services (mode normal)"
    description = "Redémarrage simplifié de services"
    category = "Système"
    request = Dict(default={},
                   doc="Arguments de la requête en cours côté frontend",
                   keys=['server', 'action', 'restart', 'stop'])

    def execute(self):
        params, self.server_nb = tools.get_request(self.request)
        if 'restart' in params or 'stop' in params:
            try:
                if 'restart' in params:
                    datas = self._restart_services(params['restart'][0])
                else:
                    datas = self._stop_services(params['stop'][0])
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name,
                                                     confirm=True)
            except Exception as mess:
                datas = dict(message=str(mess))

            return self.send_frag(datas,
                                  template="simple_services",
                                  templates=used_templates)


        result = {'titre':self.description}
        result.update(self._get_services_btns())
        return self.send_all(result=result, template="simple_services",
                             templates=used_templates)

    def _get_services_btns(self):
        """
        récupère la liste des services et renvoie les boutons de redémarrage
        """
        groups = ss_tools.get_groups()
        if groups == []:
            services = "Aucun groupe de service n'a été défini."
        else:
            services = []
            for group in groups:
                group_services = [format_label(s, c) for s, c in ss_tools.get_group_services(group)]
                entete = "%s (%s)" % (group, ','.join(group_services))
                href = ajax.call(self.server_nb, self.name, restart=group)
                reboot_btn = M.Bouton(href=href,
                                      libelle='Redémarrer',
                                      title='Redémarrer %s' % entete,
                                      icone='',
                                      _class='btn')
                stop_btn = M.Bouton(href=ajax.call(self.server_nb,
                                                   self.name,
                                                   stop=group),
                                    libelle='Arrêter',
                                    title='Arrêter %s' % entete,
                                    icone='',
                                    _class='btn')
                services.append((group, reboot_btn, stop_btn))
        return dict(services=services)

    def _restart_services(self, group):
        """ redémarre des services """
        services = ss_tools.get_group_services(group)
        if not services:
            raise Exception("La liste des services à redémarrer est vide.")
        messages = ["Redémarrage des services :"]
        for service, container in services:
            messages.append(ss_tools.restart(service, container))
        return dict(message="\\n".join(messages))

    def _stop_services(self, group):
        """ stoppe des services """
        services = ss_tools.get_group_services(group)
        if not services:
            raise Exception("La liste des services à arrêter est vide.")
        messages = ["Arrêt des services :"]
        for service, container in services:
            messages.append(ss_tools.stop(service, container))
        return dict(message="\\n".join(messages))

class SimpleServicesEditor(Action):
    """ editeur de liste de services
        créer modifier supprimer des groupes de service
    """
    user_description = Dict(default={}, doc="Description de l'exécutant",
                            keys=['name', 'role', 'ip'])
    name = "esimple_services_editor"
    libelle = "Editeur de services"
    description = "Gestion des services pour l'interface simplifiée"
    category = "Système"
    request = Dict(default={},
                   doc="Arguments de la requête en cours côté frontend",
                   keys=['server', 'action', 'edit', 'delete'])
    form_result = Dict(default={}, doc="Validation de formulaire",
                        keys=['group_name_form', 'available_services',
                              'group_services', 'new_group_name_form'])

    def execute(self):
        params, self.server_nb = tools.get_request(self.request)
        # création/modification
        if self.form_result != {}:
            try:
                datas = self.edit_group()
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name,
                                                     confirm=True)
            except Exception as mess:
                log.err("Erreur dans simple_services.py : %s" % mess)
                datas = {'message':str(mess)}
            self.form_result = {}
            return self.send_frag(datas,
                                  template='simple_services_editor',
                                  templates=used_templates)

        # suppression
        if 'delete' in params:
            group_name = params['delete'][0]
            ss_tools.del_service_group(group_name)
            datas = dict(message="Le groupe %s a bien été supprimé" % group_name)
            datas['toexec'] = tools.make_js_link(self.server_nb, self.name,
                                                 confirm=True)
            return self.send_frag(datas,
                                  template='simple_services_editor',
                                  templates=used_templates)

        # renvoie du formulaire de création ou de modification
        if 'edit' in params:
            self.group_name = params['edit'][0]
            datas = self._get_form()
            return self.send_frag(datas,
                                  template='simple_services_editor',
                                  templates=used_templates)

        # renvoie de l'ensemble de l'action
        result = {'titre': 'Edition de groupes de services'}
        result.update(self._get_menu_btns())
        result.update(self._get_main_msg())
        return self.send_all(result,
                             template='simple_services_editor',
                             templates=used_templates)

    def _get_main_msg(self):
        """ renvoie le texte d'entrée """
        return dict(contenu="<b>OUTIL DE CRÉATION DE GROUPE DE SERVICES</b>" \
                            "<br /> Cet outil vous permet de créer des groupes" \
                            "de services accessibles pour les utilisateurs ayant " \
                            "les droits sur l'action de redémarrage/arrêt des " \
                            "services (mode normal)")

    def _get_form(self):
        """
        renvoie la description du formulaire de gestion des groupes de services
        """
        group_name = F.Input(name='group_name',
                             libelle="Intitulé du nouveau groupe de service",
                             required=True)
        if self.group_name != 'new':
            group_name.disable()
            group_name.default(self.group_name)

        # construction du tableau d'inscription de service dans le groupe
        if self.group_name == 'new':
            service_list = ss_tools.SERVICES
        else:
            service_list = ss_tools.get_group_available_services(self.group_name)
        group_service_list = ss_tools.get_group_services(self.group_name)

        services = F.ListList(title = "Association de services au groupe",
                              container='simple_services_editor_listlist')

        services.set_title('left', "Services disponibles")
        services.set_title('right', 'Services associés au groupe')
        services.set_formname('left', 'available_services')
        services.set_formname('right', 'group_services')

        for service, container in service_list:
            # services disponibles
            name = "%s#%s" % (service, container)
            libelle = format_label(service, container)
            services.add_item('left', dict(name=name,
                                           libelle=libelle))
        for service, container in group_service_list:
            # services affectés au groupe
            name = "%s#%s" % (service, container)
            libelle = format_label(service, container)
            services.add_item('right', dict(name=name,
                                            libelle=libelle))
        services.add_btns()
        texte = "(*) Seul le redémarrage est possible pour ces services"
        return dict(group_name=group_name,
                        services=services,
                        validate=self._get_validate_btn(),
                        texte=texte
                        )

    def _get_menu_btns(self):
        """
        renvoie la description des boutons de menu
        (liste des groupes et bouton nouveau groupe
        """
        new_group_btn = M.Bouton(href=ajax.call(self.server_nb,
                                                self.name,
                                                container='simple_services_editor_workspace',
                                                edit='new'),
                                 icone="/image/groupe_machine.png",
                                 libelle="<b>Créer groupe</b>",
                                 _class='btn_groupe_machine')

        btns = []
        for group in ss_tools.get_groups():
            btn = (M.Bouton(href=ajax.call(self.server_nb,
                                   self.name,
                                   container='simple_services_editor_workspace',
                                   edit=group),
                             title="Editer le groupe de services %s" % group,
                             icone="",
                             libelle="<b>%s</b><br />%s" % (group, '<br />'.\
                                     join([format_label(s, c) for s, c in ss_tools.get_group_services(group)])),
                             _class="simple_services_mod"),
                    M.Bouton(href=ajax.call(self.server_nb,
                                            self.name,
                                            container='simple_services_editor_workspace',
                                            delete=group),
                             title="Supprimer le groupe de services %s"%group,
                             icone="/image/supprimer.gif",
                             libelle="",
                             _class="simple_services_btn_suppr")
                   )
            btns.append(btn)
        return dict(menu=btns, new_group_btn=new_group_btn)

    def _get_validate_btn(self):
        """
            renvoie la description du bouton valider pour la création/modification
        """
        title = 'Créer/Modifier le groupe et associer les services'
        icone = '/image/ok.gif'
        if self.group_name == 'new':
            libelle = "Créer"
            forms = ['new_group_name_form', 'group_services']
        else:
            libelle = "Modifier"
            forms = ['group_name_form', 'group_services']
        href = ajax.valid(self.server_nb,
                          self.name,
                          forms,
                          'simple_services_editor_msg')
        return M.Submit(href=href, title=title, libelle=libelle)

    def edit_group(self):
        """ edite le fichier de groupe de services """
        if 'group_name_form' in self.form_result:
            result = tools.format_form_result(self.form_result['group_name_form'])
        else:
            result = tools.format_form_result(self.form_result['new_group_name_form'])

        self.group_name = list(result.values())[0]
        services_form = tools.format_form_result(self.form_result['group_services'], '2')
        group_services = [service for service in list(services_form.keys())]

        if 'new_group_name_form' in self.form_result:
            if self.group_name in ss_tools.get_groups():
                raise Exception("Ce groupe existe déjà.")
            elif self.group_name == 'new':
                raise Exception("Le nom 'new' est un terme réservé")
            else:
                message = "Création du groupe de services %s\\n" % self.group_name
                ss_tools.create_service_group(self.group_name, group_services)
        else:
            if group_services:
                message = "Modification du groupe de services %s\\n" % self.group_name
            else:
                message = "Suppression du groupe %s." % self.group_name
            ss_tools.mod_service_group(self.group_name, group_services)
        if not group_services:
            return dict(message=message)
        return dict()

