# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action scribe_user*
# Creation d'utilisateur
# Recherche puis modification d'utilisateur
# modfication de mot de passe
# Supression d'utilisateur
#
###########################################################################
"""
Listing d'utilisateurs
"""
from pathlib import Path
import pwd
import os
from twisted.python import log
import ldap
from scribe.eoleldap import LdapEntry, Ldap
from scribe.linker import _user_factory
from scribe.ldapconf import PROF_FILTER, AD_ADDRESS, AD_REALM, AD_HOME_PATH
from ead2.backend.lib.action import Action, Dict
from ead2.lib.error import MissingKey, MissingValue
from ead2.backend.actions.lib.widgets import main as M, form as F, ajax
from ead2.backend.actions import tools
from ead2.backend.actions.scribe.tool import getform
from ead2.backend.actions.scribe.tool.user import used_templates, _get_menu


class UserListing(Action):
    """ renvoie le formuliare de listing utilisateur
    """
    user_description = Dict(default={}, doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_user_list'
    libelle = "Recherche d'utilisateur"
    category = "Gestion/Utilisateurs/"
    description = 'Lister des utilisateurs'
    request = Dict(default={}, doc="arguments de la requete en cours cote frontend",
                    keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json", keys=[])

    def execute(self):
        """ renvoit les données pour l'affichage
        """
        params, self.server_nb = tools.get_request(self.request)
        result = {'titre':"Gestion des utilisateurs",
                  'sstitre':self.description}
        result.update(_get_menu(self.server_nb, self.name))
        result.update(getform._get_user_listing_form())
        result.update({'validate':self._get_valid_btn()})
        return self.send_all(result, template='scribe_users',
                             templates=used_templates)

    def _get_valid_btn(self):
        """ renvoie la description du bouton de validation de la recherche
            la cible du retour est la balise : user_div_container
        """
        href = tools.make_form_link(self.server_nb, 'scribe_user_table', True,
                                    ['search'], 'user_div_container')
        return M.Submit(href=href, libelle='Lister')


class ExportUserList(Action):
    """
    Exportation d’une liste d’utilisateur et des mots de passe
    """
    user_description = Dict(default={}, doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_export_users_list'
    libelle = "Export de la liste des utilisateurs"
    category = None
    description = "Exporte les listes des utilisateurs pour diffusion des mots de passe temporaires"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'password_recipient'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['export_users_list'])
    recipients = {'class_admins': 'Administrateurs de la classe',
                  'self': 'Soi-même',
                  'screen': 'À l’écran',
                  }

    def execute(self):
        """ exécute l'action et renvoie les données pour la mise en forme
            1 - renvoie le formulaire d’exportation de liste
            2 - exporte la liste
        """
        params, self.server_nb = tools.get_request(self.request)
        self.user = self.user_description['name']
        result = {}
        if self.form_result != {}:
            ## 1
            try:
                password_export_report = self._valid_form()
                result['message'] = password_export_report['message']
                if 'report_path' in password_export_report:
                    result['password_export_report'] = password_export_report['report_path']
            except (MissingValue, MissingKey, Exception) as mess:
                log.err("Erreur dans valid_form")
                result['message'] = str(mess)
            self.form_result = {}

        ## 2
        result.update(self._get_form())
        return self.send_all(result, template='scribe_users',
                             templates=used_templates)

    def _get_form(self):
        destinataires = F.Select(name="password_recipient", libelle="destinataire", inline=True)
        for code, recipient in self.recipients.items():
            destinataires.add_option(code, recipient)
        return dict(icone="/image/scribe/action_pwd_over.png",
                    title="Export des identifiants provisoires",
                    password_recipient=destinataires,
                    validate=self._get_validate_btn(),
                    )

    def _get_validate_btn(self):
        """ renvoie la description du bouton valider """
        href = ajax.valid(self.server_nb,
                          self.name,
                          ['export_users_list'],
                          container='mod_user_div_container',
                          )
        title = "Exporter la liste des identifiants provisoires"
        return M.Submit(href=href, title=title)

    def _valid_form(self):
        """ valide la modification de mot de passe
        """

        def get_class_admins(school_class):
            ldap_entry = LdapEntry()
            ldap_entry.ldap_admin.connect()
            filtre = f"(&{PROF_FILTER}(Divcod={school_class}))"
            ret = ldap_entry._get_users(filtre, ['uid', 'sn', 'givenName', 'Divcod', 'homeDirectory'])
            return ret

        def format_password_file(users, school_class):
            string = f"{school_class}\n"
            string += '\n'.join([f'{u[1]} {u[0]};{u[2]};{u[3]}'
                                 for u in sorted(users)])
            return string

        def write_password_file(users, class_admin_uid, school_class):
            pwd_filename = f'Identifiants_provisoires_pour_la_classe_{school_class}.txt'
            class_admin_folder = Path(AD_HOME_PATH).joinpath(class_admin_uid, 'perso', 'export_mots_de_passe_provisoires')
            pwd_filepath = class_admin_folder.joinpath(pwd_filename)
            if not class_admin_folder.parent.exists():
                return f"Répertoire de l’administrateur {class_admin_uid} inaccessible."
            else:
                class_admin_folder.mkdir(mode=0o750, parents=True, exist_ok=True)
            with open(pwd_filepath, 'w') as password_fh:
                password_fh.write(format_password_file(users, school_class))
            uid = pwd.getpwnam(class_admin_uid).pw_uid
            os.chown(class_admin_folder, uid, 0)
            os.chown(pwd_filepath, uid, 0)
            return f"Les identifiants de la classe <b>{school_class}</b> ont été copiés dans <b>{pwd_filepath.as_posix()}</b>.<br/><br/>"

        def write_report_file(report_elements):
            report_path = Path('/tmp/ead_users_list_export') # Chemin en dur dans template/scribe_users.tmpl pour include cheetah.
            with open(report_path, 'w') as report_fh:
                report_fh.write('\n'.join(report_elements))
            return report_path.as_posix()


        if 'export_users_list' not in self.form_result:
            raise MissingKey(f"Erreur : Il manque des données pour l’export des comptes. {self.form_result}.")
        resultat = tools.format_form_result(self.form_result['export_users_list'], check=True)
        destinataire = resultat['password_recipient']
        ad_conn = ldap.initialize(f'ldaps://{AD_ADDRESS}')
        ad_conn.protocol_version = 3
        ad_conn.set_option( ldap.OPT_X_TLS_REQUIRE_CERT, False)

        log.msg("dans valid_form")
        log.msg(resultat)
        with open('/tmp/user_selection', 'r') as user_selection_fh:
            users = [u.strip() for u in user_selection_fh.readlines()]
        users_with_password = {}
        for importation_report in Path('/home/a/admin/perso/importation').glob('eleves*.csv'):
            with open(importation_report, 'r') as importation_report_fh:
                for line in importation_report_fh:
                    line = line.strip().split(';')
                    if line[3] in users:
                        try:
                            # on vérifie sur le serveur AD
                            # si ca ne serait pas un changement de mot de passe nécessaire
                            ad_conn.simple_bind_s(f'{line[3]}@{AD_REALM}', line[4])
                        except ldap.INVALID_CREDENTIALS as err:
                            #{'info': u'80090308: LdapErr: DSID-0C0903A9, comment: AcceptSecurityContext error, data 773, v1db1', 'desc': u'Invalid credentials'}
                            code = err.args[0]['info'].split("data ")[1].split(",")[0]
                            if not isinstance(code, str):
                                code = str(code)
                            if code == "773": #mot de passe à changer
                                log.msg(f"AD NEED CHANGE PASSWORD: {err}")
                                user_infos = (line[1], line[2], line[3], line[4])
                                users_with_password.setdefault(line[0], [])
                                users_with_password[line[0]].append(user_infos)
                        except Exception as err:
                            log.msg(f'Erreur : {err}')
                        else:
                            # ne devrait pas arriver
                            # => le mot de passe a été réutilisé pour le changement de mot de passe
                            log.msg(f"Mot de passe temporaire réutilisé pour {line[3]}")
                            user_infos = (line[1], line[2], line[3], line[4])
                            users_with_password.setdefault(line[0], [])
                            users_with_password[line[0]].append(user_infos)
        report = {}
        if not users_with_password:
            report["message"] = "Aucun identifiant provisoire pour la sélection courante."
            return report
        report_elements = []
        if destinataire == "class_admins":
            classes_without_admin = []
            for school_class, users in sorted(users_with_password.items()):
                class_admins = get_class_admins(school_class)
                if class_admins:
                    for class_admin in class_admins:
                        class_admin_uid = class_admin['uid'][0]
                        report.setdefault('report', [])
                        report_elements.append(write_password_file(users, class_admin_uid, school_class))
                else:
                    classes_without_admin.append(school_class)
                    report_elements.append(f"Aucun administrateur pour <b>{school_class}</b>.<br/><br/>")
            if classes_without_admin:
                classes_list = ", ".join(classes_without_admin)
                report['message'] = f"Les classes sans adminitrateurs n’ont pas été exportées : {classes_list}"
            else:
                report['message'] = "Exportation effectuée avec succès"
            report['report_path'] = write_report_file(report_elements)


            return report
        if destinataire == "self":
            for school_class, users in sorted(users_with_password.items()):
                report_elements.append(write_password_file(users, self.user, school_class))
            report['report_path'] = write_report_file(report_elements)
            report['message'] = "Exportation effectuée avec succès"
            return report
        if destinataire == "screen":
            for school_class, users in sorted(users_with_password.items()):
                report_elements.append(f"<h3>{school_class}</h3>")
                for user in sorted(users):
                    report_elements.append(f"{user[1]} {user[0]} ({user[2]}) : {user[3]}<br/>")
            report.setdefault('message', "Exportation effectuée avec succès")
            report['report_path'] = write_report_file(report_elements)
            return report
