# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""Actions permettant de gérer les groupes """
from twisted.python import log
from copy import copy
from scribe.ldapconf import SUPPORT_ETAB
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.lib.error import MissingKey, MissingValue
from ead2.backend.actions.scribe.tool.group import (typedict, used_templates,
     _get_menu, posh, ordered_list)
from ead2.backend.actions.lib.widgets import main as M, form as F, ajax
from scribe.eolegroup import Group

if posh():
    try:
        from posh.backend import db_mapper
    except:
        log.err()

## description des éléments à afficher pour la modification d'un groupe
listingdict = {'Niveau': ['Nom', 'Liste de diffusion', 'Partages', 'Membres', 'Suppression'],
               'Classe': ['Nom', 'Liste de diffusion', 'Partages', 'Membres', 'Suppression'],
               'Matiere':['Nom', 'Liste de diffusion', 'Partages', 'Membres', 'Suppression'],
               # les équipes sont supprimées en même temps que les classes ou options associées
               'Equipe': ['Nom', 'Liste de diffusion', 'Partages', 'Membres'],
               'Groupe': ['Nom', 'Liste de diffusion', 'Partages', 'Membres', 'Suppression'],
               'Option': ['Nom', 'Liste de diffusion', 'Partages', 'Membres', 'Suppression'],
               'Service':['Nom', 'Liste de diffusion', 'Partages', 'Membres', 'Suppression'],
               # FIXME : le traitement n'a jamais été codé et est-ce bien leur place ici ?
               #"Responsables de classes":["Classe", "Responsable", 'Membres'],
               #"Responsables de groupes":["Groupe", "Responsable", 'Membres'],
               'Administrateurs':["Nom", "Membres"],
              }
if SUPPORT_ETAB:
    listingdict['Etablissement'] = ['Nom', 'Liste de diffusion', 'Partages', 'Membres', 'Suppression']

class GroupListing(Action):
    """ Liste les groupes
        Permet le listing des groupes par type et par partie du nom.
    """
    user_description = Dict(default={}, doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_group_list'
    libelle = "Recherche de groupe"
    category = "Gestion/Groupes/"
    description = 'Rechercher un groupe'
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action', 'user_group'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['list_group'])

    def execute(self):
        """ renvoit les données pour l'affichage
            1 - renvoie la description du formulaire de listing
            2 - renvoie la liste des groupes
        """
        params, self.server_nb = tools.get_request(self.request)
        ## 2 -
        if self.form_result != {}:
            if 'list_group' in self.form_result:
                try:
                    groups = self._valid_form()
                except (MissingValue, MissingKey, Exception) as mess:
                    log.err("Erreur dans group.py : %s"%mess)
                    log.err()
                    groups = {'message':str(mess)}
            self.form_result = {}
            return self.send_frag(groups, template='scribe_group',
                                  templates=used_templates)
        ## 1 -
        result = {'titre': "Gestion des groupes", 'sstitre':self.description}
        result.update(_get_menu(self.server_nb, 'liste'))
        result.update(self._get_form())
        return self.send_all(result, template='scribe_group',
                             templates=used_templates)

    def _get_form(self):
        """ renvoit le formulaire de listing de groupe
        @group_type : type de groupe à lister (select)
        @namepart : partie du nom du groupe (input)
        """
        group_type = F.Select(name='group_type', libelle='Type de groupe',
                              inline=True)
        for key in ordered_list:
            # FIXME : faire des group d'option ?
            grp = typedict[key]
            group_type.add_option(grp['tgroupe'], grp['libelle'])
        namepart = F.Input(name='namepart', libelle="Partie du nom",
                           inline=True)
        members = F.Select(name='members', libelle="Membres du groupe",
                             inline=True)
        members.add_option('', 'Tous les groupes')
        members.add_option('empty', 'Groupes vides')
        members.add_option('notempty', 'Groupes non vides')
        return dict(namepart=namepart, group_type=group_type,
                    members=members, validate=self._get_valid_btn())

    def _valid_form(self):
        """
        renvoie la liste des groupes du type demandé
        et dont le nom contient namepart
        """
        if self.form_result['list_group'] == {}:
            raise MissingKey("Erreur : problème de validation de formulaire")
        resultat = tools.format_form_result(self.form_result['list_group'])
        namepart = resultat.get('namepart', '')
        namepart = namepart.strip()
        members = resultat.get('members', '')
        if 'group_type' in resultat:
            _type = resultat['group_type']
        else:
            raise MissingKey("Erreur : Problème de validation de formulaire, le type n'est pas passé")
        return self.get_groups(_type, namepart, members)

    def _get_valid_btn(self):
        """
            bouton valider Vide certaines balises au passage
        """
        href = ajax.valid(self.server_nb,
                           self.name,
                           ['list_group'],
                           container="group_list_div_container")
        for tagname in ('group_list_div_container', 'user_div_container',
                        'mod_group_div_container', 'mod_user_div_container'):
            href += "setVoid('%s');" % tagname
        libelle = "Lister"
        return M.Bouton(href=href, libelle=libelle)

    def get_groups(self, _type, name_part='', members=''):
        """ renvoit la description des groupes de type _type
            dont le nom contient namepart
            les données sont formatées pour la templatisation
        """
        ldapgroup = Group()
        ldapgroup.ldap_admin.connect()
        all_groups = ldapgroup._get_groups(_type)
        all_groups.sort()

        groups = []
        for name in all_groups:
            if name_part and name_part not in name:
                # filtrage par partie du nom
                continue
            num = len(ldapgroup._get_attr(name, 'memberUid'))
            if members == 'empty' and num != 0:
                # filtrage par groupe vide
                continue
            elif members == 'notempty' and num == 0:
                # filtrage par groupe non vide
                continue
            groups.append({'name':name, 'nb':num})

        if 'Membres' in listingdict[_type]:
            for group in groups:
                if group['nb'] != 0:
                    group['members'] = self.get_members_btn(group['name'],
                                            "Membres (%s)" % group['nb'])
                else:
                    group['members'] = {'text':'aucun membre'}

        if 'Partages' in listingdict[_type]:
            for group in groups:
                if ldapgroup._get_group_sharedirs(group['name']):
                    group['share'] = {'text':'oui'}
                else:
                    group['share'] = self.get_add_share_btn(group['name'])

        if 'Liste de diffusion' in listingdict[_type]:
            for group in groups:
                maillist = ldapgroup.get_maillist(group['name'])
                if maillist:
                    group['maillist'] = {'text':maillist}
                else:
                    group['maillist'] = self.get_maillist_btn(group['name'])

        if 'Suppression' in listingdict[_type]:
            for group in groups:
                with_share = 'text' in group.get('share', {})
                group['delete'] = self.get_delete_btn(group['name'],
                                                      with_share)

        ldapgroup.ldap_admin.close()

        entete = copy(listingdict[_type])
        if posh():
            all_posh_groups = [attr['attrvalue']\
                for attr in db_mapper.get_userattr(attrname="user_groups")]
            entete.append(self.get_posh_btns())
            for group in groups:
                if group['name'] not in all_posh_groups:
                    group['posh'] = self.get_posh_checkbox(group['name'])
                else:
                    group['posh'] = 'Dans le gestionnaire'

        if len(groups) == 0:
            entete=['<center><br/>Aucun groupe<br/><br/></center>']

        return dict(group_listed=groups, entete=entete)

    ### les liens
    def get_add_share_btn(self, name):
        """ renvoie la description du bouton d'ajout de partage """
        href = ajax.call(self.server_nb, "scribe_group_modify",
                         container='mod_group_div_container',
                         group_name=name, add_share='True')
        title = "Créer un partage pour ce groupe"
        return dict(btn=M.Lien(href=href, libelle='Créer', title=title))

    def get_maillist_btn(self, name):
        """ renvoie la description du bouton d'ajout de maillist """
        href = ajax.call(self.server_nb, "scribe_group_modify",
                         container='mod_group_div_container',
                         group_name=name,
                         add_maillist='True')
        title = "Créer une liste de diffusion pour ce groupe"
        return dict(btn=M.Lien(href=href, libelle='Créer',
                                 title=title))

    def get_delete_btn(self, name, direct=True):
        """ renvoie la description du bouton de suppression de groupe """
        href = ajax.call(self.server_nb,
                         "scribe_group_delete",
                         container='mod_group_div_container',
                         only=direct,
                         group_name=name,
                         del_group='True')
        if not direct:
            msg = "Êtes-vous sûr de vouloir supprimer ce groupe ?"
            href = ajax.confirm(msg, href)
        return dict(btn=M.Lien(href=href, libelle='Supprimer ce groupe'))

    def get_members_btn(self, name, libelle='Membres'):
        """ renvoie la description du lien vers la section de gestion des membres """
        href = ajax.call(self.server_nb, "scribe_user_table",
                         container="user_div_container",
                         alpha_key='', _type='', domain='', _class='',
                         size='10', namepart='', user_group=name)
        href +=  ajax.reduce('user_listing_tool')
        title = "Gérer les membres de ce groupe"
        return dict(btn=M.Lien(href=href, libelle=libelle, title=title))

    def get_posh_checkbox(self, name):
        """
            renvoie la description d'une checkbox pour l'envoi
            des groupes vers le gestionnaire de profil du portail
        """
        return F.Checkbox(name, libelle="")

    def get_posh_btns(self):
        """ renvoie la description du lien d'exportation de groupe vers
            le gestionnaire de profil du portail
        """
        href = ajax.valid(self.server_nb,
                          'scribe_group_modify',
                          ['posh_groups'],
                          container="user_div_container",
                          add_posh="True"
                          )
        title = "Exporter des groupes vers le gestionnaire de profil Posh"
        btn = M.Lien(href=href, libelle="Envoyer dans posh-profil",
                       title=title)
        href = "javascript:formSelectAll('posh_groups')"
        tous = M.Bouton(href=href, libelle="Tous", _class='scribe_little_btn',
                        icone='')

        href = "javascript:formSelectNone('posh_groups')"
        aucun = M.Bouton(href=href, libelle="Aucun", _class='scribe_little_btn',
                         icone='')
        return dict(btn=btn, tous=tous, aucun=aucun)

