# -*- coding: utf-8 -*-
###########################################################################
# eole ng - 2007
# copyright pole de competence eole  (ministere education - academie dijon)
# licence cecill  cf /root/licenceeole.txt
# eole@ac-dijon.fr
#
# action lshw
#
# affiche les informations matérielles du serveur
#
###########################################################################

"""Action permettant d'obtenir des renseignements systeme
elle lance les commandes suivantes :
- lshw
- df
"""
from twisted.python import log
from ead2.backend.lib.action import *
from ead2.lib.libead import uni
from twisted.internet.utils import getProcessOutput
try:
    from xml.etree import ElementTree as et
except:
    from elementtree import ElementTree as et
from os import path as osp
from ead2.backend.config.config import SCRIPT_DIR
from ead2.backend.actions.lshw_tools import _mega, _get_node_in, _get_node, get_text
from ead2.backend.actions.lshw_tools import _get_node_with_class
from ead2.backend.actions.lshw_tools import UNAVAILABLE_STR

# action
class Lshw(Action):
    """ renvoie les données matérielles du serveur
    """
    user_description = Dict(default={}, doc="description de l'éxécutant", keys=['ip', 'name', 'role'])
    name = 'lshw'
    libelle = 'Listing Matériel'
    description = 'renseignements generiques sur la machine'
    request = Dict(default={}, doc="arguments de la requete en cours cote frontend", keys=['server', 'action'])
    category = 'Système'

    def execute(self):
        """ Renvoie les données pour l'affichage
            Exécute lshw et df et parse les résultats
        """
        self.return_res = {'titre': self.description, 'retour':self.get_return_btn()}
        cmd = osp.join(SCRIPT_DIR, 'lshw.sh')
        result = getProcessOutput(cmd)
        return result.addCallbacks(self.lshw_success, self.lshw_failed)

    def lshw_success(self, result):
        """ si lshw a bien repondu on lance df """
        result = result.decode()
        xml, uptime, uname, rest = result.split('%%')
        self.return_res.update({'uptime':uptime,'uname':uname})
        self.return_res.update(parse_xml(xml))
        a = parse_df()
        a.addCallback(self.success)
        a.addErrback(self.failed)
        return a

    def success(self, result):
        self.return_res.update(result)
        self.return_res['mem'] = parse_mem()
        d = {'template':'lshw', 'data':{'content':self.return_res}}
        return 0, uni(str(d))

    def lshw_failed(self, result):
        datas = {'titre':self.description, 'retour':self.get_return_btn()}
        datas['message'] = "Erreur lors de lshw : %s" % (result)
        d = {'template':'error', 'data':{'content':datas}}
        return 0, uni(str(d))

    def failed(self, result):
        datas = {'titre':self.description, 'retour':self.get_return_btn()}
        datas['messages'] = "Erreur lors des test mémoires : %s" % (result)
        d = {'template':'error', 'data':{'content':datas}}
        return 0, uni(str(d))

    def get_return_btn(self):
        """ renvoit la description du bouton de retour vers l'acceuil """
        return {'href':"javascript:Refresh()", 'icone':"/image/back.png", 'libelle':"Revenir à la page d'accueil"}

def sort_networks(networks):
    """ tri de la liste des interfaces eth pour la présentation """
    names = [net['logicalname']for net in networks]
    names.sort()
    result = []
    for name in names:
        for net in networks:
            if name == net['logicalname']:
                result.append(net)
                break
    return result

# parsing du xml de LSHW
def parse_xml(result):
    """:result: (string) resultat de la commande lshw -xml
    :return: dictionnaire de contexte Cheetah
    """
    tree = et.fromstring(result)
    return dict(
        memory = _memory(tree),
        cpu = _cpu(tree),
        ide = list(_ide(tree)),
        network = sort_networks(list(_network(tree))),
        usb = list(_usb(tree)),
        cdrom = list(_cdrom(tree))
        )

# materiel
def _memory(tree):
    """<node id='memory' claimed='true' class='memory' handle=''>
     <description>System memory</description>
     <physid>0</physid>
     <size units='bytes'>1052119040</size>
    </node>
    :return: 105211904
    """
    node = _get_node(tree, 'memory')
    if node is not None:
        return get_text(node, 'size')
    else:
        return ''

def _cpu(tree):
    """    <node id='cp' claimed='true' class='processor' handle=''>
     <product>Intel(R) Celeron(R) CPU 2.40GHz</product>
     <vendor>Intel Corp.</vendor>
     <physid>1</physid>
     <businfo>cpu@0</businfo>
     <version>15.2.9</version>
     <size units='Hz'>18446744071814584320</size>

    :return: Intel(R) Celeron(R) CPU 2.40GHz
    """
    # une seule CPU
    node = _get_node(tree, 'cpu')
    if node is not None:
        return get_text(node, 'product')
    # plusieurs CPU (#3572)
    node = _get_node(tree, 'cpu:0')
    if node is not None:
        return get_text(node, 'product')
    return ''

def _net(net):
    """
        <node id="virtio1" claimed="true" class="network">
         <description>Ethernet interface</description>
         <physid>0</physid>
         <businfo>virtio@1</businfo>
         <logicalname>ens4</logicalname>
         <serial>02:00:c0:a8:00:6b</serial>
         <configuration>
          <setting id="autonegotiation" value="off" />
          <setting id="broadcast" value="yes" />
          <setting id="driver" value="virtio_net" />
          <setting id="driverversion" value="1.0.0" />
          <setting id="ip" value="192.168.0.31" />
          <setting id="link" value="yes" />
          <setting id="multicast" value="yes" />
         </configuration>
    """
    is_veth = False
    network = {}
    # eth0
    network['logicalname'] = get_text(net, 'logicalname')
    network['product'] = get_text(net, 'product')

    for sett in net.getiterator('setting'):
        if sett.get('id') == 'driver' and sett.get('value') == 'veth':
            is_veth = True
        if sett.get('id') in ['ip', 'broadcast', 'multicast']:
            network[sett.get('id')]=sett.get('value')
    for ress in net.getiterator('resource'):
        network['irq'] = ress.get('value')
    if not is_veth:
        return network
    return None

def _network(tree):
    """
        <node id="network" claimed="true" class="network" handle="PCI:02:01.0">
          <setting id='broadcast' value='yes' />
          <setting id='driver' value='tulip' />
          <setting id='ip' value='192.168.230.27' />
          <setting id='multicast' value='yes' />
        =======================================
        <node id="network:0" claimed="true" class="network" handle="PCI:0000:00:04.0">
          <node id="virtio1" claimed="true" class="network">
    """
    node = _get_node_in(tree, 'network')
    if node is None:
        return

    for net in node:
        iner_node = list(_get_node_with_class(net, 'network'))
        if iner_node:
            for iner_net in iner_node:
                data = _net(iner_net)
                if data:
                    # Sur les ifaces virtuelles, certaines info peuvent être situées sur le noeud parent
                    if data['product'] == UNAVAILABLE_STR:
                        data['product'] = get_text(net, 'product')
                    if 'irq' not in data:
                        for ress in net.getiterator('resource'):
                            data['irq'] = ress.get('value')
                    yield data
        else:
            data = _net(net)
            if data:
                yield data


def _ide(tree):
    node = _get_node_in(tree, 'ide')
    if node is not None:
        for ide in node:
            subnode = _get_node_in(ide, 'disk')
            if subnode is not None:
                for nd in subnode:
                    result = {}
                    result['product'] = get_text(nd, 'product')
                    result['vendor'] = get_text(nd, 'vendor')
                    result['handle'] = nd.get('handle')
                    yield result

def _usb(tree):
    node = _get_node_in(tree, 'usb')
    if node is not None:
        for usb in node:
            result = {}
            result['product'] = get_text(usb, 'product')
            result['vendor'] = get_text(usb, 'vendor')
            result['handle'] = usb.get('handle')
            yield result

def _cdrom(tree):
    node = _get_node_in(tree, 'cdrom')
    if node is not None:
        for cdrom in node:
            result = {}
            result['product'] = get_text(cdrom, 'product')
            result['vendor'] = get_text(cdrom, 'vendor')
            result['handle'] = cdrom.get('handle')
            yield result

def parse_uptime(uptime_output):
    """ parsing de l'uptime
    exemple :
    17:11:20 up  3:32,  6 users,  load average: 0.36, 0.24, 0.26
    """
    return uptime_output.split('up')[0]

# statut memoire
def parse_mem():
    """ parse le fichier /proc/meminfo et structure les données """
    mem_datas = open('/proc/meminfo').read().splitlines()
    result = {}
    for data in mem_datas:
        result[data.split()[0]] = data.split()[1]
    swaptotal = round(int(result['SwapTotal:'])/1024.0, 2)
    swapfree = round(int(result['SwapFree:'])/1024.0, 2)
    swapused = swaptotal - swapfree
    if swaptotal != 0:
        # il faut au minimum un pixel
        swapp = (swapfree/swaptotal)*149+1
    else:
        swapp = 150
    memtotal= round(int(result['MemTotal:'])/1024.0, 2)
    memfree = round(int(result['MemFree:'])/1024.0, 2)
    memused = memtotal-memfree
    # il faut au minimum un pixel
    memp = (memfree/memtotal)*149+1
    _buffer = int(result['Buffers:'])
    _cached = int(result['Cached:'])
    return dict(memtotal=memtotal,memfree=memfree,memp=memp,memused=memused,
                _buffer=_buffer,_cached=_cached, swaptotal=swaptotal,swapfree=swapfree, swapused=swapused, swapp=swapp)

# structure des points de montage
def parse_df():
    """ parse la commande df pour obtenir le nom des points de montage """
    cmd = getProcessOutput('df',
            args = ['-kP','--print-type','--exclude-type=tmpfs','--exclude-type=usbfs'],
            env = {'LC_ALL': 'C'})
    cmd.addCallback(measure_process)
    cmd.addErrback(measure_error)
    return cmd

def measure_process(mnt):
    # récupération des lignes
    lmnt = mnt.decode().splitlines()
    # traitement mnt
    lmnt = lmnt[1:]
    liste=[]
    for lmn in lmnt :
        try:
            l = {}
            l['name'] = lmn.split()[6] # montage
            l['device'] = lmn.split()[0] # partition
            l['type'] = lmn.split()[1] # jointure sur le type
            l['perc'] = lmn.split()[5] # utilisation (%)
            l['free'] = int(lmn.split()[4])/1024 # dispo
            l['used'] = int(lmn.split()[3])/1024 # utilisé
            l['size'] = int(lmn.split()[2])/1024 # taille
            # il faut au minimum un pixel
            if l['size'] == 0:
                l['freep'] = 1
            else:
                l['freep'] = (int(l['free']) * 1.0/int(l['size']))*149+1
        except Exception as err:
            log.err("Erreur dans le parsing de la commande df : {0}".format(err))
            l['graph'] = perc2img(l['perc'])
        liste.append(l)
    return {"statistique": liste}

def measure_error(mnt):
    """ la commande df ne retourne rien ?"""
    #return {'statistique': []}
    return {'statistique': [{'name':'---', 'device':'---', 'type':'---', 'perc':'---',
                            'free':'---', 'used':'---', 'size':'---', 'graph':'---'}]}

if __name__ == "__main__":
    output = """
Sys. de fich. Type 1024-blocs    Occupé Disponible Capacité Monté sur
/dev/hda1     ext3    73963548   7401936  62804412      11% /
varrun       tmpfs      513728        76    513652       1% /var/run
varlock      tmpfs      513728         4    513724       1% /var/lock
udev         tmpfs      513728       108    513620       1% /dev
devshm       tmpfs      513728         0    513728       0% /dev/shm
lrm          tmpfs      513728     18856    494872       4% /lib/modules/2.6.15-28-386/volatile

"""


