#-*-coding:utf-8-*-
"""
    Types d'objet encapsulé pour le passage à une classe d'action
    Permettent de mapper les attributs de la classe Action
"""
from os.path import exists
class TypedObject(object):
    """
        Type de base pour les paramètres des actions
        cette classe encapsule la valeur `value`
        et y affecte des contraintes diverses
    """

    def __init__(self, default='', doc=""):
        """
            :default: valeur par défaut de value
            :doc: description de la variable
        """
        self.default = default
        self.doc = doc
        self.value = default

    ## définition des accesseurs
    def __get__(self, obj, objtype=None):
        """
            accès aux attributs par délégation
            - lorsqu'on accède aux attributs directement depuis la classe `Action`
              alors c'est le type qui est renvoyé
            - lorsqu'on accède aux attributs depuis une instance, c'est self.value
              qui est retourné
        """
        # Lorsqu'on y accède depuis la classe
        if obj is None:
            return self
        # lorsqu'on y accède depuis une instance
        return self.value

    def __set__(self, obj, value):
        """
            le type du paramètre est vérifié au moment
            de sa définition (son ajout dans la classe)
        """
        self.validate(value)
        self.value = value

    def validate(self, value):
        """
            validation du typage
            cette validation se fait au moment de la création d'un parametre
        """
        raise NotImplementedError

    def get_data(self):
        """
            renvoie des données supplémentaires utiles pour la représentation
            ex : valeurs possibles pour un choix
        """
        return []

class Choice(TypedObject):
    """
        Valeur à choix multiples
    """
    def __init__(self, default='', doc="", choices=()):
        super(Choice, self).__init__(default, doc)
        self.choices = choices

    def validate(self, value):
        """
            Valide
        """
        if value not in self.choices:
            raise TypeError("value must be one of (%s)"
                            % (', '.join( repr(c) for c in self.choices )))

    def get_data(self):
        """
            Retourne les choix possibles
        """
        return self.choices

class Boolean(Choice):
    """
        Type de donnée booléen
    """
    def __init__(self, default='', doc=""):
        super(Boolean, self).__init__(default, doc, ('true', 'false'))

class Dict(TypedObject):
    """
        Type de données dico
    """
    def __init__(self, default='', doc='', keys=[]):
        super(Dict, self).__init__(default, doc)
        self.keys = keys

    def validate(self, value):
        """
            Autorise des clés
        """
        for key in list(value.keys()):
            if key not in self.keys:
                raise TypeError("value shall not have the key (%s)"%key)

class Text(TypedObject):
    """
        Type de données texte
        :default:texte par défaut
        :doc:description de la variable
        :forbidden:texte interdit
    """
    def __init__(self, default='', doc="", forbidden=[]):
        super(Text, self).__init__(default, doc)
        self.forbidden = forbidden

    def validate(self, value):
        """
            Autorise le type str
        """
        try:
            assert type(value) == str
            if self.forbidden != []:
                for text in self.forbidden:
                    assert text not in value
        except:
            raise TypeError("value must be text")

    def get_data(self):
        """
            Renvoie les interdits
        """
        return self.forbidden

class Path(Text):
    """
        Type de données chemin de fichier
    """
    def __init__(self, default='', doc="", forbidden=[]):
        super(Path, self).__init__(default, doc, forbidden)

    def validate(self, value):
        """
            value:est-ce un chemin
        """
        super(Path, self).validate(value)
        assert exists(value)

class DynamicChoice(TypedObject):
    """
        typage permettant d'appliquer
        une fonction de validation dynamiquement
    """

    def __init__(self, default='', doc="", choicefunc=None):
        """
            :choicefunc: fonction de validation
        """
        super(DynamicChoice, self).__init__(default, doc)
        self.choicefunc = choicefunc

    def validate(self, value):
        """
            Valide les possibilités
        """
        choices = self.choicefunc()
        if value not in choices:
            raise TypeError("value must be one of (%s)"
                            % (', '.join( repr(c) for c in choices )))
    def get_data(self):
        """
            Renvoie le résultat de la fonction choicefunc()
        """
        return self.choicefunc()
