# -*- coding: UTF-8 -*-
"""tests spécifiques au PermissionManager"""
from os.path import join, dirname

from ead2.perm.parser import parse_and_update
from ead2.perm.registry import PermissionManager, AUTHORIZED, UNAUTHORIZED, \
     merge_status_list

from ead2.perm.actor import Role
from ead2.perm.action import Action

def setup_module(module):
    module.admin = Role('admin')
    module.user = Role('user')
    
    module.reboot = Action('reboot')
    module.schedule = Action('schedule')
    module.logs = Action('logs')


def test_status_merge():
    A = AUTHORIZED
    NA = UNAUTHORIZED
    assert merge_status_list([A]) == A
    assert merge_status_list([A, A]) == A
    assert merge_status_list([NA, A]) == A
    assert merge_status_list([NA, NA]) == NA
    assert merge_status_list([]) == NA

class TestPermissionFileParsing:
    """Série de tests pour le parsing du fichier de déclarations"""
    def setup_method(self, method):
        self.reboot = Action('reboot')
        self.viewlogs = Action('viewlogs')
        self.perm_manager = PermissionManager([self.reboot, self.viewlogs])
        parse_and_update(self.perm_manager, join(dirname(__file__), 'perm.ini'))
        roles = self.perm_manager.get_roles()
        self._roles = dict([(role.name, role) for role in roles])

    def test_roles_parsed(self):
        """On s'assure que les rôles parsés sont bons"""
        assert len(self._roles) == 2
        assert 'admin' in self._roles
        assert 'user' in self._roles

    def test_actions_for_roles(self):
        """teste PermissionTable.get_authorized_actions()"""
        user = self._roles['user']
        admin = self._roles['admin']
        actions = self.perm_manager.authorized_actions_for([])
        assert len(actions) == 0
        actions = self.perm_manager.authorized_actions_for(Role(''))
        assert len(actions) == 0
        actions = self.perm_manager.authorized_actions_for(user)
        assert len(actions) == 1
        assert self.viewlogs in actions
        actions = self.perm_manager.authorized_actions_for(admin)
        assert len(actions) == 2
        assert self.reboot in actions
        assert self.viewlogs in actions

    def test_is_allowed(self):
        """teste le fonctionnement de is_allowed()"""
        admin = self._roles['admin']
        user = self._roles['user']
        assert self.perm_manager.role_is_allowed(admin, self.reboot)
        assert not self.perm_manager.role_is_allowed(user, self.reboot)
        assert self.perm_manager.role_is_allowed(user, self.viewlogs)
        assert self.perm_manager.role_is_allowed(admin, self.viewlogs)


class TestPermissionManager:
    """Série de tests pour le PermissionManager"""
    
    def setup_method(self, method):
        """Mise en place d'un environnement propre pour chaque test"""
        manager = PermissionManager()
        # Enregistrement des rôles
        manager.role_created(admin)
        manager.role_created(user)
        # Enregistrement des actions
        manager.register_action(reboot)
        manager.register_action(schedule)
        manager.register_action(logs)
        # Gestion des droits *admin*
        manager.allow(reboot, admin)
        manager.allow(schedule, admin)
        manager.allow(logs, admin)
        # Gestion des droits *user*
        manager.allow(logs, user)
        self.manager = manager

    def _check_invariants(self):
        """Assertions à faire à chaque fin de test
        
        (on ne peut pas utiliser teardown_method pour ça)
        """
        # *user* ne doit toujours pas avoir changé
        assert not self.manager.role_is_allowed(user, reboot)
        assert not self.manager.role_is_allowed(user, schedule)
        assert self.manager.role_is_allowed(user, logs)
        # *admin* ne doit toujours pas avoir changé
        assert self.manager.role_is_allowed(admin, reboot)
        assert self.manager.role_is_allowed(admin, schedule)
        assert self.manager.role_is_allowed(admin, logs)
        
    
    def test_basic_role_registration(self):
        """Teste les autorisations pour les rôles de base créés"""
        assert self.manager.role_is_allowed(admin, reboot)
        assert self.manager.role_is_allowed(admin, schedule)
        assert self.manager.role_is_allowed(admin, logs)
        assert not self.manager.role_is_allowed(user, reboot)
        assert not self.manager.role_is_allowed(user, schedule)
        assert self.manager.role_is_allowed(user, logs)

    def test_role_based_on_one_other_role(self):
        """Teste la création de rôles basé sur un autre rôle"""
        prof = Role('prof', baseroles=[user])
        self.manager.role_created(prof)
        # Pour l'instant, *prof* doit être une pure copie de *user*
        assert not self.manager.role_is_allowed(prof, reboot)
        assert not self.manager.role_is_allowed(prof, schedule)
        assert self.manager.role_is_allowed(prof, logs)
        # Maintenant, on autorise explicitement schedule pour *prof*
        self.manager.allow(schedule, prof)
        assert not self.manager.role_is_allowed(prof, reboot)
        assert self.manager.role_is_allowed(prof, schedule)
        assert self.manager.role_is_allowed(prof, logs)
        self._check_invariants()

    def test_role_based_on_other_roles(self):
        """Teste la création de rôles basés sur plusieurs rôles"""
        superset = Role('superset', baseroles=[user, admin])
        self.manager.role_created(superset)
        assert self.manager.role_is_allowed(superset, reboot)
        assert self.manager.role_is_allowed(superset, schedule)
        assert self.manager.role_is_allowed(superset, logs)
        # Maintenant, on autorise explicitement schedule pour *superset*
        self.manager.deny(reboot, superset)
        assert not self.manager.role_is_allowed(superset, reboot)
        assert self.manager.role_is_allowed(superset, schedule)
        assert self.manager.role_is_allowed(superset, logs)
        self._check_invariants()
        
