# -*- coding: UTF-8 -*-
"""Module de gestions des Autorisations sur actions"""
from twisted.python import log

class IPermissionManager:
    """XXX Est-ce vraiment bien utile d'avoir ça sous forme d'interface ?"""
    def deny(self, action, actor):
        """retire le droit à <actor> d'exécuter <action>

        @param action: une instance d'action (interface IAction ?)
        @param actor: un acteur (rôle ou user). Un itérable d'acteurs est
                      aussi accepté
        """

    def allow(self, action, actor):
        """autorise <actor> d'exécuter <action>

        @param action: une instance d'action (interface IAction ?)
        @param actor: un acteur (rôle ou user). Un itérable d'acteurs est
                      aussi accepté
        """


    def role_created(self, role):
        """cette méthode devrait être appelée *automatiquement* lorsqu'un
        nouveau rôle a été créé

        Ca a pour effet de mettre à jour automatiquement la liste des
        permissions, notamment pour gérer les équivalences entre rôles
        """

    def register_action(self, action):
        """Enregistre une action dans la table des permissions

        @param action: une instance d'action (interface IAction ?)
        """

    def authorized_actions_for(self, actor):
        """renvoie la liste des actions que l'acteur (rôle ou user) peut
        effectuer.

        @param actor: un acteur (rôle ou user). Un itérable d'acteurs est
                      aussi accepté
        """

## Bête Implémentation d'une Table (extrait de logilab.common)
## ATTENTION : cette classe Table ne gère proprement que des entiers
##             et met des 0 par défaut à l'ajout d'une nouvelle colonne
##             ou ligne !!
from ead2.lib.table import Table
from ead2.backend.lib.action import  get_action_dict

"""une simple table

Note: si des problèmes de lenteur se font sentir, songer à utiliser
      array (stdlib), voire carrément une matrice (numarray / Numeric)
      pour la partie 'données', ou changer l'implémentation en utilisant
      des dictionnaires ou des sets ?
"""

UNAUTHORIZED = 0
AUTHORIZED = 1

def merge_status_list(status_list):
    """renvoie un statut résultant du merge de tout les statuts de <status_list>

    Algo: Les 'AUTHORIZED' sont prioritaires
    """
    if AUTHORIZED in status_list:
        return AUTHORIZED
    return UNAUTHORIZED


class UnknownActor(Exception):
    """Exception levée quand on rencontre un acteur (rôle ou user) incconu"""
    def __init__(self, actor):
        Exception.__init__(self, actor)
        self.actor = actor

    def __str__(self):
        return "No such actor: %r" % self.actor

class UnknownAction(Exception):
    """Exception levée quand on rencontre un acteur (rôle ou user) incconu"""
    def __init__(self, action):
        Exception.__init__(self, action)
        self.action = action

    def __str__(self):
        return "No such action: %r" % self.action


class PermissionManager(object):
    """XXX: plus un registre qu'un manager ..."""
    # implements(IPermissionManager)

    def __init__(self, action_list=None):
        self._table = Table()
        action_list = action_list or []
        for action in action_list:
            self.register_action(action)
        self._actors = self._table.col_names
        self._trusted_clients = set()

    def trust(self, addr):
        """Ajoute une adresse au réseau de confiance"""
        self._trusted_clients.add(addr)

    def client_is_trusted(self, addr):
        """Teste <addr> sur la liste de clients de confiance"""
        return addr in self._trusted_clients

    # XXX ICK ! Pas de méthode get_element_by_ids dans Table !!
    def get_action_status(self, action, actor):
        """Renvoie le statut de <actor> pour <action>"""
        try:
            row_index = self._table.row_names.index(action)
        except ValueError:
            raise UnknownAction(action)
        try:
            col_index = self._table.col_names.index(actor)
        except ValueError:
            raise UnknownActor(actor)
        return self._table.get_element(row_index, col_index)


    def _change_status(self, action, actor, status):
        """change l'autorisation de <actor> sur <action> à <status>

        @param action: une instance d'action (interface IAction ?)
        @param actor: un acteur (rôle ou user). Attention, pas d'itérable ici !!
        @param status: Le statut de l'autorisation. Dans un cas simple, c'est
                       True (autorisé), False (non autorisé), mais ça pourrait
                       éventuellement être autre chose
        """
        self._table.set_cell_by_ids(action, actor, status)


    def deny(self, action, actors):
        """retire le droit à <actor> d'exécuter <action>

        @param action: une instance d'action (interface IAction ?)
        @param actors: un acteur (rôle ou user). Un itérable d'acteurs est
                       aussi accepté
        """
        # On rend <actor> itérable s'il ne l'est pas
        if not hasattr(actors, '__iter__'):
            actors = [actors]
        for actor in actors:
            if actor not in self._actors:
                # debug('Skipping unknown actor: %s' % actor.name)
                continue
            self._change_status(action, actor, UNAUTHORIZED)


    def allow(self, action, actors):
        """autorise <actor> d'exécuter <action>

        @param action: une instance d'action (interface IAction ?)
        @param actors: un acteur (rôle ou user). Un itérable d'acteurs est
                       aussi accepté
        """
        # On rend <actor> itérable s'il ne l'est pas
        if not hasattr(actors, '__iter__'):
            actors = [actors]
        for actor in actors:
            if actor not in self._actors:
                # debug('Skipping unknown actor: %s' % actor.name)
                continue
            self._change_status(action, actor, AUTHORIZED)


    def role_created(self, newrole):
        """cette méthode devrait être appelée *automatiquement* lorsqu'un
        nouveau rôle a été créé

        Ca a pour effet de mettre à jour automatiquement la liste des
        permissions, notamment pour gérer les équivalences entre rôles
        """
        if newrole.has_baseroles():
            # 1. On récupère la liste des listes de status des rôles de base
            roles_status = [self.status_for(base) for base in newrole.baseroles]
            # 2. Pour chaque ligne (<=> pour chaque action), on prend le
            #    résultat du merge sur les status des rôles de base pour
            #    cette action
            newrole_status = [merge_status_list(status_list) for status_list in
                              zip(*roles_status)]
        else:
            # newrole n'est basé sur aucun rôle => rien d'autorisé
            newrole_status = [UNAUTHORIZED] * len(self._table.row_names)
        self._table.append_column(newrole_status, newrole)


    def register_action(self, action):
        """Enregistre une action dans la table des permissions

        @param action: une instance d'action (interface IAction ?)
        """
        self._table.create_row(action)

    def authorized_actions_for(self, actors):
        """renvoie la liste des actions que l'acteur (rôle ou user) peut
        effectuer.

        @param actors: un acteur (rôle ou user). Un itérable d'acteurs est
                      aussi accepté
        """
        # On rend <actor> itérable s'il ne l'est pas
        if not hasattr(actors, '__iter__'):
            actors = [actors]
        action_set = set()
        for actor in actors:
            if actor not in self._actors:
                # debug('Skipping unknown actor: %s' % actor.name)
                continue
            authorized = set([action for action, status in
                              zip(self._table.row_names, self._table.get_column_by_id(actor))
                              if status == AUTHORIZED])
            action_set |= authorized
        return action_set


    def role_is_allowed(self, role, action):
        """renvoie True ou False suivant si <role> est autorisé à
        exécuter <action>
        """
        return self.get_action_status(action, role) == AUTHORIZED


    ## Méthodes de convenance ou pour la lisibilité
    def allow_named_action(self, action_id, actor):
        """autorise <actor> à effectuer l'action appelée <action_id>"""
        try:
            action = get_action_dict()[action_id]
            self.allow(action, actor)
        except KeyError:
            print(("!!! Attention, l'action %s est inconnue !!!" % (
                                                        action_id,)))

    def deny_named_action(self, action_id, actor):
        """interdit <actor> d'effectuer l'action appelée <action_id>"""
        action =  get_action_dict()[action_id]
        self.deny(action, actor)

    def status_for(self, role):
        """renvoie la colonne de status pour <role>"""
        return self._table.get_column_by_id(role)

    def get_roles(self):
        """renvoie la liste des roles de l'application"""
        return self._table.col_names

    def get_actions(self):
        """renvoie la liste des actions de l'application"""
        return self._table.row_names

    ## XXX HACK pprint because l.c.t.Table is expecting strings
    ## for column and row names
    def pprint(self):
        rn_backup = self._table.row_names
        cb_backup = self._table.col_names
        self._table.row_names = [str(obj) for obj in self._table.row_names]
        self._table.col_names = [str(obj) for obj in self._table.col_names]
        result = self._table.pprint()
        # Restore col_names and row_names
        self._table.row_names = rn_backup
        self._table.col_names = cb_backup
        return result


# perm_manager = PermissionManager()
# del PermissionManager
