# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
    Fonctions pour la gestion des purges des comptes utilisateurs
"""
import re
from os import stat
from os.path import isfile, getsize
from time import time

from datetime import date
from datetime import timedelta
from copy import deepcopy

from functools import cmp_to_key

from twisted.python import log
from twisted.internet.defer import Deferred

from scribe.ldapconf import SPEC_USERS
from scribe.eoletools import formate_date, format_current_date, nscd_start, nscd_stop, lsc_full_sync
from scribe.linker import get_userclass_instance
from scribe.importation.config import RAPPORTFILE

from ead2.backend.config.config import debug
from ead2.backend.actions.lib.widgets import ajax, form as F, main as M
from ead2.backend.actions.lib.error import exception, _clean_errors

from ead2.backend.actions.scribe.tool.sort import get_sort
from ead2.backend.actions.scribe.tool.store import store_by_keys

DOUBLONS = (('Comptes élève en double', 'doublon_eleve'),
            ('Comptes responsable en double', 'doublon_responsable'))

FANTOM = (('Comptes élèves non retrouvés', 'fantom_eleve'),
          ('Comptes responsables non retrouvés', 'fantom_responsable'),
          ('Comptes administratifs non retrouvés', 'fantom_administratif'),
          ('Comptes enseignants non retrouvés', 'fantom_enseignant'))

FORMNAMES = dict(userlist_action_formname='userlist_action_formname',
                 usertype_formname='usertype_formname')
PURGE_ACTIONS = dict(del_only="Supprimer<br/>\
(en conservant leurs données)",
                     full_del="Supprimer totalement",
                     update="Mettre à jour <br/>(leur date \
de mise à jour sera mise à aujourd'hui)",)

PURGE_ACTION_DICT = dict(eleve=('del_only', 'full_del', 'update'),
                         administratif=('del_only', 'full_del', 'update'),
                         enseignant=('del_only', 'full_del', 'update'),
                         default=('full_del', 'update'))
# A voir pour les enseignant et administratif
STORE_ARGUMENTS = dict(eleve=(('Meflcf', 'Niveau : %s'),
                              ('Divcod', 'Classe : %s'),
                              ('LastUpdate',
                               'Date de dernière mise à jour : %s',)),
                       default=(('LastUpdate',
                           'Date de dernière mise à jour : %s',),),
                        )
COMMON_LDAP_ATTRS = ['uid', 'LastUpdate']
SPEC_LDAP_ATTRS = dict(eleve=['Meflcf', 'Divcod'])
def add_dollar(liste):
    """
        Rajoute des dollars pour la construction d'une regexp
        (pas de dollar si * à la fin)
    """
    for user in liste:
        if user.endswith('*'):
            yield user
        else:
            yield user + '$'
SPEC_USERS_REGEXP = '|'.join(add_dollar(SPEC_USERS))

######################  Formulaire  ##########################
# Balises
def get_select_usertype():
    """
        Renvoie une balise select pour la sélection de type à lister
    """
    select = F.Select("usertype", libelle="Lister :")
    select.add_option('', '', default=True)
#    for option_libelle, option_name in DOUBLONS:
#        select.add_option(value=option_name,
#                          libelle=option_libelle,
#                          group='Comptes en double')
    for option_libelle, option_name in FANTOM:
        select.add_option(value=option_name,
                          libelle=option_libelle,
                          group='Comptes non retrouvés')
    return select

def get_date_input():
    """
        Input pour le choix de la date de mise à jour
    """
    if isfile(RAPPORTFILE) and getsize(RAPPORTFILE) != 0:
        mdate = stat(RAPPORTFILE).st_mtime
        last = int((time() - mdate) /3600 /24) + 1
    else:
        last = 1
    return F.Input(name='lastupdate',
                   libelle='Non mis à jour depuis : ',
                   default=last,
                   _type='number',
                   inline=True)

# Validation :
def get_usertype(formulaire):
    """
        Renvoie le usertype trouvé
        dans le résultat de formulaire
    """
    if debug:
        log.msg(" + Récupération du usertype")
    try:
        usertype = formulaire.pop('usertype')
    except KeyError:
        exception(" - Erreur dans la validation du formulaire \
(la variable usertype est vide).")
    if not usertype:
        exception(" - Veuillez choisir le type de compte à lister.")

    if usertype.startswith('fantom'):
        return 'fantom', usertype[7:]
    elif usertype.startswith('doublon'):
        return 'doublon', usertype[8:]
    else:
        raise Exception("Type de filtrage inconnu %s " % (usertype,))

def get_lastupdate(formulaire):
    """
        Renvoie la valeur de la clé 'lastupdate'
        au delà de laquelle on veut lister
    """
    if debug:
        log.msg(" + Récupération de la date lastupdate")
    try:
        lastupdate = int(formulaire.pop('lastupdate'))
    except KeyError:
        raise Exception(" - Il manque la variable lastupdate")
    except ValueError:
        raise Exception(" - Le nombre de jours doit être un nombre")

    if lastupdate is None:
        raise Exception(" - La variable lastupdate est vide")
    aujourdhui = date.today()
    lastupdate = aujourdhui - timedelta(days=lastupdate)
    lastupdate_ldapformat = lastupdate.isoformat().replace('-', '')
    if debug:
        log.msg("   + lastupdate = %s" % lastupdate_ldapformat)
    return lastupdate_ldapformat

######################  Recherche ldap  ##########################
# Requêtes :
def make_ldap_request(ldapuser, ldapfilter, ldapattrs):
    """
        Renvoie la liste des comptes pour:
        ldapfilter
    """
    ldapuser.ldap_admin.connect()
    for user in ldapuser.ldap_admin._search(ldapfilter, ldapattrs):
        user_dict = dict(((ldapattr, user[1][ldapattr][0])
                                    for ldapattr in ldapattrs))
        user_dict['doublons'] = get_doublons(ldapuser,
                                             user_dict['uid'])
        if not re.match(SPEC_USERS_REGEXP, user_dict['uid']):
            yield user_dict
    ldapuser.ldap_admin.close()

def get_doublons(ldapuser, userid):
    """
        Renvoie les doublons de l'utilisateur d'id user_id
    """
    if userid.endswith('*'):
        userid = userid[0:-1]
    ldapfilter = "(&%s%s)" % (ldapuser.filtre, "(uid=%s*)" % userid)
    doublons = ldapuser.ldap_admin._search(ldapfilter, 'uid')
    if len(doublons) > 1:
        uids = [doublon[1]['uid'][0] for doublon in doublons]
        uids.remove(userid)
        return uids
    else:
        return []

def map_user_list(user_list):
    """
        Rajoute des données dans les users récupérés depuis le ldap
    """
    for user in user_list:
        lastupdate = "Dernière mise à jour le %s" % (
                                        formate_date(user['LastUpdate']))
        user['checkbox'] = F.Checkbox(name=user['uid'], value=user['uid'])
        user['LastUpdate'] = formate_date(user['LastUpdate'])
        if user['doublons']:
            msg = "Homonymes : <br> - %s" % '<br> - '.join(user['doublons'])
            user['infos']  = (lastupdate, msg)
        else:
            user['infos']  = (lastupdate, '')
        yield user

def make_ret_dict(userlist, usertype, server_nb, actionname):
    """
        Crée le dictionnaire de retour
    """
    boutons_purge = get_boutons_purge(server_nb,
                                                actionname,
                                get_purge_actions(usertype))
    return dict(users=userlist,
                boutons_purge=boutons_purge)

def get_boutons_purge(server_nb, actionname, actions):
    """
        Renvoie la liste des liens d'action disponibles
    """
    boutons_purge = []
    for action in actions:
        libelle = PURGE_ACTIONS[action]
        bouton = M.Lien(
          href=ajax.valid(server_nb,
                          actionname,
                          formnames=[FORMNAMES['userlist_action_formname']],
                          container='div_action_return_msg',
                          purge_action=action),
          title="Effectuer cette action pour \
tous les comptes sélectionnés",
          libelle=libelle,
          _class='scribe_little_btn underlinedtag')
        boutons_purge.append(bouton)
    return boutons_purge

def action_del_only(userlist, ldapuser):
    """
        Supprime les comptes de la liste
        en conservant les données
    """
    if debug:
        log.msg(" + Appel de l'action del_only + ")
    if not userlist:
        exception("Aucun compte n'a été sélectionné.")
    ldapuser.ldap_admin.connect()
    if len(userlist) == 1:
        message = "Le compte : \\n"
    else:
        message = "Les comptes : \\n"
    nscd_stop()
    for login in userlist:
        if debug:
            log.msg(" + Suppression de : %s" % login)
        try:
            ldapuser._delete(login, remove_data=False, delete_resp=True)
            message += " - %s \\n" % login
        except:
            log.err()
            exception(" - Erreur à la suppression du compte : %s " % (
                                                                    login,))
    lsc_full_sync()
    nscd_start()
    if len(userlist) == 1:
        message += "a été supprimé."
    else:
        message += "ont été supprimés."
    return message

def action_full_del(userlist, ldapuser):
    """
        Supprime les comptes de la liste
    """
    if debug:
        log.msg(" + Appel de l'action full_del +")
    if not userlist:
        exception("Aucun compte n'a été sélectionné.")
    ldapuser.ldap_admin.connect()
    if len(userlist) == 1:
        message = "Le compte : \\n"
    else:
        message = "Les comptes : \\n"
    nscd_stop()
    for login in userlist:
        if debug:
            log.msg(" + Suppression de : %s" % login)
        try:
            ldapuser._delete(login, remove_data=True, delete_resp=True)
            message += " - %s \\n" % login
        except:
            log.err()
            exception(" - Erreur à la suppression du compte : %s " % (
                                                               login,))
    lsc_full_sync()
    nscd_start()
    if len(userlist) == 1:
        message += "a été supprimé, ainsi que ses données."
    else:
        message += "ont été supprimés, ainsi que leur données."
    return message

def action_update(userlist, ldapuser):
    """
        Met à jour les comptes à la date d'aujourd'hui
    """
    if debug:
        log.msg(" + Appel de l'action update +")
    if not userlist:
        exception("Aucun compte n'a été sélectionné.")
    str_aujourdhui = format_current_date()
    ldapuser.ldap_admin.connect()
    if len(userlist) == 1:
        message = "Le compte : \\n"
    else:
        message = "Les comptes : \\n"
    for login in userlist:
        if debug:
            log.msg(" + Mise à jour de %s" %  login)
        try:
            ldapuser._set_attr(login, 'LastUpdate', str_aujourdhui)
            message += " - %s \\n" % login
        except:
            log.err()
            exception(" - Erreur à la mise à jour du compte : %s " % (
                                                               login,))
    if len(userlist) == 1:
        message += "a été mis à jour à la date d'aujourd'hui."
    else:
        message += "ont été mis à jour à la date d'aujourd'hui."
    return message

def valid_action(actionname, ldapuser, formulaire):
    """
        éxécute les actions
    """
    if debug:
        log.msg(" + Validation de l'action %s, formulaire =  %s " % (
                                                actionname, formulaire))
    deferred = Deferred()
    deferred.callback(formulaire)
    # userlist = formulaire.keys()
    deferred.addCallback(lambda formulaire:list(formulaire.keys()))
    deferred.addCallback(globals().get('action_%s' % actionname),
                                                ldapuser=ldapuser)
    deferred.addErrback(_clean_errors,
            "Erreur à l'éxécution de l'action : %s" % actionname)
    deferred.addCallback(lambda message:dict(message=message))
    return deferred

def get_ldapsearch_attrs(usertype):
    """
        Renvoie les attributs ldap demandés lors de la requête
    """
    dico = deepcopy(COMMON_LDAP_ATTRS)
    dico.extend(SPEC_LDAP_ATTRS.get(usertype, []))
    return dico

def get_store_attrs(usertype):
    """
        Renvoie les arguments de tri utilisé pour 'usertype'
    """
    return STORE_ARGUMENTS.get(usertype, STORE_ARGUMENTS['default'])

def get_purge_actions(usertype):
    """
        Renvoie les actions disponibles pour le type
        d'utilisateur usertype
    """
    return PURGE_ACTION_DICT.get(usertype, PURGE_ACTION_DICT['default'])

def make_fantom_ldap(lastupdatefilter, usertype):
    """
        Renvoie le résultat de la requête ldap
    """
    try:
        ldapuser = get_userclass_instance(usertype)
    except:
        exception("Erreur à la récupération de" + \
                " l'objet de gestion du type %s." % usertype)
    ldapfilter = "(&%s%s)" % (ldapuser.filtre, lastupdatefilter)
    try:
        ldap_attrs = get_ldapsearch_attrs(usertype)
    except:
        exception("Erreur à la récupération des comptes " + \
                  "auprès de l'annuaire")
    return make_ldap_request(ldapuser,
                                           ldapfilter,
                                           ldap_attrs)

def sort_user_list(userlist):
    """
        Tri la liste des comptes en fonction de l'uid
    """
    userlist = list(userlist)
    sorted(userlist, key=cmp_to_key(get_sort('uid')))
    return userlist

def _list_fantom(memory):
    """
        Liste les comptes 'fantom' (non retrouvés)
    """
    deferred = Deferred()
    deferred.callback(memory['lastupdate'])
    deferred.addCallback(lambda lastupdate:"(LastUpdate<=%s)" % (lastupdate,))
    deferred.addCallback(make_fantom_ldap, usertype=memory['usertype'])
    deferred.addErrback(_clean_errors,
            "Erreur au listing des comptes de type %s" % (
                                            memory['usertype'],))
    deferred.addCallback(map_user_list)
    deferred.addErrback(_clean_errors,
            "Erreur au formattage pour l'affichage des comptes.")
    deferred.addCallback(sort_user_list)
    deferred.addErrback(_clean_errors,
            "Erreur au tri des comptes utilisateurs.")
    store_arguments = get_store_attrs(memory['usertype'])
    deferred.addCallback(store_by_keys,
                         keynames=store_arguments)
    deferred.addErrback(_clean_errors,
            "Erreur au rangement des comptes utilisateurs.")
    return deferred

def _list_doublon():
    """
        Liste les comptes 'doublons'
    """
    exception("Not implemented yet")

def get_search_vars(formulaire):
    """
        Renvoie les critères de recherche
        :formulaire: {'usertype':"<filter_type>_<usertype>",
                      "lastupdate":"pas mis à jour depuis"}
    """
    filtertype, usertype = get_usertype(formulaire)
    lastupdate = get_lastupdate(formulaire)
    return filtertype, usertype, lastupdate

