# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
fonctions de filtrage des données
"""
from scribe.eoletools import deformate_date, get_days_from_epoch, get_nt_time_from_date
from scribe import eoleuser
from scribe.ldapconf import MAIL_DOMAIN, LDAP_MODE
from ead2.backend.actions import tools
from ead2.lib.error import MissingKey
from ead2.backend.actions.scribe.tool.user import ADRESSE

def filter_date(datas, modif):
    """
    gestion de la date de naissance modifiée
    datas : données originales
    modif : données du formulaire
    """
    old_date = datas.get('dateNaissance', [None])[0]
    new_user_date = modif.get('dateNaissance', 0)
    if not new_user_date:
        raise Exception("Erreur : Il manque la date de naissance.")
    if not tools.test_date(new_user_date):
        raise Exception("Erreur : le format de la date de naissance est incorrect (jj/mm/aaaa)")
    new_date = deformate_date(new_user_date)
    if new_date == old_date:
        modif.pop('dateNaissance')
    else:
        modif['dateNaissance'] = new_date
    return modif

def filter_exp_date(datas, modif):
    """
    gestion de la date d'expiration modifiée
    datas : données originales
    modif : données du formulaire
    """
    new_user_date = modif.get('exp_date', '')
    if new_user_date and not tools.test_date(new_user_date):
        raise Exception("Erreur : le format de la date d'expiration est incorrect (jj/mm/aaaa)")

    if LDAP_MODE == 'openldap':
        old_unix_days = datas.get('shadowExpire', [None])[0]
        new_unix_days = str(get_days_from_epoch(new_user_date)) if new_user_date else '-1'
        if new_unix_days != old_unix_days:
            modif['shadowExpire'] = new_unix_days
    else:
        old_nt_time = datas.get('accountExpires', [None])[0]
        new_nt_time = str(get_nt_time_from_date(new_user_date)) if new_user_date else '0'
        if new_nt_time != old_nt_time:
            modif['accountExpires'] = new_nt_time

    if 'exp_date' in modif:
        modif.pop('exp_date')

    return modif

def filter_mail(datas, modif):
    """
    gestion du mail modifié (hors élèves)
    datas : données originales
    modif : données du formulaire
    """
    if 'mailtype' not in modif:
        # mail ni modifié ni initialisé
        modif.pop('mailperso')
    elif modif['mailtype'] == 'perso':
        if 'mail' in datas and \
            datas['mail'][0] == modif['mailperso']:
            # mail distant inchangé
            modif.pop('mailperso')
            modif.pop('mailtype')
        elif not tools.test_mail(modif['mailperso']):
            raise Exception("Erreur : l'adresse mail personalisée n'est pas valide !")
    elif 'mailDir' in datas:
        if datas['mail'][0].endswith(MAIL_DOMAIN['restreint']):
            if modif['mailperso'] == 'restreint':
                modif.pop('mailtype')
                modif.pop('mailperso')
        elif modif['mailperso'] == 'internet':
            modif.pop('mailtype')
            modif.pop('mailperso')
    return modif

def filter_quota(user, modif):
    """
    gestion du quota modifié
    :param user: login de l'utilisateur
    :param modif: données du formulaire
    """
    quota = eoleuser.get_quota(user)
    if int(quota) == int(modif['quota']):
        modif.pop('quota')
    else:
        if not tools.test_number(modif['quota']):
            raise Exception("Erreur : Le quota doit être un nombre.")
        modif['_quota'] = int(modif['quota'])
        modif.pop('quota')
    return modif

def filter_shell(datas, modif):
    """
    gestion du shell modifié
    :param datas: données originales
    :param modif: données du formulaire
    """
    old_shell = 'loginShell' in datas and \
                datas['loginShell'][0] == '/bin/bash'
    modif['shell'] = 'shell' in modif
    if modif['shell'] == old_shell:
        modif.pop('shell')
    return modif

def filter_profil(user, modif):
    """
    gestion du profil modifié
    :param user: login de l'utilisateur
    :param modif: données du formulaire
    """
    profil = eoleuser.User().get_profil(user)
    if '_sambaProfilePath' in modif:
        modif['_sambaProfilePath'] = int(modif['_sambaProfilePath'])
        if modif['_sambaProfilePath'] == profil:
            modif.pop('_sambaProfilePath')
    return modif

def filter_nameandforname(datas, modif):
    """
    gestion des nom et prénom modifiés
    :param datas: données originales
    :param modif: données du formulaire
    """
    for key in ['name', 'forname']:
        if not tools.test_ldap_attribute(modif[key]):
            raise MissingKey("Erreur : Format d'entrée incorrect %s." % (
                                                               modif[key]))

    if modif['name'] == datas['sn'][0]:
        modif.pop('name')

    if modif['forname'] == datas['givenName'][0]:
        modif.pop('forname')
    return modif

def _filter_pupil_modif(user, datas, modif):
    """
    filtre les modifications renvoie le dictionnaire de
    modification en supprimant les champs inchangés
    user  : login de l'utilisateur
    datas : données originales
    modif : données du formulaire
    """

    # date de naissance
    modif = filter_date(datas, modif)

    # clés génériques (qui n'ont pas besoin d'un traitement particulier)
    for key, item in list(datas.items()):
        if key in list(modif.keys()):
            if modif[key] == item[0]:
                modif.pop(key)

    # nom/prénom
    modif = filter_nameandforname(datas, modif)

    #quota
    modif = filter_quota(user, modif)

    # date d'expiration
    modif = filter_exp_date(datas, modif)

    # shell
    modif = filter_shell(datas, modif)

    # numéro élève
    if 'employeeNumber' in modif:
        if not tools.test_number(modif['employeeNumber']):
            raise Exception("Erreur : Le numéro d'élève doit être un nombre.")
    # profils
    modif = filter_profil(user, modif)

    # mail et domaine eleve
    if 'mail' in datas:
        if datas['mail'][0] == '%s@%s' % (modif['email'], modif['maildomain']):
            # domaine inchangé
            modif.pop('maildomain')

    # classe
    if '_Divcod' in modif:
        if modif['_Divcod'] == datas['Divcod'][0]:
            modif.pop('_Divcod')

    # attributs inutiles par la suite
    modif.pop('email')
    modif.pop('login')
    return modif


def _filter_teacher_modif(user, datas, modif):
    """
    filtre les modifications renvoie le dictionnaire de
    modification en supprimant les champs inchangés
    user  : login de l'utilisateur
    datas : données originales
    modif : données du formulaire
    """

    # date de naissance
    modif = filter_date(datas, modif)

    # clé général
    for key, item in list(datas.items()):
        if key in list(modif.keys()):
            if modif[key] == item[0]:
                modif.pop(key)

    for key, item in list(modif.items()):
        if key in ['displayName', 'mailperso']:
            if not tools.test_ldap_attribute(item):
                raise MissingKey("Erreur : Format d'entrée incorrect %s." % item)

    # nom/prénom
    modif = filter_nameandforname(datas, modif)

    # quotas
    modif = filter_quota(user, modif)

    # date d'expiration
    modif = filter_exp_date(datas, modif)

    # shell
    modif = filter_shell(datas, modif)

    # profil
    modif = filter_profil(user, modif)

    ## mail
    modif = filter_mail(datas, modif)

    # attributs inutiles par la suite
    modif.pop('login')
    return modif

def _filter_responsable_modif(datas, modif):
    """
    filtre les modifications renvoie le dictionnaire de
    modification en supprimant les champs inchangés
    datas : données originales
    modif : données du formulaire
    """
    # date de naissance
    if modif['dateNaissance'] == '':
        modif.pop('dateNaissance')
    else:
        modif = filter_date(datas, modif)

    if modif['addassoc'] == '':
        modif.pop('addassoc')

    attrs = [add[0] for add in ADRESSE]
    attrs.extend(['displayName', 'codecivilite'])
    for key in attrs:
        if key not in modif:
            continue
        if modif[key] == datas.get(key, [''])[0]:
            modif.pop(key)

    #for key, item in modif.items():
    #    if key in ['displayName', 'mailperso']:
    #        # FIXME : cette fonction ne fait RIEN !
    #        if not tools.test_ldap_attribute(item):
    #            raise MissingKey, "Erreur : Format d'entrée incorrect %s." % item

    # nom/prénom
    modif = filter_nameandforname(datas, modif)

    ## mail
    modif = filter_mail(datas, modif)

    # attributs inutiles par la suite
    modif.pop('login')
    return modif

def _filter_other_modif(datas, modif):
    """
    filtre les modifications et renvoie le dictionnaire de
    modification en supprimant les champs inchangés
    :pramam datas: données originales
    :param modif: données du formulaire
    """
    # date de naissance
    if modif['dateNaissance'] == '':
        modif.pop('dateNaissance')
    else:
        modif = filter_date(datas, modif)

    # clé général
    for key, item in list(datas.items()):
        if key in list(modif.keys()):
            if modif[key] == item[0]:
                modif.pop(key)

    for key, item in list(modif.items()):
        if key in ['displayName', 'mailperso']:
            if not tools.test_ldap_attribute(item):
                raise MissingKey("Erreur : Format d'entrée incorrect %s." % item)

    # nom/prénom
    modif = filter_nameandforname(datas, modif)

    ## mail
    modif = filter_mail(datas, modif)

    # attributs inutiles par la suite
    modif.pop('login')
    return modif

