# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action spécifique aux profs
# gestion des préférences
#
# modification du mail, section utilisateur, création de groupe
#
###########################################################################
from twisted.python import log
from scribe import eoleuser
from scribe import eolegroup
from scribe import linker
from ead2.lib.error import MissingKey, MissingValue
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.backend.actions.lib.widgets import main as M, form as F
from ead2.backend.actions.scribe.tool import scribe_tools, formencode, \
group, getform, usercreate, validate
from ead2.backend.actions.scribe.tool.prof import get_menu
from ead2.backend.actions.scribe.tool.user import manage_groups
from scribe.ldapconf import SUPPORT_ETAB

USED_TEMPLATES = ['main', 'formselect', 'form', 'listlist',
                  'accordion', 'checklist']

class ProfPreference(Action):
    """
        page d'entrée pour l'édition des préférences d'un prof:
        - modification de son mail
        - modification du mot de passe
        - inscription au groupe
    """
    user_description = Dict(default={},
                            doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_prof_preference'
    libelle = "Préférences"
    category = ""
    description = "Modifier vos préférences"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={},
                       doc="Retour de formulaire en Json", keys=[])

    def execute(self):
        """ renvoie les données pour l'affichage (le menu) """
        self.params, self.server_nb = tools.get_request(self.request)
        result = {'titre':self.description}
        result.update(get_menu(self.server_nb, self.user_description['name']))
        return self.send_all(result, template='scribe_prof_preference',
                             templates=USED_TEMPLATES)


class ProfModMail(Action):
    """ Modifie le mail d'un prof """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_prof_mod_mail'
    libelle = "Mail"
    category = None
    description = "Modification du mail d'un utilisateur"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={},
                       doc="Retour de formulaire en Json",
                       keys=['mod_mail'])

    def execute(self):
        """ renvoie les données pour l'affichage """
        params, self.server_nb = tools.get_request(self.request)
        self.user = self.user_description['name']
        result = {}
        if self.form_result != {}:
            try:
                result.update(self._valid_form())
            except (MissingValue, MissingKey, Exception) as mess:
                log.err("Erreur dans prof.py : %s" % mess)
                result['message'] = str(mess)
            self.form_result = {}
        result.update(self._get_form())
        return self.send_all(result, template='scribe_prof_preference',
                             templates=USED_TEMPLATES)

    def _get_form(self, _type=None):
        """
        renvoie la description du formulaire d'édition
        de l'adresse mail
        """
        user = F.Input(name='user_name', default=self.user)
        datas = eoleuser.User().get_attrs(self.user, [])
        mail = formencode.make_mailinput(self.user, datas)
        return dict(user_name=user,
                    mail=mail,
                    validate=self._get_validate_btn(),
                    icone='/image/scribe/action_liste_over.png')

    def _get_validate_btn(self):
        """ renvoie la description du bouton valider """
        href = tools.make_form_link(self.server_nb, self.name,
                                    True, ['mod_mail'],
                                    balise='mod_user_div_container')
        return M.Submit(href=href, title="Modifier le mail")

    def _valid_form(self):
        """ valide la modification du mail du prof """
        mod_mail = self.form_result.get('mod_mail', None)
        if not mod_mail:
            raise MissingKey("Erreur : Il manque des données pour la modification du mail.")
        resultat = tools.format_form_result(mod_mail)
        mailtype = resultat.get('mailtype', None)
        mailperso = resultat.get('mailperso', '')
        if not mailtype:
            raise MissingKey("Erreur : il manque des valeurs pour la modification du mail.")
        elif mailtype == 'perso':
            mailperso = validate.test_mailperso(mailperso)
        ldapuser = linker._user_factory(self.user)
        ldapuser._mod_mail(self.user, mailtype, mailperso)
        ldapuser.ldap_admin.close()
        return dict(message = "L'adresse mail a bien été modifiée.")


class ProfModGroupe(Action):
    """ Inscrit un utilisateur à des groupes
        Action spécifique aux profs ou prof_admin
    """
    user_description = Dict(default={}, doc="description de l'exécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_prof_mod_groupe'
    libelle = "Groupes"
    category = None
    description = "Modification des groupes d'un utilisateur"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['groupe_form'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie le formulaire d'inscription aux groupes
            2 - valide les inscriptions
        """
        params, self.server_nb = tools.get_request(self.request)
        self.user = self.user_description['name']
        if SUPPORT_ETAB:
            self.etab = eoleuser.User().get_etab(self.user)
        else:
            self.etab = None
        result = {'sstitre':"Inscription à un groupe"}
        ## 2 -
        if self.form_result != {}:
            try:
                result.update(self._valid_form())
            except (MissingValue, MissingKey, Exception) as mess:
                log.err("Erreur dans prof.py : %s" % mess)
                result['message'] = str(mess)
            self.form_result = {}
        ## 1 -
        result.update(self._get_form())
        return self.send_all(result, template='scribe_prof_preference',
                             templates=USED_TEMPLATES)

    def _get_form(self, _type=None):
        """ renvoie la description du formulaire d'édition de mot de passe """
        user_name = F.Input(name='user_name',
                            default=self.user)

        ## gestion des groupes
        ldapgroup = eolegroup.Group()
        ldapgroup.ldap_admin.connect()
        user_all_groups = set(ldapgroup._get_user_groups(self.user))
        all_groups = set(ldapgroup._get_groups('Groupe', etab=self.etab))
        ldapgroup.ldap_admin.close()
        user_groups = scribe_tools.sort_list(all_groups.intersection(user_all_groups))
        available_groups = scribe_tools.sort_list(all_groups.difference(user_all_groups))
        available_groups.sort()
        user_groups.sort()
        _liste = {'title':"Vos groupes",
                  'formid':'groupe_form',
                  'id':'groupe_table',
                  'datas':user_groups}
        _select = F.Select(name='groupe_select',
                           libelle="Groupes disponibles",
                           _id='groupe_select',
                           multi=True, size=10)
        for _gr in available_groups:
            _select.add_option(_gr)
        groupe = {'name':"groupe",
                  'btns':formencode.get_transf_btns('groupe'),
                  'select':_select,
                  'liste':_liste,
                  'title':""}

        return dict(user_name=user_name,
                    groupe=groupe,
                    validate=self._get_validate_btn(),
                    icone='/image/scribe/action_groupe_over.png')

    def _get_validate_btn(self):
        """ bouton de validation """
        href = tools.make_form_link(self.server_nb, self.name,
                                    True, ['groupe_form'],
                                    balise='mod_user_div_container')
        return M.Submit(href=href,
                        title="Modifier les inscriptions aux groupes")

    def _valid_form(self):
        """ valide l'inscription aux groupes """
        if 'groupe_form' not in self.form_result:
            raise MissingKey("Erreur : il manque des données pour la gestion des groupes.")
        groupe = tools.format_form_result(self.form_result['groupe_form'], '2')
        ldapgroup = eolegroup.Group()
        ldapgroup.ldap_admin.connect()
        manage_groups(self.user, 'Groupe', ldapgroup, list(groupe.keys()))
        ldapgroup.ldap_admin.close()
        message = "La modification des groupes de %s s'est bien passée."
        return dict(message=message % self.user)


class ProfUser(Action):
    """ action d'entrée pour la gestion des élèves
        renvoie vers ProfUserCreate et ProfModUser
    """
    user_description = Dict(default={},
                            doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_prof_user'
    libelle = "Mes Élèves"
    category = "Gestion"
    description = "Gestion des élèves"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=[])

    def execute(self):
        params, self.server_nb = tools.get_request(self.request)
        result = {'titre':"Gestion des élèves"}
        result.update(self._get_menu())
        return self.send_all(result, template='scribe_prof_user', templates=USED_TEMPLATES)

    def _get_menu(self):
        """ renvoie la description du menu """
        container = 'main_div_container'
        create = M.Bouton(href=tools.make_js_link(self.server_nb,
                                                  'scribe_prof_user_create',
                                                  container),
                          libelle="Créer élève",
                          _class='scribe_action_btn',
                          title="Création d'utilisateur",
                          icone="/image/scribe/action_user_create.png")
        modify = M.Bouton(href=tools.make_js_link(self.server_nb,
                                                  'scribe_prof_user_modify',
                                                  container),
                          libelle="Modifier élève",
                          title="Modification d'utilisateur",
                          icone="/image/scribe/action_user_modify.png",
                          _class='scribe_action_btn')
        return dict(menus=(create, modify))

class ProfCreateUser(Action):
    """
    action de création d'élève
    pour les prof administrateur de classe (prof_admin)
    """
    user_description = Dict(default={},
                            doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_prof_user_create'
    libelle = "Utilisateurs"
    category = None
    description = "Création d'un élève"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['add_user'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie du formulaire de création d'élève
            2 - création de l'élève avec contrôle des droits du prof
        """
        params, self.server_nb = tools.get_request(self.request)
        self.user = self.user_description['name']

        self.user_type = scribe_tools.get_role(self.user_description['role'])

        result = {'sstitre':"Création d'un utilisateur"}
        ## 2 -
        if self.form_result != {}:
            try:
                result.update(self._valid_form())
                result['toexec'] = tools.make_js_link(self.server_nb,
                                                      self.name, confirm=True,
                                                      balise='main_div_container')
            except (MissingValue, MissingKey, Exception) as mess:
                log.err("Erreur dans prof.py : %s" % mess)
                result['message'] = str(mess)
            self.form_result = {}
            return self.send_frag(result, template='scribe_prof_user',
                                  templates=USED_TEMPLATES)
        ## 1 -
        result.update(self._get_form())
        result.update(self._get_validate_btn())
        result.update(self._get_help())
        return self.send_all(result, template='scribe_prof_user',
                             templates=USED_TEMPLATES)

    def _get_form(self):
        """ renvoie la description du formulaire de création d'utilisateur """
        form = getform._get_pupil_form(self.user_type, self.user)
        form.update(dict(icone='/image/scribe/action_user_create_over.png'))
        return form

    def _get_help(self):
        """ renvoie la description du lien vers l'aide """
        return dict(help=dict(href=tools.make_help_link(self.server_nb,
                                   'help', help_page='scribe_user')))

    def _get_validate_btn(self):
        """ renvoie la description du bouton valider """
        href = tools.make_form_link(self.server_nb, self.name,
                                    True,
                                    ['add_user'],
                                    'return_msg_container')
        return dict(validate=M.Submit(href=href,
                                      title="Créer l'utilisateur"))

    def _valid_form(self):
        """ valide les retours formulaires, en l'occurence crée un élève """
        return usercreate.create_pupil(self.form_result)

class ProfModUser(Action):
    """
        entrée pour la modification d'élève
        pour les prof administrateur de classe (prof_admin)
        s'appuye sur le listing (scribe_user_table, la modification,
        modification de mdp, suppression d'élève de user.py)
    """
    user_description = Dict(default={}, doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_prof_user_modify'
    libelle = "Modifier"
    category = None
    description = "modification d'un élève"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['add_user'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie le formulaire pour lister les élèves
            2 - filtre les élèves depuis un formulaire
        """
        params, self.server_nb = tools.get_request(self.request)
        self.user = self.user_description['name']
        self.user_type = scribe_tools.get_role(self.user_description['role'])
        ## 2 -
        if self.form_result != {}:# validation de formulaire
            # FIXME -> self._valid_form()
            filtered_datas = self._valid_form()
            self.form_result = {}
            return self.send_frag(filtered_datas, template='scribe_prof_user',
                                  templates=USED_TEMPLATES)
        ## 1 -
        result = {}
        result.update(getform._get_user_listing_form(self.user))
        result.update(dict(validate=self._get_validate_btn()))
        return self.send_all(result, template='scribe_prof_user',
                             templates=USED_TEMPLATES)

    def _get_validate_btn(self):
        """ renvoie la description du bouton valider """
        href = tools.make_form_link(self.server_nb, 'scribe_user_table',
                                    True,
                                    ['search'],
                                    'user_div_container')
        return M.Submit(href=href,
                        title='Lister selon les filtres',
                        libelle="Lister")

class ProfGroup(Action):
    """ action d'entrée pour la gestion des groupes """
    user_description = Dict(default={}, doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_prof_group'
    libelle = "Les groupes"
    category = "Gestion"
    description = "gestion des groupes"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json", keys=[])

    def execute(self):
        """ renvoie les données pour l'affichage (le menu )"""
        params, self.server_nb = tools.get_request(self.request)
        result = {'titre':"Gestion des groupes"}
        result.update(self._get_menu())
        return self.send_all(result, template='scribe_prof_group',
                             templates=USED_TEMPLATES)

    def _get_menu(self):
        """ renvoie la description du menu """
        html_container = "main_div_container"
        create_action_name = "scribe_prof_group_create"
        modify_action_name = "scribe_prof_user_modify"
        create = M.Bouton(href=tools.make_js_link(self.server_nb,
                                                  create_action_name,
                                                  html_container),
                          libelle="Créer groupe",
                          title="Création de groupe",
                          icone="/image/scribe/action_group_create.png",
                          _class='scribe_action_btn')
        modify = M.Bouton(href=tools.make_js_link(self.server_nb,
                                                  modify_action_name,
                                                  html_container),
                          libelle="Modifier un élève",
                          title="Modification d'utilisateur",
                          icone="/image/scribe/action_user_modify.png",
                          _class="scribe_action_btn")
        return dict(menus=(create,))

class ProfGroupCreate(Action):
    """ outil de création de groupe par un prof_admin
    (que des groupes de type Groupe)"""
    user_description = Dict(default={}, doc="description de l'éxécutant",
                            keys=['ip', 'name', 'role'])
    name = 'scribe_prof_group_create'
    libelle = "Créer"
    category = None
    description = "Création de groupe"
    request = Dict(default={},
                   doc="arguments de la requete en cours cote frontend",
                   keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                       keys=['add_group'])

    def execute(self):
        """ renvoie les données pour l'affichage de la création de groupe
            1 - renvoie le formulaire de création de groupe
            2 - valide la création
        """
        params, self.server_nb = tools.get_request(self.request)
        ## 2 -
        if self.form_result != {}:
            try:
                datas = self.create_group()
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name,
                                    confirm=True, balise="main_div_container")
            except Exception as mess:
                log.err("Erreur dans prof.py : %s" % mess)
                datas = dict(message = str(mess))
            self.form_result = {}
            return self.send_frag(datas, template='scribe_prof_group',
                                  templates=USED_TEMPLATES)
        ## 1 -
        result = {'icone':'/image/scribe/action_group_create_over.png'}
        result.update(getform.groupe_form('groupe', "du groupe"))
        result.update(self._get_valid_btn())
        return self.send_all(result, template='scribe_prof_group',
                             templates=USED_TEMPLATES)

    def _get_valid_btn(self):
        """ bouton valider pour la creation
        d'user passe en meme temps le type en cours (classe, nivo...)"""
        href = tools.make_form_link(self.server_nb, self.name, True,
                               ['add_group'], 'return_msg_container')
        return dict(validate=M.Submit(href=href,
                                      title="Créer ce groupe"))

    def create_group(self):
        """ cree un groupe """
        if 'add_group' not in self.form_result:
            raise MissingKey("Erreur : il manque des données pour la création du groupe.")
        creation = tools.format_form_result(self.form_result['add_group'])
        return dict(message=group.groupe_create(creation, 'Groupe', 'du groupe'))

