# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Action gestion_group*
#
# Cree modifie et supprime des groupes
#
###########################################################################

"""Actions permettant de gérer les utilisateurs, compte, partage, groupe (application Gestion)"""
from twisted.python import log
from ead2.backend.lib.action import Action, Dict
from ead2.backend.actions import tools
from ead2.lib.libead import uni
from horus import backend as horus # import des actions (get_groups, mod_user, inscription ... )
from ead2.backend.actions.lib.widgets import main, form as F
from ead2.backend.actions.horus import gestion_tools as G
from ead2.lib.error import BadGroupName, BadShareName

titre = 'Gestion des groupes'
used_templates = ['main', 'alphas', 'listlist', 'accordion', 'form', 'formselect', 'checklist' ]
spec_groups = ['DomainAdmins', 'DomainUsers', 'applidos', 'minedu', 'DomainComputers', 'PrintOperators']

def _get_groups():
    """ recupere la liste des groupes existants"""
    try:
        groups = horus.get_groups()
        for num, grp in enumerate(groups):
            if grp in spec_groups:
                groups[num] = "- " + grp + "<br />  (Réservé eole)"
            else:
                groups[num] = "- " + grp
        groups.insert(0, '<b>Groupes existants:</b>')
    except:
        groups = ["Il n'y a aucun groupe"]
    return groups

class GestionGroupCreate(Action):
    """ creation d'un groupe """
    user_description = Dict(default={}, doc="description de l'éxécutant", keys=['ip', 'name', 'role'])
    name = 'gestion_group_create'
    libelle = "Création"
    category = "Gestion/Groupes"
    description = 'créer un groupe'
    request = Dict(default={}, doc="arguments de la requete en cours cote frontend",
                        keys=['server', 'action', 'user_alpha'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json",
                        keys=['group', 'active', 'unactive', 'share_form', 'alpha_key'])

    def execute(self):
        """ renvoit les données pour l'affichage
            1 - envoie la description du formulaire de création de groupe
            2 - renvoie les utilisateurs dont le nom commence par la lettre 'user_alpha'
            3- valide la création et renvoie un message
        """
        params, self.server_nb = tools.get_request(self.request)
        menu = G.get_group_menu(self.server_nb, self.name)
        sstitre = self.description
        result = {'sstitre':sstitre, 'titre':titre, 'menus':menu}
        result['retour']  = G.get_return_btn(self.server_nb)
        retour = ''
        ## 2 -
        self.user_alpha = ''
        if 'user_alpha' in params:
            self.user_alpha = params['user_alpha'][0]
            return self.user_frag()
        ## 3 -
        if self.form_result != {}:
            try:
                retour = self.create_group(self.form_result)
                retour['toexec'] = tools.make_js_link(self.server_nb, self.name, confirm=True)
            except Exception as mess:
                log.err()
                retour = {'message':str(mess)}
            self.form_result = {}
            return self.send_frag(retour, template='horus_gestion_group', templates=used_templates)
        ## 1 -
        result['leftpart'] = _get_groups()
        result['message'] = retour
        result.update(self._get_form())
        result.update(self._get_valid_btn())
        return self.send_all(result, template='horus_gestion_group', templates=used_templates)

    def user_frag(self):
        """ renvoit le fragment listant les utilisateurs filtré par la première lettre du login"""
        result = {}
        result['datas'] = self._get_users()
        data = {'template':'checklist', 'data':{'content':result, 'templates':['checklist']}}
        return 0, uni(str(data))

    def _get_form(self):
        """ renvoit les données pour le formulaire """
        #input pour le nom d'utilisateur
        group_name = F.Input(name='group_name', libelle="Nom du groupe", js="testVoid('group_name', 'error_group_name')")
        # section ajout d'utilisateur
        left = {'title':"Utilisateurs disponibles", 'form':"active", 'datas':self._get_users()}
        if self.user_alpha != 'tous' and self.user_alpha:
            left['title'] = "Utilisateurs disponibles <br /> Lettre %s" % self.user_alpha
        right = {'title':"Utilisateurs du groupe", 'form':"unactive", 'datas':[]}
        title = ''
        name = 'group_users'
        group_users = {'name':name, 'btns':self._get_transf_btns1(), 'left':left, 'right':right, 'title':title}
        alphas = self._get_alpha_links()
        # section ajout de partage
        group_shares = {}
        group_shares['btns'] = self._get_transf_btns2()
        group_shares['input'] = F.Input(libelle='Nom du partage (sans accent)', name='share')
        group_shares['liste'] = {'title':'Liste des partages associés au groupe', 'values':[], 'formid':'share_form',
                            'id':'share_table'}
        group_shares['title'] = "Ajouter des partages et les lier au groupe"
        return dict(group_name=group_name, group_users=group_users, group_shares=group_shares,
                    user_alphas=alphas, focus='group_name')

    def _get_users(self):
        """ renvoit les utilisateurs disponibles dont le nom commence par user_alpha """
        try:
            usrs = horus.get_users()
        except:
            usrs = []
        result = []
        if self.user_alpha:
            alpha = self.user_alpha.lower()
        else:
            alpha = 'tous'
        if alpha == 'tous': return [{'name':usr, 'libelle':usr}for usr in usrs]
        if alpha:
            result = [{'name':usr, 'libelle':usr}for usr in tools.filter_by_alpha(usrs, alpha)]
        else:
            result = [{'name':usr, 'libelle':usr}for usr in usrs]
        return result

    def _get_alpha_links(self):
        """ renvoit la description des liens alphabetique """
        links = []
        alpha = self.user_alpha
        for i in range(97, 123):
            if chr(i) != alpha:href = tools.make_js_link(self.server_nb, self.name, balise='active_table', user_alpha=chr(i))
            else: href = ''
            libelle = chr(i).upper()
            links.append({'href':href, 'libelle':libelle})
        if 'tous' != alpha:
            links.append({'href':tools.make_js_link(self.server_nb, self.name, balise='active_table', user_alpha='tous'), 'libelle':'Tous'})
        else:
            links.append({'href':'', 'libelle':'Tous'})
        return links

    def _get_transf_btns1(self):
        """ renvoit les btns de transfert de donnees liste a liste pour les utilisateurs"""
        href2 = "javascript:formTransferListElement('active','unactive_table');"
        href1 = "javascript:formTransferListElement('unactive','active_table');"
        icone1 = "/image/rightleft_red.png"
        icone2 = "/image/leftright.png"
        return [main.Bouton(href=href2, libelle="Ajouter", icone=icone2),
                main.Bouton(href=href1, libelle="Retirer", icone=icone1)]

    def _get_transf_btns2(self):
        """ renvoit les btns de transfert de donnees input a liste pour les partages"""
        href1 = "javascript:formRemoveFromList('share_form');"
        href2 = "javascript:formAddNameToList1('share','share_table');"
        icone1 = "/image/rightleft_red.png"
        icone2 = "/image/leftright.png"
        return [main.Bouton(href=href2, libelle="Ajouter", icone=icone2),
                main.Bouton(href=href1, libelle="Retirer", icone=icone1)]

    def _get_valid_btn(self):
        """ renvoit la description du bouton valider """
        form_names = ['group', 'unactive', 'share_form', 'alpha_key']
        container = 'group_msg_div_container'
        href = tools.make_form_link(self.server_nb, self.name, True, form_names, container)
        validation = tools.make_key_form_link(self.server_nb, self.name, False, form_names, container)
        return dict(validate=main.Bouton(href=href, libelle="Valider", title="Créer ce groupe"),
                    validation=validation)

    def create_group(self, dico):
        """ cree un groupe depuis le retour formulaire du frontend
            group: le nom du groupe
            share_form: les partages associés au groupe
            unactive: les utilisateurs du groupe
        """
        group = ''
        ## récupération des valeurs venant du client
        if 'alpha_key' in dico:
            self.user_alpha = dico['alpha_key'][0]['value']
        if 'group' in dico:
            result = tools.format_form_result(dico['group'])
            if 'group_name' in result:
                group = result['group_name'] # nom du groupe cree
                if not tools.test_login(group):
                    raise BadGroupName
            else:
                raise Exception("Erreur lors de la validation du formulaire")

        shares = {}
        if 'share_form' in dico:
            share_form = dico['share_form']
            resultat = tools.format_form_result(share_form, True)
            keys = list(resultat.keys())
            for key in keys:
                if key.startswith('sticky'):
                    share_name = key[6:]
                    if not tools.test_login(share_name):
                        raise BadShareName
                    if share_name in keys:
                        shares[share_name] = resultat[key]

        # creation du groupe
        if group:
            if horus.is_group(group):
                raise Exception("Erreur : un groupe de ce nom existe déjà.")
            add = horus.add_group(group)
            if not add:
                raise Exception("Erreur : une erreur est survenue à la création du groupe.")
            else:
                message = "Le groupe %s a bien été créé.\\n\\n" % group
            # inscription des utilisateurs
            if 'unactive' in dico:
                users = list(tools.format_form_result(dico['unactive'], True).keys())
                if users != []:
                    message += "Inscription d'utilisateur : \\n"
                    for nom in users:
                        if horus.inscription(nom, group):
                            message += "   - %s a bien été inscrit au groupe.\\n" % nom
                        else:
                            message += "   - erreur lors de l'inscription de %s. \\n" % nom
           # gestion des partages attaches au groupe on recupere le nom, le sticky_bit et on cree
            if shares != {}:
                message += "Attachement de partage : \\n"
                for share, sticky in list(shares.items()):
                    if not horus.add_share(share, group, sticky=sticky):
                        message += " - une erreur est survenue lors de l'attachement du partage %s au groupe.\\n" % share
                    else:
                        message += "   - Le partage %s a bien été créé et attaché au groupe.\\n" % share
            return dict(message=message)
        raise Exception("Erreur: une erreur est survenue lors de la création du groupe.")

class GestionGroupModify(Action):
    """ modification d'un groupe """
    user_description = Dict(default={}, doc="description de l'éxécutant", keys=['ip', 'name', 'role'])
    name = 'gestion_group_modify'
    libelle = "Modification"
    category = "Gestion/Groupes"
    description = 'Modifier un groupe'
    request = Dict(default={}, doc="arguments de la requete en cours cote frontend",
                    keys=['server', 'action', 'current_group', 'user_alpha'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json", keys=['group', 'active', 'unactive', 'share_form', 'alpha_key'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - récupère le groupe en cours d'édition
            2 - renvoie le formulaire de modification de groupe
            3 - renvoie la liste des utilisateurs filtrés sur la première lettre de login
            4 - valide la modification
        """
        params, self.server_nb = tools.get_request(self.request)
        menu = G.get_group_menu(self.server_nb, self.name)
        sstitre = self.description
        result = {'sstitre':sstitre, 'titre':titre, 'menus':menu}
        result['retour']  = G.get_return_btn(self.server_nb)
        self.user_alpha = ''
        ## 4 -
        if self.form_result != {}:
            try:
                datas = self.modify_group(self.form_result)
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name, confirm=True,
                                        current_group=self.current_group)
            except (Exception) as mess:
                log.err()
                datas = dict(message=str(mess))
            self.form_result = {}
            return self.send_frag(datas, template='horus_gestion_group', templates=used_templates)
        ## 1 -
        if 'current_group' in list(params.keys()):
            self.current_group = params['current_group'][0]
        else:
            self.current_group = ''
        ## 3 -
        if 'user_alpha' in params:
            self.user_alpha = params['user_alpha'][0]
            return self.user_frag()
        ## 2 -
        result.update(self._get_form())
        result.update(self._get_valid_btn())
        result['leftpart'] = _get_groups()
        return self.send_all(result, template='horus_gestion_group', templates=used_templates)

    def user_frag(self):
        """ renvoit le fragment listant les utilisateurs"""
        result = {}
        try:
            members = horus.get_members(self.current_group)
        except:
            members = []
        result['datas'] = [F.Checkbox(user, libelle=user)for user in self._get_users()if user not in members]
        return self.send_frag(result, template='checklist', templates=used_templates)

    def _get_form(self):
        """ renvoit les données pour le formulaire """
        try:
            groups = horus.get_groups()
        except:
            groups = []

        group_name = F.Select(name='group_name', libelle="Nom du groupe",
                                 onchange=tools.make_js_link(self.server_nb, self.name, code=True,
                                 current_group='this.options[this.selectedIndex].value'))
        for group in groups:
            if group in spec_groups:
                libelle = "%s*" % group
            else:
                libelle = group
            if group == self.current_group:
                group_name.add_option(group, libelle, default=True)
            else:
                group_name.add_option(group, libelle)

        if not self.current_group:
            # aucune sélection -> affichage
            group_name.add_option('', default=True)
            return dict(group_select=group_name)

        # un groupe sélectionné -> on continue

        # section ajout d'utilisateur
        members = []
        if self.current_group:
            try:
                members = horus.get_members(self.current_group)
            except:
                members = []
            users = [user for user  in self._get_users() if user not in members]
        else:
            users = self._get_users()

        left_opts = [{'name':user}for user in users]
        right_opts = [{'name':member}for member in members]
        left = {'title':"Utilisateurs disponibles", 'form':"active", 'datas':left_opts}
        if self.user_alpha != 'tous' and self.user_alpha:
            left['title'] = "Utilisateurs disponibles <br /> Lettre %s" % self.user_alpha
        right = {'title':"Utilisateurs du groupe", 'form':"unactive", 'datas':right_opts}
        title = ''
        name = 'group_users'
        group_users = {'name':name, 'btns':self._get_transf_btns1(), 'left':left, 'right':right, 'title':title}
        alphas = self._get_alpha_links()

        if self.current_group in spec_groups:
            # affichage sans les partages (group_shares)
            return dict(group_select=group_name, group_users=group_users, user_alphas=alphas)

        # section ajout de partage
        group_shares = {}
        group_shares['btns'] = self._get_transf_btns2()
        group_shares['input'] = F.Input(name='share', libelle='Nom du partage (sans accent)')
        try:
            shares = horus.get_group_shares(self.current_group)
        except:
            shares = []
        group_shares['liste'] = {'title':'Liste des partages associés au groupe', 'values':shares, 'formid':'share_form',
                            'id':'share_table'}
        group_shares['title'] = "Ajouter des partages et les lier au groupe"
        return dict(group_select=group_name, group_users=group_users,
                    group_shares=group_shares, user_alphas=alphas)


    def _get_users(self):
        """ renvoit les utilisateurs disponibles dont le nom commence par user_alpha """
        try:
            usrs = horus.get_users()
        except:
            usrs = []
        if self.user_alpha:
            alpha = self.user_alpha.lower()
        else:
            alpha = 'tous'
        if alpha == 'tous': return usrs
        if alpha:
            return tools.filter_by_alpha(usrs, alpha)
        else:
            return usrs

    def _get_alpha_links(self):
        """ renvoit la description des liens alphabetique """
        links = []
        alpha = self.user_alpha
        for i in range(97, 123):
            if chr(i) != alpha:
                href = tools.make_js_link(self.server_nb, self.name,
                                          balise='active_table',
                                          user_alpha=chr(i),
                                          current_group=self.current_group)
            else:
                href = ''
            libelle = chr(i).upper()
            links.append({'href':href, 'libelle':libelle})
        if 'tous' != alpha:
            links.append({'href':tools.make_js_link(self.server_nb, self.name, balise='active_table',
                          user_alpha='tous', current_group=self.current_group), 'libelle':'Tous'})
        else:
            links.append({'href':'', 'libelle':'Tous'})
        return links

    def _get_transf_btns1(self):
        """ renvoit les btns de transfert de donnees liste a liste pour les utilisateurs"""
        href2 = "javascript:formTransferListElement('active','unactive_table');"
        href1 = "javascript:formTransferListElement('unactive','active_table');"
        icone1 = "/image/rightleft_red.png"
        icone2 = "/image/leftright.png"
        return [main.Bouton(href=href2, libelle="Ajouter", icone=icone2),
                main.Bouton(href=href1, libelle="Retirer", icone=icone1)]

    def _get_transf_btns2(self):
        """ renvoit les btns de transfert de donnees input a liste pour les partages"""
        href1 = "javascript:formRemoveFromList('share_form');"
        href2 = "javascript:formAddNameToList1('share','share_table');"
        icone1 = "/image/rightleft_red.png"
        icone2 = "/image/leftright.png"
        return [main.Bouton(href=href2, libelle="Ajouter", icone=icone2),
                main.Bouton(href=href1, libelle="Retirer", icone=icone1)]

    def _get_valid_btn(self):
        """ renvoit la description du bouton valider """
        form_names = ['group','active', 'unactive', 'share_form', 'alpha_key']
        container = 'group_msg_div_container'
        href = tools.make_form_link(self.server_nb, self.name, True,
                                    form_names, container)
        validation = tools.make_key_form_link(self.server_nb, self.name, False,
                                form_names, container)
        return dict(validate=main.Bouton(href=href, libelle='Valider', title="Modifier ce groupe"),
                    validation=validation)

    def modify_group(self, dico):
        """ modifie un groupe
            :dico: - unactive est la liste des users du groupe
                   - active est la liste de users qui ne sont pas du groupe
                   - share_form est la liste des partages associés au groupe
        """
        if 'alpha_key' in dico:
            self.user_alpha = dico['alpha_key'][0]['value']

        ## recuperation du nom du groupe
        if 'group' not in dico:
            raise Exception("Il manque des données pour la modification du groupe.")
        group_form = tools.format_form_result(dico['group'])
        if 'group_name' not in group_form:
            raise Exception("Il manque des données pour la modification du groupe.")
        group = group_form['group_name']
        self.current_group = group
        if not group: raise Exception("Aucun groupe n'a été sélectionné.")

        ## membres du groupe
        message2 = ""
        if 'unactive' in dico and 'unactive' in dico:
            try:
                myusers = horus.get_members(group)
            except:
                myusers = []
            users = list(tools.format_form_result(dico['unactive'], True).keys())
            no_users = list(tools.format_form_result(dico['active'], True).keys())
            for user in users:
                if user not in myusers:
                    if horus.inscription(user, group): message2 += "- %s a bien été inscrit\\n" % user
                    else:  message2 += "erreur à l'inscription de %s\\n" % user

            for user in no_users:
                if group == horus.get_gid(user):
                    message2 += "erreur : le groupe %s est le groupe principal de %s, impossible de le(la) désinscrire\\n" % (group, user)
                elif user in myusers:
                    if horus.desinscription(user, group): message2 += "- %s a bien été désinscrit\\n" % user
                    else: message2 += "erreur à la désinscription de %s\\n" % user

        if message2 == "": message2 = "  Aucune modification\\n"
        message = "Modification des inscriptions au groupe %s :\\n%s" % (group, message2)

        ## partage associé
        if 'share_form' in dico:
            message += "\\nModification des partages du groupe %s :\\n" % group
            message3 = ""
            try:
                myshares = horus.get_group_shares(group)
            except:
                myshares = []
            shares = {}
            if 'share_form' in dico:
                share_form = dico['share_form']
                resultat = tools.format_form_result(share_form, True)
                keys = list(resultat.keys())
                for key in keys:
                    if key.startswith('sticky'):
                        share_name = key[6:]
                        if not tools.test_login(share_name): raise BadShareName
                        if share_name in keys: shares[share_name] = resultat[key]
                    elif 'sticky'+key not in keys:
                        shares[key] = False
            if shares != {}:
                for share, sticky in list(shares.items()):
                    if share not in myshares:
                        if not horus.add_share(share, group, sticky=sticky):
                            message3 += "- erreur lors de l'association du partage %s au groupe\\n" % share
                        else:
                            message3 += "- création et association du partage %s réussie\\n" % share
            # on supprime les partages qui ont ete enleve
            for name in myshares:
                if name not in shares:
                    if horus.del_share(name): message += "- suppression du partage %s\\n" % name
                    else: message += "- echec de la suppression du partage %s\\n" % name
            if message3 == "": message3 = "  Aucune modification\\n"
            message += message3

        return dict(message=message)

class GestionGroupDelete(Action):
    """ gere la suppression de partage """
    user_description = Dict(default={}, doc="description de l'éxécutant", keys=['ip', 'name', 'role'])
    name = 'gestion_group_suppr'
    libelle = "Suppression"
    category = "Gestion/Groupes"
    description = 'supprimer un groupe'
    request = Dict(default={}, doc="arguments de la requete en cours cote frontend", keys=['server', 'action'])
    form_result = Dict(default={}, doc="Retour de formulaire en Json", keys=['group'])

    def execute(self):
        """ renvoie les données pour l'affichage
            1 - renvoie le formulaire de suppression de groupe
            2 - supprime le groupe
        """
        params, self.server_nb = tools.get_request(self.request)
        menu = G.get_group_menu(self.server_nb, self.name)
        sstitre = self.description
        result = {'sstitre':sstitre, 'titre':titre, 'menus':menu}
        ## 2 -
        if self.form_result != {}:
            try:
                datas = self.del_group(self.form_result)
                datas['toexec'] = tools.make_js_link(self.server_nb, self.name)
            except Exception as mess:
                log.err()
                datas = dict(message=str(mess))
            self.form_result = {}
            return self.send_frag(datas, template='horus_gestion_group', templates=used_templates)
        ## 1 -
        result['group_select'] = self._get_form()
        result['validate'] = self._get_valid_btn()
        result['retour']  = G.get_return_btn(self.server_nb)
        return self.send_frag(result, template='horus_gestion_group', templates=used_templates)

    def _get_form(self):
        """ renvoie la description du formulaire de suppresion de groupe """
        form = []
        groups = [group for group in self._get_groups()if group not in spec_groups]
        group_select = F.Select(name='group_name', libelle="Nom du Groupe")
        for group in groups:
            group_select.add_option(group)
        return group_select

    def _get_groups(self):
        """ recupere la liste des groupes existants"""
        try:
            groups = horus.get_groups()
        except:
            groups = []
        return groups

    def del_group(self, dico):
        """ supprimer un groupe """
        if 'group' in dico:
            result = tools.format_form_result(dico['group'])
            if 'group_name' not in result:
                raise Exception("Il manque des données pour la suppression de groupe.")
            name = result['group_name']
            if name:
                if horus.del_group(name):
                    return dict(message="Le groupe %s a bien été supprimé" % name)
                else:
                    raise Exception("Erreur : erreur lors de la suppression du groupe %s" % name)
        raise Exception("Erreur : il manque des données pour la suppression de groupe")

    def _get_valid_btn(self):
        """ renvoit la description du bouton valider """
        href = tools.make_form_link(self.server_nb, self.name, False, ['group'], 'group_msg_div_container')
        href = tools.make_confirm_link("Êtes-vous sûr de vouloir supprimer ce groupe ?", href)
        return main.Bouton(href=href, libelle="Valider", title="Supprimer ce groupe")
