#!/usr/bin/env python
#-*- coding: utf-8 -*-
"""

utilitaire d'importation de configuration config.eol 2.2 ou config.eol 2.3
vers config.eol 2.4

usage :

   %prog <config_file_name>

"""
from ConfigParser import ConfigParser

from tiramisu.option import SymLinkOption, ChoiceOption
from .eosfunc import is_empty
from .var_loader import convert_value
import re
from itertools import product

from .i18n import _

# ____ logger utility ____
# log_filename = u'/var/log/creole.log'
# try:
#     from pyeole.log import init_logging
# except:
#     # compatibilité pour Zéphir 2.3
#     from pyeole.log import make_logger
#     log = make_logger(u'creole3.upgrade',
#                       logfile=log_filename,
#                       loglevel='INFO')
# else:
#     log = init_logging(name=u'creoleUpgrade',
#                        level='info',
#                        console=False,
#                        syslog=None,
#                        filename=log_filename)

import logging
log = logging.getLogger(__name__)

KEYS = ['val', 'valprec', 'valdefault']


def migration_23_to_tiramisu(opt, val):
    if not opt.impl_is_multi():
        if (val == [] or val == ['']) and not isinstance(opt, ChoiceOption):
            val = None
        else:
            if val == []:
                val = None
            else:
                try:
                    val = convert_value(opt, val[0])
                except ValueError:
                    #s'il y une erreur sur la conversion de la variable
                    #met la valeur incorrect pour que la valeur soit
                    #marquée en erreur dans tiramisu (donc affiché dans
                    #l'interface)
                    val = val[0]
    else:
        if val == ['']:
            val = []
        else:
            new_val = []
            for v in val:
                if v == '':
                    new_val.append(None)
                else:
                    try:
                        new_val.append(convert_value(opt, v))
                    except ValueError:
                        #s'il y une erreur sur la conversion de la variable
                        #met la valeur incorrect pour que la valeur soit
                        #marquée en erreur dans tiramisu (donc affiché dans
                        #l'interface)
                        new_val.append(v)
            val = new_val
    return val

class Dico(ConfigParser):

    def get_val(self, var, default=''):
        """
        Renvoie la valeur d'une variable
        """
        if self.has_section(var):
            return self.get(var, 'val')
        return default

    def copy(self, old, new, keep=True):
        """
        Renomme ou copie une variable
        vers une autre
        """
        if self.has_section(old):
            if not self.has_section(new):
                self.add_section(new)
            for key in KEYS:
                value = self.get(old, key)
                self.set(new, key, value)
            if keep:
                log.info(_(u"Variable {0} has been copied in {1}").format(old, new))
            else:
                self.remove_section(old)
                log.info(_(u"Variable {0} has been renamed to {1}").format(old, new))

    def move(self, old, new):
        """
        Renomme ou copie une variable
        vers une autre
        """
        self.copy(old, new, keep=False)

    def remove(self, old):
        if self.has_section(old):
            self.remove_section(old)
            log.info(_(u"Variable {0} has been removed").format(old))

    def simple2multi(self, src, new):
        """
        n variables simples => 1 multi
        """
        res = []
        for var in src:
            if self.has_section(var):
                try:
                    value = eval(self.get(var, 'val'))[0]
                    if value != '':
                        res.append(value)
                except:
                    log.error(_(u"Source variable {0} invalid").format(var))
        if res != []:
            self.fill_var(new, res)

    def fill_var(self, var, val, valprec=[], valdefault=[]):
        """
        Crée ou met à jour une variable
        """
        if type(val) != list:
            val = [val]
        if not self.has_section(var):
            self.add_section(var)
        log.info(_(u"Variable updated: {0} = {1}").format(var, val))
        self.set(var, 'val', str(val))
        self.set(var, 'valprec', valprec)
        self.set(var, 'valdefault', valdefault)

    def save(self, fichier):
        """
        Enregistre le résultat
        """
        fic = file(fichier, 'w')
        self.write(fic)
        fic.close()

def upgrade(config, configfile):
    """
    Mise à jour d'un fichier .eol
    de 2.2 vers 2.4
    ou de 2.3 vers 2.4

    :param dico: configparser instance
    :param version: config.eol version ('2.3' ou '2.4')
    """
    log.info(_(u"config.eol upgrade started"))
    dico = Dico()
    dico.read(configfile)
    version = get_version(dico)
    if version == '2.2':
        upgrade22to23(dico)
    upgrade23to24(dico)
    # FIXME do stuff on 2.4 variables
    # chargement des valeurs par default depuis les dicos XML 2.4
    owner = u"upgrade"
    store_dico = export_to_store(dico, config)
    return store_dico, version

def export_to_store(dico, config):
    """
    exporte depuis un dico vers un dico qui a été mis à jour par les
    valeurs par défaut creole 2.4::

        {"libelle_etab": {"owner": "gen_config", "val": "monchapet"},
        {"owner": "gen_config", "val": ["0.0.0.0"]}

    :param dico: configparser dict
    :returns: config parser dico
    """
    default_owner = u'upgrade'
    store = {}
    # modification des settings pour accéder aux options disabled
    config.cfgimpl_get_settings().remove('disabled')
    old_format = False
    for section in dico.sections():
        val = eval(dico.get_val(section))
        try:
            path = config.find_first(byname=section, type_='path', check_properties=False)
            if not path.startswith('creole.') or path.startswith('creole.containers.'):
                continue

            opt = config.unwrap_from_path(path)
            if isinstance(opt, SymLinkOption):
                continue
            val = migration_23_to_tiramisu(opt, val)
        except AttributeError:
            log.error(_(u"Unknown variable: {0}").format(section))
            old_format = True
        if val is None or val == []:
            continue
        store[section] = {"owner": default_owner, "val":  val}
        if old_format:
            store[section]['old_format'] = True
    return store

def upgrade22to23(dico):
    """
    Mise à jour d'un fichier .eol
    de 2.2 vers 2.3

    :param dico: configparser instance
    """
    log.info(_(u"Starting {0} to {1} upgrade").format('2.2', '2.3'))
    # famille General
    dico.move('proxy', 'activer_proxy_client')
    dico.move('proxy_server', 'proxy_client_adresse')
    dico.move('proxy_port', 'proxy_client_port')
    dico.simple2multi(['serveur_maj', 'serveur_maj2'], 'serveur_maj')
    # spécifique Amon
    domaine = dico.get_val('nom_domaine_academique')
    if domaine != '':
        if '.' in domaine:
            ac, dom = eval(domaine)[0].rsplit('.', 1)
        else:
            # gère le cas particulier de sphynx ou le suffixe n'était pas
            # dans le domaine académique (.fr par défaut)
            ac = eval(domaine)[0]
            dom = 'fr'
        dico.fill_var('nom_academie', ac)
        dico.fill_var('suffixe_domaine_academique', dom)
    # rien sur Zéphir 2.2
    if dico.has_section('ip_ssh_eth0'):
        # ip/netmask facultatifs sur Scribe-2.2
        if 'oui' in dico.get_val('ssh_eth0') and dico.get_val('ip_ssh_eth0') == '[]':
            dico.fill_var('ip_ssh_eth0', '0.0.0.0')
            dico.fill_var('netmask_ssh_eth0', '0.0.0.0')
        # pas de ssh_eth0 sur Horus-2.2
        if not dico.has_section('ssh_eth0'):
            # FIXME ip_ssh_eth0 semble faculatif
            dico.fill_var('ssh_eth0', 'oui')
        # familles Interface-X
        for num in range(0, 5):
            dico.copy('ssh_eth%s' % num, 'admin_eth%s' % num)
            dico.copy('ip_ssh_eth%s' % num, 'ip_admin_eth%s' % num)
            dico.copy('netmask_ssh_eth%s' % num, 'netmask_admin_eth%s' % num)
    dico.move('agregation', 'activer_agregation')

    # famille Services
    dico.move('cups', 'activer_cups')
    dico.move('ftp_perso', 'activer_proftpd')
    dico.move('ead_web', 'activer_ead_web')
    dico.move('apache', 'activer_apache')
    dico.move('mysql', 'activer_mysql')
    dico.move('xinet_interbase', 'activer_interbase')
    if 'oui' in dico.get_val('sso'):
        dico.fill_var('activer_sso', 'local')
    else:
        dico.fill_var('activer_sso', 'non')

    # migration DHCP
    dhcp = dico.get_val('dhcp', None)
    if dhcp is not None:
        dico.move('dhcp', 'activer_dhcp')
        if dico.get_val('adresse_network_dhcp', None) is None:
            #migration d'un Horus 2.2
            len_dhcp = len(eval(dico.get_val('ip_basse_dhcp', "[]")))
            #recuperation des variables a migrer
            adresse_network_dhcp = eval(dico.get_val("adresse_network_eth0"))
            dico.fill_var("adresse_network_dhcp", adresse_network_dhcp * len_dhcp)
            adresse_netmask_dhcp = eval(dico.get_val("adresse_netmask_eth0"))
            dico.fill_var("adresse_netmask_dhcp", adresse_netmask_dhcp * len_dhcp)
            adresse_ip_gw_dhcp = eval(dico.get_val("adresse_ip_gw", "[]"))
            if adresse_ip_gw_dhcp != []:
                dico.fill_var("adresse_ip_gw_dhcp", adresse_ip_gw_dhcp * len_dhcp)
            nom_domaine_dhcp = eval(dico.get_val("nom_domaine_local", "[]"))
            if nom_domaine_dhcp != []:
                dico.fill_var("nom_domaine_dhcp", nom_domaine_dhcp * len_dhcp)
            adresse_ip_dns_dhcp = eval(dico.get_val("adresse_ip_dns", "[]"))
            if adresse_ip_dns_dhcp != []:
                dico.fill_var("adresse_ip_dns_dhcp", [adresse_ip_dns_dhcp[0]] * len_dhcp)

    # famille Messagerie
    dico.move('passerelle_smtp_aca', 'passerelle_smtp')
    dico.move('spamassassin', 'activer_spamassassin')
    if 'oui' in dico.get_val('courier_imap'):
        if 'oui' in dico.get_val('courier_pop'):
            dico.fill_var('activer_courier', 'pop - imap')
        else:
            dico.fill_var('activer_courier', 'imap')
    elif 'oui' in dico.get_val('courier_pop'):
        dico.fill_var('activer_courier', 'pop')
    else:
        dico.fill_var('activer_courier', 'non')
    # Zéphir
    dico.move('serveur_smtp', 'passerelle_smtp')
    dico.move('compte_smtp', 'system_mail_from')
    if '465' in dico.get_val('port_smtp'):
        dico.fill_var('tls_smtp', 'port 465')

    # famille Client_ldap
    dico.move('base_ldap', 'ldap_base_dn')
    serveur_ldap = dico.get_val('serveur_ldap', '[]')
    if serveur_ldap != '[]':
        dico.move('serveur_ldap', 'adresse_ip_ldap')
        if eval(serveur_ldap)[0] not in ['127.0.0.1', 'localhost']:
            dico.fill_var('activer_client_ldap', 'distant')

    # famille Eole-sso
    dico.move('adresse_ip_sso', 'eolesso_adresse')
    dico.move('port_sso', 'eolesso_port')
    # eolesso_ldap (multi)
    dico.move('ldap_sso', 'eolesso_ldap')
    dico.move('port_ldap_sso', 'eolesso_port_ldap')
    dico.move('base_ldap_sso', 'eolesso_base_ldap')
    dico.move('sso_ldap_label', 'eolesso_ldap_label')
    dico.move('sso_ldap_reader', 'eolesso_ldap_reader')
    dico.move('sso_ldap_reader_passfile', 'eolesso_ldap_reader_passfile')
    # la "suite"
    dico.move('adresse_sso_parent', 'eolesso_adresse_parent')
    dico.move('port_sso_parent', 'eolesso_port_parent')
    dico.move('sso_pam_securid', 'eolesso_pam_securid')
    dico.move('sso_cert', 'eolesso_cert')
    dico.move('sso_ca_location', 'eolesso_ca_location')
    dico.move('sso_session_timeout', 'eolesso_session_timeout')
    dico.move('sso_css', 'eolesso_css')

    # famille Applications web
    dico.move('phpmyadmin', 'activer_phpmyadmin')
    dico.move('posh', 'activer_envole')
    dico.move('web_domain', 'web_url')
    dico.move('web_default', 'web_redirection')
    posh_path = dico.get_val('posh_path', '[]')
    if posh_path != '[]' and eval(posh_path)[0] != '':
        dico.fill_var('alias_envole', '/' + eval(posh_path)[0])

    # famille Bacula
    if 'oui' in "%s%s%s" % (dico.get_val('active_bacula_dir'),
                            dico.get_val('active_bacula_fd'),
                            dico.get_val('active_bacula_sd')):
        dico.fill_var('activer_bacula', 'oui')
        dico.move('active_bacula_dir', 'activer_bacula_dir')
        dico.move('active_bacula_sd', 'activer_bacula_sd')
        # bacula_fd n'est plus géré
    else:
        dico.fill_var('activer_bacula', 'non')

    # famille Clamav
    dico.move('enable_clamd', 'dansguardian_clam')

    # famille Certifs-ssl
    dico.move('ssl_serveur_name', 'ssl_server_name')

    # famille Authentification
    dico.move('active_nufw', 'activer_nufw')
    dico.move('freeradius', 'activer_freeradius')

    # famille Logs
    if 'Oui' in dico.get_val('activate_tls'):
        dico.fill_var('rsyslog_tls', 'oui')

    # famille Reverse proxy
    revprox = dico.get_val('revprox_domainname', '[]')
    if revprox != '[]' and eval(revprox)[0] != '':
        dico.fill_var('revprox_activate_http', 'oui')

    # famille réseau avancé
    route = dico.get_val('route_adresse', '[]')
    if route != '[]' and eval(route)[0] != '':
        dico.fill_var('activer_route', 'oui')

    # famille Vpn-pki
    dico.simple2multi(['url_crl1', 'url_crl2'], 'url_crl')


def upgrade23to24(dico):
    """
    Mise à jour d'un fichier .eol
    de 2.3 vers 2.4

    :param dico: configparser instance
    """
    log.info(_(u"Starting {0} to {1} upgrade").format('2.3', '2.4'))
    cache_dir = dico.get_val('cache_dir', '[]')
    if cache_dir != '[]' and eval(cache_dir)[0] == '/var/spool/squid':
        dico.fill_var('cache_dir', '')

    system_mail_to = dico.get_val('system_mail_to', '[]')
    if system_mail_to != '[]' and eval(system_mail_to)[0] == 'postmaster':
        dico.fill_var('system_mail_to', '')

    varname = 'alias_gw_eth0'
    var = dico.get_val(varname, '[]')
    if var != '[]' and eval(var)[0] == 'aucun':
        dico.fill_var(varname, '')

    for i in range(0, 5):
        dico.move('adresse_ip_vlan_eth{0}'.format(i), 'vlan_ip_eth{0}'.format(i))
        dico.move('adresse_netmask_vlan_eth{0}'.format(i), 'vlan_netmask_eth{0}'.format(i))
        dico.move('adresse_network_vlan_eth{0}'.format(i), 'vlan_network_eth{0}'.format(i))
        dico.move('adresse_broadcast_vlan_eth{0}'.format(i), 'vlan_broadcast_eth{0}'.format(i))
        dico.move('adresse_gw_vlan_eth{0}'.format(i), 'vlan_gw_eth{0}'.format(i))
        dico.move('id_vlan_eth{0}'.format(i), 'vlan_id_eth{0}'.format(i))

    varname = 'vlan_gw_eth0'
    var = dico.get_val(varname, '[]')
    if var != '[]' and eval(var)[0] == 'aucun':
        dico.fill_var(varname, '')

    dico.move('proxy_eth0_adresse', 'proxy_eth0_ip')
    dico.move('proxy_eth0_network', 'proxy_eth0_network')
    dico.move('nom_interface1', 'nom_zone_eth1')
    dico.move('era_proxy_bypass', 'era_proxy_bypass_eth1')
    dico.move('smb_adresse_ip_wins', 'smb_wins_server')

    dico.remove('http_port')
    dico.remove('http_port_2')
    dico.remove('test_nutmaster')
    dico.remove('test_activer_routage_ipv6')
    dico.remove('test_activer_kerberos')
    dico.remove('test_activer_clam_proxy')
    dico.remove('test_activer_proxy_eth0')
    dico.remove('revprox_poshadmin')
    dico.remove('ip_client_logs_udp')
    dico.remove('adresse_ip_conteneur_dns')

    dico.simple2multi(['test_distant_domaine1', 'test_distant_domaine2'], 'test_distant_domaine')
    dico.remove('test_distant_domaine1')
    dico.remove('test_distant_domaine2')
    dico.simple2multi(['ssl_subjectaltname_ip', 'ssl_subjectaltname_ns'], 'ssl_subjectaltname')
    dico.remove('ssl_subjectaltname_ip')
    dico.remove('ssl_subjectaltname_ns')

    old_serveur_maj = eval(dico.get_val('serveur_maj', '[]'))
    if old_serveur_maj != []:
        serveur_maj = []
        for maj in old_serveur_maj:
            if maj == 'eoleng.ac-dijon.fr':
                maj = 'eole.ac-dijon.fr'
            if maj == 'test-eoleng.ac-dijon.fr':
                maj = 'test-eole.ac-dijon.fr'
            serveur_maj.append(maj)
        dico.fill_var('serveur_maj', serveur_maj)

    ssl_country_name = eval(dico.get_val('ssl_country_name', '[""]'))[0].upper()
    dico.fill_var('ssl_country_name', ssl_country_name)

    tmp_short_name = []
    tmp_long_name = []
    tmp_ip = []
    nom_domaine_local = eval(dico.get_val('nom_domaine_local', "['']"))[0]
    def _append_tmp_name(ip, long_name, short_name="NONE"):
        splitted_labels = long_name.split('.')
        if short_name == "NONE":
            short_name = splitted_labels[0]
        # ajout si non déjà défini dans Réseau avancé
        if long_name not in tmp_long_name:
            if short_name not in tmp_short_name:
                #le nom court n'existe pas dans la liste, donc l'ajoute
                tmp_short_name.append(short_name)
            else:
                if '.'.join(splitted_labels[1:]) == nom_domaine_local:
                    # le nom court est déjà présent
                    # privilégie le nom court pour le nom de domaine local
                    tmp_short_name[tmp_short_name.index(short_name)] = None
                    tmp_short_name.append(short_name)
                else:
                    # ne pas doublonner le nom court
                    tmp_short_name.append(None)
            if len(splitted_labels) > 1:
                tmp_long_name.append(long_name)
            else:
                # si nom court, transforme en nom long
                tmp_long_name.append(long_name + '.' + nom_domaine_local)
            tmp_ip.append(ip)

    if eval(dico.get_val('activer_ajout_hosts', '["non"]'))[0] == 'oui':
        # récupération et passage en minuscules des
        # nom_court_hosts et nom_long_hosts existants #11473
        ips = eval(dico.get_val('adresse_ip_hosts', '[]').lower())
        long_names = eval(dico.get_val('nom_long_hosts', '[]').lower())
        for idx, short_name in enumerate(eval(dico.get_val('nom_court_hosts', '[]').lower())):
            _append_tmp_name(ips[idx], long_names[idx], short_name)

    # Migration des variables hosts #2795
    # noms d'hôtes forcés en minuscules #9790
    nom_host_dns = eval(dico.get_val('nom_host_dns', '[]').lower())
    if not is_empty(nom_host_dns):
        ips = eval(dico.get_val('ip_host_dns'))
        # transforme les nom_host_dns en nom_court_hosts et nom_long_hosts
        # donc force activer_ajout_hosts à oui
        dico.fill_var('activer_ajout_hosts', 'oui')
        for idx, long_name in enumerate(nom_host_dns):
            _append_tmp_name(ips[idx], long_name)

    if not is_empty(tmp_short_name):
        dico.fill_var('adresse_ip_hosts', tmp_ip)
        dico.fill_var('nom_court_hosts', tmp_short_name)
        dico.fill_var('nom_long_hosts', tmp_long_name)
    dico.remove('nom_host_dns')
    dico.remove('ip_host_dns')

    # Ajout du point devant chaque zone #7008
    old_nom_zone_dns_cache = eval(dico.get_val('nom_zone_dns_cache', '[]'))
    if not is_empty(old_nom_zone_dns_cache):
        nom_zone_dns_cache = []
        for old in old_nom_zone_dns_cache:
            nom_zone_dns_cache.append('.' + old)
        dico.fill_var('nom_zone_dns_cache', nom_zone_dns_cache)

    # Modification du chemin de la corbeille Samba #7463
    smb_trash_dir = eval(dico.get_val('smb_trash_dir', '["/"]'))[0]
    if not smb_trash_dir.startswith('/'):
        dico.fill_var('smb_trash_dir', 'perso/{0}'.format(smb_trash_dir))

    # antivirus temps réel => remis à default #19833
    if dico.get_val('smb_vscan', "['non']") == "['oui']":
        dico.remove('smb_vscan')

    # Famille Proxy parent #7823
    if not is_empty(eval(dico.get_val('nom_cache_pere', '[]'))):
        dico.fill_var('activer_cache_pere', 'oui')
    if not is_empty(eval(dico.get_val('nom_cache_pere_zone', '[]'))):
        dico.fill_var('activer_cache_pere_zone', 'oui')
    if not is_empty(eval(dico.get_val('proxy_sibling_ip', '[]'))):
        dico.fill_var('activer_proxy_sibling', 'oui')

    # Autorisation proxy eth0 #8167
    if not is_empty(eval(dico.get_val('proxy_eth0_ip', '[]'))):
        dico.fill_var('activer_supp_proxy_eth0', 'oui')

    # Famille Rvp #8164
    if not is_empty(eval(dico.get_val('adresse_network_zone_rvp', '[]'))):
        dico.fill_var('acces_proxy_zone_rvp', 'oui')

    # half_closed_clients => remise à default #19813
    if dico.get_val('half_closed_clients', "['off']") == "['on']":
        dico.remove('half_closed_clients')

    ##
    ## Modification de la configuration exim
    ##
    # passerelle SMTP
    log.info(_(u"Migrating SMTP parameters"))
    passerelle_smtp = dico.get_val('passerelle_smtp', '[]')
    dico.move('passerelle_smtp', 'exim_relay_smtp')
    if is_empty(passerelle_smtp):
        # No SMTP gateway
        dico.fill_var('activer_exim_relay_smtp', u'non')

    # Type de serveur SMTP
    exim_mail_type = eval(dico.get_val('exim_mail_type', '["satellite"]'))[0]
    log.info("Migration de exim_mail_type: '{0}'".format(exim_mail_type))
    dico.remove('exim_mail_type')
    if exim_mail_type == 'satellite':
        # Nothing to do
        pass
    elif exim_mail_type == 'local':
        # Local is smarthost without relay, should not happen
        dico.fill_var('exim_smarthost', u'oui')
    elif exim_mail_type == 'smarthost':
        dico.fill_var('exim_smarthost', u'oui')
    elif exim_mail_type == 'mailhub':
        dico.fill_var('exim_relay', u'oui')
        dico.fill_var('exim_relay_manual_routes', u'oui')
    elif exim_mail_type == 'internet':
        dico.fill_var('activer_exim_relay_smtp', u'non')
        dico.fill_var('exim_relay', u'oui')
        dico.fill_var('exim_relay_manual_routes', u'oui')
    else:
        log.warn(_(u'Mail configuration not recognised, not processed'))

    # Réécriture
    mail_rewrite_domain = eval(dico.get_val('mail_rewrite_domain', '["non"]'))[0]
    dico.remove('mail_rewrite_domain')
    if mail_rewrite_domain == 'oui':
        dico.fill_var('exim_qualify_domain', 'nom de domaine local')

    # Modèle Era utilisé (#9082)
    mapping = {'2zones-amonecole-nginx' : u'2zones-amonecole',
               '3zones-scribe-nginx'    : u'3zones-dmz',
               '3zones-scribe'          : u'3zones-dmz',
               '4zones-scribe-nginx'    : u'4zones',
               '4zones-scribe-nufw'     : u'4zones',
               '4zones-scribe'          : u'4zones',
               '5zones-scribe-nginx'    : u'5zones',
               '5zones-scribe'          : u'5zones',
              }
    model = eval(dico.get_val('type_amon', '[""]'))[0]
    if model in mapping:
        dico.fill_var('type_amon', mapping[model])

    # Migration des modules ecdl
    if dico.get_val('ecdl_regles_filtrage_supplementaires', 'Pas un eCDL') != 'Pas un eCDL':
        dico.move('ecdl_ldap_machine_suffix', 'ldap_machine_suffix')
        dico.move('ecdl_ldap_group_suffix', 'ldap_group_suffix')
        dico.move('ecdl_smb_share_model', 'smb_share_model')
        dico.move('ecdl_smb_vscan', 'smb_vscan')
        dico.move('ecdl_smb_ports', 'smb_ports')
        dico.move('ecdl_smb_server_string', 'smb_server_string')
        dico.move('ecdl_smb_trash', 'smb_trash')
        dico.move('ecdl_smb_trash_dir', 'smb_trash_dir')
        dico.move('ecdl_smb_trash_purge', 'smb_trash_purge')
        dico.move('ecdl_smb_quotawarn' , 'smb_quotawarn')
        dico.move('ecdl_smb_guest', 'smb_guest')
        dico.move('ecdl_smb_wins_support', 'smb_wins_support')
        dico.move('ecdl_smb_adresse_ip_wins', 'smb_wins_server')
        dico.move('ecdl_smb_dns_proxy', 'smb_dns_proxy')
        dico.move('ecdl_smb_oplocks', 'smb_oplocks')
        dico.move('ecdl_smb_dos_attributes', 'smb_dos_attributes')
        dico.move('ecdl_smb_unixextensions', 'smb_unixextensions')
        dico.move('ecdl_smb_partage_nom', 'smb_partage_nom')
        dico.move('ecdl_smb_partage_path', 'smb_partage_path')
        dico.move('ecdl_smb_partage_visibilite', 'smb_partage_visibilite')
        dico.move('ecdl_smb_partage_ecriture', 'smb_partage_ecriture')
        dico.move('ecdl_regles_filtrage_supplementaires', 'activer_regles_filtrage_port_source')
        dico.move('ecdl_smb_os_level', 'smb_os_level')
        dico.move('ecdl_smb_domain_master', 'smb_domain_master')
        dico.move('ecdl_ca_cert', 'ldap_ca_cert')
        dico.move('meddtl_suffixe_ldap_nss_base_passwd', 'ldap_nss_base_passwd_filter')
        dico.move('meddtl_suffixe_ldap_nss_base_group', 'ldap_nss_base_group_filter')
        dico.move('ecdl_ldap_timeout', 'ldap_timeout')
        dico.move('ecdl_smb_netbios_name', 'smb_netbios_name')
        dico.move('ecdl_smb_workgroup', 'smb_workgroup')
        dico.move('ecdl_smb_usershare_max_shares', 'smb_usershare_max_shares')
        dico.move('ecdl_smb_activer_partages', 'smb_activer_partages')
        dico.remove('ecdl_smb_log_level')
    # fin de migration des modules ecdl

    # migration des modules esbl
    if dico.get_val('activer_lister_repertoires_apache', 'Pas un eSBL') != 'Pas un eSBL':
        dico.fill_var('smb_log_level', 0)
        smb_activer_ordre_resolution_nom = dico.get_val('smb_activer_ordre_resolution_nom', 'non')
        if smb_activer_ordre_resolution_nom == 'oui':
            smb_name_resolve_order = " ".join(eval(dico.get_val('smb_procede_recherche_nom')))
            dico.fill_var('smb_name_resolve_order', smb_name_resolve_order)
        smb_ad_nom_long_controleur = dico.get_val('smb_ad_nom_long_controleur', "['']")
        if smb_ad_nom_long_controleur != "['']":
            dico.fill_var('smb_ad_server', smb_ad_nom_long_controleur)
        smb_ad_realm = dico.get_val('smb_ad_realm', "['']")
        if smb_ad_realm != "['']":
            dico.fill_var('smb_realm', smb_ad_realm)
        dico.move('activer_lister_repertoires_apache', 'apache_lister_repertoires')

        # répartition des variables pour les répertoires ftp
        ftps = {}
        for ftp_rep, ftp_anon in zip(eval(dico.get_val('acces_ftp', '[]')),
                                      eval(dico.get_val('acces_ftp_anonymous', '[]'))):
            ftps[ftp_anon] = ftps.get(ftp_anon, []) + [ftp_rep]
        # si len(ftps['oui']) > 1, pas de reprise automatique
        # sinon ftps['oui'] -> ftp_anonymous_directory
        # ftps['non'] -> ftp_access_directory

        if 'oui' in ftps and len(ftps['oui']) == 1:
            dico.fill_var('ftp_anonymous_directory', ftps['oui'][0])
            dico.fill_var('activer_ftp_anonymous_access', 'oui')
        if 'non' in ftps:
            dico.fill_var('ftp_access_directory', ftps['non'])
            dico.fill_var('activer_ftp_access', 'oui')
        ftp_maxretrievefilesize = dico.get_val('ftp_maxretrievefilesize', '')
        if ftp_maxretrievefilesize != '':
            ftp_maxretrievefilesize = re.search(r'[0-9]+', ftp_maxretrievefilesize).group()
            dico.fill_var('ftp_maxretrievefilesize', ftp_maxretrievefilesize)
        ftp_maxstorefilesize = dico.get_val('ftp_maxstorefilesize', '')
        if ftp_maxstorefilesize != '':
            ftp_maxstorefilesize = re.search(r'[0-9]+', ftp_maxstorefilesize).group()
            dico.fill_var('ftp_maxstorefilesize', ftp_maxstorefilesize)

        dico.move('activer_pare_feu', 'activer_firewall')
    # fin de migration des modules esbl

    # migration des modules essl
    if dico.get_val('ecdl_serveurs_ip', "Pas un eSSL") != "Pas un eSSL":
        # variables ftp_max*
        ftp_maxretrievefilesize = dico.get_val('ftp_maxretrievefilesize', '')
        if ftp_maxretrievefilesize != '':
            ftp_maxretrievefilesize = re.search(r'[0-9]+', ftp_maxretrievefilesize).group()
            dico.fill_var('ftp_maxretrievefilesize', ftp_maxretrievefilesize)
        ftp_maxstorefilesize = dico.get_val('ftp_maxstorefilesize', '')
        if ftp_maxstorefilesize != '':
            ftp_maxstorefilesize = re.search(r'[0-9]+', ftp_maxstorefilesize).group()
            dico.fill_var('ftp_maxstorefilesize', ftp_maxstorefilesize)
        # variables renommées
        dico.move('sites_distants_morea_ip', 'sites_distants_ip')
        dico.move('sites_distants_morea_netmask', 'sites_distants_netmask')
        dico.move('nagios_morea_ip', 'nagios_dist_ip')
        dico.move('nagios_morea_netmask', 'nagios_dist_netmask')
        dico.move('morea_routeur_ip', 'wan_routeur_ip')
        dico.move('morea_interface', 'wan_interface')
        dico.move('surf_lan_ip', 'sites_dist_ip')
        dico.move('surf_lan_netmask', 'sites_dist_netmask')
        dico.move('morea_route_adresse', 'wan_route_adresse')
        dico.move('morea_route_netmask', 'wan_route_netmask')
        # conversions de valeurs
        variante_type_mapping = {'standard': 'production',
                                 'Applis Web': 'Applis_Web',
                                 'eSSL Morea': 'eSSL',
                                 'eSSL Internet': 'eSSL_Internet',
                                 'eSSL SPC': 'eSSL_SPC',
                                 'ppp': 'PPP',
                                 '': 'production'}
        variante_type = eval(dico.get_val('variante_type', "['']"))[0]
        dico.fill_var('variante_type', variante_type_mapping[variante_type])

    # migration des variables dhcp
    exxl_dhcp = dico.has_section('dhcp_lease_max')
    if dico.get_val('activer_dhcp', "['non']") == "['oui']" and exxl_dhcp:
        # récupération des valeurs de la multi
        ip_basse = eval(dico.get_val('ip_basse_dhcp', '[""]'))
        ip_haute = eval(dico.get_val('ip_haute_dhcp', '[""]'))
        restriction = eval(dico.get_val('activer_dhcp_hotes_autorises', "['']"))
        lease_default = eval(dico.get_val('dhcp_lease_default', "['']"))
        lease_max = eval(dico.get_val('dhcp_lease_max', "['']"))
        # récupération des valeurs communes simples
        network = [eval(dico.get_val('adresse_network_dhcp', "['']"))[0]]
        netmask = [eval(dico.get_val('adresse_netmask_dhcp', "['']"))[0]]
        nom_domaine_dhcp = [eval(dico.get_val('nom_domaine_dhcp', "['']"))[0]]
        gateway_dhcp = [eval(dico.get_val('adresse_ip_gw_dhcp', "['']"))[0]]
        # récupération des valeurs communes multiples
        dns_dhcp = eval(dico.get_val('adresse_ip_dns_dhcp', "['']"))
        wins = eval(dico.get_val('adresse_ip_wins_dhcp', "['']"))
        wins_primaire = wins[0]
        if len(wins) > 1:
            wins_secondaire = wins[1]
        else:
            wins_secondaire = wins_primaire
        ntp_dhcp = eval(dico.get_val('adresse_ip_ntp_dhcp', "['']"))
        # création des nouvelles listes, produit cartésien
        ranges, dns_dhcp, ntp_dhcp = zip(*list(product(zip(ip_basse, ip_haute, restriction, lease_default, lease_max), dns_dhcp, ntp_dhcp)))
        dns_dhcp = list(dns_dhcp)
        ntp_dhcp = list(ntp_dhcp)
        ip_basse, ip_haute, restriction, lease_default, lease_max = [list(l) for l in zip(*ranges)]
        nb_ranges = len(ip_basse)
        nom_domaine_dhcp = nom_domaine_dhcp*nb_ranges
        gateway_dhcp = gateway_dhcp*nb_ranges
        wins_primaire = [wins_primaire]*nb_ranges
        wins_secondaire = [wins_secondaire]*nb_ranges
        network = network*nb_ranges
        netmask = netmask*nb_ranges
        # chargement des valeurs dans le dictionnaire
        dico.fill_var('adresse_network_dhcp', network)
        dico.fill_var('adresse_netmask_dhcp',netmask)
        dico.fill_var('ip_basse_dhcp', ip_basse)
        dico.fill_var('ip_haute_dhcp', ip_haute)
        dico.fill_var('nom_domaine_dhcp', nom_domaine_dhcp)
        dico.fill_var('adresse_ip_gw_dhcp', gateway_dhcp)
        dico.fill_var('adresse_ip_dns_dhcp', dns_dhcp)
        dico.fill_var('adresse_ip_wins_primaire_dhcp', wins_primaire)
        dico.fill_var('adresse_ip_wins_secondaire_dhcp', wins_secondaire)
        dico.fill_var('adresse_ip_ntp_dhcp', ntp_dhcp)
        dico.fill_var('interdire_hotes_inconnus', restriction)
        dico.fill_var('dhcp_lease_default', lease_default)
        dico.fill_var('dhcp_lease_max', lease_max)

    #envole
    if dico.get_val('activer_envole', "['non']") == "['oui']" and dico.get_val('force_envole', "['non']") == "['oui']":
        alias_envole = eval(dico.get_val('alias_envole'))[0]
        if alias_envole != '/':
            dico.fill_var('web_redirection', alias_envole)
            dico.remove('alias_envole')

def get_version(dico):
    """
    recupère la version en fonction de la présence ou non
    de la variable 'serveur_maj2'

    :param dico: ConfigParser object
    :return version: '2.2' ou '2.3'
    """
    # ________ version du config.eol ________

    if dico.has_section('serveur_maj2') and not \
       dico.has_section('activer_bash_completion'):
        version = '2.2'
    else:
        version = '2.3'
    return version


def main(config_file):
    """main entry point"""
    pass

if __name__ == '__main__':
    import sys
    if len(sys.argv) != 2:
        print (__doc__)
        sys.exit(1)
    main(sys.argv[1])
