# -*- coding: utf-8 -*-
###########################################################################
#
# Eole NG - 2010
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################
"""
Librairie pour la gestion des mises à jour
"""
from os import system
from dateutil import parser
from pyeole.schedule import ManageSchedule, list_once, add_schedule, \
    del_schedule, apply_schedules, DAY_TO_STRING
from pyeole.process import system_out
from .client import CreoleClient

from .i18n import _

# fichier d'information pour la mise à jour unique
DIFF_FILENAME = '/var/lib/eole/reports/maj-diff.txt'

#########################################
## Mise à jour hebdomadaire (maj_auto) ##
#########################################

client = CreoleClient()

def maj_enabled():
    """
    vérifie si la mise à jour est activée ou non
    """
    return client.get('/schedule/majauto/day') == 'weekly'

def get_maj_day():
    """
    renvoie le jour et l'heure des mises à jour
    par exemple :
    {'hour': 5, 'minute': 4, 'weekday': 'vendredi'}
    """
    shed = client.get('/schedule/schedule')
    shed.pop('monthday')
    shed['weekday'] = DAY_TO_STRING[shed['weekday']]
    return shed

def enable_maj_auto():
    """
    active la mise à jour hebdomadaire
    """
    if not maj_enabled():
        manage_schedule = ManageSchedule()
        manage_schedule.add('majauto', 'weekly')
        manage_schedule.save()
        apply_schedules()

def disable_maj_auto():
    """
    désactive la mise à jour hebdomadaire
    """
    if maj_enabled():
        manage_schedule = ManageSchedule()
        manage_schedule.delete('majauto')
        manage_schedule.save()
        apply_schedules()


#########################################
##  Mise à jour unique (schedule once) ##
#########################################

def maj_once_enabled():
    """
    vérifie si la mise à jour est activée ou non
    """
    return 'majauto' in list_once('post')

def enable_maj_once():
    """
    active la mise à jour 'once'
    """
    if not maj_once_enabled():
        cancel_maj_differee()
        add_schedule('once', 'post', 'majauto')
        write_diff(True, 'ce soir')
    return True

def disable_maj_once():
    """
    désactive la mise à jour 'once'
    """
    if maj_once_enabled():
        del_schedule('once', 'post', 'majauto')


#########################################
##  Mise à jour unique (maj_differee)  ##
#########################################

def write_diff(enable, heure=None):
    """ ecrit les informations du gestionnaire de mise a jour
        dans le fichier de config de l'ead """
    fic = file(DIFF_FILENAME, 'w')
    if enable:
        fic.write(_(u'An update is scheduled at {0}').format(heure))
    else:
        fic.write("")
    fic.close()

def cancel_maj_differee():
    """
    déprogramme les mises à jour differées
    """
    disable_maj_once()
    cmd = """for i in `grep -l "Maj-Auto" /var/spool/cron/atjobs/* 2>/dev/null`; do rm -f $i ;  done;"""
    system(cmd)
    write_diff(False)
    return True

def prog_maj_differee(heure, options='-R'):
    """
    Programmation une mise à jour différée de quelques heures
    Elle est lancée via la commande at pour l'utilisateur root
    options : options à passer à Maj-Auto
    """
    if heure == 'once':
        return enable_maj_once()
    # suppression des éventuelles autres maj différées
    cancel_maj_differee()
    stdin = "rm -f %s\nMaj-Auto %s\n" % (DIFF_FILENAME, options)
    env_path = {'PATH': '/usr/share/eole:/usr/share/eole/sbin:/usr/local/sbin:'
                        '/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin',
                'LC_ALL': 'fr_FR.UTF-8'}
    ret = system_out(['/usr/bin/at', 'now', '+', str(heure), 'hours'], stdin=stdin, env=env_path)
    if ret[0] != 0:
        return False
    scheduled_maj = " ".join(ret[2].splitlines()[1].split()[3:7])
    scheduled_maj = parser.parse(scheduled_maj)
    scheduled_day = "{0:0=2d}".format(scheduled_maj.day)
    scheduled_month = "{0:0=2d}".format(scheduled_maj.month)
    scheduled_year = "{0:0=2d}".format(scheduled_maj.year)
    scheduled_hour = "{0:0=2d}".format(scheduled_maj.hour)
    scheduled_minute = "{0:0=2d}".format(scheduled_maj.minute)
    scheduled_maj = _(u'{0} the {1}').format(":".join((scheduled_hour, scheduled_minute)), \
                    "/".join((scheduled_day, scheduled_month, scheduled_year)))
    write_diff(True, scheduled_maj)
    return True
