# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2008
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Outil de gestion des configurations dansguardian
#   édite les fichiers locaux de configuration utilisateur
#
###########################################################################

"""
    Gestion des configurations de DansGuardian/e2guardian
"""
import re
from os.path import join, isfile
from pyeole.service import manage_services
from amon.guardian import config
from amon.guardian.config_dans import desactive, active_c, active_m
#from amon.backend import get_filter_zones, is_auth

columns_list = range(0, len(config.optional_policies))
columns_libelle = ['Défaut'] + [str(x) for x in range(1, len(config.optional_policies))]
categ_re = re.compile(r"(?P<commented>#?)(?P<include>.Include</var/lib/blacklists/eole/(?P<categ>[^/]+)/.*)")

#association mot-clé<->nom de fichier
kw_to_files_dict = {
        'blacklist':   join(config.DANS_LOCAL_SUBDIR, 'domains'),
        'whitelist':   join(config.DANS_LOCAL_SUBDIR, 'whites'),
        'extension':  join(config.DANS_LOCAL_SUBDIR, 'extensions'),
        'mime':        join(config.DANS_LOCAL_SUBDIR, 'types_mime'),
        "filtres_opt": [join(config.DANS_LOCAL_SUBDIR, "bannedsitelist"),
                        join(config.DANS_LOCAL_SUBDIR, "bannedurllist")],
        "greysitelist": join(config.DANS_LOCAL_SUBDIR, "site_liste_blanche")
            }

def restart():
    """ relance e2guardian """
    manage_services('restart', 'eole-guardian', container='proxy')
    return True

def reload():
    """ reload e2guardian """
    return restart()

def get_groupslist(zone, _type='user'):
    """
        renvoie ipgroupslist ou filtergroupslist
        selon le type demandé
    """
    if _type == 'user': # is_auth(zone):
        return config.filtergroupslist % zone
    else:
        return config.ipgroupslist % zone

# Modification de la configuration d'une zone
def set_policy(zone, policy_index, item, _type='user'):
    """Ajoute l'item (utilisateur ou groupe de machines) de type _type à la politique de
       filtrage optionnelle désignée par policy_index, pour la zone zone.
       @param : zone = 0/1
       @policy_index : 0/1/2/3/4/5/6/7/8
       @item : nom ou plage d'ip
       @_type : user/ip
    """
    filename = get_groupslist(zone, _type)
    with open(filename, 'r') as filename_stream:
        conf_datas =filename_stream.read()
    if item in conf_datas:
        conf_datas = conf_datas.splitlines()
        for index, line in enumerate(conf_datas):
            if item in line:
                conf_datas[index] = "%s=filter%d" % (item, policy_index)
                break
        to_w = "\n".join(conf_datas)
        to_w += "\n"
        with open(filename, 'w') as filename_stream:
            filename_stream.write(to_w)
    else:
        with open(filename, 'a') as filename_stream:
            filename_stream.write("%s=filter%d\n" % (item, policy_index))

def unset_policy(zone, item, _type='user'):
    """
        Supprime l'item (utilisateur ou groupe de machines) de type _type
        de la liste des items associés à des politiques.
    """
    #filename = config.filtergroupslist%zone
    filename = get_groupslist(zone, _type)
    with open(filename, 'r') as filename_stream:
        conf_datas = filename_stream.read()
    if item in conf_datas:
        conf_datas = conf_datas.splitlines()
        for index, line in enumerate(conf_datas):
            if item in line:
                conf_datas.remove(line)
                break
        to_w = "\n".join(conf_datas)
        to_w += '\n'
        with open(filename, 'w') as filename_stream:
            filename_stream.write(to_w)

def get_policies(zone, _type='user'):
    """ renvoie le contenu de config.filtergroupslist ou ipgroupslist formaté """
    #filename = config.filtergroupslist%zone
    filename = get_groupslist(zone, _type=_type)
    with open(filename, 'r') as filename_stream:
        conf_datas = filename_stream.read().splitlines()
    policies = {}
    for policy in config.policies:
        policies[policy] = []
    for line in conf_datas:
        line = line.split('=')
        for policy in config.policies:
            _filter = 'filter%d' % policy
            if len(line)>1:
                if _filter == line[1].strip():
                    policies[policy].append(line[0].strip())
                    break
    return policies

def get_policy(zone, item, _type='user'):
    """ renvoie la police appliquée à l'item item pour la zone zone """
    for policy, items in get_policies(zone, _type=_type).items():
        if item in items:
            return policy
    return None

def get_specials(zone, keyword=None):
    """ renvoie la liste des items (user et ip) spéciaux (filtre 2,3,4)
        keyword: 2 pour modérateur, 3 pour user interdits, 4 pour les users listes blanches
    """
    if keyword is None:
        return dict([(policy, items)
                     for policy, items in get_policies(zone).items()
                     if policy in config.spec_policies])
    else:
        return dict([(policy, items)
                     for policy, items in get_policies(zone).items()
                     if policy == keyword])


        ########### modele de filtrage
def set_filtrage(zone, mode=None):
    """ met en place le modèle de filtrage (aucun, meta ou toute la page)
        pour la zone zone
        mode = 'meta':activation du filtrage par méta
               'on'  :activation du filtrage de la page complète
               sinon désactivation du filtrage de contenu
    """
    if mode is None:
        default_mode = '1' #filtrage sur meta par défaut
        statusfile = config.dans_status % zone
        if isfile(statusfile):
            with open(statusfile, 'r') as statusfile_stream:
                mode = statusfile_stream.read().splitlines()[0]
            if mode not in ['0', '1', '2']:
                print("Suppression du fichier %s (format invalide)" % statusfile)
                remove(statusfile)
                mode = default_mode
        else:
            mode = default_mode
    # petit hack pour permettre l'appel avec les valeurs numeriques du fichier sauvegardant le status
    filter_keys = {'1':'meta', '2':'on', '0':'default'}
    if mode in filter_keys:
        mode = filter_keys[mode]

    if mode == 'meta':
        active_m(zone)
    elif mode == 'on':
        active_c(zone)
    else:
        desactive(zone)
# Modification des configurations personnalisée par politique

def kw_to_files(keyword, zone, indice):
    """ renvoie le nom des fichiers """
    filename = kw_to_files_dict[keyword]
    if type(filename) == list:
        return [f % (zone, indice) for f in filename]
    return filename % (zone, indice)

def handle_policy_files(function, keyword, zone, collection):
    for index, indice in enumerate(config.optional_policies):
        filename = kw_to_files(keyword, zone, indice)
        function(collection, filename, index)
    return collection

def load_site_list(keyword='blacklist', zone=''):
    """ charge la liste des sites interdits/autorisés, les listes d'extensions
        et de type mime interdits de la zone zone pour les politiques de
        filtrage.
    """
    def add_index_to_sites_from_file(sites, filename, index=None):
        with open(filename, 'r') as filename_stream:
            for line in filename_stream.readlines():
                line = line.strip()
                if line in sites:
                    sites[line][index] = 1
                else:
                    new = [0]*len(config.optional_policies)
                    new[index] = 1
                    sites[line] = new
    return handle_policy_files(add_index_to_sites_from_file, keyword,
            zone, {})

def save_site_list(sites, keyword="blacklist", zone=""):
    """ Écrit la liste des sites interdits/autorisés de la zone zone
        pour les politiques de filtrage.
    """
    def write_sites_to_file(files, filename, i):
        print("ecriture de %s dans %s" % (files[i], filename))
        with open(filename, 'w') as filename_stream:
            for site in files[i]:
                filename_stream.write(site + "\n")

    # On range d'abord les listes par fichier dans files :
    files = []
    for i in config.optional_policies:
        files.append([])
    for site, booleans in sites.items():
        for i, b in enumerate(booleans):
            if b:
                files[i].append(site)

    # On peut maintenant écrire la liste dans chaque fichier.
    handle_policy_files(write_sites_to_file, keyword, zone, files)

def load_category_list(keyword="filtres_opt", zone=""):
    """Charge une liste de catégories de la zone zone."""
    def read_status_of_categories(categs, filename, index):
        if type(filename) == list:
            filename = filename[0]
        with open(filename, 'r') as filename_stream:
            for line in filename_stream.readlines():
                m = categ_re.search(line)
                if not m or len(m.groups()) < 2:
                    continue
                enabled = m.group("commented") == ""
                categ = m.group("categ")
                if not categ in categs:
                    # New categ, all enabled by default:
                    categs[categ] = [1]*len(config.optional_policies)
                if not enabled:  # Only disable (since categories are enabled
                                 # by default):
                    categs[categ][index] = 0
        return categs
    return handle_policy_files(read_status_of_categories, keyword, zone, {})

def save_category_list(categs, keyword="filtres_opt", zone=""):
    """Sauve une liste de catégories de la zone zone."""
    def write_status_of_categories(categs, filenames, index):
        if type(filenames) != list:
            filenames = list(filenames)
        for filename in filenames:
            with open(filename, 'r') as filename_stream:
                lines_to_write = []  # Lignes à écrire plus tard dans le fichier
                                     # qu'on écrasera.
                for line in filename_stream.readlines():
                    m = categ_re.search(line)
                    if not m or len(m.groups()) < 2:
                        continue
                    if categs[m.group("categ")][index]:
                        lines_to_write.append(m.group("include"))
                    else:
                        lines_to_write.append("#" + m.group("include"))
            with open(filename, 'w') as filename_stream:
                for line in lines_to_write:
                    filename_stream.write(line + "\n")
        return categs
    return handle_policy_files(write_status_of_categories, keyword, zone,
                               categs)

def option_and_index_to_name(option, index):
    """ forme un name à partir d'un nom d'option et d'un index (entier) """
    return "%s_%s" % (option, index)

def name_to_option_and_index(name):
    """ Récupère option et index (l'inverse d'option_and_index_to_name) """
    where = name[::-1].find("_")
    if where == -1:
        raise ValueError("Aucun indice de colonne trouvé dans le nom \"%s\""
                % name)
    try:
        return name[:-where-1], int(name[len(name)-where:])
    except ValueError:  # La partie après le séparateur n'est pas un entier.
        raise ValueError("Aucun indice de colonne trouvé dans le nom \"%s\""
                % name)

def load_greysitelist(zone):
    """ renvoie la liste blanche de sites pour la zone zone """
    return open(kw_to_files('greysitelist', zone, 4)).read().splitlines()

def save_greysitelist(zone, sitelist):
    """ écrit une liste de site interdits """
    with open(kw_to_files('greysitelist', zone, 4), 'w') as sitelist_f:
        sitelist_f.write('\n'.join(sitelist))
