# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Gere les horaires pour les groupes de machine
#
###########################################################################
"""
Gère les horaires pour les groupes de machine
"""
from os.path import isfile
import pickle

from amon.ipset.config import ipset

# dico avec pour clé nom du groupe et pour valeur une liste de plage horaire
# {'groupe':{'lundi':[{hdeb:'10:00', hfin:'12:00'}...], 'mardi':...}
time_file = ipset['schedule']

def get_schedules(zone, group=None, jour=None):
    """ renvoie les horaires
        @group: groupe dont on veut les horaires
        @jour: jour demandé (nécessite un nom de groupe)
    """
    schedules = load_schedules(zone)
    if group:
        if group in schedules:
            if jour is not None:
                if  jour in schedules[group]:
                    return schedules[group][jour]
                else:
                    return []
            else:
                return schedules[group]
        else:
            return {}
    return schedules

def add_schedule(zone, group, jour, hdeb, hfin):
    """ crée une plage horaire
        @group: nom du groupe
        @jour: jour de la plage
        @hdeb: heure de début de plage
        @hfin: heure de fin de plage
    """
    if test_group_schedule(zone, group, jour, hdeb, hfin):
        schedules = get_schedules(zone)
        if group in schedules:
            if jour in schedules[group]:
                schedules[group][jour].append({'hdeb':hdeb, 'hfin':hfin})
            else:
                schedules[group][jour] = [{'hdeb':hdeb, 'hfin':hfin}]
        else:
            schedules[group] = {jour:[{'hdeb':hdeb, 'hfin':hfin}]}
        save_schedules(zone, schedules)
        return True
    return False

def del_schedule(zone, group, jour=None, hdeb=None, hfin=None):
    """ supprime une plage horaire ou l'ensemble des plages horaires du groupe si jour, hdeb et hfin = None
        @group: nom du groupe
        @jour: jour de la plage à supprimer
        @hdeb: heure de début de la plage
        @hfin: heure de fin de la plage
    """
    schedules = get_schedules(zone)
    if None in [jour, hdeb, hfin]:
        if group in schedules:
            schedules.pop(group)
    else:
        schedule = {'hdeb':hdeb, 'hfin':hfin}
        if group in schedules:
            if jour in schedules[group]:
                if schedule in schedules[group][jour]:
                    schedules[group][jour].remove(schedule)
    # enregistrement des modifications
    save_schedules(zone, schedules)
    return True

def copy_schedules(zone, group_from, group_to):
    """ copie les horaires d'un groupe à un autre
        @group_from : nom du groupe dont on copie les horaires
        @group_to : nom du groupe auquel on associe les horaires
    """
    def copylist(a):
        for i in a:
            yield copypy(i)

    def copydict(a):
        dico = {}
        for key, value in a.items():
            dico[key] = copypy(value)
        return dico

    def copypy(a):
        if type(a) == dict:
            return copydict(a)
        elif type(a) == list:
            return list(copylist(a))
        else:
            return a

    schedules = get_schedules(zone)
    if group_to in schedules:
        schedules.pop(group_to)
    if group_from in schedules:
        schedules[group_to] = copypy(schedules[group_from])
        save_schedules(zone, schedules)
        return True
    raise Exception("Erreur : le groupe %s n'a pas d'horaire configuré." % group_from)

def load_schedules(zone):
    """ charge les horaires
    """
    if isfile(time_file%zone):
        f = open(time_file%zone, 'rb')
        schedules = pickle.load(f)
        f.close()
        return schedules
    return {}

def save_schedules(zone, schedules):
    """ sauvegarde les horaires """
    f = open(time_file%zone, 'wb')
    pickle.dump(schedules, f)
    f.close()
    return True

def get_time_list():
    """ récupère la liste des horaires """
    return [hour['name'] for hour in get_hours()]

def get_days_list():
    """ récupère la liste des jours """
    return [day['libelle'] for day in get_days()]

def test_time_range(hdeb, hfin):
    """ compare heure de début et heure de fin"""
    if int(hdeb.split(':')[0]) > int(hfin.split(':')[0]):
        raise Exception("Erreur:L'heure de début doit précéder l'heure de fin de plage.")
    elif int(hdeb.split(':')[0]) == int(hfin.split(':')[0]):
        if int(hdeb.split(':')[1]) >= int(hfin.split(':')[1]):
            raise Exception("Erreur:L'heure de début doit précéder l'heure de fin de plage.")
    return True

def test_group_schedule(zone, group, jour, hdeb, hfin):
    """ teste la validité de la plage horaire pour le groupe groupe """
    if test_time_range(hdeb, hfin):
##        tester les intersections de plage avec des sets
        # on teste les intersections pour la nouvelle entrée horaire
        hour_list = get_time_list()
        schedule_as_set = set(hour_list[hour_list.index(hdeb):hour_list.index(hfin)])
        for plage in get_schedules(zone, group, jour):
            plage = set(hour_list[hour_list.index(plage['hdeb']): hour_list.index(plage['hfin'])])
            if plage.intersection(schedule_as_set) != set([]):
                # FIXME: on pourrait joindre les plages horaires lorsqu'elles s'intersectent
                raise Exception("Erreur: les plages horaires doivent être distinctes (intersection vide)")
        return True
    return False

def get_hours(default=''):
    """ renvoie les heures (et demi-heure) de la journée formatées pour la présentation dans une balise select """
    hours = []
    for i in range(24):
        val1  = '%d:00'%i
        val2 = '%d:30'%i
        hour = {'libelle':val1, 'name':val1}
        halfandhour = {'libelle':val2, 'name':val2}
        if default == val1:
            hour['default'] = 'ok'
        elif default == val2:
            halfandhour['default'] = 'ok'
        hours.append(hour)
        hours.append(halfandhour)
    hours.append({'libelle':'24:00', 'name':'23:59'})
    return hours

def get_days(default=''):
    """ renvoie les jours de la semaine formatés pour la présentation dans une balise select """
    days = ['lundi', 'mardi', 'mercredi', 'jeudi', 'vendredi', 'samedi', 'dimanche']
    ret_days = []
    for day in days:
        if day == default: ret_days.append({'name':day, 'libelle':day, 'default':'ok'})
        else: ret_days.append({'name':day, 'libelle':day})
    return ret_days
