/*
 *  $Id: surface.c 28965 2025-12-10 10:34:12Z yeti-dn $
 *  Copyright (C) 2011-2025 David Nečas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <stdlib.h>
#include "libgwyddion/macros.h"
#include "libgwyddion/math.h"
#include "libgwyddion/rand-gen-set.h"
#include "libgwyddion/serializable-utils.h"
#include "libgwyddion/surface.h"
#include "libgwyddion/stats.h"

#include "libgwyddion/internal.h"

#define TYPE_NAME "GwySurface"

enum {
    SGNL_DATA_CHANGED,
    NUM_SIGNALS
};

enum {
    ITEM_UNIT_XY, ITEM_UNIT_Z,
    ITEM_DATA,
    NUM_ITEMS
};

static void             finalize              (GObject *object);
static void             serializable_init     (GwySerializableInterface *iface);
static void             serializable_itemize  (GwySerializable *serializable,
                                               GwySerializableGroup *group);
static gboolean         serializable_construct(GwySerializable *serializable,
                                               GwySerializableGroup *group,
                                               GwyErrorList **error_list);
static GwySerializable* serializable_copy     (GwySerializable *serializable);
static void             serializable_assign   (GwySerializable *destination,
                                               GwySerializable *source);
static void             copy_info             (GwySurface *dest,
                                               GwySurface *src);
static void             ensure_ranges         (GwySurface *surface);
static void             ensure_checksum       (GwySurface *surface);

static guint signals[NUM_SIGNALS];
static GObjectClass *parent_class = NULL;

static GwySerializableItem serializable_items[NUM_ITEMS] = {
    { .name = "unit_xy", .ctype = GWY_SERIALIZABLE_OBJECT,       },
    { .name = "unit_z",  .ctype = GWY_SERIALIZABLE_OBJECT,       },
    { .name = "data",    .ctype = GWY_SERIALIZABLE_DOUBLE_ARRAY,
                         .flags = GWY_SERIALIZABLE_BIG_DATA      },
};

G_DEFINE_TYPE_WITH_CODE(GwySurface, gwy_surface, G_TYPE_OBJECT,
                        G_ADD_PRIVATE(GwySurface)
                        G_IMPLEMENT_INTERFACE(GWY_TYPE_SERIALIZABLE, serializable_init));

static void
serializable_init(GwySerializableInterface *iface)
{
    iface->itemize   = serializable_itemize;
    iface->construct = serializable_construct;
    iface->copy      = serializable_copy;
    iface->assign    = serializable_assign;

    serializable_items[ITEM_UNIT_XY].aux.object_type = GWY_TYPE_UNIT;
    serializable_items[ITEM_UNIT_Z].aux.object_type = GWY_TYPE_UNIT;
}

static void
gwy_surface_class_init(GwySurfaceClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    GType type = G_TYPE_FROM_CLASS(klass);

    parent_class = gwy_surface_parent_class;

    gobject_class->finalize = finalize;

    /**
     * GwySurface::data-changed:
     * @gwysurface: The #GwySurface which received the signal.
     *
     * The ::data-changed signal is never emitted by surface itself.  It is intended as a means to notify others data
     * field users they should update themselves.
     **/
    signals[SGNL_DATA_CHANGED] = g_signal_new("data-changed", type,
                                              G_SIGNAL_RUN_FIRST,
                                              G_STRUCT_OFFSET(GwySurfaceClass, data_changed),
                                              NULL, NULL,
                                              g_cclosure_marshal_VOID__VOID,
                                              G_TYPE_NONE, 0);
    g_signal_set_va_marshaller(signals[SGNL_DATA_CHANGED], type, g_cclosure_marshal_VOID__VOIDv);
}

static void
gwy_surface_init(GwySurface *surface)
{
    surface->priv = gwy_surface_get_instance_private(surface);
}

static void
alloc_data(GwySurface *surface, gsize n)
{
    if (surface->n == n)
        return;

    GwySurfacePrivate *priv = surface->priv;
    GWY_FREE(priv->data);
    if (n)
        priv->data = g_new(GwyXYZ, n);
    surface->n = n;
}

static void
finalize(GObject *object)
{
    GwySurface *surface = GWY_SURFACE(object);
    GwySurfacePrivate *priv = surface->priv;

    if (priv->checksummer) {
        g_checksum_free(priv->checksummer);
        priv->checksummer = NULL;
    }
    g_clear_object(&priv->unit_xy);
    g_clear_object(&priv->unit_z);
    g_free(priv->data);

    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
copy_info(GwySurface *dest, GwySurface *src)
{
    gwy_surface_copy_units(src, dest);
}

/**
 * gwy_surface_new: (constructor)
 *
 * Creates a new empty surface.
 *
 * The surface will not contain any points. This parameterless constructor exists mainly for language bindings,
 * gwy_surface_new_from_data() is usually more useful.
 *
 * Returns: (transfer full):
 *          A newly created surface.
 **/
GwySurface*
gwy_surface_new(void)
{
    return g_object_new(GWY_TYPE_SURFACE, NULL);
}

/**
 * gwy_surface_new_sized:
 * @n: Number of points.
 *
 * Creates a new surface with preallocated size.
 *
 * The surface will contain the speficied number of points with uninitialised
 * values.
 *
 * Returns: (transfer full):
 *          A newly created surface.
 **/
GwySurface*
gwy_surface_new_sized(guint n)
{
    GwySurface *surface = g_object_new(GWY_TYPE_SURFACE, NULL);
    alloc_data(surface, n);
    return surface;
}

/**
 * gwy_surface_new_from_data:
 * @points: (array length=n): Array of @n points with the surface data.
 * @n: Number of points.
 *
 * Creates a new surface, filling it with provided points.
 *
 * Returns: (transfer full):
 *          A newly created surface.
 **/
GwySurface*
gwy_surface_new_from_data(const GwyXYZ *points,
                          guint n)
{
    GwySurface *surface;

    g_return_val_if_fail(!n || points, NULL);

    surface = g_object_new(GWY_TYPE_SURFACE, NULL);
    alloc_data(surface, n);
    gwy_assign(surface->priv->data, points, n);
    return surface;
}

/**
 * gwy_surface_new_alike:
 * @model: A surface to use as the template.
 *
 * Creates a new empty surface similar to another surface.
 *
 * The units of the new surface will be identical to those of @model but the new surface will not contain any points.
 * Use gwy_surface_copy() to completely duplicate a surface including data.
 *
 * Returns: (transfer full):
 *          A newly created surface.
 **/
GwySurface*
gwy_surface_new_alike(GwySurface *model)
{
    GwySurface *surface;

    g_return_val_if_fail(GWY_IS_SURFACE(model), NULL);

    surface = g_object_new(GWY_TYPE_SURFACE, NULL);
    copy_info(surface, model);
    return surface;
}

/**
 * gwy_surface_new_part:
 * @surface: A surface.
 * @xfrom: Minimum x-coordinate value.
 * @xto: Maximum x-coordinate value.
 * @yfrom: Minimum y-coordinate value.
 * @yto: Maximum y-coordinate value.
 *
 * Creates a new surface as a part of another surface.
 *
 * The new surface consits of data with lateral coordinates within the specified ranges (inclusively).  It may be
 * empty.
 *
 * Data are physically copied, i.e. changing the new surface data does not change @surface's data and vice versa.
 *
 * Returns: (transfer full):
 *          A newly created surface.
 **/
GwySurface*
gwy_surface_new_part(GwySurface *surface,
                     gdouble xfrom,
                     gdouble xto,
                     gdouble yfrom,
                     gdouble yto)
{
    GwySurface *part;

    g_return_val_if_fail(GWY_IS_SURFACE(surface), NULL);

    part = gwy_surface_new_alike(surface);
    if (!surface->n || xfrom > xto || yfrom > yto)
        return part;

    const GwyXYZ *data = surface->priv->data;
    gsize n = 0;
    for (gsize i = 0; i < surface->n; i++) {
        gdouble x = data[i].x, y = data[i].y;
        if (x >= xfrom && x <= xto && y >= yfrom && y <= yto)
            n++;
    }
    alloc_data(part, n);

    GwyXYZ *pdata = part->priv->data;
    n = 0;
    for (gsize i = 0; i < surface->n; i++) {
        gdouble x = data[i].x, y = data[i].y;
        if (x >= xfrom && x <= xto && y >= yfrom && y <= yto)
            pdata[n++] = pdata[i];
    }

    return part;
}

static void
copy_field_to_surface(GwyField *field,
                      GwySurface *surface,
                      GwyNield *mask,
                      GwyMaskingType masking,
                      gint col, gint row,
                      gint width, gint height)
{
    gdouble dx = gwy_field_get_dx(field), dy = gwy_field_get_dy(field);
    gdouble xoff = 0.5*dx + field->xoff, yoff = 0.5*dy + field->yoff;
    GwySurfacePrivate *priv = surface->priv;

    gint xres = field->xres, yres = field->yres;
    gsize n = (gsize)xres * (gsize)yres;
    const gdouble *datapos = field->priv->data + xres*row + col;
    const gint *maskpos = NULL;
    if (mask) {
        maskpos = mask->priv->data + xres*row + col;
        n = gwy_nield_area_count(mask, NULL, GWY_MASK_IGNORE, col, row, width, height);
    }

    alloc_data(surface, n);

    GwyXYZ *data = surface->priv->data;
    gsize k = 0;
    for (gint i = 0; i < height; i++) {
        for (gint j = 0; j < width; j++) {
            if (nielded_included(maskpos + i*xres + j, masking)) {
                data[k].x = dx*j + xoff;
                data[k].y = dy*i + yoff;
                data[k].z = datapos[i*xres + j];
                k++;
            }
        }
    }

    gwy_field_copy_units_to_surface(field, surface);

    gwy_surface_invalidate(surface);
    if (n == xres*yres) {
        priv->cached_ranges = TRUE;
        priv->min.x = xoff;
        priv->min.y = yoff;
        priv->max.x = dx*(xres - 1) + xoff;
        priv->max.y = dy*(yres - 1) + yoff;
        gwy_field_get_min_max(field, &priv->min.z, &priv->max.z);
    }
}

/**
 * gwy_surface_get_data:
 * @surface: A surface.
 *
 * Gets the raw XYZ data array of a surface.
 *
 * The returned buffer is not guaranteed to be valid through whole data surface life time.
 *
 * This function invalidates any cached information, use gwy_surface_get_data_const() if you are not going to change
 * the data.
 *
 * See gwy_surface_invalidate() for some discussion.
 *
 * Returns: The surface XYZ data as a pointer to an array of gwy_surface_get_npoints() items.
 **/
GwyXYZ*
gwy_surface_get_data(GwySurface *surface)
{
    g_return_val_if_fail(GWY_IS_SURFACE(surface), NULL);
    gwy_surface_invalidate(surface);
    return surface->priv->data;
}

/**
 * gwy_surface_get_data_const:
 * @surface: A surface.
 *
 * Gets the raw XYZ data array of a surface, read-only.
 *
 * The returned buffer is not guaranteed to be valid through whole data field life time.
 *
 * Use gwy_surface_get_data() if you want to change the data.
 *
 * See gwy_surface_invalidate() for some discussion.
 *
 * Returns: The surface XYZ data as a pointer to an array of gwy_surface_get_npoints() items.
 **/
const GwyXYZ*
gwy_surface_get_data_const(GwySurface *surface)
{
    g_return_val_if_fail(GWY_IS_SURFACE(surface), NULL);
    return surface->priv->data;
}

/**
 * gwy_surface_get_npoints:
 * @surface: A surface.
 *
 * Gets the number of points in an XYZ surface.
 *
 * Returns: The number of points.
 **/
guint
gwy_surface_get_npoints(GwySurface *surface)
{
    g_return_val_if_fail(GWY_IS_SURFACE(surface), 0);
    return surface->n;
}

/**
 * gwy_surface_data_changed:
 * @surface: A surface.
 *
 * Emits signal GwySurface::data-changed on a surface.
 **/
void
gwy_surface_data_changed(GwySurface *surface)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_signal_emit(surface, signals[SGNL_DATA_CHANGED], 0);
}

/**
 * gwy_surface_copy_data:
 * @surface: Source surface.
 * @target: Destination surface.
 *
 * Copies the data of a surface to another surface of the same dimensions.
 *
 * Only the data points are copied.  To make a surface completely identical to another, including units and change of
 * dimensions, you can use gwy_surface_assign().
 **/
void
gwy_surface_copy_data(GwySurface *surface, GwySurface *target)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_return_if_fail(GWY_IS_SURFACE(target));
    g_return_if_fail(target->n == surface->n);
    if (surface == target)
        return;
    gwy_assign(target->priv->data, surface->priv->data, surface->n);
    gwy_surface_invalidate(target);

    GwySurfacePrivate *dpriv = target->priv, *spriv = surface->priv;

    dpriv->cached_ranges = spriv->cached_ranges;
    dpriv->min = spriv->min;
    dpriv->max = spriv->max;
    dpriv->cached_checksum = spriv->cached_checksum;
    gwy_assign(dpriv->checksum, spriv->checksum, G_N_ELEMENTS(spriv->checksum));
}

/**
 * gwy_surface_invalidate:
 * @surface: A surface.
 *
 * Invalidates cached surface statistics.
 *
 * Cached statistics include ranges returned by gwy_surface_get_xrange(), gwy_surface_get_yrange() and
 * gwy_surface_get_min_max(), the fingerprint for gwy_surface_xy_is_compatible() and and possibly other
 * characteristics in the future.
 *
 * See gwy_field_invalidate() for discussion of invalidation and examples.
 **/
void
gwy_surface_invalidate(GwySurface *surface)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    surface->priv->cached_ranges = FALSE;
    surface->priv->cached_checksum = FALSE;
}

/**
 * gwy_surface_set_from_field:
 * @surface: A surface.
 * @field: A two-dimensional data field.
 *
 * Fills the data of a surface from a data field.
 *
 * The number of points in the new surface will be equal to the number of points in the field.  Lateral coordinates
 * will be equal to the corresponding @field coordinates; values will be created in regular grid according to
 * @field's physical size and offset.
 *
 * Lateral and value units will correspond to @field's units.  This means the field needs to have identical @x
 * and @y units.
 **/
void
gwy_surface_set_from_field(GwySurface *surface,
                           GwyField *field)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_return_if_fail(GWY_IS_FIELD(field));
    copy_field_to_surface(field, surface, NULL, GWY_MASK_IGNORE, 0, 0, field->xres, field->yres);
}

/**
 * gwy_surface_set_from_field_mask:
 * @surface: A surface.
 * @field: A two-dimensional data field.
 * @mask: (nullable): Mask of pixels to include from/exclude, or %NULL
 * @masking: Masking mode to use.
 *
 * Fills the data of a surface from a data field, possibly using masking.
 **/
void
gwy_surface_set_from_field_mask(GwySurface *surface,
                                GwyField *field,
                                GwyField *mask,
                                GwyMaskingType masking)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    GwyNield *nield = oldstyle_mask_to_nield(mask, masking);
    copy_field_to_surface(field, surface, nield, masking, 0, 0, field->xres, field->yres);
    g_clear_object(&nield);
}

/**
 * gwy_surface_set_from_NIELD:
 * @surface: A surface.
 * @field: A two-dimensional data field.
 * @mask: (nullable): Mask of pixels to include from/exclude, or %NULL
 * @masking: Masking mode to use.
 * @col: Column index.
 * @row: Row index.
 * @width: Area width (number of columns).
 * @height: Area height (number of rows)
 *
 * Fills the data of a surface from a data field, possibly using masking.
 **/
void
gwy_surface_set_from_NIELD(GwySurface *surface,
                           GwyField *field,
                           GwyNield *mask,
                           GwyMaskingType masking,
                           gint col,
                           gint row,
                           gint width,
                           gint height)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    if (!_gwy_field_check_area(field, col, row, width, height, FALSE)
        || !_gwy_NIELD_check_mask(field, &mask, &masking))
        return;
    copy_field_to_surface(field, surface, mask, masking, col, row, width, height);
}

/**
 * gwy_surface_get_unit_xy:
 * @surface: A surface.
 *
 * Returns lateral SI unit of a surface.
 *
 * The returned object can be modified to change the surface lateral units.
 *
 * Returns: (transfer none):
 *          SI unit corresponding to the lateral (XY) dimensions of the surface. Its reference count is not
 *          incremented.
 **/
GwyUnit*
gwy_surface_get_unit_xy(GwySurface *surface)
{
    g_return_val_if_fail(GWY_IS_SURFACE(surface), NULL);

    if (!surface->priv->unit_xy)
        surface->priv->unit_xy = gwy_unit_new(NULL);

    return surface->priv->unit_xy;
}

/**
 * gwy_surface_get_unit_z:
 * @surface: A surface.
 *
 * Returns value SI unit of a surface.
 *
 * The returned object can be modified to change the surface value units.
 *
 * Returns: (transfer none):
 *          SI unit corresponding to the "height" (Z) dimension of the surface. Its reference count is not
 *          incremented.
 **/
GwyUnit*
gwy_surface_get_unit_z(GwySurface *surface)
{
    g_return_val_if_fail(GWY_IS_SURFACE(surface), NULL);

    if (!surface->priv->unit_z)
        surface->priv->unit_z = gwy_unit_new(NULL);

    return surface->priv->unit_z;
}

/**
 * gwy_surface_get_value_format_xy:
 * @surface: A surface.
 * @style: Unit format style.
 * @format: (nullable): A SI value format to modify, or %NULL to allocate a new one.
 *
 * Finds value format good for displaying coordinates of a surface.
 *
 * Returns: The value format.  If @format is %NULL, a newly allocated format is returned, otherwise (modified) @format
 *          itself is returned.
 **/
GwyValueFormat*
gwy_surface_get_value_format_xy(GwySurface *surface,
                                GwyUnitFormatStyle style,
                                GwyValueFormat *format)
{
    gdouble xmin, xmax, ymin, ymax;
    gdouble max, unit;
    GwyUnit *siunit;

    g_return_val_if_fail(GWY_IS_SURFACE(surface), NULL);

    siunit = gwy_surface_get_unit_xy(surface);
    if (!surface->n)
        return gwy_unit_get_format_with_resolution(siunit, style, 1.0, 0.1, format);

    if (surface->n == 1) {
        max = MAX(fabs(surface->priv->data[0].x), fabs(surface->priv->data[0].y));
        if (!max)
            max = 1.0;
        return gwy_unit_get_format_with_resolution(siunit, style, max, max/10.0, format);
    }

    gwy_surface_get_xrange(surface, &xmin, &xmax);
    gwy_surface_get_yrange(surface, &ymin, &ymax);
    max = MAX(MAX(fabs(xmax), fabs(xmin)), MAX(fabs(ymax), fabs(ymin)));
    if (!max)
        max = 1.0;
    unit = hypot(ymax - ymin, xmax - xmin)/sqrt(surface->n);
    if (!unit)
        unit = max/10.0;

    return gwy_unit_get_format_with_resolution(siunit, style, max, 0.2*unit, format);
}

/**
 * gwy_surface_get_value_format_z:
 * @surface: A surface.
 * @style: Unit format style.
 * @format: (nullable): A SI value format to modify, or %NULL to allocate a new one.
 *
 * Finds value format good for displaying values of a surface.
 *
 * Returns: The value format.  If @format is %NULL, a newly allocated format is returned, otherwise (modified) @format
 *          itself is returned.
 **/
GwyValueFormat*
gwy_surface_get_value_format_z(GwySurface *surface,
                               GwyUnitFormatStyle style,
                               GwyValueFormat *format)
{
    GwyUnit *siunit;
    gdouble max, min;

    g_return_val_if_fail(GWY_IS_SURFACE(surface), NULL);

    siunit = gwy_surface_get_unit_z(surface);
    gwy_surface_get_min_max(surface, &min, &max);
    if (max == min) {
        max = ABS(max);
        min = 0.0;
    }

    return gwy_unit_get_format_with_digits(siunit, style, max - min, 3, format);
}

/**
 * gwy_surface_copy_units:
 * @surface: A surface.
 * @target: Target surface.
 *
 * Sets lateral and value units of a surface to match another surface.
 **/
void
gwy_surface_copy_units(GwySurface *surface,
                       GwySurface *target)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_return_if_fail(GWY_IS_SURFACE(target));

    _gwy_copy_unit(surface->priv->unit_xy, &target->priv->unit_xy);
    _gwy_copy_unit(surface->priv->unit_z, &target->priv->unit_z);
}

/**
 * gwy_field_copy_units_to_surface:
 * @field: A two-dimensional data field.
 * @surface: A surface.
 *
 * Sets lateral and value units of a surface to match a data field.
 **/
void
gwy_field_copy_units_to_surface(GwyField *field,
                                GwySurface *surface)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_return_if_fail(GWY_IS_FIELD(field));

    GwySurfacePrivate *priv = surface->priv;
    GwyFieldPrivate *fpriv = field->priv;
    _gwy_copy_unit(fpriv->unit_xy, &priv->unit_xy);
    _gwy_copy_unit(fpriv->unit_z, &priv->unit_z);
}

/**
 * gwy_surface_copy_units_to_field:
 * @surface: A surface.
 * @field: A two-dimensional data field.
 *
 * Sets lateral and value units of a data field to match a surface.
 **/
void
gwy_surface_copy_units_to_field(GwySurface *surface,
                                GwyField *field)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_return_if_fail(GWY_IS_FIELD(field));

    GwySurfacePrivate *priv = surface->priv;
    GwyFieldPrivate *fpriv = field->priv;
    _gwy_copy_unit(priv->unit_xy, &fpriv->unit_xy);
    _gwy_copy_unit(priv->unit_z, &fpriv->unit_z);
}

/**
 * gwy_surface_get:
 * @surface: A surface.
 * @pos: Position in @surface.
 *
 * Obtains a single XYZ surface point.
 *
 * This function exists <emphasis>only for language bindings</emphasis> as it is very slow compared to simply
 * accessing @data in #GwySurface directly in C.
 *
 * Returns: The point at @pos.
 **/
GwyXYZ
gwy_surface_get(GwySurface *surface,
                guint pos)
{
    GwyXYZ nullpoint = { 0.0, 0.0, 0.0 };

    g_return_val_if_fail(GWY_IS_SURFACE(surface), nullpoint);
    g_return_val_if_fail(pos < surface->n, nullpoint);
    return surface->priv->data[pos];
}

/**
 * gwy_surface_set:
 * @surface: A surface.
 * @pos: Position in @surface.
 * @point: Point to store at given position.
 *
 * Sets a single XYZ surface value.
 *
 * This function exists <emphasis>only for language bindings</emphasis> as it is very slow compared to simply
 * accessing @data in #GwySurface directly in C.
 **/
void
gwy_surface_set(GwySurface *surface,
                guint pos,
                GwyXYZ point)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_return_if_fail(pos < surface->n);
    surface->priv->data[pos] = point;
}

/**
 * gwy_surface_get_data_full:
 * @surface: A surface.
 * @n: (out): Location to store the count of extracted data points.
 *
 * Provides the values of an entire surface as a flat array.
 *
 * This function, paired with gwy_surface_set_data_full() can be namely useful in language bindings.
 *
 * Note that this function returns a pointer directly to @surface's data.
 *
 * Returns: (array length=n): The array containing the surface points.
 **/
const GwyXYZ*
gwy_surface_get_data_full(GwySurface *surface,
                          gsize *n)
{
    g_return_val_if_fail(GWY_IS_SURFACE(surface), NULL);
    *n = surface->n;
    return surface->priv->data;
}

/**
 * gwy_surface_set_data_full:
 * @surface: A surface.
 * @points: (array length=n): Data points to copy to the surface.  They replace whatever points are in @surface now.
 * @n: The number of points in @data.
 *
 * Puts back values from a flat array to an entire data surface.
 *
 * If you do not have an array of #GwyXYZ and want to fill the data sequentially, use gwy_surface_resize() and
 * gwy_surface_get_data().
 *
 * See gwy_surface_get_data_full() for a discussion.
 **/
void
gwy_surface_set_data_full(GwySurface *surface,
                          const GwyXYZ *points,
                          gsize n)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    g_return_if_fail(points || !n);
    alloc_data(surface, n);
    if (n)
        gwy_assign(surface->priv->data, points, n);
}

/**
 * gwy_surface_get_xrange:
 * @surface: A surface.
 * @min: (out): Location where to store the minimum value.
 * @max: (out): Location where to store the maximum value.
 *
 * Gets the range of X coordinates of an XYZ surface.
 *
 * This information is cached.
 **/
void
gwy_surface_get_xrange(GwySurface *surface,
                       gdouble *min,
                       gdouble *max)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    ensure_ranges(surface);
    if (min)
        *min = surface->priv->min.x;
    if (max)
        *max = surface->priv->max.x;
}

/**
 * gwy_surface_get_yrange:
 * @surface: A surface.
 * @min: (out): Location where to store the minimum value.
 * @max: (out): Location where to store the maximum value.
 *
 * Gets the range of Y coordinates of an XYZ surface.
 *
 * This information is cached.
 **/
void
gwy_surface_get_yrange(GwySurface *surface,
                       gdouble *min,
                       gdouble *max)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    ensure_ranges(surface);
    if (min)
        *min = surface->priv->min.y;
    if (max)
        *max = surface->priv->max.y;
}

/**
 * gwy_surface_get_min_max:
 * @surface: A surface.
 * @min: (out): Location where to store the minimum value.
 * @max: (out): Location where to store the maximum value.
 *
 * Gets the range of Z values of an XYZ surface.
 *
 * This information is cached.
 **/
void
gwy_surface_get_min_max(GwySurface *surface,
                        gdouble *min,
                        gdouble *max)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    ensure_ranges(surface);
    if (min)
        *min = surface->priv->min.z;
    if (max)
        *max = surface->priv->max.z;
}

/**
 * gwy_surface_multiply:
 * @surface: A surface.
 * @value: The value to multiply all data with.
 *
 * Multiplies all values in an XYZ surface by given constant.
 **/
void
gwy_surface_multiply(GwySurface *surface,
                     gdouble value)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));

    gsize n = surface->n;
    GwyXYZ *data = surface->priv->data;
    for (gsize i = 0; i < n; i++)
        data[i].z *= value;

    gwy_surface_invalidate(surface);
}

/**
 * gwy_surface_add:
 * @surface: A surface.
 * @value: The value to add to all data.
 *
 * Adds a constant to all data in an XYZ surface.
 **/
void
gwy_surface_add(GwySurface *surface,
                gdouble value)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));

    gsize n = surface->n;
    GwyXYZ *data = surface->priv->data;
    for (gsize i = 0; i < n; i++)
        data[i].z += value;

    gwy_surface_invalidate(surface);
}

/**
 * gwy_surface_scale_xy:
 * @surface: A surface.
 * @xfactor: Factor to multiply all X coordinates with.
 * @yfactor: Factor to multiply all Y coordinates with.
 * @xoffset: Offset to add to all X coordinates.
 * @yoffset: Offset to add to all Y coordinates.
 *
 * Rescales and/or shifts all X and Y coordinates of an XYZ surface.
 *
 * The multiplicative factors are applied before the offsets.
 **/
void
gwy_surface_scale_xy(GwySurface *surface,
                     gdouble xfactor,
                     gdouble yfactor,
                     gdouble xoffset,
                     gdouble yoffset)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));

    gsize n = surface->n;
    GwyXYZ *data = surface->priv->data;
    for (gsize i = 0; i < n; i++) {
        data[i].x = xfactor*data[i].x + xoffset;
        data[i].y = yfactor*data[i].y + yoffset;
    }

    gwy_surface_invalidate(surface);
}

static void
ensure_ranges(GwySurface *surface)
{
    GwySurfacePrivate *priv = surface->priv;
    GwyXYZ min, max;
    guint i;

    if (priv->cached_ranges)
        return;

    priv->cached_ranges = TRUE;
    if (!surface->n) {
        gwy_clear(&priv->min, 1);
        gwy_clear(&priv->max, 1);
        return;
    }

    const GwyXYZ *data = surface->priv->data;
    min = max = data[0];
    for (i = 1; i < surface->n; i++) {
        GwyXYZ pt = data[i];

        max.x = fmax(max.x, pt.x);
        max.y = fmax(max.y, pt.y);
        max.z = fmax(max.z, pt.z);
        min.x = fmin(min.x, pt.x);
        min.y = fmin(min.y, pt.y);
        min.z = fmin(min.z, pt.z);
    }

    priv->min = min;
    priv->max = max;
    return;
}

/**
 * gwy_surface_xy_is_compatible:
 * @surface: A surface.
 * @othersurface: Another surface.
 *
 * Checks whether the XY positions of two surfaces are compatible.
 *
 * Compatible XY positions mean the XY units are the same and the points are the same.  Two surfaces that have the
 * same set of XY points but in different orders are <emphasis>not</emphasis> considered compatible.  This is because
 * the points at the same index in @data are different and thus calculations involving data from the two surfaces are
 * impossible.  It is necessary to match the points order in the two surfaces to make this possible.
 *
 * This information is cached.
 *
 * Returns: %TRUE if the surfaces are XY-position compatible, %FALSE if they are not.
 **/
gboolean
gwy_surface_xy_is_compatible(GwySurface *surface,
                             GwySurface *othersurface)
{
    GwyUnit *sunit, *ounit;
    gboolean sempty, oempty;

    g_return_val_if_fail(GWY_IS_SURFACE(surface), FALSE);
    g_return_val_if_fail(GWY_IS_SURFACE(othersurface), FALSE);

    sunit = surface->priv->unit_xy;
    ounit = othersurface->priv->unit_xy;
    sempty = (!sunit || gwy_unit_equal_string(sunit, NULL));
    oempty = (!ounit || gwy_unit_equal_string(ounit, NULL));
    if ((sempty && !oempty) || (!sempty && oempty) || (!sempty && !oempty && !gwy_unit_equal(sunit, ounit)))
        return FALSE;

    ensure_checksum(surface);
    ensure_checksum(othersurface);
    return !memcmp(surface->priv->checksum, othersurface->priv->checksum, sizeof(surface->priv->checksum));
}

static void
ensure_checksum(GwySurface *surface)
{
    GwySurfacePrivate *priv = surface->priv;
    gdouble *xydata;
    guint i, n, k;

    if (priv->cached_checksum)
        return;

    n = surface->n;
    xydata = g_new(gdouble, 2*n);
    for (i = k = 0; i < n; i++) {
        xydata[k++] = priv->data[i].x;
        xydata[k++] = priv->data[i].y;
    }

    GChecksum *checksummer = priv->checksummer;
    if (!checksummer)
        priv->checksummer = checksummer = g_checksum_new(G_CHECKSUM_MD5);
    else
        g_checksum_reset(checksummer);

    g_checksum_update(checksummer, (gchar*)xydata, 2*n*sizeof(gdouble));
    gsize len = G_N_ELEMENTS(priv->checksum);
    g_checksum_get_digest(checksummer, priv->checksum, &len);
    g_assert(len == G_N_ELEMENTS(priv->checksum));
    g_free(xydata);

    priv->cached_checksum = TRUE;
}

/**
 * gwy_surface_resize:
 * @surface: A surface.
 * @npoints: New number of points in the surface.
 *
 * Changes the number of points in a surface.
 *
 * If the number of points decreases then the first @npoints points will be kept.  If the number of points increases
 * the new points will be uninitialised.  Although usually you would overwrite all the points after using this
 * function.
 *
 * Use gwy_surface_set_data_full() if you already have an array of #GwyXYZ to fill the surface data with.
 **/
void
gwy_surface_resize(GwySurface *surface, gsize npoints)
{
    g_return_if_fail(GWY_IS_SURFACE(surface));
    if (npoints == surface->n)
        return;

    surface->priv->data = g_renew(GwyXYZ, surface->priv->data, npoints);
    surface->n = npoints;
    gwy_surface_invalidate(surface);
}

/**
 * gwy_surface_reduce_points:
 * @surface: A surface.
 * @npoints: Requested number of points in the reduced surface.  If it is not smaller than the number of points in
 *           @surface then the function behaves like gwy_surface_copy().
 *
 * Creates a similar surface with smaller number of points.
 *
 * The functions attempts to choose points from the original surface to cover its full area, even though points from
 * dense regions are still more likely to be found in the result than points from sparse regions.  As the main purpose
 * to enable quick rough operations that may take long time with the full surface, the focus is on speed not fidelity.
 *
 * The function may employ random selection and thus be be non-deterministic.
 *
 * Returns: (transfer full): A newly created #GwySurface with reduced number of points.
 **/
GwySurface*
gwy_surface_reduce_points(GwySurface *surface, gsize npoints)
{
    GwyXYZ *srcdata, *destdata;
    GwySurface *retval;
    GwyRandGenSet *rngset;
    GRand *rng;
    gdouble xmin, xmax, ymin, ymax, xlen, ylen, d;
    gboolean x_is_degenerate, y_is_degenerate;
    gint k, n, xres = 1, yres = 1, i, j;
    guint *redindex, *pts = NULL;
    gint *grid;
    guint ngrid = 0;

    g_return_val_if_fail(GWY_IS_SURFACE(surface), NULL);
    n = surface->n;
    if (npoints >= surface->n)
        return gwy_surface_copy(surface);
    if (!npoints)
        return gwy_surface_new_alike(surface);

    gwy_debug("reducing from %u to %u", surface->n, npoints);
    rngset = gwy_rand_gen_set_new(1);
    rng = gwy_rand_gen_set_rng(rngset, 0);
    srcdata = surface->priv->data;

    retval = gwy_surface_new_alike(surface);
    alloc_data(retval, npoints);
    destdata = retval->priv->data;

    /* If a reasonable number of points is requested try to spread them. */
    if (npoints > 3) {
        gwy_surface_get_xrange(surface, &xmin, &xmax);
        gwy_surface_get_yrange(surface, &ymin, &ymax);
        xlen = xmax - xmin;
        ylen = ymax - ymin;
        x_is_degenerate = (xmin >= xmax);
        y_is_degenerate = (ymin >= ymax);
        if (x_is_degenerate && y_is_degenerate) {
            /* Choose points randomly */
            gwy_debug("both x and y are degenerate");
            d = 0.5*fmax(fabs(xmax), fabs(ymax));
            if (d == 0.0)
                d = 1.0;
        }
        if (x_is_degenerate && !y_is_degenerate) {
            gwy_debug("x is degenerate");
            /* Vertical line */
            xres = 1;
            yres = 3*npoints;
            d = 0.5*ylen/yres;
        }
        else if (y_is_degenerate && !x_is_degenerate) {
            gwy_debug("y is degenerate");
            /* Horizontal line */
            xres = 3*npoints;
            yres = 1;
            d = 0.5*xlen/xres;
        }
        else {
            gwy_debug("neither x nor y is degenerate");
            d = sqrt(4.0*xlen*ylen/npoints);
            if (xlen >= ylen) {
                yres = (gint)floor(ylen/d + 0.5);
                xres = (4*npoints + yres-1)/yres;
                gwy_debug("choosing using x");
            }
            else {
                xres = (gint)floor(xlen/d + 0.5);
                yres = (4*npoints + xres-1)/xres;
                gwy_debug("choosing using y");
            }
            d *= 0.5;
        }
        if (!xres || !yres)
            xres = yres = 1;
        xmin -= d;
        ymin -= d;
        xmax += d;
        ymax += d;
        xlen = xmax - xmin;
        ylen = ymax - ymin;
        gwy_debug("xres %d, yres %d", xres, yres);
    }

    if (xres == 1 && yres == 1) {
        redindex = gwy_rand_gen_set_choose_shuffle(rngset, 0, n, npoints);
        for (k = 0; k < (gint)npoints; k++)
            destdata[k] = srcdata[redindex[k]];
        g_free(redindex);
        gwy_rand_gen_set_free(rngset);
        gwy_surface_invalidate(retval);
        return retval;
    }

    /* Count the number of points in each rectangle. */
    grid = g_new0(gint, xres*yres);
    ngrid = 0;
    for (k = 0; k < n; k++) {
        j = (gint)floor((srcdata[k].x - xmin)/xlen*xres);
        i = (gint)floor((srcdata[k].y - ymin)/ylen*yres);
        j = GWY_CLAMP(j, 0, xres-1);
        i = GWY_CLAMP(i, 0, yres-1);
        if (!grid[i*xres + j])
            ngrid++;
        grid[i*xres + j]++;
    }
    gwy_debug("points in rectangles %u", ngrid);

    /* Select one point in each rectangle by generating the index of point we should retain. */
    if (ngrid < npoints) {
        /* Use a point from each rectangle and then fill the rest randomly. */
        for (i = 0; i < xres*yres; i++) {
            if (grid[i] > 1)
                grid[i] = g_rand_int_range(rng, 1, grid[i]+1);
        }
        /* Find the indices of selected points. */
        pts = g_new0(guint, surface->n);
        for (k = ngrid = 0; k < n; k++) {
            j = (gint)floor((srcdata[k].x - xmin)/xlen*xres);
            i = (gint)floor((srcdata[k].y - ymin)/ylen*yres);
            j = GWY_CLAMP(j, 0, xres-1);
            i = GWY_CLAMP(i, 0, yres-1);
            if (grid[i*xres + j] > 1)
                grid[i*xres + j]--;
            else if (grid[i*xres + j] == 1) {
                destdata[ngrid++] = srcdata[k];
                pts[k] = TRUE;
                grid[i*xres + j] = 0;
            }
        }

        /* Start filling the new surface with points from rectangles. */
        gwy_debug("filled from grid %u", ngrid);

        /* Fill the rest with randomly selected points. */
        for (i = j = 0; i < n; i++) {
            if (!pts[i])
                pts[j++] = i;
        }
        redindex = gwy_rand_gen_set_choose_shuffle(rngset, 0,
                                                   surface->n - ngrid,
                                                   npoints - ngrid);
        for (k = 0; k < (gint)(npoints - ngrid); k++)
            destdata[ngrid + k] = srcdata[pts[redindex[k]]];

        g_free(pts);
        g_free(redindex);
        gwy_debug("filled randomly %u", (npoints - ngrid));
    }
    else {
        if (ngrid > npoints) {
            /* We have more points in rectangles than requested so remove some of them randomly. */
            gwy_debug("removing %u rectangles", ngrid - npoints);
            pts = g_new0(guint, ngrid);
            for (i = j = 0; i < xres*yres; i++) {
                if (grid[i])
                    pts[j++] = i;
            }
            redindex = gwy_rand_gen_set_choose_shuffle(rngset, 0, ngrid, ngrid - npoints);
            for (i = 0; i < ngrid - npoints; i++)
                grid[pts[redindex[i]]] = 0;
            g_free(redindex);
            g_free(pts);
        }

        /* Use a point from each remaining rectangle. */
        for (i = 0; i < xres*yres; i++) {
            if (grid[i] > 1)
                grid[i] = g_rand_int_range(rng, 1, grid[i]+1);
        }
        for (k = ngrid = 0; k < n; k++) {
            j = (gint)floor((srcdata[k].x - xmin)/xlen*xres);
            i = (gint)floor((srcdata[k].y - ymin)/ylen*yres);
            j = GWY_CLAMP(j, 0, xres-1);
            i = GWY_CLAMP(i, 0, yres-1);
            if (grid[i*xres + j] > 1)
                grid[i*xres + j]--;
            else if (grid[i*xres + j] == 1) {
                destdata[ngrid++] = srcdata[k];
                grid[i*xres + j] = 0;
            }
        }
    }
    g_free(grid);
    gwy_rand_gen_set_free(rngset);

    return retval;
}

static void
serializable_itemize(GwySerializable *serializable, GwySerializableGroup *group)
{
    GwySurface *surface = GWY_SURFACE(serializable);
    GwySurfacePrivate *priv = surface->priv;

    gwy_serializable_group_append_unit(group, serializable_items + ITEM_UNIT_XY, priv->unit_xy);
    gwy_serializable_group_append_unit(group, serializable_items + ITEM_UNIT_Z, priv->unit_z);
    gwy_serializable_group_append_double_array(group, serializable_items + ITEM_DATA,
                                               (gdouble*)priv->data, 3*surface->n);
    gwy_serializable_group_itemize(group);
}

static gboolean
serializable_construct(GwySerializable *serializable, GwySerializableGroup *group, GwyErrorList **error_list)
{
    GwySerializableItem its[NUM_ITEMS], *it;
    gboolean ok = FALSE;
    gwy_assign(its, serializable_items, NUM_ITEMS);
    gwy_deserialize_filter_items(its, NUM_ITEMS, group, TYPE_NAME, error_list);

    GwySurface *surface = GWY_SURFACE(serializable);
    GwySurfacePrivate *priv = surface->priv;

    /* Botched up data dimensions is a hard fail. */
    it = its + ITEM_DATA;
    if (!gwy_check_data_dimension(error_list, TYPE_NAME, 1, 0, it->array_size)
        || !gwy_check_data_length_multiple(error_list, TYPE_NAME, it->array_size, 3)) {
        g_free(it->value.v_double_array);
        goto fail;
    }
    g_free(priv->data);
    priv->data = (GwyXYZ*)it->value.v_double_array;
    surface->n = it->array_size/3;

    /* The rest is already validated by pspec. */
    priv->unit_xy = (GwyUnit*)its[ITEM_UNIT_XY].value.v_object;
    its[ITEM_UNIT_XY].value.v_object = NULL;
    priv->unit_z = (GwyUnit*)its[ITEM_UNIT_Z].value.v_object;
    its[ITEM_UNIT_Z].value.v_object = NULL;

    ok = TRUE;

fail:
    g_clear_object(&its[ITEM_UNIT_XY].value.v_object);
    g_clear_object(&its[ITEM_UNIT_Z].value.v_object);
    return ok;
}

static GwySerializable*
serializable_copy(GwySerializable *serializable)
{
    GwySurface *surface = GWY_SURFACE(serializable);
    GwySurface *copy = gwy_surface_new_alike(surface);
    alloc_data(copy, surface->n);
    gwy_surface_copy_data(surface, copy);
    return GWY_SERIALIZABLE(copy);
}

static void
serializable_assign(GwySerializable *destination, GwySerializable *source)
{
    GwySurface *destsurface = GWY_SURFACE(destination), *srcsurface = GWY_SURFACE(source);

    alloc_data(destsurface, srcsurface->n);
    gwy_surface_copy_data(srcsurface, destsurface);
    copy_info(srcsurface, destsurface);
}

/**
 * gwy_surface_copy:
 * @surface: A XYZ surface to duplicate.
 *
 * Create a new XYZ surface as a copy of an existing one.
 *
 * This function is a convenience gwy_serializable_copy() wrapper.
 *
 * Returns: (transfer full):
 *          A copy of the XYZ surface.
 **/
GwySurface*
gwy_surface_copy(GwySurface *surface)
{
    /* Try to return a valid object even on utter failure. Returning NULL probably would crash something soon. */
    if (!GWY_IS_SURFACE(surface)) {
        g_assert(GWY_IS_SURFACE(surface));
        return g_object_new(GWY_TYPE_SURFACE, NULL);
    }
    return GWY_SURFACE(gwy_serializable_copy(GWY_SERIALIZABLE(surface)));
}

/**
 * gwy_surface_assign:
 * @destination: Target XYZ surface.
 * @source: Source XYZ surface.
 *
 * Makes one XYZ surface equal to another.
 *
 * This function is a convenience gwy_serializable_assign() wrapper.
 **/
void
gwy_surface_assign(GwySurface *destination, GwySurface *source)
{
    g_return_if_fail(GWY_IS_SURFACE(destination));
    g_return_if_fail(GWY_IS_SURFACE(source));
    if (destination != source)
        gwy_serializable_assign(GWY_SERIALIZABLE(destination), GWY_SERIALIZABLE(source));
}

/**
 * SECTION:surface
 * @title: GwySurface
 * @short_description: General two-dimensional data
 *
 * #GwySurface represents general, i.e. possibly unevenly spaced, two-dimensional data, also called XYZ data.
 *
 * GwySurfacePrivate points are stored in a flat array #GwySurface-struct.data of #GwyXYZ values.
 *
 * Unlike #GwyField, a surface can also be empty, i.e. contain zero points.
 **/

/**
 * GwySurface:
 * @n: Number of points.
 * @data: GwySurfacePrivate data.  See the introductory section for details.
 *
 * Object representing surface data.
 *
 * The #GwySurface struct contains some public fields that can be directly accessed for reading.  To set them, you
 * must use the #GwySurface methods.
 **/

/**
 * GwySurfaceClass:
 *
 * Class of surfaces.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
