// Copyright 2016 Martin Hebnes Pedersen (LA5NTA). All rights reserved.
// Use of this source code is governed by the MIT-license that can be
// found in the LICENSE file.

package lzhuf

import (
	"bufio"
	"bytes"
	"io"
	"io/ioutil"
	"os"
	"path/filepath"
	"testing"
)

var testdataPath = "testdata/"

func TestRoundtrip(t *testing.T) {
	files, err := ioutil.ReadDir(testdataPath)
	if err != nil {
		t.Fatal("Unable to open testdata directory:", err)
	}

	for _, fi := range files {
		if fi.IsDir() || filepath.Ext(fi.Name()) == ".lzh" || fi.Name()[0] == '.' {
			continue
		}

		t.Logf("Running %s...", fi.Name())
		file, err := os.Open(filepath.Join(testdataPath, fi.Name()))
		if err != nil {
			t.Fatal(err)
		}

		var orig bytes.Buffer

		// Compress (and tee to orig for comparison)
		var compressed bytes.Buffer
		w := NewB2Writer(&compressed)
		io.Copy(w, io.TeeReader(file, &orig))
		w.Close()

		r, err := NewB2Reader(&compressed)
		if err != nil {
			t.Errorf("%s: Unexpected NewB2Reader error: %s", fi.Name(), err)
			continue
		}

		rd := bufio.NewReader(r)
		for i := 0; orig.Len() > 0; i++ {
			c, _ := orig.ReadByte()
			d, _ := rd.ReadByte()

			if c != d {
				t.Errorf("%s: Byte idx %d not matching. Skipping rest of compare.", fi.Name(), i)
				break
			}
		}

		if err := r.Close(); err != nil {
			t.Errorf("%s: Unexpected Close error: %s", fi.Name(), err)
		}

		file.Close()
	}
}

func TestReaderUnexpectedEOF(t *testing.T) {
	lz, _ := NewReader(bytes.NewReader(samples[2].compressed[:10]), true)
	if _, err := io.Copy(ioutil.Discard, lz); err != io.ErrUnexpectedEOF {
		t.Errorf("Read: Expected io.ErrUnexpectedEOF, got '%v'", err)
	}
	if err := lz.Close(); err != io.ErrUnexpectedEOF {
		t.Errorf("Close: Expected io.ErrUnexpectedEOF, got '%s'", err)
	}
}

func TestB2ReaderInvalidChecksum(t *testing.T) {
	data := make([]byte, len(samples[0].compressed))
	copy(data, samples[0].compressed)
	data[0] = 0x1 // Invalid checksum

	lz, _ := NewB2Reader(bytes.NewReader(data))
	io.Copy(ioutil.Discard, lz)
	if err := lz.Close(); err != ErrChecksum {
		t.Error("Did not receive ErrChecksum from Close")
	}
}

func TestReaderShortRead(t *testing.T) {
	// With crc16 checksum
	lz, _ := NewB2Reader(bytes.NewReader(samples[4].compressed))
	io.CopyN(ioutil.Discard, lz, int64(len(samples[4].compressed)-1))
	if err := lz.Close(); err != ErrChecksum {
		t.Error("Did not receive ErrChecksum from Close on short read with crc16", err)
	}

	// Without crc16 checksum
	lz, _ = NewReader(bytes.NewReader(samples[4].compressed[2:]), false)
	io.CopyN(ioutil.Discard, lz, int64(len(samples[4].compressed[2:])-1))
	if err := lz.Close(); err != ErrChecksum {
		t.Error("Did not receive ErrChecksum from Close on short read without crc16", err)
	}
}

func TestReaderInvalidHeader(t *testing.T) {
	var err error

	_, err = NewB2Reader(bytes.NewReader([]byte{0x0}))
	if err != io.ErrUnexpectedEOF {
		t.Error("Expected io.ErrUnexpectedEOF on too short crc16 checksum, got:", err)
	}

	_, err = NewReader(bytes.NewReader([]byte{0x0}), false)
	if err != io.ErrUnexpectedEOF {
		t.Error("Expected io.ErrUnexpectedEOF on too filesize header, got:", err)
	}
}

func TestReader(t *testing.T) {
	for i, sample := range samples {
		lz, err := NewReader(bytes.NewReader(sample.compressed), true)
		if err != nil {
			t.Errorf("Unexpected NewReader error: %s", err)
			continue
		}

		var buf bytes.Buffer
		_, err = io.Copy(&buf, lz)
		if err != nil {
			t.Errorf("Unexpected error: %s", err)
		}

		if !bytes.Equal(buf.Bytes(), sample.plain) {
			t.Errorf("Sample %d failed", i)
		}

		if err := lz.Close(); err != nil {
			t.Errorf("Sample %d failed on close: %s", i, err)
		}
	}
}

func TestWriterTestdata(t *testing.T) {
	files, err := ioutil.ReadDir(testdataPath)
	if err != nil {
		t.Fatal("Unable to open testdata directory:", err)
	}

	for _, fi := range files {
		if fi.IsDir() || filepath.Ext(fi.Name()) == ".lzh" || fi.Name()[0] == '.' {
			continue
		}

		t.Logf("Running %s...", fi.Name())
		file, err := os.Open(filepath.Join(testdataPath, fi.Name()))
		if err != nil {
			t.Fatal(err)
		}

		// Compress (and tee to orig for comparison)
		var compressed bytes.Buffer
		w := NewB2Writer(&compressed)
		io.Copy(w, file)
		w.Close()

		f, err := os.Open(filepath.Join("testdata", fi.Name()+".lzh"))
		if err != nil {
			t.Fatal(err)
		}

		rd := bufio.NewReader(f)
		for i := 0; compressed.Len() > 0; i++ {
			c, _ := compressed.ReadByte()
			d, _ := rd.ReadByte()

			if c != d {
				t.Errorf("%s: Byte idx %d not matching. Skipping rest of compare.", fi.Name(), i)
				break
			}
		}

		file.Close()
		f.Close()
	}
}

func TestWriter(t *testing.T) {
	for i, sample := range samples {
		var buf bytes.Buffer
		lz := NewWriter(&buf, true)

		_, err := io.Copy(lz, bytes.NewReader(sample.plain))
		if err != nil {
			t.Errorf("Unexpected error: %s", err)
		}

		if err := lz.Close(); err != nil {
			t.Errorf("Close error on sample %d: %s", i, err)
		}

		if !bytes.Equal(buf.Bytes()[0:2], sample.compressed[0:2]) {
			t.Errorf("Sample %d failed: checksum mismatch", i)
		}

		if !bytes.Equal(buf.Bytes()[2:6], sample.compressed[2:6]) {
			t.Errorf("Sample %d failed: length header mismatch", i)
		}

		if !bytes.Equal(buf.Bytes()[6:], sample.compressed[6:]) {
			t.Errorf("Sample %d failed", i)
		}
	}
}

type sample struct {
	plain      []byte
	compressed []byte
}

var samples = []sample{
	sample{
		[]byte("\n"),
		[]byte{0xe, 0x8f, 0x1, 0x0, 0x0, 0x0, 0xcb, 0x0},
	},
	sample{
		[]byte("foo"),
		[]byte{0xb6, 0x47, 0x3, 0x0, 0x0, 0x0, 0xf9, 0x7e, 0xf1, 0x0},
	},
	sample{
		[]byte("The quick brown fox jumps over the lazy dog\r\nThe quick brown fox jumps over the lazy dog"),
		[]byte{0x76, 0x25, 0x58, 0x0, 0x0, 0x0, 0xf0, 0x7d, 0x3e, 0x3a, 0xcf, 0xe8, 0xf, 0xd7, 0xdf, 0xf7, 0xc2, 0xf7, 0x7f, 0xbf, 0x60, 0x7f, 0xab, 0x7f, 0x2b, 0xa0, 0x4b, 0x7f, 0x6c, 0xf, 0xcf, 0xf3, 0xff, 0x55, 0x60, 0x2c, 0x3b, 0xba, 0x80, 0x23, 0x3, 0xdf, 0x8f, 0x68, 0x30, 0x2d, 0x3f, 0xa, 0xff, 0x3c, 0xce, 0x5b, 0xf2, 0x2c},
	},
	sample{
		[]byte("bar"),
		[]byte{0xc7, 0xef, 0x03, 0x00, 0x00, 0x00, 0xf7, 0x7b, 0x7f, 0xc0},
	},
	sample{ // from wikipedia
		[]byte("Unicode is a computing industry standard for the consistent encoding, representation, and handling of text expressed in most of the world's writing systems. Developed in conjunction with the Universal Character Set standard and published as The Unicode Standard, the latest version of Unicode contains a repertoire of more than 110,000 characters covering 100 scripts and multiple symbol sets. The standard consists of a set of code charts for visual reference, an encoding method and set of standard character encodings, a set of reference data computer files, and a number of related items, such as character properties, rules for normalization, decomposition, collation, rendering, and bidirectional display order (for the correct display of text containing both right-to-left scripts, such as Arabic and Hebrew, and left-to-right scripts).[1] As of June 2014, the most recent version is Unicode 7.0. The standard is maintained by the Unicode Consortium."),
		[]byte{0x3f, 0x3e, 0xbd, 0x3, 0x0, 0x0, 0xf0, 0xfe, 0xbe, 0xbe, 0xff, 0xdf, 0xc3, 0xe3, 0xac, 0xc3, 0xff, 0xdf, 0x7b, 0x6e, 0x61, 0x60, 0xfe, 0x7f, 0x80, 0x20, 0x16, 0x71, 0x3e, 0x77, 0x6d, 0x57, 0xc0, 0xb7, 0xbd, 0xb6, 0xff, 0x2, 0xd1, 0x4e, 0x95, 0xe5, 0x9a, 0xb3, 0xac, 0x2a, 0xd1, 0xf2, 0x9f, 0x26, 0x8a, 0xdf, 0x4b, 0xfa, 0xa0, 0x8e, 0x5a, 0xf2, 0x41, 0x14, 0x39, 0x62, 0xa2, 0x6c, 0xb3, 0x83, 0xa8, 0x42, 0xdd, 0xc3, 0xa7, 0xbf, 0x70, 0xdc, 0x10, 0x4a, 0x9, 0xd4, 0x92, 0xa, 0x53, 0x3f, 0xb9, 0xa1, 0x9b, 0xa7, 0x5b, 0x1c, 0x7e, 0x2a, 0xa4, 0x5b, 0x14, 0xce, 0x92, 0x30, 0x48, 0x5, 0xd1, 0x11, 0x22, 0x47, 0xe3, 0x46, 0x12, 0x75, 0xd7, 0x2c, 0xc0, 0x9c, 0x40, 0x64, 0xc4, 0x90, 0x1b, 0x8c, 0xc5, 0x2d, 0xb3, 0x2b, 0x9b, 0x9a, 0x69, 0xac, 0xbc, 0x3d, 0x39, 0xc9, 0xf5, 0x68, 0xdd, 0x37, 0xa0, 0x50, 0x22, 0xa9, 0x7d, 0x5e, 0x24, 0x62, 0x72, 0xe3, 0xec, 0xa4, 0x6a, 0x94, 0x94, 0xd, 0x46, 0xe9, 0x50, 0x22, 0x33, 0x4e, 0x7f, 0x32, 0x19, 0xe3, 0x90, 0xc7, 0xfc, 0xf5, 0x45, 0x1, 0x25, 0xa4, 0x30, 0x7f, 0xdf, 0x31, 0x22, 0xe7, 0xb2, 0xc3, 0x25, 0x82, 0xe8, 0x70, 0x9e, 0x1c, 0x13, 0x10, 0xca, 0x1f, 0xd0, 0x5, 0xf1, 0xaa, 0x74, 0xd1, 0x32, 0x74, 0x8e, 0x9, 0xea, 0x63, 0xd, 0xc9, 0xe, 0x63, 0x53, 0xa9, 0x1e, 0x11, 0x46, 0x63, 0xda, 0x76, 0x27, 0x42, 0x66, 0x4e, 0x79, 0xe4, 0x65, 0x45, 0xb4, 0xa4, 0xe0, 0x2f, 0x1e, 0x80, 0xe4, 0x88, 0x1c, 0x71, 0xb0, 0x39, 0xb6, 0x94, 0xf1, 0x1a, 0x50, 0x8c, 0x2b, 0xf1, 0xd1, 0xf, 0xcd, 0x45, 0x26, 0x31, 0x36, 0x35, 0xd9, 0xb5, 0x9a, 0xcc, 0xf9, 0xd8, 0x5c, 0x9e, 0xc3, 0xfe, 0xac, 0x74, 0x81, 0xf1, 0x9f, 0xab, 0x40, 0xb8, 0x7, 0x51, 0xaf, 0xf4, 0xa3, 0xa2, 0x57, 0x35, 0xee, 0xb7, 0xc7, 0xd2, 0xcc, 0xcd, 0x88, 0xb1, 0xec, 0x90, 0xb9, 0x6b, 0x99, 0x49, 0xa6, 0xe2, 0xe, 0xc1, 0x1d, 0xc6, 0x5f, 0x8a, 0xf, 0x5, 0xfe, 0x61, 0xdb, 0xdf, 0x5b, 0x8f, 0xd3, 0x9f, 0x2d, 0xd3, 0x6f, 0x7e, 0xc2, 0xf1, 0xea, 0xf7, 0x16, 0x62, 0x9a, 0xf1, 0x9b, 0x6e, 0xe4, 0x52, 0x80, 0xe0, 0x1f, 0xa5, 0x93, 0xb8, 0x6a, 0xaf, 0x81, 0xa1, 0x5b, 0xe3, 0xbe, 0x51, 0x37, 0xa1, 0x1f, 0xae, 0xfa, 0x5e, 0x9b, 0x94, 0xf8, 0xac, 0x47, 0x95, 0xe4, 0x2, 0x79, 0x6b, 0x7a, 0xef, 0xe, 0xbe, 0xf4, 0x64, 0x27, 0xa4, 0x5d, 0x15, 0x71, 0x2d, 0x52, 0x2c, 0xce, 0x9e, 0xeb, 0xdb, 0x97, 0x29, 0xae, 0x7c, 0x62, 0x8f, 0x90, 0x61, 0xa1, 0x3e, 0xfe, 0x50, 0x78, 0xf7, 0xa0, 0x83, 0xd6, 0xf2, 0xf0, 0xd2, 0x4f, 0xef, 0x6b, 0x90, 0xde, 0xae, 0xc7, 0xbe, 0x28, 0x53, 0xed, 0x58, 0x70, 0x34, 0x7, 0x74, 0xda, 0xd8, 0x61, 0xdb, 0x43, 0x30, 0x8d, 0x34, 0xc2, 0xc7, 0xb0, 0x5d, 0x75, 0x31, 0xd9, 0x5f, 0x8e, 0x38, 0x69, 0x4e, 0xe3, 0x65, 0x79, 0x65, 0xbe, 0xc6, 0xb8, 0x84, 0x8, 0xf4, 0x4c, 0xd9, 0x84, 0x92, 0xa0, 0x8d, 0x68, 0x79, 0x9c, 0x2, 0xa0, 0xd2, 0x2a, 0xc5, 0x19, 0x2a, 0x1e, 0x14, 0x9b, 0xb5, 0xe7, 0x3f, 0xa6, 0x6a, 0x61, 0xe7, 0x86, 0xd0, 0xed, 0x2e, 0x32, 0x3a, 0x28, 0x19, 0x75, 0xfb, 0xcb, 0x72, 0x8d, 0xa6, 0x29, 0x4b, 0xd3, 0x7d, 0xf7, 0xef, 0xc5, 0x9b, 0xb, 0x2c, 0x34, 0x6c, 0x7b, 0x58, 0x6a, 0x4d, 0xe3, 0x6c, 0xe3, 0xba, 0x9c, 0xeb, 0x2a, 0x66, 0x3b, 0xb3, 0x15, 0x5f, 0x26, 0xe5, 0xa9, 0xbe, 0xd, 0x2f, 0x62, 0xd8, 0xd6, 0xe6, 0x7d, 0x64, 0xce, 0xd3, 0xdd, 0x79, 0x6f, 0xb7, 0x1, 0x6d, 0xb8, 0x26, 0xfc, 0x2b, 0x7d, 0x79, 0x8c, 0x5a, 0x1f, 0x7d, 0x4e, 0xa8, 0x86, 0x9a, 0xb6, 0xe, 0x8d, 0x58, 0xb, 0x81, 0x65, 0xcd, 0x26, 0x56, 0x72, 0xec, 0x51, 0x80},
	},
	sample{
		[]byte(`UTF-8 encoded sample plain-text file
‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾‾

Markus Kuhn [ˈmaʳkʊs kuːn] <http://www.cl.cam.ac.uk/~mgk25/> — 2002-07-25


The ASCII compatible UTF-8 encoding used in this plain-text file
is defined in Unicode, ISO 10646-1, and RFC 2279.


Using Unicode/UTF-8, you can write in emails and source code things such as

Mathematics and sciences:

  ∮ E⋅da = Q,  n → ∞, ∑ f(i) = ∏ g(i),      ⎧⎡⎛┌─────┐⎞⎤⎫
                                            ⎪⎢⎜│a²+b³ ⎟⎥⎪
  ∀x∈ℝ: ⌈x⌉ = −⌊−x⌋, α ∧ ¬β = ¬(¬α ∨ β),    ⎪⎢⎜│───── ⎟⎥⎪
                                            ⎪⎢⎜⎷ c₈   ⎟⎥⎪
  ℕ ⊆ ℕ₀ ⊂ ℤ ⊂ ℚ ⊂ ℝ ⊂ ℂ,                   ⎨⎢⎜       ⎟⎥⎬
                                            ⎪⎢⎜ ∞     ⎟⎥⎪
  ⊥ < a ≠ b ≡ c ≤ d ≪ ⊤ ⇒ (⟦A⟧ ⇔ ⟪B⟫),      ⎪⎢⎜ ⎲     ⎟⎥⎪
                                            ⎪⎢⎜ ⎳aⁱ-bⁱ⎟⎥⎪
  2H₂ + O₂ ⇌ 2H₂O, R = 4.7 kΩ, ⌀ 200 mm     ⎩⎣⎝i=1    ⎠⎦⎭

Linguistics and dictionaries:

  ði ıntəˈnæʃənəl fəˈnɛtık əsoʊsiˈeıʃn
  Y [ˈʏpsilɔn], Yen [jɛn], Yoga [ˈjoːgɑ]

APL:

  ((V⍳V)=⍳⍴V)/V←,V    ⌷←⍳→⍴∆∇⊃‾⍎⍕⌈

Nicer typography in plain text files:

  ╔══════════════════════════════════════════╗
  ║                                          ║
  ║   • ‘single’ and “double” quotes         ║
  ║                                          ║
  ║   • Curly apostrophes: “We’ve been here” ║
  ║                                          ║
  ║   • Latin-1 apostrophe and accents:      ║
  ║                                          ║
  ║   • ‚deutsche‘ „Anführungszeichen“       ║
  ║                                          ║
  ║   • †, ‡, ‰, •, 3–4, —, −5/+5, ™, …      ║
  ║                                          ║
  ║   • ASCII safety test: 1lI|, 0OD, 8B     ║
  ║                      ╭─────────╮         ║
  ║   • the euro symbol: │ 14.95 € │         ║
  ║                      ╰─────────╯         ║
  ╚══════════════════════════════════════════╝

Combining characters:

  STARGΛ̊TE SG-1, a = v̇ = r̈, a⃑ ⊥ b⃑

Greek (in Polytonic):

  The Greek anthem:

  Σὲ γνωρίζω ἀπὸ τὴν κόψη
  τοῦ σπαθιοῦ τὴν τρομερή,
  σὲ γνωρίζω ἀπὸ τὴν ὄψη
  ποὺ μὲ βία μετράει τὴ γῆ.

  ᾿Απ᾿ τὰ κόκκαλα βγαλμένη
  τῶν ῾Ελλήνων τὰ ἱερά
  καὶ σὰν πρῶτα ἀνδρειωμένη
  χαῖρε, ὦ χαῖρε, ᾿Ελευθεριά!

  From a speech of Demosthenes in the 4th century BC:

  Οὐχὶ ταὐτὰ παρίσταταί μοι γιγνώσκειν, ὦ ἄνδρες ᾿Αθηναῖοι,
  ὅταν τ᾿ εἰς τὰ πράγματα ἀποβλέψω καὶ ὅταν πρὸς τοὺς
  λόγους οὓς ἀκούω· τοὺς μὲν γὰρ λόγους περὶ τοῦ
  τιμωρήσασθαι Φίλιππον ὁρῶ γιγνομένους, τὰ δὲ πράγματ᾿
  εἰς τοῦτο προήκοντα,  ὥσθ᾿ ὅπως μὴ πεισόμεθ᾿ αὐτοὶ
  πρότερον κακῶς σκέψασθαι δέον. οὐδέν οὖν ἄλλο μοι δοκοῦσιν
  οἱ τὰ τοιαῦτα λέγοντες ἢ τὴν ὑπόθεσιν, περὶ ἧς βουλεύεσθαι,
  οὐχὶ τὴν οὖσαν παριστάντες ὑμῖν ἁμαρτάνειν. ἐγὼ δέ, ὅτι μέν
  ποτ᾿ ἐξῆν τῇ πόλει καὶ τὰ αὑτῆς ἔχειν ἀσφαλῶς καὶ Φίλιππον
  τιμωρήσασθαι, καὶ μάλ᾿ ἀκριβῶς οἶδα· ἐπ᾿ ἐμοῦ γάρ, οὐ πάλαι
  γέγονεν ταῦτ᾿ ἀμφότερα· νῦν μέντοι πέπεισμαι τοῦθ᾿ ἱκανὸν
  προλαβεῖν ἡμῖν εἶναι τὴν πρώτην, ὅπως τοὺς συμμάχους
  σώσομεν. ἐὰν γὰρ τοῦτο βεβαίως ὑπάρξῃ, τότε καὶ περὶ τοῦ
  τίνα τιμωρήσεταί τις καὶ ὃν τρόπον ἐξέσται σκοπεῖν· πρὶν δὲ
  τὴν ἀρχὴν ὀρθῶς ὑποθέσθαι, μάταιον ἡγοῦμαι περὶ τῆς
  τελευτῆς ὁντινοῦν ποιεῖσθαι λόγον.

  Δημοσθένους, Γ´ ᾿Ολυνθιακὸς

Georgian:

  From a Unicode conference invitation:

  გთხოვთ ახლავე გაიაროთ რეგისტრაცია Unicode-ის მეათე საერთაშორისო
  კონფერენციაზე დასასწრებად, რომელიც გაიმართება 10-12 მარტს,
  ქ. მაინცში, გერმანიაში. კონფერენცია შეჰკრებს ერთად მსოფლიოს
  ექსპერტებს ისეთ დარგებში როგორიცაა ინტერნეტი და Unicode-ი,
  ინტერნაციონალიზაცია და ლოკალიზაცია, Unicode-ის გამოყენება
  ოპერაციულ სისტემებსა, და გამოყენებით პროგრამებში, შრიფტებში,
  ტექსტების დამუშავებასა და მრავალენოვან კომპიუტერულ სისტემებში.

Russian:

  From a Unicode conference invitation:

  Зарегистрируйтесь сейчас на Десятую Международную Конференцию по
  Unicode, которая состоится 10-12 марта 1997 года в Майнце в Германии.
  Конференция соберет широкий круг экспертов по  вопросам глобального
  Интернета и Unicode, локализации и интернационализации, воплощению и
  применению Unicode в различных операционных системах и программных
  приложениях, шрифтах, верстке и многоязычных компьютерных системах.

Thai (UCS Level 2):

  Excerpt from a poetry on The Romance of The Three Kingdoms (a Chinese
  classic 'San Gua'):

  [----------------------------|------------------------]
    ๏ แผ่นดินฮั่นเสื่อมโทรมแสนสังเวช  พระปกเกศกองบู๊กู้ขึ้นใหม่
  สิบสองกษัตริย์ก่อนหน้าแลถัดไป       สององค์ไซร้โง่เขลาเบาปัญญา
    ทรงนับถือขันทีเป็นที่พึ่ง           บ้านเมืองจึงวิปริตเป็นนักหนา
  โฮจิ๋นเรียกทัพทั่วหัวเมืองมา         หมายจะฆ่ามดชั่วตัวสำคัญ
    เหมือนขับไสไล่เสือจากเคหา      รับหมาป่าเข้ามาเลยอาสัญ
  ฝ่ายอ้องอุ้นยุแยกให้แตกกัน          ใช้สาวนั้นเป็นชนวนชื่นชวนใจ
    พลันลิฉุยกุยกีกลับก่อเหตุ          ช่างอาเพศจริงหนาฟ้าร้องไห้
  ต้องรบราฆ่าฟันจนบรรลัย           ฤๅหาใครค้ำชูกู้บรรลังก์ ฯ

  (The above is a two-column text. If combining characters are handled
  correctly, the lines of the second column should be aligned with the
  | character above.)

Ethiopian:

  Proverbs in the Amharic language:

  ሰማይ አይታረስ ንጉሥ አይከሰስ።
  ብላ ካለኝ እንደአባቴ በቆመጠኝ።
  ጌጥ ያለቤቱ ቁምጥና ነው።
  ደሀ በሕልሙ ቅቤ ባይጠጣ ንጣት በገደለው።
  የአፍ ወለምታ በቅቤ አይታሽም።
  አይጥ በበላ ዳዋ ተመታ።
  ሲተረጉሙ ይደረግሙ።
  ቀስ በቀስ፥ ዕንቁላል በእግሩ ይሄዳል።
  ድር ቢያብር አንበሳ ያስር።
  ሰው እንደቤቱ እንጅ እንደ ጉረቤቱ አይተዳደርም።
  እግዜር የከፈተውን ጉሮሮ ሳይዘጋው አይድርም።
  የጎረቤት ሌባ፥ ቢያዩት ይስቅ ባያዩት ያጠልቅ።
  ሥራ ከመፍታት ልጄን ላፋታት።
  ዓባይ ማደሪያ የለው፥ ግንድ ይዞ ይዞራል።
  የእስላም አገሩ መካ የአሞራ አገሩ ዋርካ።
  ተንጋሎ ቢተፉ ተመልሶ ባፉ።
  ወዳጅህ ማር ቢሆን ጨርስህ አትላሰው።
  እግርህን በፍራሽህ ልክ ዘርጋ።

Runes:

  ᚻᛖ ᚳᚹᚫᚦ ᚦᚫᛏ ᚻᛖ ᛒᚢᛞᛖ ᚩᚾ ᚦᚫᛗ ᛚᚪᚾᛞᛖ ᚾᚩᚱᚦᚹᛖᚪᚱᛞᚢᛗ ᚹᛁᚦ ᚦᚪ ᚹᛖᛥᚫ

  (Old English, which transcribed into Latin reads 'He cwaeth that he
  bude thaem lande northweardum with tha Westsae.' and means 'He said
  that he lived in the northern land near the Western Sea.')

Braille:

  ⡌⠁⠧⠑ ⠼⠁⠒  ⡍⠜⠇⠑⠹⠰⠎ ⡣⠕⠌

  ⡍⠜⠇⠑⠹ ⠺⠁⠎ ⠙⠑⠁⠙⠒ ⠞⠕ ⠃⠑⠛⠔ ⠺⠊⠹⠲ ⡹⠻⠑ ⠊⠎ ⠝⠕ ⠙⠳⠃⠞
  ⠱⠁⠞⠑⠧⠻ ⠁⠃⠳⠞ ⠹⠁⠞⠲ ⡹⠑ ⠗⠑⠛⠊⠌⠻ ⠕⠋ ⠙⠊⠎ ⠃⠥⠗⠊⠁⠇ ⠺⠁⠎
  ⠎⠊⠛⠝⠫ ⠃⠹ ⠹⠑ ⠊⠇⠻⠛⠹⠍⠁⠝⠂ ⠹⠑ ⠊⠇⠻⠅⠂ ⠹⠑ ⠥⠝⠙⠻⠞⠁⠅⠻⠂
  ⠁⠝⠙ ⠹⠑ ⠡⠊⠑⠋ ⠍⠳⠗⠝⠻⠲ ⡎⠊⠗⠕⠕⠛⠑ ⠎⠊⠛⠝⠫ ⠊⠞⠲ ⡁⠝⠙
  ⡎⠊⠗⠕⠕⠛⠑⠰⠎ ⠝⠁⠍⠑ ⠺⠁⠎ ⠛⠕⠕⠙ ⠥⠏⠕⠝ ⠰⡡⠁⠝⠛⠑⠂ ⠋⠕⠗ ⠁⠝⠹⠹⠔⠛ ⠙⠑
  ⠡⠕⠎⠑ ⠞⠕ ⠏⠥⠞ ⠙⠊⠎ ⠙⠁⠝⠙ ⠞⠕⠲

  ⡕⠇⠙ ⡍⠜⠇⠑⠹ ⠺⠁⠎ ⠁⠎ ⠙⠑⠁⠙ ⠁⠎ ⠁ ⠙⠕⠕⠗⠤⠝⠁⠊⠇⠲

  ⡍⠔⠙⠖ ⡊ ⠙⠕⠝⠰⠞ ⠍⠑⠁⠝ ⠞⠕ ⠎⠁⠹ ⠹⠁⠞ ⡊ ⠅⠝⠪⠂ ⠕⠋ ⠍⠹
  ⠪⠝ ⠅⠝⠪⠇⠫⠛⠑⠂ ⠱⠁⠞ ⠹⠻⠑ ⠊⠎ ⠏⠜⠞⠊⠊⠥⠇⠜⠇⠹ ⠙⠑⠁⠙ ⠁⠃⠳⠞
  ⠁ ⠙⠕⠕⠗⠤⠝⠁⠊⠇⠲ ⡊ ⠍⠊⠣⠞ ⠙⠁⠧⠑ ⠃⠑⠲ ⠔⠊⠇⠔⠫⠂ ⠍⠹⠎⠑⠇⠋⠂ ⠞⠕
  ⠗⠑⠛⠜⠙ ⠁ ⠊⠕⠋⠋⠔⠤⠝⠁⠊⠇ ⠁⠎ ⠹⠑ ⠙⠑⠁⠙⠑⠌ ⠏⠊⠑⠊⠑ ⠕⠋ ⠊⠗⠕⠝⠍⠕⠝⠛⠻⠹
  ⠔ ⠹⠑ ⠞⠗⠁⠙⠑⠲ ⡃⠥⠞ ⠹⠑ ⠺⠊⠎⠙⠕⠍ ⠕⠋ ⠳⠗ ⠁⠝⠊⠑⠌⠕⠗⠎
  ⠊⠎ ⠔ ⠹⠑ ⠎⠊⠍⠊⠇⠑⠆ ⠁⠝⠙ ⠍⠹ ⠥⠝⠙⠁⠇⠇⠪⠫ ⠙⠁⠝⠙⠎
  ⠩⠁⠇⠇ ⠝⠕⠞ ⠙⠊⠌⠥⠗⠃ ⠊⠞⠂ ⠕⠗ ⠹⠑ ⡊⠳⠝⠞⠗⠹⠰⠎ ⠙⠕⠝⠑ ⠋⠕⠗⠲ ⡹⠳
  ⠺⠊⠇⠇ ⠹⠻⠑⠋⠕⠗⠑ ⠏⠻⠍⠊⠞ ⠍⠑ ⠞⠕ ⠗⠑⠏⠑⠁⠞⠂ ⠑⠍⠏⠙⠁⠞⠊⠊⠁⠇⠇⠹⠂ ⠹⠁⠞
  ⡍⠜⠇⠑⠹ ⠺⠁⠎ ⠁⠎ ⠙⠑⠁⠙ ⠁⠎ ⠁ ⠙⠕⠕⠗⠤⠝⠁⠊⠇⠲

  (The first couple of paragraphs of "A Christmas Carol" by Dickens)

Compact font selection example text:

  ABCDEFGHIJKLMNOPQRSTUVWXYZ /0123456789
  abcdefghijklmnopqrstuvwxyz £©µÀÆÖÞßéöÿ
  –—‘“”„†•…‰™œŠŸž€ ΑΒΓΔΩαβγδω АБВГДабвгд
  ∀∂∈ℝ∧∪≡∞ ↑↗↨↻⇣ ┐┼╔╘░►☺♀ ﬁ�⑀₂ἠḂӥẄɐː⍎אԱა

Greetings in various languages:

  Hello world, Καλημέρα κόσμε, コンニチハ

Box drawing alignment tests:                                          █
                                                                      ▉
  ╔══╦══╗  ┌──┬──┐  ╭──┬──╮  ╭──┬──╮  ┏━━┳━━┓  ┎┒┏┑   ╷  ╻ ┏┯┓ ┌┰┐    ▊ ╱╲╱╲╳╳╳
  ║┌─╨─┐║  │╔═╧═╗│  │╒═╪═╕│  │╓─╁─╖│  ┃┌─╂─┐┃  ┗╃╄┙  ╶┼╴╺╋╸┠┼┨ ┝╋┥    ▋ ╲╱╲╱╳╳╳
  ║│╲ ╱│║  │║   ║│  ││ │ ││  │║ ┃ ║│  ┃│ ╿ │┃  ┍╅╆┓   ╵  ╹ ┗┷┛ └┸┘    ▌ ╱╲╱╲╳╳╳
  ╠╡ ╳ ╞╣  ├╢   ╟┤  ├┼─┼─┼┤  ├╫─╂─╫┤  ┣┿╾┼╼┿┫  ┕┛┖┚     ┌┄┄┐ ╎ ┏┅┅┓ ┋ ▍ ╲╱╲╱╳╳╳
  ║│╱ ╲│║  │║   ║│  ││ │ ││  │║ ┃ ║│  ┃│ ╽ │┃  ░░▒▒▓▓██ ┊  ┆ ╎ ╏  ┇ ┋ ▎
  ║└─╥─┘║  │╚═╤═╝│  │╘═╪═╛│  │╙─╀─╜│  ┃└─╂─┘┃  ░░▒▒▓▓██ ┊  ┆ ╎ ╏  ┇ ┋ ▏
  ╚══╩══╝  └──┴──┘  ╰──┴──╯  ╰──┴──╯  ┗━━┻━━┛  ▗▄▖▛▀▜   └╌╌┘ ╎ ┗╍╍┛ ┋  ▁▂▃▄▅▆▇█
`),
		[]byte{0x2c, 0x55, 0xa0, 0x36, 0x0, 0x0, 0xf0, 0xf8, 0x3a, 0x5b, 0x9e, 0x26, 0xb3, 0xe3, 0xfa, 0xf7, 0xfe, 0xfe, 0x17, 0xf7, 0x78, 0x1f, 0xff, 0x6f, 0xe7, 0xf9, 0xf8, 0xb3, 0x66, 0xde, 0xda, 0xe7, 0xeb, 0x7d, 0x84, 0x1, 0x68, 0x12, 0xc2, 0xcf, 0xca, 0xc6, 0xb4, 0xf9, 0x66, 0xe0, 0xc4, 0xaf, 0x8, 0xb, 0xf0, 0x2, 0xab, 0x8f, 0xd9, 0x99, 0xfe, 0xfb, 0x80, 0xdd, 0x45, 0xd7, 0x9e, 0xfa, 0x4a, 0x8b, 0xce, 0xae, 0x29, 0x71, 0x45, 0x58, 0xfe, 0x7e, 0x48, 0x5a, 0x34, 0x8f, 0x61, 0x4c, 0xe, 0x4d, 0xf4, 0xc9, 0xc8, 0x9c, 0x7d, 0xd, 0xd7, 0x1b, 0x76, 0xc6, 0x7, 0x0, 0xaf, 0x75, 0x7b, 0x59, 0xa5, 0x9a, 0xee, 0x59, 0x30, 0xf9, 0xe4, 0x58, 0xcd, 0x82, 0x9e, 0xfc, 0xd1, 0xdf, 0x70, 0x5d, 0xf2, 0x8d, 0x55, 0x52, 0x40, 0x67, 0x2d, 0xe3, 0x52, 0x86, 0x44, 0xf8, 0x6d, 0x5b, 0x38, 0xd2, 0x15, 0x2c, 0x45, 0x9, 0x23, 0x9b, 0xdf, 0xe7, 0xf5, 0x4e, 0x50, 0x55, 0xe0, 0x90, 0x97, 0x29, 0x4f, 0xbb, 0xdc, 0x92, 0x72, 0x93, 0x95, 0xc3, 0x39, 0xa4, 0x58, 0xeb, 0x29, 0xf8, 0x4, 0xa4, 0xa6, 0x63, 0x7f, 0x8b, 0xa8, 0x1, 0x21, 0x22, 0x29, 0x57, 0x1e, 0x7f, 0x7c, 0x45, 0x8a, 0x92, 0x7d, 0x55, 0x5d, 0xb8, 0x41, 0x95, 0xf9, 0x1a, 0x37, 0xad, 0x10, 0x48, 0x2, 0xff, 0x47, 0x7, 0x9c, 0x63, 0xb5, 0x5c, 0x93, 0x61, 0xaf, 0x73, 0xc7, 0xc6, 0x56, 0x2f, 0xa5, 0x65, 0xa0, 0x61, 0x4f, 0x49, 0x64, 0xe8, 0xaf, 0xc2, 0x7c, 0xad, 0xfd, 0xe4, 0x77, 0xf0, 0x4e, 0x89, 0xa7, 0x3d, 0x7f, 0x41, 0x98, 0xea, 0x7a, 0x11, 0x79, 0x7, 0xae, 0x7a, 0xc8, 0x30, 0x1e, 0x12, 0x58, 0x6e, 0x51, 0x72, 0xca, 0x71, 0x66, 0xd4, 0x79, 0x8b, 0xb7, 0x8e, 0x1b, 0xc0, 0x81, 0xb2, 0xef, 0x4c, 0xf1, 0x29, 0x32, 0xb3, 0xf5, 0x15, 0x5b, 0x95, 0xc6, 0x22, 0xbb, 0xb2, 0x1, 0x99, 0xf3, 0xa4, 0x10, 0x2, 0x7b, 0xd9, 0xd2, 0xe6, 0xef, 0xb5, 0x43, 0x3e, 0xe1, 0x3b, 0x35, 0xfb, 0x67, 0xe9, 0x52, 0xff, 0xd, 0x82, 0xcc, 0x28, 0x1c, 0x28, 0x58, 0x21, 0x5e, 0x80, 0x20, 0x1e, 0x74, 0x3f, 0x8e, 0xbf, 0x9a, 0x83, 0x38, 0x31, 0x51, 0x33, 0xff, 0x68, 0x2a, 0xe1, 0x6, 0x66, 0xf3, 0xd1, 0x10, 0x37, 0xf5, 0xe5, 0x3d, 0xc5, 0x6b, 0xe8, 0xd0, 0xde, 0x26, 0xae, 0xae, 0xd3, 0xd7, 0x1, 0x4, 0xd9, 0xe3, 0xf2, 0xc6, 0x62, 0xf3, 0x7a, 0x67, 0x9d, 0x9, 0xff, 0x91, 0xe6, 0xf5, 0xb0, 0xf1, 0xd7, 0x36, 0x53, 0xa2, 0x49, 0x43, 0xbf, 0xca, 0xd3, 0xbc, 0x48, 0x7f, 0x6b, 0x5c, 0x62, 0x1d, 0x4a, 0x70, 0xc, 0x24, 0x5e, 0xee, 0x19, 0xb4, 0x3a, 0x7b, 0x93, 0xe1, 0xec, 0xfa, 0x5e, 0xb2, 0x4e, 0xa5, 0xa5, 0xdb, 0x97, 0x41, 0x39, 0x41, 0xe, 0xe1, 0xb1, 0xf3, 0x9e, 0xc1, 0x7, 0x0, 0x55, 0x3, 0xff, 0x13, 0x5a, 0x4c, 0x4f, 0x51, 0xf5, 0xe4, 0x6a, 0x33, 0x86, 0xbb, 0x7a, 0xb3, 0xd7, 0x5a, 0xcd, 0xee, 0x68, 0x27, 0x0, 0x48, 0x88, 0xb1, 0xb1, 0x87, 0xc7, 0xe0, 0xf4, 0xa6, 0x7, 0x1f, 0x5c, 0x31, 0x5d, 0xdc, 0x9f, 0x10, 0x7b, 0x2b, 0x5, 0x77, 0xf4, 0xdb, 0xaa, 0xe, 0x81, 0xf1, 0xb7, 0x12, 0xad, 0x64, 0x7e, 0x9a, 0x80, 0xf6, 0x7b, 0x71, 0x49, 0x97, 0x94, 0xe0, 0x57, 0x66, 0x6c, 0xf, 0xc2, 0x64, 0x81, 0xf6, 0xc8, 0x41, 0xe8, 0x9b, 0x2e, 0x1c, 0xf9, 0x43, 0xa1, 0x88, 0x28, 0x4c, 0x80, 0xf9, 0xf5, 0x2d, 0x17, 0xcf, 0x9c, 0xe6, 0x35, 0xdc, 0xf1, 0xe, 0xf9, 0x6f, 0x23, 0xdd, 0x28, 0x9a, 0xe0, 0xb6, 0xd7, 0xf5, 0x4a, 0xf6, 0xc0, 0x2f, 0x85, 0x60, 0x60, 0x15, 0xd9, 0x6a, 0x44, 0xb, 0xca, 0x90, 0x76, 0x67, 0x80, 0x25, 0x3d, 0x58, 0xad, 0xbb, 0xef, 0xb5, 0x16, 0x4c, 0x98, 0xa0, 0x30, 0xd2, 0xde, 0xd5, 0x37, 0x7e, 0xb9, 0x6e, 0xc1, 0x1, 0xca, 0x62, 0xc9, 0xf6, 0x74, 0x6e, 0x64, 0xff, 0x38, 0x73, 0xc0, 0xb1, 0x48, 0xdd, 0x7a, 0x76, 0x18, 0x94, 0x2, 0x68, 0x8, 0xb9, 0xb7, 0x20, 0xb1, 0x7a, 0xdb, 0xc6, 0xba, 0x9c, 0x6, 0x1c, 0x1, 0x92, 0x41, 0x3, 0xd6, 0xaf, 0xcd, 0xa4, 0x44, 0xe7, 0x34, 0x5e, 0x76, 0x5a, 0x59, 0x2d, 0x21, 0xf1, 0xd2, 0x86, 0x4e, 0x53, 0xcb, 0x9b, 0x15, 0x3e, 0x19, 0x84, 0x9a, 0x98, 0x4a, 0xa3, 0x1d, 0xd9, 0x3c, 0xe2, 0xc4, 0xc0, 0x84, 0xb6, 0x51, 0xfa, 0x59, 0x0, 0xf, 0x72, 0x92, 0x61, 0xfa, 0x36, 0x50, 0xe4, 0x87, 0xc, 0x6b, 0x81, 0xe0, 0x7, 0x2b, 0xba, 0xf6, 0x64, 0x92, 0xd7, 0x18, 0x77, 0xcd, 0x36, 0x20, 0x38, 0x46, 0x95, 0x58, 0xa0, 0x66, 0x86, 0x26, 0x56, 0x7b, 0xb5, 0xc1, 0x96, 0x70, 0x8d, 0x8d, 0x1c, 0x68, 0xc7, 0xc1, 0x7, 0xe8, 0x41, 0x30, 0x78, 0x85, 0xad, 0xe8, 0xae, 0xe1, 0x4d, 0xd3, 0x5f, 0xd0, 0x5d, 0x54, 0x1c, 0xf2, 0x80, 0x7f, 0x71, 0xb1, 0xbd, 0x22, 0x65, 0x42, 0x76, 0xc3, 0xbb, 0x1d, 0xa, 0xa1, 0xa4, 0xb8, 0xc2, 0x4e, 0x61, 0x6a, 0x9b, 0xa9, 0x7a, 0xf1, 0x30, 0xeb, 0xe0, 0x5a, 0x7f, 0x3c, 0xd1, 0xaa, 0x46, 0xf3, 0x6f, 0x1, 0xa0, 0xee, 0x27, 0x62, 0x1, 0x4, 0x51, 0x5d, 0x45, 0xb4, 0x15, 0xd6, 0xd8, 0xb4, 0x17, 0x9b, 0x5b, 0x6, 0x83, 0x6c, 0xb7, 0x60, 0x5f, 0x0, 0x5b, 0x3f, 0xbf, 0xd8, 0x2b, 0x2f, 0x88, 0xda, 0xf0, 0xf0, 0xf3, 0x5a, 0x67, 0xaf, 0x9a, 0x5, 0xa2, 0x18, 0x2f, 0xb7, 0xd0, 0x6f, 0xdd, 0xb7, 0x7d, 0xd8, 0x3b, 0xc5, 0x85, 0xd2, 0xb3, 0x7f, 0xde, 0x5e, 0x56, 0xb8, 0xea, 0x46, 0xcc, 0x32, 0x3c, 0xec, 0xc5, 0xb5, 0x50, 0xb6, 0x18, 0x9c, 0x2a, 0xbe, 0x31, 0x28, 0x56, 0xee, 0xbd, 0x17, 0x75, 0xa9, 0x84, 0x8a, 0xb9, 0x9b, 0xb3, 0xff, 0xf3, 0x2b, 0xe3, 0xee, 0x6, 0x68, 0xb, 0xb0, 0x2, 0xee, 0x2, 0xac, 0xc7, 0x31, 0x81, 0x60, 0x8a, 0x29, 0xbf, 0x1d, 0xce, 0x19, 0x78, 0x4c, 0xb7, 0x60, 0x3a, 0xc2, 0xdb, 0x2f, 0x72, 0xb0, 0xd, 0xbe, 0x22, 0xc3, 0x40, 0xbc, 0xb2, 0xce, 0x8d, 0x1b, 0xba, 0x80, 0xab, 0x22, 0x6d, 0xef, 0xe, 0x4, 0x54, 0xeb, 0xe6, 0x3c, 0x75, 0x8a, 0xf0, 0x3e, 0x64, 0x6d, 0x27, 0x9d, 0xca, 0x79, 0x50, 0x49, 0x11, 0x29, 0x28, 0xda, 0xc9, 0xac, 0xb1, 0x94, 0xeb, 0x2f, 0x9f, 0xca, 0x8f, 0x4a, 0xad, 0xb6, 0x39, 0xd6, 0xcf, 0x28, 0xbc, 0x4b, 0x2b, 0x73, 0x15, 0xbf, 0x1d, 0xdd, 0xcd, 0x3f, 0xf2, 0x1c, 0xe2, 0xbc, 0xb8, 0x47, 0x79, 0x4, 0xe9, 0x52, 0x1b, 0x2e, 0xab, 0x11, 0x52, 0x42, 0xc5, 0x50, 0x55, 0xd3, 0xfc, 0x73, 0xc9, 0x63, 0x6b, 0x1d, 0x21, 0xb4, 0xa3, 0x90, 0xd3, 0x8d, 0xe7, 0x78, 0xec, 0xc9, 0x5f, 0xfe, 0xf0, 0x9a, 0x90, 0x69, 0x9f, 0x82, 0xf8, 0xf9, 0x26, 0xe0, 0x58, 0xcf, 0xa1, 0x61, 0xb0, 0xf6, 0x18, 0xea, 0x72, 0xea, 0x2, 0x44, 0xad, 0x2, 0x74, 0xac, 0x13, 0x87, 0xfb, 0xab, 0xfe, 0xca, 0xf0, 0xaa, 0x6c, 0x30, 0xfd, 0x59, 0xcc, 0xb, 0xbb, 0x15, 0xf0, 0x7b, 0x93, 0xe0, 0xcf, 0xb2, 0x40, 0x80, 0x2e, 0x78, 0x8d, 0x2, 0x93, 0x72, 0x89, 0x1, 0x12, 0x53, 0xd9, 0xad, 0xa4, 0x95, 0xbf, 0x91, 0x9, 0xe5, 0x88, 0x7a, 0x7a, 0x98, 0x27, 0xe5, 0xc7, 0x9d, 0x5a, 0x3e, 0xaf, 0xb9, 0xd, 0x60, 0x74, 0x94, 0xfe, 0xe3, 0x3f, 0xee, 0xb1, 0xf2, 0x63, 0x90, 0x5e, 0xb2, 0x13, 0x2e, 0xba, 0x3f, 0x98, 0xd7, 0xe4, 0x1a, 0xbd, 0xde, 0x9c, 0x8, 0x5b, 0x3f, 0xeb, 0x7b, 0x19, 0xb0, 0xdb, 0x3d, 0xe8, 0xc7, 0xda, 0xd2, 0x89, 0xfb, 0x1, 0x79, 0xb0, 0x6a, 0x11, 0xb, 0x3a, 0x1b, 0x2a, 0x68, 0x18, 0x6a, 0x68, 0x1a, 0xfb, 0xd7, 0x9, 0x9a, 0xa6, 0x40, 0x24, 0x4, 0x6b, 0x25, 0xfb, 0xf4, 0xe0, 0x32, 0x3e, 0x33, 0xb3, 0xd3, 0x60, 0xed, 0x6a, 0xaa, 0xe5, 0x4d, 0xcd, 0xc6, 0xcf, 0xab, 0x65, 0xf, 0xba, 0x61, 0x69, 0x15, 0xb9, 0x50, 0x64, 0xcb, 0xe3, 0xe6, 0x47, 0x67, 0xfe, 0xc, 0xcf, 0x3e, 0xcf, 0x62, 0x38, 0xfd, 0x87, 0xbc, 0x12, 0x87, 0x80, 0xbc, 0xa9, 0x19, 0xbc, 0xe, 0xe1, 0x3b, 0xe3, 0x98, 0xa8, 0xfb, 0xf0, 0x21, 0x45, 0x69, 0x5d, 0xb7, 0xdc, 0xab, 0x5f, 0xfa, 0xb5, 0xd, 0x29, 0xdb, 0xde, 0x95, 0x3b, 0x33, 0x80, 0xf1, 0xc2, 0x21, 0xf7, 0xcb, 0x87, 0xe8, 0x39, 0x5e, 0x5e, 0x5d, 0xb3, 0x1d, 0xae, 0x9a, 0x7f, 0x53, 0xea, 0x7d, 0x98, 0xa6, 0xc3, 0x7f, 0xe9, 0xea, 0x7, 0xb9, 0xcb, 0x86, 0x8b, 0xea, 0x67, 0x15, 0x14, 0x65, 0xc4, 0xdb, 0xa2, 0xd3, 0x41, 0xe6, 0x3e, 0x6e, 0x9e, 0x67, 0x43, 0xef, 0x32, 0xd0, 0x8a, 0xca, 0xc7, 0xb8, 0xbe, 0x59, 0xa2, 0x73, 0xa9, 0x7d, 0x1f, 0x9, 0x31, 0x9a, 0x57, 0x3c, 0x86, 0x59, 0xc9, 0x77, 0x46, 0x3c, 0xfe, 0x31, 0x5d, 0xb9, 0x1e, 0xe, 0xf0, 0xcb, 0x2b, 0xa7, 0x8f, 0xdf, 0xc6, 0xc3, 0x7, 0xc7, 0x15, 0xa3, 0x92, 0x81, 0x35, 0x36, 0xe, 0x6c, 0xcd, 0xe, 0x41, 0x59, 0x21, 0x14, 0xe5, 0xb0, 0x65, 0x20, 0xb, 0x14, 0xbc, 0x5c, 0xb9, 0x1f, 0xc1, 0x47, 0xc0, 0x8a, 0x33, 0x1c, 0x73, 0x29, 0xb9, 0x7c, 0xde, 0x9e, 0xb4, 0xd8, 0xc9, 0x20, 0x69, 0x12, 0x64, 0x4e, 0x99, 0x4b, 0xb4, 0xe3, 0xdd, 0xd8, 0xa9, 0x93, 0x74, 0x89, 0x64, 0x7b, 0xa9, 0xe0, 0x18, 0x21, 0x8b, 0xf8, 0x31, 0xb8, 0x68, 0xc8, 0x92, 0x82, 0x38, 0xbe, 0x1a, 0xff, 0xa2, 0x3d, 0xa1, 0x82, 0x90, 0x4, 0x75, 0x3c, 0xd7, 0x2, 0x3, 0x0, 0x92, 0x9c, 0x41, 0x44, 0x4b, 0x12, 0x65, 0x61, 0xe, 0x89, 0x5b, 0xce, 0x68, 0x5e, 0x6b, 0x6c, 0x6a, 0x1a, 0x41, 0xa8, 0xc, 0x8f, 0x3d, 0x14, 0x6b, 0xe8, 0x4e, 0xa4, 0x25, 0x14, 0x83, 0x2b, 0x5f, 0x2d, 0x0, 0xab, 0xcc, 0x8a, 0xb8, 0xe2, 0x1, 0x3a, 0x22, 0x84, 0x14, 0xb0, 0x43, 0x43, 0xc5, 0x59, 0x51, 0x49, 0x98, 0xa, 0xb5, 0x42, 0x4e, 0x4e, 0xa9, 0xea, 0x9b, 0x8c, 0x40, 0x45, 0xca, 0x45, 0x6, 0x1b, 0x15, 0xe5, 0x81, 0xd, 0x66, 0x64, 0x99, 0x95, 0x66, 0xa0, 0x20, 0xc7, 0xb8, 0x8b, 0xad, 0x9c, 0x6e, 0xe9, 0x96, 0x73, 0x15, 0x76, 0xec, 0x98, 0xad, 0xd9, 0x63, 0x6f, 0xb0, 0xce, 0xc2, 0xcc, 0x66, 0xe0, 0xc1, 0xfc, 0xb2, 0x52, 0x8a, 0x11, 0x7e, 0x9a, 0xe6, 0x53, 0x9e, 0xbc, 0x5e, 0x9d, 0x5e, 0xef, 0x47, 0x8b, 0x88, 0x6a, 0x86, 0x6b, 0xd7, 0x41, 0xd7, 0xb6, 0xaf, 0xfd, 0x68, 0xe7, 0x95, 0x58, 0xd5, 0xa4, 0x2f, 0xb5, 0x7d, 0x9a, 0xcf, 0x31, 0xd3, 0x6, 0xd1, 0x81, 0x94, 0x21, 0x66, 0x66, 0x34, 0xb0, 0x8, 0x67, 0x90, 0x17, 0x5a, 0x1b, 0x19, 0x3a, 0xd8, 0x57, 0xb, 0x3e, 0x14, 0xe1, 0x2a, 0x59, 0x26, 0x3a, 0x14, 0x2a, 0x59, 0x42, 0x53, 0x28, 0x35, 0x38, 0x61, 0x1d, 0xff, 0x5a, 0x76, 0xdc, 0xc8, 0xde, 0x2, 0x31, 0xd7, 0x44, 0x9d, 0x4a, 0xf2, 0x97, 0xa5, 0x17, 0x50, 0xf5, 0x6d, 0x9f, 0xf6, 0x26, 0x62, 0xc9, 0x6b, 0xca, 0xea, 0x58, 0x2d, 0x64, 0xc1, 0xc6, 0xb4, 0x48, 0xbf, 0x2, 0x5e, 0x5f, 0x67, 0x4d, 0x72, 0xe6, 0xeb, 0x2e, 0xbd, 0x3e, 0x79, 0x2e, 0xb4, 0xdc, 0xbd, 0x5e, 0xf0, 0xd2, 0x97, 0xf5, 0xf5, 0x5c, 0x4, 0x84, 0x55, 0xa3, 0x65, 0xf5, 0xc4, 0x35, 0xce, 0x57, 0xc2, 0x17, 0x30, 0x89, 0x64, 0xef, 0x3b, 0x18, 0x7b, 0xc0, 0x48, 0x65, 0xee, 0xb7, 0x7a, 0x81, 0xb1, 0x2c, 0xb, 0x61, 0x1, 0x23, 0x36, 0x97, 0xfc, 0x97, 0x32, 0x30, 0xf8, 0xf2, 0x21, 0x12, 0x2f, 0x29, 0x9b, 0x85, 0xbb, 0x5e, 0x8f, 0xa8, 0xc4, 0xee, 0x16, 0x76, 0xab, 0x61, 0x17, 0x0, 0x8c, 0xca, 0xd6, 0x5e, 0x85, 0x22, 0xaa, 0xb4, 0xa3, 0x8a, 0xd0, 0xe, 0xdf, 0x4f, 0x1e, 0x95, 0x33, 0xc, 0x62, 0xd, 0xba, 0x9b, 0x19, 0x27, 0xec, 0xa1, 0xc7, 0xff, 0xb7, 0xc, 0xae, 0xb0, 0x67, 0x1f, 0xcb, 0x6f, 0xbd, 0x94, 0x19, 0x55, 0x20, 0x7e, 0x8a, 0x72, 0xe, 0xff, 0x50, 0xf4, 0xaa, 0x26, 0xf6, 0x1e, 0xd5, 0x57, 0xd2, 0x9b, 0x34, 0xab, 0xf5, 0x61, 0x64, 0x65, 0xf6, 0xc8, 0x9d, 0x41, 0x3a, 0x51, 0xfc, 0xf6, 0xb9, 0x1a, 0x5b, 0x32, 0xb9, 0x19, 0x78, 0xa9, 0x3d, 0xd4, 0xa9, 0x20, 0x8f, 0x82, 0x7e, 0x83, 0xd1, 0xad, 0x52, 0xdb, 0x87, 0xd6, 0x49, 0x10, 0x4, 0x92, 0x1b, 0xf7, 0x8, 0x82, 0x1, 0xca, 0xd7, 0x16, 0x73, 0xf6, 0xa3, 0xdc, 0x7d, 0x1d, 0x3f, 0x47, 0x23, 0xc2, 0x1e, 0x88, 0xba, 0xa2, 0x75, 0xbb, 0x99, 0x85, 0x60, 0x52, 0xfd, 0x3d, 0x75, 0x74, 0x4d, 0xc3, 0x34, 0xcb, 0x9d, 0xfd, 0x30, 0x7e, 0xd0, 0xa6, 0x9b, 0x58, 0x7, 0x93, 0xdd, 0x73, 0x7b, 0x26, 0xfa, 0x78, 0x3d, 0x3c, 0x74, 0x8d, 0x72, 0x1e, 0x60, 0xda, 0x83, 0xd0, 0xf9, 0xe1, 0xf9, 0x97, 0xbc, 0xd9, 0x47, 0x94, 0x71, 0x1a, 0x66, 0x6a, 0xa7, 0xe9, 0x0, 0xf2, 0x4f, 0xd3, 0x2d, 0xf6, 0xfb, 0xf6, 0x96, 0x64, 0x8c, 0xef, 0x83, 0x66, 0xf4, 0xa3, 0x42, 0xa1, 0x3a, 0x7c, 0xcb, 0x42, 0x46, 0x90, 0x47, 0x72, 0xce, 0x9d, 0x66, 0x0, 0xf9, 0xd3, 0x41, 0xaf, 0x6a, 0x83, 0x88, 0x19, 0x42, 0xe6, 0xf4, 0x81, 0x32, 0xcf, 0xa1, 0xa7, 0xb6, 0x1e, 0x59, 0x92, 0x15, 0x22, 0xb3, 0x48, 0xf8, 0x26, 0xa4, 0xc7, 0xba, 0x52, 0x22, 0x4a, 0xd0, 0xa7, 0xb7, 0xa1, 0xd7, 0x81, 0x83, 0xa2, 0x7, 0x96, 0x9f, 0x45, 0xa3, 0x47, 0xd2, 0x5b, 0xc, 0x56, 0xe4, 0x88, 0x57, 0x91, 0x7f, 0x4e, 0x39, 0x25, 0x4d, 0xe7, 0x6, 0x61, 0x50, 0x1c, 0xeb, 0x9c, 0x7d, 0xe5, 0x5e, 0xe5, 0x2e, 0xe3, 0xac, 0x67, 0x64, 0x24, 0xc5, 0x7e, 0xe, 0xaa, 0x78, 0xa4, 0x69, 0xe7, 0x41, 0xa6, 0xa2, 0x5c, 0x75, 0xa0, 0xc0, 0x2d, 0x2a, 0x44, 0xe9, 0x3a, 0x6b, 0xa4, 0x50, 0x48, 0xec, 0xd5, 0x11, 0xe7, 0xd4, 0x54, 0xd9, 0x88, 0x75, 0x97, 0xe6, 0xfd, 0x7d, 0x5a, 0xc0, 0x8d, 0x96, 0x11, 0xa8, 0xcf, 0x3c, 0x10, 0x9f, 0x30, 0xfb, 0x4a, 0x0, 0x6d, 0xd6, 0xb8, 0xa4, 0x5b, 0x8d, 0xee, 0x93, 0x75, 0x50, 0xdf, 0x6d, 0x51, 0x93, 0x5e, 0xcb, 0x8b, 0xfd, 0xa3, 0x9d, 0xf9, 0x45, 0xf3, 0xee, 0xfb, 0x54, 0x54, 0x13, 0xdb, 0xa, 0xdf, 0xe2, 0xa7, 0x1e, 0xad, 0xfc, 0x8c, 0x57, 0x22, 0xe4, 0x36, 0x16, 0xc5, 0x34, 0xb2, 0xc4, 0xbf, 0xe, 0x64, 0x69, 0xff, 0x61, 0x4b, 0x81, 0x1d, 0x2d, 0xad, 0x31, 0x54, 0x5b, 0xb6, 0x2c, 0x52, 0x1, 0x29, 0x45, 0xf4, 0xe6, 0xfd, 0xb1, 0x5e, 0x96, 0x12, 0x6b, 0x83, 0xa0, 0x81, 0x2b, 0x9a, 0xae, 0x29, 0xb3, 0x2, 0x85, 0xf9, 0xc1, 0xeb, 0xef, 0x8f, 0xff, 0x73, 0x4d, 0x7b, 0x62, 0xac, 0x79, 0xdd, 0x57, 0x84, 0x30, 0xf6, 0xc6, 0x47, 0x8, 0x71, 0x7d, 0xd9, 0xf5, 0x6d, 0xe3, 0x49, 0x6a, 0x8f, 0x17, 0xd7, 0xfc, 0x88, 0x27, 0x15, 0xd0, 0x90, 0xf6, 0x2c, 0x85, 0x66, 0xe0, 0xd1, 0x33, 0x95, 0x2d, 0xb5, 0x1e, 0x5, 0xeb, 0xbe, 0x16, 0x44, 0xca, 0xb8, 0x30, 0xcb, 0x53, 0x89, 0xf8, 0xd4, 0x6b, 0x27, 0xb6, 0x3a, 0x91, 0x71, 0xf9, 0xe0, 0x51, 0xfd, 0x32, 0x72, 0xa7, 0x2d, 0x8c, 0xf1, 0x87, 0xe6, 0xb6, 0xeb, 0x13, 0xaa, 0x45, 0x56, 0x7f, 0x61, 0xcb, 0xee, 0xeb, 0x44, 0x37, 0x52, 0x44, 0x58, 0x63, 0x83, 0xaa, 0x55, 0xe9, 0xec, 0x20, 0x25, 0xd, 0x80, 0x64, 0x67, 0x6b, 0xcc, 0x70, 0xc2, 0xd3, 0x9c, 0xe7, 0xab, 0xc9, 0x5c, 0xca, 0x39, 0x3e, 0xbd, 0x57, 0xde, 0x6, 0x85, 0xb7, 0x7c, 0x2, 0xba, 0xd5, 0x7b, 0xee, 0x88, 0xf2, 0xab, 0x75, 0x66, 0xf6, 0x75, 0x7a, 0xa4, 0x7c, 0x8c, 0x62, 0xad, 0x17, 0x2e, 0xb1, 0x9a, 0x51, 0x1, 0xa4, 0x1f, 0xb, 0x7c, 0x66, 0xca, 0x57, 0x89, 0x1f, 0x14, 0x79, 0x94, 0x4c, 0x28, 0xf4, 0xec, 0x93, 0x58, 0x6f, 0xbb, 0x2a, 0xdf, 0x25, 0x3b, 0x45, 0xe5, 0xbe, 0xad, 0xe2, 0x9a, 0x23, 0x4d, 0x49, 0xb1, 0x27, 0xd4, 0x75, 0xe6, 0x1f, 0x3e, 0x73, 0x91, 0x1c, 0x39, 0x9, 0xcd, 0x6d, 0xa7, 0x30, 0x25, 0xeb, 0xd1, 0xe6, 0xa6, 0xee, 0xa9, 0x1e, 0x8f, 0x35, 0xa, 0xac, 0xb3, 0x81, 0x71, 0x55, 0xdb, 0xc8, 0x4f, 0x58, 0x6a, 0xe2, 0x1a, 0x6e, 0xcc, 0x88, 0x2e, 0x46, 0x73, 0x4f, 0x8b, 0x2, 0x97, 0x6, 0x32, 0x53, 0xa7, 0xcb, 0x96, 0xbf, 0xac, 0xe3, 0x3c, 0xd4, 0x9d, 0x57, 0xf1, 0xc3, 0x5e, 0x1c, 0x92, 0x58, 0x35, 0xdc, 0x6e, 0x44, 0x74, 0x77, 0x94, 0xf9, 0x73, 0xd, 0x17, 0x45, 0x1e, 0xda, 0xfa, 0xe8, 0xe7, 0x8e, 0xe9, 0x34, 0xd1, 0x7e, 0x95, 0x7e, 0xd6, 0x73, 0x6, 0x0, 0x47, 0xb8, 0xfe, 0xd6, 0xf9, 0xe0, 0xa3, 0xbf, 0xf8, 0x2f, 0xa3, 0x95, 0x41, 0x27, 0xfe, 0x73, 0x53, 0xfb, 0xce, 0x63, 0xa8, 0x1, 0x60, 0x25, 0x5f, 0xb0, 0x7c, 0x37, 0x60, 0x80, 0x12, 0xfd, 0xf0, 0x4a, 0x1, 0x88, 0xd, 0x81, 0x5b, 0xdc, 0x15, 0xc8, 0xa0, 0x2, 0x48, 0xf0, 0x81, 0x1, 0x8f, 0xe1, 0xf1, 0xb3, 0x1d, 0x8, 0x81, 0xb, 0xf3, 0xc1, 0x55, 0x92, 0xfe, 0xab, 0xf1, 0x0, 0x18, 0x67, 0xba, 0x30, 0x4, 0x48, 0x2c, 0xa4, 0x96, 0x23, 0x42, 0x51, 0x92, 0xc, 0x91, 0x24, 0x24, 0x47, 0x5e, 0x8f, 0x49, 0xc2, 0xe, 0xfc, 0x92, 0x19, 0xe9, 0xc7, 0xec, 0xe8, 0x53, 0xef, 0x52, 0x8b, 0xc6, 0xe8, 0xb0, 0x81, 0x48, 0x3a, 0x32, 0x9d, 0x4b, 0x3e, 0x58, 0x8, 0xe6, 0x4f, 0x48, 0x20, 0x92, 0x84, 0xcf, 0xcb, 0x5f, 0x38, 0x69, 0x7a, 0x53, 0x52, 0x4c, 0x96, 0xc6, 0x7e, 0x4, 0xc7, 0x29, 0x9, 0x3e, 0x30, 0x23, 0xa8, 0xf1, 0x38, 0x2, 0x90, 0xc9, 0xc0, 0x21, 0xc6, 0xdb, 0xbb, 0x58, 0x31, 0x2d, 0x42, 0x3f, 0x6, 0x9e, 0x94, 0x9c, 0x42, 0x6e, 0xa9, 0x7b, 0xd0, 0xe3, 0x6a, 0xb4, 0xc, 0xc8, 0xf1, 0x6, 0x89, 0x40, 0x58, 0x52, 0x54, 0x92, 0x4d, 0x28, 0xc2, 0x43, 0xe4, 0xfd, 0x52, 0xbe, 0x90, 0xfc, 0x8d, 0x9c, 0x1a, 0x28, 0x49, 0xd2, 0x44, 0x92, 0x4, 0x3f, 0xc9, 0xff, 0x12, 0x11, 0x2b, 0x19, 0x69, 0x58, 0x72, 0xd4, 0xb5, 0xa5, 0x55, 0xf9, 0x4f, 0xa9, 0x14, 0x20, 0xb5, 0x9f, 0x2b, 0xa2, 0x40, 0x57, 0xbe, 0x1d, 0x8d, 0x67, 0x33, 0x8b, 0x25, 0x66, 0x22, 0x54, 0xb9, 0x4a, 0x8b, 0x96, 0x9c, 0xf5, 0x7c, 0x55, 0x93, 0xcf, 0xa2, 0x22, 0x94, 0x2f, 0x91, 0x5a, 0x16, 0xcf, 0xe2, 0xc6, 0x8b, 0x61, 0x25, 0xdc, 0x9c, 0x8f, 0x4d, 0xe0, 0x4a, 0x11, 0x39, 0x49, 0x89, 0x4c, 0x41, 0xd4, 0x37, 0xa2, 0x42, 0x31, 0xc2, 0x41, 0x5a, 0x2d, 0x11, 0x88, 0x37, 0x65, 0x5, 0x20, 0x55, 0x6c, 0x10, 0x52, 0xd5, 0xa1, 0x1e, 0x6c, 0xc4, 0x7f, 0xd9, 0xb9, 0x68, 0x69, 0x5f, 0xb1, 0xd1, 0x45, 0x82, 0x31, 0xfb, 0xc5, 0xaf, 0xa5, 0x4c, 0x95, 0x1d, 0xff, 0x3a, 0xe8, 0x9d, 0xaf, 0xfd, 0xa4, 0x27, 0xbc, 0x41, 0x50, 0x17, 0x33, 0x7b, 0x55, 0x20, 0x92, 0x84, 0x8f, 0x22, 0xfa, 0x1c, 0xc9, 0xdd, 0xca, 0xbf, 0xf9, 0x57, 0x0, 0xd5, 0x70, 0x45, 0x1a, 0x38, 0x65, 0x6f, 0xa6, 0x5, 0x1d, 0x64, 0xca, 0x4a, 0x2b, 0x91, 0x7b, 0xdd, 0xa2, 0x8f, 0xd4, 0x5c, 0x9, 0xa4, 0xe3, 0x65, 0x1f, 0x7a, 0x3f, 0x39, 0xaa, 0xb9, 0xaa, 0xd4, 0x33, 0x34, 0x7b, 0x12, 0x3a, 0xb3, 0xd4, 0xe2, 0x91, 0xec, 0x9d, 0xcd, 0x93, 0x77, 0xc2, 0x3e, 0x49, 0xd9, 0x4, 0xb0, 0x59, 0xcb, 0x53, 0x1, 0xce, 0xf5, 0xd7, 0x6d, 0xce, 0xec, 0x50, 0x1c, 0xc2, 0xdd, 0xd0, 0x81, 0x8a, 0x99, 0xef, 0x85, 0x1b, 0xbd, 0xb2, 0x96, 0xaa, 0x67, 0xf1, 0x8a, 0xfe, 0x54, 0xc2, 0xcd, 0xdb, 0x8d, 0xed, 0x48, 0xa7, 0xfe, 0x3c, 0x5e, 0x38, 0x37, 0x34, 0x76, 0x58, 0x40, 0x2d, 0x72, 0xb5, 0x1f, 0x64, 0xd6, 0x46, 0xbb, 0xec, 0x16, 0xaa, 0x54, 0xa2, 0xb1, 0xbd, 0xb5, 0x87, 0xda, 0x7a, 0x82, 0xd0, 0x30, 0x8a, 0x36, 0x81, 0x30, 0xd8, 0x7a, 0x1c, 0xa0, 0xc1, 0xa, 0x33, 0xf4, 0x53, 0x99, 0xd1, 0xef, 0x51, 0x82, 0x7e, 0x1a, 0xdb, 0x50, 0xbb, 0x13, 0xb4, 0x4d, 0x60, 0x1d, 0x90, 0x80, 0x63, 0x60, 0x11, 0x8d, 0x58, 0x75, 0x90, 0x3b, 0x1e, 0x52, 0x6f, 0x5c, 0xd7, 0xd, 0x58, 0x58, 0x33, 0x3c, 0xd2, 0xb3, 0x22, 0xa0, 0xa4, 0xe3, 0x6d, 0x46, 0xca, 0x66, 0x31, 0x94, 0x1c, 0x1, 0xd9, 0xbd, 0x90, 0xae, 0xa6, 0x49, 0x6e, 0x13, 0x6, 0xb2, 0xbd, 0x41, 0x99, 0x33, 0x7a, 0x20, 0x38, 0x2c, 0x57, 0xe0, 0xcc, 0x63, 0xb9, 0xe6, 0x51, 0xb2, 0xa6, 0xba, 0x15, 0x53, 0xe5, 0x4, 0xfb, 0x36, 0x6f, 0x33, 0x3c, 0x2, 0x22, 0x1b, 0xf2, 0xcf, 0x58, 0x2a, 0xd9, 0xa3, 0x8d, 0x6, 0x2e, 0x64, 0xc, 0x1f, 0x94, 0x58, 0x5f, 0x7, 0x84, 0xfc, 0xb4, 0x63, 0x16, 0x6a, 0xdd, 0x41, 0xd0, 0xc0, 0x94, 0xb5, 0x29, 0x19, 0xd6, 0x2b, 0xe9, 0xf7, 0xd8, 0x29, 0xda, 0x7, 0x3e, 0x22, 0xe5, 0x41, 0xb2, 0xf3, 0xd6, 0xc6, 0xfa, 0x4c, 0xdc, 0xfa, 0x90, 0xc2, 0xc8, 0x38, 0xc3, 0x5, 0x78, 0x33, 0xed, 0x92, 0xff, 0x37, 0x19, 0x68, 0xd0, 0x86, 0x4b, 0x65, 0xb2, 0x5a, 0x4, 0x8c, 0x9a, 0xab, 0xec, 0xb9, 0xed, 0x26, 0xbe, 0x47, 0xe4, 0x97, 0x7d, 0xfe, 0x88, 0xe5, 0x76, 0x5f, 0xdd, 0xc9, 0xd3, 0x26, 0x5, 0x45, 0xa9, 0x89, 0x5b, 0x15, 0x19, 0xa8, 0xca, 0xb9, 0x3e, 0xba, 0xe8, 0x94, 0x3c, 0x8d, 0x47, 0xa4, 0x6a, 0xcf, 0x1f, 0x63, 0x51, 0x6d, 0x57, 0x30, 0xff, 0xc0, 0x54, 0xa3, 0x8b, 0x5, 0xda, 0xd8, 0x40, 0x5, 0x2d, 0x64, 0xfc, 0xca, 0x92, 0x1d, 0x24, 0x72, 0x97, 0x85, 0x97, 0x77, 0x85, 0x27, 0x74, 0x70, 0x7b, 0xb5, 0x83, 0x41, 0x5e, 0x41, 0x9e, 0x78, 0xb2, 0x68, 0x55, 0x2f, 0xb6, 0x44, 0xb2, 0x3d, 0xb6, 0xae, 0x69, 0xb6, 0x95, 0xd9, 0xbd, 0x7c, 0x8a, 0xa6, 0x73, 0x26, 0x58, 0x58, 0x30, 0xd5, 0x87, 0xdd, 0x6b, 0x22, 0xe9, 0x88, 0xb1, 0x4b, 0x38, 0x7e, 0xa0, 0x31, 0x63, 0x1, 0xf, 0xfc, 0xc0, 0x20, 0x41, 0xc9, 0x97, 0x14, 0x8d, 0x13, 0xcc, 0x69, 0x42, 0x57, 0xf4, 0xa5, 0x35, 0xcc, 0xf, 0xb2, 0x1a, 0x94, 0x2d, 0x3, 0xd9, 0xf5, 0x4b, 0x62, 0x89, 0x64, 0x7a, 0x68, 0x4c, 0x92, 0x64, 0x25, 0x95, 0xd4, 0x8b, 0xe6, 0xd1, 0xa6, 0x5c, 0xf0, 0x36, 0x1b, 0xaf, 0x65, 0xc, 0x9, 0xb, 0xb7, 0x22, 0x76, 0x59, 0xd9, 0xd1, 0xb0, 0xe4, 0x5b, 0x39, 0xf9, 0xd3, 0xfe, 0xbb, 0x4b, 0xdf, 0x96, 0x5f, 0x45, 0xe5, 0x8d, 0xb, 0x73, 0x69, 0x52, 0x2, 0x12, 0x43, 0x8d, 0x9d, 0x74, 0xad, 0x6, 0x7b, 0x46, 0x93, 0xed, 0x9b, 0xbb, 0xad, 0x8b, 0x6e, 0x88, 0xa1, 0xad, 0x9f, 0xb5, 0xd3, 0x13, 0xe, 0x97, 0x7c, 0x7c, 0x3a, 0xd6, 0x6f, 0xb9, 0xaa, 0x0, 0x3b, 0xec, 0x86, 0x34, 0x30, 0xec, 0xa3, 0xff, 0xf9, 0x3f, 0x33, 0xbf, 0x7e, 0x7d, 0x88, 0xfa, 0xd8, 0x82, 0x30, 0xc2, 0x8b, 0x3c, 0xcd, 0x6b, 0x6b, 0xe8, 0xf4, 0xe5, 0xdb, 0x5e, 0xcb, 0xff, 0x48, 0x3, 0x76, 0xa8, 0x4c, 0x4, 0xb9, 0xb1, 0x3c, 0x78, 0x40, 0x96, 0x0, 0x0, 0x2c, 0x6b, 0x6, 0x0, 0x8c, 0xd8, 0xdf, 0x4, 0x56, 0xa7, 0x55, 0x80, 0xfa, 0x10, 0x38, 0xc5, 0x94, 0xf1, 0xaa, 0x43, 0xae, 0x37, 0xaa, 0x4b, 0x1d, 0xf3, 0x93, 0x24, 0x44, 0xf8, 0x20, 0x87, 0xb4, 0x4f, 0xd9, 0x10, 0x53, 0xba, 0x27, 0x6d, 0x76, 0x58, 0xf2, 0x9, 0xff, 0xc1, 0xd8, 0xa, 0x5c, 0x54, 0x8, 0xf, 0x4e, 0xfb, 0x2b, 0x40, 0x6c, 0xac, 0x81, 0x9f, 0x82, 0xd8, 0xf9, 0x87, 0x58, 0xc9, 0x81, 0x7b, 0x36, 0x32, 0x20, 0xfc, 0x5f, 0x64, 0xe7, 0xd0, 0x1c, 0xc9, 0xfd, 0x6a, 0x2f, 0x89, 0x79, 0xfc, 0x79, 0x59, 0xb, 0x75, 0xec, 0x10, 0xcb, 0x17, 0x7f, 0x62, 0x5, 0x64, 0x4a, 0x39, 0xce, 0xf8, 0x74, 0x1d, 0xcb, 0x22, 0x32, 0x74, 0x21, 0x16, 0x8, 0xbe, 0xac, 0x21, 0x93, 0x67, 0x2, 0x8, 0x65, 0xcd, 0x15, 0xc5, 0x5a, 0x2, 0x58, 0x18, 0x6f, 0x5c, 0x4a, 0x16, 0x6e, 0xb1, 0x93, 0x2a, 0xaa, 0x3e, 0x64, 0x6d, 0xa4, 0x25, 0xb4, 0x29, 0x15, 0xc, 0x3b, 0x74, 0x8c, 0x7e, 0x4, 0x8a, 0xba, 0x9d, 0x8c, 0xd7, 0x64, 0xb, 0x22, 0x99, 0x21, 0x8a, 0x47, 0x65, 0xd2, 0x98, 0xfe, 0x39, 0xc, 0xbd, 0x70, 0xa0, 0x84, 0x66, 0xc4, 0xa8, 0x89, 0x51, 0x77, 0xe4, 0xa5, 0x49, 0xa8, 0x99, 0x22, 0x24, 0x42, 0xa, 0x92, 0x69, 0x8f, 0xe5, 0x35, 0x68, 0xe9, 0xa2, 0x41, 0x99, 0xca, 0xc8, 0x86, 0x48, 0xb0, 0x5b, 0x34, 0x34, 0x89, 0x21, 0x64, 0xa5, 0x38, 0x80, 0x5d, 0x17, 0x5b, 0xd0, 0xaa, 0x46, 0x46, 0x35, 0xa2, 0xce, 0x46, 0x7c, 0xb2, 0xe9, 0x28, 0x8f, 0xc6, 0x40, 0x30, 0x8b, 0x56, 0xcd, 0x64, 0x44, 0x91, 0xd, 0xaa, 0xc2, 0xae, 0xe2, 0x9b, 0x84, 0x59, 0xfe, 0xd5, 0xec, 0x51, 0x51, 0x8c, 0xaf, 0x32, 0x72, 0x5c, 0x95, 0xd4, 0xb6, 0x86, 0xd0, 0xc6, 0xd2, 0x78, 0x12, 0x50, 0x7b, 0x19, 0xa, 0x11, 0x20, 0x71, 0x80, 0x45, 0x19, 0xd, 0x3b, 0xa9, 0x5a, 0x5a, 0xe0, 0xa1, 0xb9, 0x9d, 0x18, 0x21, 0x29, 0xb2, 0x3b, 0x89, 0xaa, 0x46, 0xeb, 0x8, 0x14, 0xcc, 0x8b, 0x62, 0x67, 0x72, 0x72, 0x2d, 0x4b, 0x77, 0x5a, 0x8e, 0xcd, 0xe1, 0xd3, 0x10, 0x48, 0xd4, 0x8c, 0x10, 0xe7, 0x4, 0x29, 0x63, 0x1f, 0x81, 0x4d, 0x5e, 0x9c, 0xc9, 0x86, 0xaf, 0x5b, 0xe9, 0xe6, 0x6d, 0x1a, 0x95, 0x34, 0x22, 0xa1, 0x45, 0x1c, 0x8a, 0xf1, 0xbc, 0xf8, 0x11, 0x82, 0x98, 0xfd, 0xa3, 0x82, 0xc6, 0x49, 0xd6, 0x98, 0x64, 0x9c, 0x37, 0x9, 0x78, 0x61, 0x26, 0xb6, 0x64, 0x86, 0x52, 0x6, 0xdc, 0x8c, 0xb9, 0x55, 0x21, 0xa5, 0xd6, 0x54, 0x52, 0xb0, 0x51, 0xb7, 0x30, 0x2e, 0x7f, 0x7b, 0x26, 0x38, 0x6f, 0x2, 0x6c, 0x15, 0x58, 0x80, 0x54, 0x4d, 0x4f, 0x59, 0xd, 0xd3, 0x54, 0x38, 0x63, 0x73, 0xce, 0x3a, 0xa9, 0xcc, 0xda, 0xcb, 0x68, 0x91, 0x76, 0x19, 0xfd, 0xa, 0x4a, 0xb7, 0x7a, 0x82, 0x59, 0x8, 0xa8, 0x6e, 0x63, 0x9c, 0x1b, 0x5b, 0x8, 0xee, 0x71, 0x44, 0xc, 0x3b, 0xc, 0x8b, 0x13, 0xce, 0x26, 0x58, 0xa1, 0x1, 0xaa, 0xb, 0xb0, 0x14, 0xd6, 0x8e, 0x62, 0xfc, 0x3d, 0xaa, 0xb7, 0x6c, 0x41, 0xa0, 0x5b, 0x20, 0xd0, 0xe5, 0xa4, 0x5b, 0x74, 0x48, 0x21, 0x9a, 0x11, 0xd4, 0xac, 0xa4, 0xed, 0x52, 0xc8, 0xf3, 0x66, 0x92, 0x23, 0xb7, 0x71, 0x56, 0x1d, 0x28, 0xf4, 0x20, 0xab, 0x3f, 0x53, 0x63, 0xa0, 0x5a, 0xd, 0x60, 0x8a, 0x89, 0x49, 0x9a, 0x66, 0x6a, 0x11, 0xb8, 0x5, 0x62, 0x4d, 0x74, 0x76, 0x48, 0x49, 0xe6, 0xda, 0xb4, 0x82, 0x2b, 0x7c, 0xdc, 0x99, 0xee, 0x1f, 0x73, 0x95, 0x79, 0xd8, 0x2d, 0x65, 0x5, 0xaa, 0x41, 0x9a, 0x73, 0xed, 0x74, 0x6d, 0x36, 0xda, 0xaf, 0xa5, 0x4d, 0x9f, 0x1a, 0xd6, 0x77, 0x2, 0xac, 0x1c, 0xc1, 0xfa, 0xe9, 0x2c, 0x24, 0x15, 0xb8, 0x92, 0xc1, 0xc5, 0x5f, 0xd, 0x16, 0x14, 0xa2, 0x2d, 0x3b, 0xd, 0xf6, 0x2c, 0x7b, 0x29, 0x36, 0x50, 0x29, 0x9, 0x49, 0xfd, 0xce, 0xf0, 0x44, 0x39, 0x46, 0xe5, 0x5f, 0xf0, 0x41, 0x40, 0x5c, 0x3c, 0x72, 0x45, 0xa9, 0x37, 0x2e, 0x69, 0x63, 0xbc, 0xe3, 0x7a, 0x45, 0x13, 0x5d, 0x3d, 0x99, 0xcb, 0x3c, 0xd5, 0x38, 0xaf, 0x96, 0xab, 0x6, 0xb3, 0xd5, 0xc, 0x61, 0x22, 0x52, 0x9a, 0xdf, 0xba, 0x58, 0x93, 0xed, 0x62, 0xf5, 0x8, 0x22, 0xeb, 0x42, 0x22, 0xd6, 0x75, 0x67, 0x0, 0xcd, 0xb7, 0xab, 0x34, 0xc3, 0xad, 0xcf, 0x26, 0xdd, 0x62, 0x7f, 0x41, 0xe2, 0x6b, 0x25, 0xac, 0xce, 0x78, 0x9e, 0x29, 0xe3, 0x47, 0xf0, 0x5, 0x0, 0x89, 0x6f, 0x2f, 0x3a, 0x8, 0x82, 0xbc, 0x1d, 0x5a, 0x19, 0x61, 0x45, 0x4e, 0xdd, 0xc9, 0x21, 0x46, 0x7f, 0xd8, 0x76, 0x5e, 0x2f, 0x4f, 0xfb, 0xb9, 0x4f, 0x6b, 0xd3, 0xfa, 0x66, 0xb7, 0x3a, 0xa9, 0x20, 0x22, 0x2f, 0xae, 0x30, 0x17, 0x0, 0xc9, 0xcf, 0x7d, 0x0, 0x6c, 0x3a, 0x42, 0xa1, 0x2b, 0x80, 0xe8, 0x64, 0xc8, 0x68, 0xe5, 0x42, 0x96, 0xa4, 0x42, 0x28, 0x1e, 0x4f, 0xac, 0xb1, 0x82, 0xf8, 0x7e, 0x6, 0xb4, 0x86, 0x60, 0x9f, 0xca, 0x1e, 0x1a, 0x8f, 0xc, 0xc5, 0xeb, 0xf5, 0xef, 0x8a, 0xc4, 0x15, 0xec, 0x61, 0x7, 0x4c, 0x13, 0x30, 0x13, 0x28, 0x5a, 0x6a, 0xce, 0x3c, 0x8d, 0x4e, 0x1b, 0x87, 0x89, 0x78, 0x5a, 0x69, 0xdc, 0x26, 0x9a, 0x10, 0x83, 0xc3, 0x62, 0xad, 0x48, 0x78, 0x4b, 0xd4, 0x2b, 0xa3, 0xd8, 0xae, 0x2f, 0xb2, 0x37, 0x23, 0x9f, 0xaa, 0x25, 0x3a, 0x6, 0x2, 0x59, 0x3a, 0x12, 0x12, 0x60, 0x48, 0xf0, 0x28, 0x3d, 0xe0, 0x47, 0x8c, 0x50, 0x8a, 0x36, 0x7f, 0xc3, 0x2b, 0x14, 0x2d, 0x45, 0x98, 0x4, 0xcb, 0xb6, 0x50, 0x38, 0xfd, 0x7e, 0x4b, 0x56, 0xb2, 0xa1, 0xbe, 0x41, 0x6, 0xce, 0x11, 0x6f, 0xfd, 0xd0, 0x48, 0x6c, 0xc, 0xa4, 0x1e, 0xb0, 0x44, 0x7, 0x9, 0x53, 0x2f, 0x44, 0xa2, 0x3a, 0x9, 0x65, 0x4d, 0xe2, 0xb5, 0x49, 0x15, 0x5d, 0x11, 0x4a, 0x38, 0x9a, 0xce, 0x53, 0xca, 0x92, 0x99, 0xe3, 0x1c, 0x2a, 0x28, 0x59, 0x6c, 0xde, 0xa2, 0x8e, 0x56, 0x5f, 0x82, 0xd0, 0xb3, 0x4f, 0xbb, 0x1a, 0x5c, 0x9, 0xb5, 0xeb, 0xcf, 0xca, 0x2c, 0xa4, 0x74, 0xe4, 0xc6, 0x44, 0x4e, 0x7e, 0x73, 0x27, 0x76, 0xc, 0xda, 0x8a, 0xc, 0xf9, 0xd1, 0xb5, 0x27, 0xb, 0xae, 0x6a, 0xd2, 0x67, 0xea, 0x9, 0xc3, 0xb2, 0x96, 0x74, 0xf6, 0xa5, 0x98, 0xe, 0xc6, 0x37, 0xa0, 0x5d, 0xc4, 0x32, 0xa9, 0xd0, 0x40, 0x91, 0xfa, 0x9, 0xec, 0xe5, 0xb5, 0x2b, 0x20, 0xe7, 0xf8, 0x6f, 0xf0, 0x8c, 0x8d, 0x4f, 0x20, 0x39, 0x93, 0xc1, 0xaf, 0x13, 0x77, 0x3b, 0x1d, 0x8, 0xa6, 0x15, 0x4b, 0xc2, 0xa7, 0xdd, 0x2a, 0xa9, 0xb4, 0xc5, 0x27, 0xdf, 0x30, 0x2f, 0xea, 0x8c, 0xf3, 0xe2, 0x61, 0x16, 0x69, 0xab, 0x1a, 0x71, 0x53, 0x62, 0xc, 0x1a, 0xf, 0x83, 0xcb, 0xfa, 0xb1, 0x4e, 0x6b, 0x5e, 0xb, 0x48, 0x4, 0xba, 0x99, 0xa, 0x16, 0x72, 0x15, 0xa4, 0x27, 0xc9, 0x35, 0x1e, 0x32, 0xa2, 0xac, 0x78, 0x1a, 0xea, 0xe6, 0x42, 0xfb, 0x6f, 0x82, 0xc6, 0xa7, 0x12, 0xa8, 0xed, 0x7, 0xbf, 0x56, 0x55, 0x95, 0xec, 0x85, 0x7d, 0xc, 0x7, 0x8c, 0xde, 0x8, 0x2a, 0xa8, 0x6a, 0x23, 0xb, 0x56, 0x4, 0x7f, 0x8d, 0x7f, 0x7a, 0xa0, 0xe6, 0x18, 0xd4, 0xab, 0xbb, 0x8c, 0xa8, 0x18, 0x43, 0xf8, 0x66, 0xb1, 0x6a, 0xda, 0xdf, 0xbb, 0xc4, 0x1d, 0x71, 0x74, 0x7b, 0x52, 0xea, 0x84, 0x53, 0x23, 0x7a, 0x94, 0x33, 0x79, 0x91, 0x32, 0x65, 0xdc, 0x2d, 0x7f, 0x43, 0xcc, 0xb5, 0xa3, 0xc1, 0x51, 0x50, 0xae, 0x90, 0xe1, 0x3a, 0x2, 0xaa, 0x52, 0x8e, 0xea, 0x87, 0xad, 0xf5, 0xcb, 0xa6, 0x29, 0xac, 0xe3, 0x40, 0xb1, 0x88, 0xea, 0x40, 0x92, 0x27, 0xf8, 0xb9, 0xaa, 0x77, 0xf9, 0x48, 0x6b, 0x10, 0xa6, 0xba, 0x92, 0x3e, 0xfa, 0x66, 0xd5, 0x61, 0x45, 0x70, 0xe6, 0xf3, 0xd6, 0xed, 0x7a, 0x16, 0xc, 0x4b, 0xda, 0xca, 0x7b, 0x34, 0x68, 0x53, 0x1c, 0xa5, 0x11, 0x58, 0xe5, 0xd0, 0xb2, 0xa5, 0x9b, 0x84, 0xb2, 0x23, 0x25, 0x9c, 0x1f, 0x46, 0xe6, 0x66, 0xfe, 0x5d, 0x10, 0x14, 0xbd, 0xb, 0x0, 0xe7, 0x3e, 0x5d, 0x79, 0x78, 0x4b, 0xa4, 0x27, 0x25, 0xb6, 0x8d, 0x7, 0xd, 0xd9, 0xc2, 0xf0, 0xb4, 0x2d, 0xc9, 0x94, 0xc5, 0xcf, 0x19, 0x37, 0x6a, 0x7a, 0x0, 0xd8, 0xaa, 0xa7, 0x63, 0x79, 0x45, 0xe6, 0x23, 0x99, 0xe, 0x69, 0x53, 0x21, 0xe9, 0x2, 0x25, 0xa3, 0xe7, 0x53, 0x3, 0xe2, 0x43, 0x48, 0xd7, 0xf9, 0x1a, 0xaf, 0x7, 0x7a, 0x30, 0xbb, 0x4a, 0xeb, 0x5c, 0x9, 0x57, 0xd2, 0x81, 0xa2, 0x66, 0x20, 0xc9, 0xa3, 0x68, 0xcc, 0x12, 0x93, 0x80, 0x57, 0xe1, 0x32, 0x2, 0x56, 0x91, 0xea, 0xe7, 0x9c, 0x83, 0xf7, 0x46, 0xcb, 0xcc, 0xd2, 0xdf, 0xff, 0x94, 0xe6, 0x9c, 0xaa, 0x1c, 0xf8, 0x8c, 0x9e, 0x61, 0xb9, 0x43, 0xa2, 0xba, 0x90, 0xec, 0xcc, 0xc7, 0x8d, 0xe4, 0x65, 0xac, 0x4d, 0x3e, 0x5b, 0x9, 0x2a, 0x29, 0xfd, 0x27, 0xf9, 0x5b, 0x4a, 0x31, 0xa4, 0x26, 0x49, 0x2c, 0x9c, 0xcb, 0x69, 0xc5, 0xe6, 0x96, 0xdc, 0xaf, 0x1, 0x58, 0x30, 0x5b, 0x97, 0xd5, 0x6b, 0xea, 0x3f, 0x92, 0x4b, 0xdf, 0xe7, 0x82, 0xf0, 0x1b, 0x96, 0x2e, 0x4b, 0x28, 0xbb, 0x71, 0x48, 0xc3, 0xe6, 0xb8, 0x19, 0x7a, 0xa0, 0x4f, 0xea, 0x6d, 0xc0, 0x3a, 0x40, 0xd7, 0x9c, 0xd0, 0x8e, 0xc3, 0x36, 0xb7, 0x85, 0x46, 0xfb, 0x57, 0x94, 0x38, 0x89, 0x2, 0x58, 0x3a, 0xc7, 0x6b, 0xcf, 0x4b, 0x4, 0xfa, 0x88, 0x17, 0x10, 0x47, 0x66, 0x82, 0x26, 0x58, 0x80, 0x40, 0xf7, 0x5, 0xf, 0xe8, 0x85, 0xc3, 0x1a, 0xa, 0xda, 0xf9, 0x2a, 0xbd, 0x3d, 0x8, 0x15, 0xee, 0x5d, 0x22, 0xc6, 0xa, 0x5d, 0x58, 0x2f, 0x59, 0xc7, 0xa4, 0x23, 0x1, 0x2f, 0xef, 0xd4, 0x95, 0x83, 0x81, 0x58, 0x56, 0xb5, 0x9b, 0xbd, 0x0, 0x47, 0xf6, 0x9e, 0x86, 0xb9, 0xad, 0x20, 0x41, 0x6f, 0x79, 0x78, 0x8, 0x1b, 0xda, 0xc6, 0x61, 0x32, 0x17, 0x15, 0x87, 0xac, 0xb2, 0x49, 0x98, 0x5f, 0x78, 0x86, 0xe2, 0x5a, 0xf8, 0xbc, 0xe0, 0x96, 0xc6, 0xc2, 0x57, 0xc1, 0xca, 0x7, 0x3b, 0xe1, 0x56, 0x67, 0x81, 0x63, 0x7c, 0x9b, 0xcd, 0x6d, 0xd1, 0xd2, 0x27, 0xfd, 0x95, 0xfa, 0x12, 0xe3, 0x9, 0x84, 0x66, 0x76, 0xc7, 0xab, 0x4, 0xbc, 0xd6, 0xc0, 0xcb, 0x9, 0x25, 0xfe, 0xf6, 0x29, 0x90, 0x8f, 0xb8, 0x50, 0xd5, 0x7e, 0x13, 0x11, 0xfa, 0xfa, 0x4c, 0xe8, 0x5c, 0xa5, 0xf4, 0xa1, 0x12, 0xf5, 0x91, 0xd0, 0x98, 0x91, 0x18, 0x92, 0x4b, 0xa2, 0x92, 0x2a, 0x42, 0xb1, 0x78, 0xd4, 0x6e, 0x9f, 0xf5, 0xe8, 0x41, 0x16, 0xd4, 0x18, 0x69, 0x2e, 0xdd, 0x4b, 0x81, 0x67, 0x1c, 0x67, 0x5d, 0xb8, 0xec, 0x7b, 0x39, 0x69, 0xfb, 0x51, 0xcb, 0xdb, 0xea, 0x3, 0x20, 0x72, 0x17, 0xbc, 0x70, 0x21, 0x84, 0xe8, 0xf1, 0x5e, 0xb, 0xc2, 0xa6, 0xd2, 0x15, 0xbe, 0x7a, 0x9b, 0x7, 0x60, 0xab, 0xae, 0xa6, 0xf7, 0x4a, 0xa9, 0x4f, 0x74, 0x74, 0x50, 0x11, 0x61, 0xef, 0x1d, 0x7, 0xb8, 0x4d, 0x65, 0x6c, 0x40, 0x9e, 0x22, 0xb0, 0x22, 0xcf, 0x81, 0xf0, 0x2c, 0x36, 0xc1, 0x62, 0x48, 0x16, 0x63, 0xab, 0x86, 0x97, 0xa2, 0xd7, 0x74, 0x26, 0xac, 0xb1, 0x2f, 0x80, 0xdc, 0xe, 0x98, 0x58, 0x27, 0x24, 0x8, 0xa7, 0x46, 0x38, 0x7f, 0xca, 0x83, 0x78, 0x3c, 0x9d, 0x9e, 0x3c, 0x28, 0x4a, 0x62, 0x37, 0x10, 0xd8, 0xf2, 0xf0, 0x62, 0x4f, 0x16, 0x38, 0x3d, 0x84, 0x97, 0xdc, 0x1f, 0xc7, 0x47, 0x43, 0xf0, 0x92, 0x72, 0x48, 0x18, 0x84, 0x2d, 0x31, 0x33, 0x92, 0xc9, 0x19, 0xe4, 0xb, 0x83, 0xa7, 0x2d, 0x5, 0x6e, 0x20, 0x4c, 0x14, 0x53, 0x84, 0x78, 0x90, 0x46, 0xb7, 0x43, 0x14, 0xb6, 0x20, 0xa6, 0xb6, 0x3d, 0xf1, 0xab, 0x7, 0xe8, 0xba, 0x52, 0xfa, 0xa, 0xc, 0x33, 0xca, 0x9b, 0x1f, 0xd8, 0x41, 0x82, 0x59, 0x4a, 0xe3, 0x8f, 0x1f, 0xef, 0x4f, 0x28, 0x22, 0x29, 0x4e, 0x5, 0xb1, 0x67, 0x89, 0x1c, 0x91, 0xe1, 0xe6, 0x45, 0x31, 0x73, 0x83, 0xf8, 0x4b, 0xa, 0xa5, 0xaf, 0x48, 0x5f, 0x25, 0xb8, 0xb8, 0x8d, 0x4c, 0x6c, 0x86, 0x49, 0xf9, 0x98, 0xfe, 0x4d, 0x32, 0x39, 0x8d, 0x3c, 0x3e, 0x94, 0xc3, 0xa7, 0x36, 0x9, 0x66, 0x45, 0x80, 0x59, 0x32, 0x94, 0xfc, 0x82, 0x3f, 0x30, 0x32, 0x85, 0x41, 0x49, 0x40, 0xad, 0x2d, 0xc0, 0xe, 0xc4, 0x6e, 0x92, 0x85, 0xe5, 0x15, 0x91, 0x79, 0x3c, 0xf8, 0x5b, 0x26, 0x51, 0xfe, 0x2a, 0x24, 0xe2, 0xca, 0xfa, 0xda, 0xa3, 0x91, 0x7c, 0x8b, 0xf5, 0xd6, 0x81, 0x8e, 0xbf, 0x1f, 0xfe, 0x22, 0x65, 0x45, 0xf2, 0xa8, 0xb1, 0x33, 0xc5, 0x19, 0xcd, 0xcf, 0x28, 0x7b, 0x1c, 0xc6, 0xea, 0x76, 0x2a, 0xba, 0xf2, 0x49, 0x89, 0x5c, 0x7a, 0x4a, 0x63, 0x53, 0xb7, 0xc4, 0xce, 0x55, 0x52, 0xd8, 0x3f, 0x19, 0x96, 0x67, 0x80, 0x55, 0x4, 0x89, 0xf2, 0x27, 0xe8, 0x2d, 0x9a, 0xee, 0x5d, 0x99, 0x5b, 0x95, 0xf0, 0x82, 0x13, 0x9, 0xc6, 0x4, 0xef, 0x7c, 0xfd, 0xbc, 0x8d, 0x65, 0xbe, 0x99, 0x11, 0x40, 0x79, 0xa2, 0x58, 0x52, 0x62, 0x6b, 0x1f, 0x38, 0x4e, 0xc1, 0xcb, 0x22, 0x1b, 0x88, 0xcb, 0x22, 0xda, 0x4a, 0x64, 0xe9, 0x95, 0x5d, 0xf7, 0x12, 0x39, 0x7, 0xcc, 0x83, 0x4b, 0x5d, 0x8e, 0xea, 0x62, 0x45, 0xc3, 0xbc, 0xdd, 0x72, 0xa9, 0xf, 0xfa, 0xa6, 0xba, 0xe, 0x66, 0xbe, 0xcd, 0x54, 0x86, 0xb, 0x49, 0x5, 0xdc, 0x47, 0x8c, 0x3e, 0x43, 0x1a, 0xd6, 0xbe, 0x44, 0xfe, 0xf5, 0x5e, 0xe9, 0x44, 0xca, 0xdf, 0x4e, 0x1e, 0x7b, 0x3e, 0x11, 0x76, 0xed, 0xd9, 0x1f, 0x1e, 0x6e, 0x75, 0x68, 0xc4, 0x6e, 0xd3, 0xdb, 0xe5, 0x1f, 0x6f, 0xf0, 0x75, 0x50, 0xb7, 0x9f, 0x3d, 0xca, 0x72, 0x63, 0xd0, 0xad, 0xec, 0x70, 0xbd, 0x25, 0x6e, 0x7a, 0xfb, 0xe6, 0xa5, 0x4f, 0xb5, 0x75, 0xdb, 0x57, 0x3e, 0x69, 0x23, 0xa4, 0x73, 0x6d, 0x91, 0x58, 0xf7, 0x73, 0x34, 0x84, 0xd5, 0x12, 0xab, 0x83, 0xd1, 0x36, 0x5c, 0xeb, 0x53, 0x6f, 0x70, 0xda, 0xa6, 0xd9, 0x40, 0x5c, 0x88, 0x6f, 0xb1, 0x53, 0xcd, 0x43, 0x29, 0x29, 0x65, 0xa7, 0xff, 0x25, 0xe7, 0x81, 0x45, 0x84, 0x9e, 0x50, 0x15, 0x12, 0xe6, 0xd7, 0x1d, 0x3c, 0xd8, 0x76, 0xa6, 0x0, 0x8f, 0x39, 0x1e, 0x54, 0x81, 0x7f, 0x55, 0x55, 0x64, 0x4a, 0xf7, 0xcb, 0xe8, 0x8b, 0x61, 0x2b, 0x4a, 0xc, 0x28, 0xe3, 0x35, 0xf9, 0x9c, 0x8f, 0x55, 0x60, 0x8a, 0xf0, 0xae, 0x92, 0xd3, 0x6d, 0x4d, 0x6e, 0xe9, 0xd5, 0x9, 0xc7, 0xf5, 0x1c, 0xf, 0x8c, 0x29, 0x9b, 0x46, 0x6b, 0x1a, 0x58, 0x84, 0xe8, 0x82, 0xe4, 0x8, 0x8c, 0x9f, 0x2a, 0xa1, 0x8a, 0x58, 0xed, 0x74, 0x8f, 0x55, 0x76, 0xad, 0x93, 0xbb, 0x4a, 0x9e, 0xb6, 0x32, 0x48, 0x35, 0x65, 0x26, 0x77, 0x89, 0x35, 0xb6, 0x47, 0xf9, 0x57, 0xe6, 0x78, 0x58, 0x31, 0x71, 0x28, 0x9f, 0x58, 0x23, 0x53, 0x6a, 0x9, 0xa, 0x3f, 0x79, 0x1d, 0xd2, 0xf5, 0x4a, 0xa, 0x38, 0x5c, 0xd, 0xb1, 0x90, 0x1a, 0x3e, 0x75, 0x9e, 0xc2, 0x6e, 0x27, 0x1d, 0x6e, 0xfc, 0x14, 0xee, 0xc5, 0xee, 0xd7, 0xe7, 0x28, 0x2b, 0xa6, 0x2f, 0xe7, 0xaf, 0xc4, 0x41, 0x4c, 0x8a, 0x94, 0x76, 0x94, 0x12, 0x37, 0x99, 0xda, 0x58, 0xca, 0xde, 0xc7, 0x7, 0x9, 0x71, 0x37, 0x8d, 0xa2, 0x97, 0x97, 0x1c, 0xc, 0x51, 0xde, 0xe5, 0xb7, 0xb8, 0xda, 0x92, 0xa3, 0xef, 0x17, 0x99, 0x6a, 0x1b, 0xf9, 0x9a, 0xbd, 0xcb, 0xfe, 0x5c, 0xc3, 0xf6, 0xda, 0xb1, 0x86, 0x16, 0x30, 0x97, 0x47, 0x27, 0x14, 0x30, 0x92, 0xc1, 0x80, 0xcb, 0x46, 0x27, 0xdb, 0x3d, 0x9d, 0x1d, 0x9a, 0x6, 0xb6, 0x83, 0xc6, 0x12, 0x94, 0x77, 0x36, 0xda, 0xc5, 0xb0, 0x85, 0xa8, 0x5b, 0x10, 0xc1, 0x90, 0xbd, 0xb5, 0xc5, 0x83, 0x51, 0x13, 0x24, 0xb, 0x12, 0xb8, 0x3f, 0xdf, 0xd9, 0xa2, 0xf1, 0x77, 0xfc, 0x7d, 0x46, 0xcf, 0x38, 0xa2, 0x78, 0xff, 0xda, 0x3c, 0x20, 0x3d, 0x5, 0x9e, 0xe4, 0x7a, 0xe, 0x30, 0x1, 0x56, 0xfc, 0xa2, 0x43, 0xcc, 0x27, 0x6f, 0x49, 0x9b, 0xe8, 0x65, 0xc9, 0xe7, 0x65, 0x82, 0x4e, 0xc8, 0x2c, 0x0, 0x79, 0xb7, 0x70, 0x0, 0x1, 0x77, 0x2d, 0x34, 0x46, 0xea, 0x34, 0x95, 0xc4, 0x26, 0xdd, 0xff, 0xa, 0x7f, 0xcb, 0x48, 0x8, 0xd, 0xb1, 0x17, 0x40, 0xaf, 0xb3, 0x73, 0x76, 0xa2, 0x81, 0x18, 0x87, 0x6d, 0x7d, 0x4b, 0x6b, 0xa, 0x15, 0x68, 0xbd, 0xc7, 0x22, 0xea, 0xea, 0xc2, 0x4, 0xb, 0x1a, 0x81, 0x43, 0xf9, 0x85, 0x75, 0x38, 0xde, 0x2a, 0x7a, 0xb8, 0xb3, 0xbf, 0x4d, 0xa6, 0x3d, 0x14, 0x33, 0xc4, 0x6c, 0x1c, 0x26, 0x1c, 0xb1, 0x33, 0xc2, 0xf5, 0xd6, 0xd9, 0x11, 0xf8, 0x54, 0xb6, 0xa8, 0x4, 0xa, 0x77, 0xf7, 0xb7, 0xf2, 0x53, 0x2e, 0x21, 0x6e, 0x84, 0xc4, 0x9c, 0x26, 0xe0, 0x72, 0x9f, 0x70, 0xca, 0xa8, 0x8e, 0x55, 0xdf, 0x19, 0x97, 0xf1, 0x4c, 0xeb, 0xe3, 0x53, 0xab, 0x25, 0x4f, 0x2e, 0x6c, 0x5a, 0x1c, 0x72, 0xde, 0x57, 0x7c, 0xf2, 0x3b, 0xbb, 0x2b, 0x81, 0xed, 0xb0, 0x7f, 0xac, 0x5f, 0x76, 0x1f, 0x5c, 0x20, 0xf7, 0xa8, 0x56, 0x52, 0xad, 0x53, 0xa8, 0xaf, 0x43, 0x5e, 0xe6, 0xff, 0xdf, 0x92, 0x8e, 0x24, 0xfb, 0x45, 0x7d, 0xa6, 0x78, 0x87, 0x44, 0xcf, 0xda, 0xac, 0xe3, 0x5b, 0x5a, 0xe6, 0x91, 0xcd, 0xfb, 0xa1, 0x29, 0x26, 0x2d, 0x3f, 0xae, 0xe4, 0x3d, 0x77, 0xb9, 0xda, 0xba, 0x6b, 0x6f, 0x47, 0xe5, 0xde, 0x3d, 0xd7, 0x51, 0xab, 0x71, 0xeb, 0x36, 0xd6, 0x1, 0x9e, 0x5c, 0x1b, 0xdc, 0xbc, 0x51, 0xf9, 0x7f, 0xe2, 0xf4, 0x6b, 0x4d, 0x87, 0x77, 0xde, 0xcf, 0x38, 0x9c, 0xfc, 0x70, 0xee, 0xf5, 0xca, 0x97, 0xeb, 0xe4, 0xb8, 0x14, 0xb, 0xf8, 0x2c, 0xdd, 0x30, 0xde, 0xdf, 0xc1, 0xa5, 0xd0, 0x54, 0xde, 0xdd, 0x7e, 0x84, 0x29, 0x51, 0x45, 0x45, 0x4e, 0xeb, 0x9f, 0xb, 0xb5, 0xac, 0x5a, 0x58, 0x71, 0xd0, 0x66, 0xf7, 0x50, 0x3f, 0x9e, 0x56, 0x2a, 0x40, 0x4d, 0xc6, 0x98, 0xb5, 0xb6, 0xef, 0xa, 0x56, 0x28, 0x50, 0xa5, 0x59, 0xe7, 0xba, 0x4, 0xb, 0x15, 0x98, 0x4, 0xc, 0x18, 0x69, 0x56, 0x1f, 0xa1, 0x2b, 0x30, 0x46, 0x90, 0xcd, 0xa4, 0xd4, 0x6f, 0x7e, 0xe9, 0xac, 0x16, 0x6d, 0xf4, 0x69, 0xfa, 0x42, 0xc9, 0xe6, 0xc2, 0xab, 0x7f, 0x98, 0x58, 0x60, 0xc7, 0xe7, 0x7a, 0xd5, 0xb2, 0xe3, 0xeb, 0xc8, 0xae, 0x2e, 0xaa, 0x2f, 0xdd, 0xcb, 0x89, 0x63, 0xc8, 0x91, 0x72, 0xd, 0xef, 0x3d, 0xf1, 0xf3, 0xee, 0x4b, 0xd5, 0xc1, 0xcb, 0xdf, 0xc4, 0xb0, 0xa, 0xa1, 0xb9, 0x96, 0xae, 0x49, 0x10, 0x51, 0x4b, 0x17, 0xe4, 0x3f, 0x37, 0x25, 0x6f, 0x69, 0x4e, 0x6e, 0x28, 0xa4, 0xa3, 0x52, 0x1d, 0xc7, 0x6, 0xfe, 0x65, 0xfd, 0x5f, 0xb9, 0xd6, 0xf4, 0xfb, 0x26, 0x24, 0x0, 0xb, 0x43, 0xe6, 0xbb, 0x6c, 0xf0, 0x1, 0x51, 0x13, 0x8f, 0xb, 0x7a, 0x15, 0x7b, 0x61, 0x80, 0xb3, 0xbf, 0x1, 0xe, 0x3a, 0x16, 0xf7, 0x1a, 0x2f, 0xe7, 0xe6, 0x2, 0x57, 0xfc, 0x4, 0x36, 0xc0, 0xb7, 0x9, 0xc3, 0x42, 0xce, 0xdc, 0x63, 0x7, 0x5, 0x83, 0x66, 0x44, 0x1f, 0x44, 0xc0, 0x5e, 0xdf, 0x81, 0x5, 0x10, 0x16, 0x4, 0x22, 0xbc, 0x93, 0xe, 0x2c, 0x42, 0x23, 0x31, 0xcc, 0x80, 0x20, 0x17, 0x45, 0xdc, 0xee, 0x4, 0x46, 0x5e, 0xe8, 0x99, 0x63, 0x7, 0x3d, 0xc7, 0xae, 0x17, 0x8, 0xb2, 0xb1, 0x1, 0x2, 0xfd, 0xf4, 0x4, 0x67, 0xeb, 0x2e, 0xa7, 0x4e, 0x86, 0x7b, 0xa7, 0x1d, 0x8, 0xc1, 0xfb, 0xc0, 0xa, 0x2f, 0x81, 0xea, 0x23, 0x42, 0x3a, 0x85, 0xbe, 0x86, 0x16, 0xfa, 0x1a, 0x51, 0xa6, 0xcc, 0x2c, 0x56, 0x64, 0x25, 0x38, 0xe8, 0x17, 0x71, 0xa8, 0x59, 0x18, 0x44, 0xdf, 0x36, 0xa4, 0xd9, 0x64, 0xd0, 0x8a, 0x3a, 0xc5, 0xca, 0xc4, 0x5e, 0x6e, 0x2, 0x43, 0xa7, 0x83, 0xf0, 0x0, 0xdf, 0xa0, 0xb9, 0x3f, 0xcb, 0x8, 0x1b, 0x34, 0x83, 0xb1, 0x91, 0x2e, 0xa4, 0xc0, 0x66, 0x19, 0xfa, 0x29, 0x40, 0x90, 0x93, 0x10, 0xde, 0xa0, 0xb6, 0xf5, 0x99, 0xa0, 0x55, 0x8, 0xd3, 0xd0, 0x5a, 0x78, 0xdc, 0x7f, 0xd4, 0x9e, 0x4e, 0x41, 0x10, 0x3e, 0xa9, 0xf9, 0xe, 0xf4, 0x21, 0x4d, 0x7d, 0xc0, 0x6d, 0x11, 0xb, 0x84, 0x8a, 0x6b, 0xb5, 0x12, 0xd3, 0xd6, 0x88, 0x4a, 0x69, 0x74, 0x60, 0x67, 0x8e, 0x11, 0xb, 0xfd, 0x48, 0x8e, 0x20, 0xd7, 0xed, 0x31, 0x63, 0x65, 0x77, 0xda, 0xc6, 0xa8, 0x4b, 0x4f, 0x99, 0x1a, 0x15, 0xd0, 0x4b, 0x72, 0x39, 0xa7, 0xcf, 0x13, 0xa7, 0x4d, 0xc8, 0xa, 0xee, 0x5d, 0xfc, 0x64, 0x5e, 0xb3, 0x48, 0x2f, 0xea, 0xa, 0xfa, 0x19, 0x86, 0x6e, 0x54, 0x7b, 0x74, 0xec, 0x83, 0x8, 0x97, 0xca, 0x9d, 0xea, 0xdb, 0xa1, 0x9, 0x50, 0x15, 0xd1, 0xfd, 0x2, 0xee, 0x16, 0x67, 0x98, 0x70, 0xb3, 0xa4, 0x16, 0x41, 0x6d, 0x11, 0xde, 0xe9, 0x21, 0xb1, 0xa1, 0x2c, 0x6a, 0xad, 0xf8, 0x11, 0x1b, 0x41, 0x2b, 0x96, 0xb1, 0xb3, 0xd4, 0x47, 0xac, 0x29, 0x93, 0xb1, 0xde, 0x5b, 0x35, 0x5e, 0x74, 0x4, 0xe5, 0x54, 0x95, 0xb5, 0xf9, 0x77, 0xe1, 0x64, 0x2, 0x97, 0x61, 0x96, 0xef, 0xcb, 0x64, 0xfe, 0x31, 0x5c, 0x3c, 0x24, 0x21, 0xbb, 0xae, 0x17, 0x35, 0xc2, 0x6d, 0xd2, 0xe1, 0x9a, 0x6e, 0xc0, 0x21, 0x2, 0x1, 0x27, 0x10, 0xb, 0x8e, 0x1, 0x63, 0x5d, 0xa6, 0x5e, 0x46, 0xcc, 0x3c, 0x9e, 0x2d, 0xed, 0x92, 0xc, 0xd8, 0xcd, 0x33, 0xb4, 0xf2, 0xc5, 0x30, 0xff, 0x2f, 0xbc, 0xfd, 0x7, 0x8e, 0x1f, 0xfa, 0x9d, 0x43, 0xf6, 0x43, 0x87, 0x2b, 0xb3, 0x2, 0xa4, 0x3b, 0x85, 0x18, 0x43, 0xf6, 0x1f, 0xe5, 0x40, 0x28, 0x5b, 0xda, 0x70, 0xff, 0xb2, 0xf5, 0x97, 0xf8, 0xb0, 0xff, 0xa4, 0xdf, 0xa1, 0xf8, 0x2a, 0x3d, 0x1d, 0x7d, 0x3, 0x6f, 0x7e, 0x34, 0x74, 0x9e, 0xd2, 0x6, 0xa2, 0x95, 0xa9, 0x65, 0x28, 0x51, 0xcb, 0x6e, 0x1f, 0x1a, 0x8a, 0x73, 0xc7, 0x53, 0x9d, 0x37, 0xf1, 0x10, 0x5a, 0x0, 0xdc, 0x37, 0x82, 0xd3, 0x2, 0xbd, 0x45, 0x22, 0x23, 0xd1, 0x4a, 0xe4, 0x9d, 0x74, 0x95, 0x3, 0x38, 0xa8, 0x50, 0x75, 0x4b, 0x2c, 0x2a, 0x60, 0x82, 0xa6, 0x88, 0x52, 0x38, 0x9d, 0x56, 0x21, 0xa4, 0x5, 0x8a, 0x7f, 0x38, 0x9d, 0x32, 0x3a, 0x31, 0x52, 0xfa, 0x25, 0x57, 0x4b, 0xf, 0xb5, 0x53, 0xa7, 0xc2, 0xe1, 0x55, 0x20, 0xf4, 0x61, 0x74, 0x50, 0xab, 0xa9, 0x18, 0x9b, 0x39},
	},
	sample{ // From danplanet.com (I agree Dan. Radio amateurs should use well defined open standards, not LZHUF... It's 2015!).
		[]byte(`Winlink 2000 is a hot topic in the amateur community right now.  It provides a mechanism to access Internet email over several different transports, including an Internet connection, local V/UHF packet radio, and long-distance HF Pactor.  The idea is great, and in practice, it works reasonably well.  There are a lot of complaints to be made about the design and implementation of the system, but at the end of the day, those guys put in the time, effort and money and made it work.

One of the (many) complaints I have is their use of the ancient B2F forwarding protocol.  It’s fine to use that over slow Pactor links (I suppose), but why aren’t we just using something like POP3 for the Internet hops?  Rather silly, I think.  Anyway, one of the design points of the B2F protocol is the use of an even more ancient compression algorithm called “lzhuf”.

The algorithm and code for this was written in Japan in 1988 to run on a 16-bit machine.  The source, and many disparate alterations since then are sprinkled around the Internet and are easy to find via google.  However, most people use either the command-line LZHUF_1.EXE file that has been around forever, or the DLL-ized version that the Winlink applications deliver and use.  This effectively limits its use to Windows machines (and dosemu-installed Linux boxes).  When I tried to compile several of the variants under Linux, I found that the code makes a bunch of assumptions about type sizes and thus crashes and/or fails to decode compressed text as a result.  In fact, depending on where it fails, sometimes it runs off in an endless loop writing garbage to the output file until you kill it!

After spending a lot of time trying to find someone who had fixed the code to compile on a modern 32-bit system, I finally found a copy of the source that would compile on my system with g++ and actually run properly. I found the updated source code here and have archived a copy of the source on my system, as well as a static Linux binary in case you don’t want to compile it yourself.  If you run the binary with no arguments, it will print a usage message.

Now, you might ask “Dan, why does Winlink 2000 use this old, unmaintained, fragile, and obscure compression algorithm?”.  Well, in the days of freely available code, algorithms, and libraries to do advanced compression, encoding, etc, I can assure you that the top notch Winlink engineers have a good reason.  ...Right?  I figured that this obscure gem from the golden age of 4MHz PCs must be an undiscovered compression miracle, one that makes the extremely slow Pactor connections able to transfer data as efficiently as possible.  So, I decided to compress some test files with lzhuf, as well as the freely-available-and-well-regarded gzip and bzip2 algorithms and compare the results.

As input, I used the lzhuf source code itself, which is about 19KB in size.  That’s a pretty good sized email even with a potential file attachment.  Below are the results:

...

So there you go: with bzip2, you’d get almost a kilobyte less data to transfer than you would using lzhuf.  Does a kilobyte really matter?  Well, Pactor-I is 200 baud (at most), with very small block sizes.  Yes, I think I’d rather save that kilobyte.

So, I ask the Winlink 2000 developers: Why not move to bzip2 compression?  It’s free.  It’s widely available.  It’s considered one of the best.  You put “2000″ in your name to sound like the system is new, fresh, and modern, why not use a compression algorithm to match?
`),
		[]byte{0x75, 0xfb, 0xd8, 0xd, 0x0, 0x0, 0xf1, 0xfd, 0x7f, 0x5f, 0x8c, 0x4c, 0x3f, 0xbe, 0xb3, 0x7d, 0xbc, 0xbe, 0xb9, 0x81, 0x73, 0xff, 0x6f, 0xdb, 0x73, 0xe9, 0xfb, 0x0, 0xae, 0xc6, 0xd3, 0xf9, 0x77, 0xdf, 0x5d, 0x36, 0x2b, 0xab, 0x6b, 0xf1, 0xae, 0xdb, 0xf3, 0x3e, 0xad, 0x40, 0xf, 0xfa, 0x55, 0x33, 0xe9, 0x65, 0x52, 0x58, 0x9e, 0x40, 0x58, 0xcd, 0xcf, 0xf3, 0x95, 0x24, 0x60, 0xa2, 0x7, 0xba, 0x8c, 0x7a, 0xb2, 0x46, 0xca, 0x13, 0xc0, 0x53, 0xfc, 0x1f, 0x55, 0x47, 0x94, 0x20, 0xc0, 0x99, 0x4d, 0xf5, 0xdb, 0xb7, 0x6e, 0x20, 0xa0, 0xc4, 0x45, 0x58, 0x70, 0xfa, 0x12, 0x8f, 0x48, 0xc4, 0x77, 0x9d, 0x41, 0xba, 0xdb, 0x4b, 0xc2, 0x36, 0x23, 0x48, 0x91, 0xa4, 0x57, 0xe0, 0x2b, 0x6b, 0x8e, 0x31, 0x3f, 0x22, 0xd4, 0x2c, 0x81, 0xa8, 0x2c, 0x98, 0x70, 0x6a, 0xa2, 0x61, 0xa1, 0xaf, 0xb8, 0x6c, 0x5c, 0x22, 0xdd, 0x21, 0xf4, 0x1a, 0x5c, 0x82, 0x7b, 0x28, 0xf9, 0x56, 0x4e, 0x93, 0xaa, 0x33, 0xfc, 0x95, 0xc1, 0x7, 0xff, 0x41, 0x1e, 0xc0, 0xb7, 0x70, 0x1e, 0xa7, 0x48, 0x57, 0xff, 0xf6, 0x8, 0x64, 0xef, 0x9f, 0xac, 0x3f, 0xe1, 0x8b, 0x18, 0xf6, 0x40, 0x3a, 0xa0, 0xdb, 0x9f, 0xa7, 0x60, 0x22, 0xe7, 0xa6, 0xe, 0x84, 0x4c, 0x41, 0x7a, 0x3f, 0x7c, 0xe8, 0xed, 0x40, 0x76, 0x87, 0xf8, 0x75, 0x11, 0xfd, 0x2a, 0x2f, 0xd9, 0x88, 0x3c, 0x8, 0xc3, 0xd4, 0xa7, 0x9f, 0x61, 0x11, 0x9f, 0xf3, 0x8, 0xbd, 0x1f, 0xa5, 0xbf, 0xfb, 0xb7, 0x8f, 0x10, 0x3f, 0xc7, 0xfb, 0x2b, 0xbb, 0xbc, 0xd7, 0xb8, 0xf3, 0xdc, 0x11, 0x21, 0xf7, 0xd7, 0x52, 0x1d, 0x40, 0x3c, 0x14, 0x83, 0xf0, 0x39, 0xc2, 0x29, 0x5e, 0x9a, 0x5a, 0xda, 0xaf, 0x14, 0xd3, 0x2e, 0x6e, 0x3d, 0xda, 0xfe, 0x32, 0x1e, 0xa5, 0x2, 0xdc, 0xac, 0x6d, 0x15, 0xbb, 0x52, 0xf4, 0xd5, 0x9e, 0x42, 0x97, 0x62, 0x9c, 0x7f, 0xda, 0x59, 0x7d, 0x68, 0x27, 0x21, 0xe7, 0xc1, 0x7, 0x1a, 0xfc, 0x53, 0x13, 0xa2, 0xf7, 0xd6, 0xc6, 0x17, 0xeb, 0x19, 0xe3, 0x68, 0x19, 0x60, 0x3b, 0xfa, 0xfd, 0x55, 0xb6, 0x26, 0x8f, 0x19, 0xf6, 0xaa, 0xf0, 0xf9, 0xf7, 0x4d, 0x89, 0x75, 0xcc, 0x5e, 0x93, 0xbb, 0x67, 0x6e, 0x7e, 0xe5, 0x71, 0x5a, 0x40, 0x13, 0x6c, 0x23, 0x6c, 0xe3, 0xac, 0x9, 0x84, 0xc3, 0xc4, 0x41, 0xcf, 0x7f, 0x6d, 0x13, 0xe, 0xd9, 0x2c, 0x91, 0xed, 0x4c, 0x2b, 0x6d, 0x3e, 0xa9, 0x5c, 0xf5, 0xb8, 0x76, 0xf5, 0xa7, 0xd8, 0xb7, 0x67, 0xfd, 0x2c, 0x4f, 0x57, 0xb8, 0xa5, 0x57, 0x30, 0xb1, 0x95, 0xf4, 0xe8, 0x63, 0x60, 0xe2, 0x34, 0x67, 0x62, 0x94, 0x2d, 0x75, 0xe5, 0xb1, 0x46, 0x6d, 0xf1, 0x86, 0x70, 0xa1, 0x48, 0x70, 0x44, 0x65, 0xf4, 0x6d, 0xbd, 0xad, 0x29, 0xb1, 0x42, 0xc9, 0xf5, 0x3, 0xb9, 0x65, 0x74, 0x16, 0xa3, 0x64, 0x8a, 0xd, 0x9a, 0xaf, 0xa5, 0xb5, 0xf2, 0x21, 0x78, 0x6d, 0x15, 0x39, 0x3a, 0x1e, 0xc8, 0x6f, 0xa, 0xca, 0x61, 0x37, 0x43, 0x1e, 0xc, 0xdf, 0xdc, 0x85, 0x70, 0x65, 0xd7, 0x5a, 0xce, 0x99, 0x39, 0xb, 0x81, 0x4a, 0x96, 0xd9, 0x79, 0x21, 0x81, 0x5f, 0x4, 0x4, 0x5d, 0x45, 0x39, 0x29, 0xda, 0xbe, 0x46, 0x24, 0xea, 0xb, 0xf4, 0x3d, 0x24, 0x5a, 0x8a, 0x41, 0x68, 0x9e, 0xd4, 0xd8, 0xfb, 0x62, 0xc2, 0x11, 0x54, 0xea, 0xb9, 0xf4, 0x21, 0x30, 0x57, 0x4c, 0x1c, 0xe8, 0x5d, 0xc6, 0xd1, 0x56, 0x4, 0xa9, 0x8c, 0x32, 0xd0, 0xcc, 0xc3, 0xd1, 0x6e, 0x22, 0x78, 0xf9, 0x28, 0x2a, 0x41, 0xc8, 0x45, 0x63, 0x28, 0x95, 0xc5, 0xf, 0x30, 0x6, 0x61, 0x5b, 0xa5, 0x6e, 0x19, 0xba, 0xd9, 0x52, 0x7, 0x6e, 0x83, 0xa8, 0xb6, 0x76, 0x8d, 0x19, 0xac, 0x34, 0x6e, 0x6a, 0xd3, 0x25, 0x41, 0x29, 0xec, 0x62, 0x63, 0xf1, 0xdb, 0x65, 0xdd, 0x52, 0xc3, 0x4a, 0x37, 0x36, 0xdb, 0x65, 0x7d, 0x9d, 0xf8, 0xe4, 0xc4, 0x4a, 0x70, 0xd4, 0xbe, 0x58, 0xe0, 0xe8, 0x4c, 0xaf, 0x4c, 0x11, 0x90, 0x1d, 0xf, 0xdb, 0x45, 0x1d, 0x40, 0x6d, 0x8a, 0x2b, 0x79, 0x63, 0xe0, 0x8d, 0xc7, 0x11, 0xc4, 0x1d, 0x4a, 0x5a, 0x43, 0x2b, 0xdd, 0x4f, 0x94, 0xa9, 0x28, 0xd5, 0xb6, 0xa2, 0xbd, 0xa9, 0x49, 0x1b, 0x3d, 0x4a, 0x48, 0x28, 0x96, 0x58, 0x18, 0x35, 0x43, 0x31, 0xdb, 0xa3, 0x42, 0xf9, 0x50, 0x2a, 0xb1, 0x27, 0xcd, 0x87, 0x9a, 0xff, 0x26, 0xe2, 0x9f, 0x49, 0x7d, 0x4c, 0x2e, 0x42, 0x3d, 0x51, 0xfc, 0x8, 0xf8, 0x80, 0x97, 0x88, 0x14, 0xfb, 0xe, 0xb8, 0x91, 0xe8, 0x7, 0xd9, 0xd6, 0xad, 0x37, 0x1c, 0x6e, 0x12, 0x88, 0xfb, 0xed, 0xe6, 0x8, 0x6b, 0x9e, 0xcb, 0x4d, 0x71, 0x2b, 0xce, 0x77, 0x8e, 0xa0, 0x5c, 0xc5, 0xdf, 0xe2, 0x13, 0x7e, 0x87, 0xd4, 0xa7, 0xb2, 0xe7, 0x64, 0xf1, 0x15, 0xff, 0x55, 0x7e, 0x82, 0x48, 0x1b, 0xc6, 0xdd, 0xde, 0xe8, 0xb4, 0x41, 0x42, 0xe1, 0x4b, 0xbb, 0x3b, 0xb1, 0xfe, 0xfe, 0xd4, 0xa, 0xee, 0x72, 0x8f, 0xe4, 0x4a, 0xfe, 0x26, 0x81, 0x1c, 0x5, 0x89, 0xdd, 0x21, 0x61, 0x3a, 0x29, 0xb6, 0xa6, 0x9f, 0x63, 0x14, 0x29, 0x3f, 0x64, 0xb7, 0x96, 0xcb, 0x12, 0x38, 0x74, 0xee, 0x2b, 0x9d, 0x83, 0x94, 0x51, 0x5f, 0x65, 0xd5, 0xe1, 0xed, 0x6d, 0xbf, 0xdb, 0xd5, 0xae, 0x1d, 0xbf, 0x55, 0xdf, 0xe9, 0x24, 0xf1, 0xae, 0x59, 0xaa, 0x5b, 0xd4, 0x85, 0x8, 0x11, 0xf2, 0x6b, 0x84, 0xf3, 0x9, 0x20, 0xe5, 0x1a, 0xeb, 0xec, 0x4e, 0x37, 0x4c, 0x99, 0x81, 0xba, 0xf0, 0x6, 0x6c, 0x2a, 0xbf, 0x2b, 0xff, 0xd5, 0xfa, 0x1f, 0x55, 0x6b, 0x8, 0x45, 0xc6, 0x44, 0x60, 0x84, 0x39, 0x19, 0xd8, 0xc4, 0xa, 0x42, 0x51, 0xa9, 0xaa, 0xd9, 0x9, 0x41, 0x4, 0x8f, 0x15, 0xfe, 0x8f, 0xf1, 0x4, 0xc5, 0x77, 0x8b, 0x24, 0x3e, 0x6f, 0xec, 0xd, 0xd2, 0xa8, 0xb9, 0x29, 0xdd, 0x45, 0xbc, 0x6b, 0x5b, 0xdc, 0xed, 0x79, 0x58, 0x17, 0xb7, 0x68, 0x64, 0xae, 0x32, 0x6c, 0x26, 0x42, 0x76, 0xf, 0x52, 0x9, 0x7a, 0x8, 0x86, 0xac, 0xe4, 0x32, 0xb6, 0xdd, 0x56, 0x82, 0xbd, 0x54, 0xe4, 0x85, 0x4f, 0xe0, 0x63, 0x57, 0x3b, 0xbd, 0xa3, 0x4a, 0x52, 0x77, 0x2c, 0x9, 0x44, 0x54, 0x89, 0x8, 0x62, 0xcd, 0x28, 0x47, 0xf1, 0x5, 0xd2, 0x3f, 0xb0, 0x85, 0x4, 0xcb, 0x36, 0x3, 0x6c, 0x3a, 0x54, 0x40, 0xd8, 0xd9, 0xf3, 0xb, 0x85, 0x1d, 0x39, 0xb1, 0xce, 0x34, 0x4d, 0xdb, 0xa4, 0x30, 0xe6, 0x9d, 0x83, 0xdb, 0xbb, 0x19, 0x18, 0x66, 0x6c, 0xd, 0xd, 0x62, 0x69, 0xcf, 0xb2, 0xd6, 0x57, 0x2f, 0x1d, 0xeb, 0x90, 0xbe, 0xad, 0xc8, 0xc1, 0x5e, 0xe6, 0x20, 0xd3, 0x5f, 0x55, 0xbf, 0xa0, 0x77, 0xe4, 0xd7, 0x7, 0xe1, 0x5f, 0xe4, 0x68, 0x68, 0x6c, 0x17, 0xc7, 0xb5, 0xad, 0x97, 0x11, 0x18, 0x26, 0xbf, 0x15, 0xfe, 0xa3, 0x7d, 0x1d, 0x2d, 0xe0, 0xac, 0xd2, 0x7d, 0x5, 0x5a, 0xf8, 0xdd, 0xeb, 0x1f, 0xe7, 0x0, 0x5c, 0xad, 0x5f, 0x78, 0x8c, 0xd5, 0x96, 0x11, 0x36, 0xf1, 0x79, 0x63, 0xdd, 0x26, 0x14, 0x18, 0x26, 0xa1, 0x83, 0xc7, 0xb5, 0x90, 0xad, 0xf6, 0xb8, 0xb9, 0x85, 0x6, 0xb, 0x80, 0xc1, 0x42, 0x81, 0x39, 0x9b, 0x51, 0xab, 0x77, 0xcd, 0xd9, 0x50, 0x4c, 0x11, 0xa9, 0xeb, 0x10, 0xe2, 0x61, 0x5e, 0x49, 0x9d, 0xac, 0x29, 0xec, 0x7a, 0x87, 0xe1, 0x79, 0x27, 0x53, 0xed, 0xa5, 0xb0, 0xa, 0x45, 0xcd, 0x5f, 0x39, 0x98, 0x45, 0x9e, 0x16, 0xb4, 0xfc, 0x21, 0x71, 0xfe, 0x4a, 0x97, 0xe6, 0x3b, 0xc6, 0xb8, 0x6e, 0xb3, 0x61, 0xe, 0x49, 0x5f, 0xfe, 0x62, 0x17, 0xbc, 0x5c, 0x6a, 0xee, 0xc0, 0x7f, 0xd4, 0xfd, 0x25, 0xd8, 0x79, 0x97, 0xc4, 0x82, 0xcd, 0x6b, 0xa4, 0x8d, 0x2f, 0xce, 0xf0, 0x6, 0x7a, 0xad, 0xa5, 0xa7, 0x3a, 0x40, 0xe6, 0x55, 0x50, 0x1f, 0x89, 0xef, 0x9b, 0x9a, 0x78, 0x4d, 0x71, 0xe5, 0x17, 0x3, 0x6e, 0xdc, 0x2, 0xeb, 0x59, 0x94, 0x39, 0x8c, 0xf3, 0x29, 0x78, 0xe7, 0xbd, 0xd1, 0xa3, 0xc4, 0x94, 0x4d, 0x3d, 0xe3, 0x9d, 0xe2, 0xe7, 0x34, 0x23, 0x68, 0xd7, 0xad, 0x2b, 0x1e, 0xc2, 0xd2, 0x31, 0x2a, 0xea, 0x6a, 0xae, 0xda, 0xfb, 0x81, 0x74, 0x15, 0x44, 0xf1, 0xd1, 0x18, 0x4f, 0xa5, 0xbe, 0xfa, 0x55, 0xc6, 0x6c, 0x4b, 0x17, 0x54, 0x5, 0x60, 0xa3, 0xe4, 0x3a, 0x52, 0x64, 0xd6, 0x4d, 0x16, 0xcc, 0x9b, 0x17, 0x4d, 0x99, 0xd9, 0x76, 0xcc, 0xcf, 0x1a, 0x91, 0x1d, 0x16, 0xb9, 0xf8, 0x76, 0xd9, 0x1c, 0xfb, 0x81, 0xe6, 0x1b, 0x4a, 0x6a, 0xf8, 0x14, 0xa2, 0x2b, 0x6, 0x6d, 0x45, 0xdf, 0x96, 0xcc, 0xd8, 0x97, 0xa1, 0x57, 0x9b, 0x69, 0xbe, 0xe5, 0xac, 0xe3, 0x3e, 0x69, 0xc0, 0xf3, 0x85, 0x27, 0x52, 0xd1, 0x83, 0x57, 0x61, 0x77, 0xda, 0x24, 0xcd, 0xd1, 0x61, 0x38, 0x98, 0x60, 0x3a, 0x30, 0x6e, 0xf3, 0xd3, 0xb3, 0x77, 0xb, 0x57, 0xd, 0xda, 0x9b, 0xc, 0x9d, 0x4c, 0x8e, 0x76, 0xfd, 0xfe, 0xcd, 0xb7, 0xb9, 0x9a, 0x6a, 0x5a, 0x1c, 0x6c, 0x20, 0x93, 0x27, 0xaa, 0xbe, 0x3e, 0x67, 0xab, 0xcf, 0xf5, 0x3c, 0xa4, 0x8f, 0x51, 0xac, 0x4e, 0x9d, 0xd6, 0xcb, 0x35, 0xd2, 0x3b, 0x88, 0xab, 0xe4, 0xdb, 0x85, 0xb7, 0x3c, 0x9b, 0x29, 0x1a, 0x5e, 0x50, 0x62, 0x68, 0xd1, 0x69, 0x90, 0x28, 0xcd, 0x29, 0xe5, 0x84, 0xcb, 0x8a, 0x20, 0xf6, 0x41, 0x48, 0x9, 0xeb, 0x7, 0x79, 0xb8, 0xc9, 0x8, 0x79, 0xad, 0x96, 0xb2, 0xeb, 0x7, 0x26, 0x75, 0xfa, 0x92, 0x26, 0x8b, 0xf5, 0x27, 0x3b, 0x5d, 0x57, 0xd5, 0x1d, 0xa2, 0x2d, 0x2b, 0x36, 0x6, 0xe2, 0xf6, 0xa4, 0xd5, 0x98, 0xf9, 0x16, 0x6a, 0x93, 0x1e, 0x92, 0xd3, 0x9b, 0xfe, 0x3a, 0xb3, 0x98, 0xd1, 0x78, 0x46, 0x65, 0x24, 0x75, 0x28, 0x9a, 0x14, 0xf3, 0x89, 0xf1, 0x95, 0xff, 0x73, 0x34, 0xe6, 0x42, 0x44, 0x7d, 0x56, 0x91, 0x7a, 0xd5, 0x81, 0x5a, 0x34, 0xfc, 0x83, 0x2f, 0xbd, 0xf3, 0xb1, 0x10, 0x8e, 0xd6, 0x7c, 0x65, 0x23, 0x44, 0x67, 0xa, 0x69, 0xf1, 0x65, 0xce, 0x73, 0x23, 0xd1, 0x10, 0x98, 0xc7, 0xcd, 0x4c, 0xed, 0x6d, 0xdc, 0xa8, 0xa2, 0x37, 0xb3, 0x50, 0x96, 0x4b, 0x4a, 0x2c, 0x33, 0xaa, 0xcb, 0x33, 0x2e, 0xb6, 0xf1, 0xa3, 0x29, 0x18, 0x65, 0x2f, 0x5f, 0x65, 0x2a, 0xed, 0x31, 0x9a, 0x93, 0xbb, 0xc7, 0x93, 0xc4, 0x7a, 0x2c, 0xa4, 0x4b, 0x77, 0x91, 0x7e, 0x99, 0x4, 0x25, 0xb8, 0x29, 0x19, 0x26, 0xc5, 0xe2, 0x59, 0x96, 0x99, 0x92, 0x7a, 0x2b, 0xb4, 0xde, 0x11, 0xcb, 0x15, 0x75, 0xa9, 0xa5, 0x34, 0xf9, 0xb3, 0x7c, 0xa9, 0xe4, 0xce, 0xbd, 0xaa, 0x1a, 0x79, 0xe5, 0x13, 0x58, 0xb9, 0x8f, 0xd9, 0x39, 0x45, 0x12, 0x46, 0xbc, 0x36, 0x59, 0x69, 0x6f, 0xbc, 0x8d, 0x2e, 0x1b, 0x30, 0xcb, 0x82, 0xe, 0x3, 0xba, 0xd3, 0xce, 0x13, 0x27, 0x71, 0x54, 0xfa, 0x65, 0x5e, 0x75, 0x4, 0x17, 0x71, 0xdb, 0x33, 0xd9, 0x7f, 0x9a, 0xe4, 0x89, 0x59, 0xe, 0x7a, 0xc2, 0x8d, 0x61, 0xab, 0x6e, 0xd6, 0xf2, 0x6d, 0x53, 0xe5, 0xcf, 0xdc, 0x1e, 0xb3, 0x36, 0x8b, 0x30, 0x58, 0x93, 0x65, 0x70, 0x75, 0x64, 0xbb, 0x22, 0x4a, 0xab, 0x57, 0xb8, 0x5e, 0x6e, 0xe6, 0xbe, 0xbe, 0xb8, 0x57, 0xd2, 0x6b, 0xf4, 0xdb, 0x9d, 0x5e, 0x66, 0x9e, 0x2b, 0x2e, 0x5b, 0x6d, 0x95, 0xdb, 0x88, 0x44, 0xc0, 0xda, 0x34, 0x3a, 0xd6, 0x1b, 0x62, 0xfe, 0x78, 0xb8, 0xe9, 0xb6, 0xd5, 0xb4, 0x89, 0x44, 0x71, 0x15, 0x55, 0xca, 0xb8, 0x38, 0x5e, 0x47, 0x1e, 0x31, 0xdc, 0x46, 0xec, 0xb6, 0xd7, 0x26, 0x51, 0x6e, 0xaf, 0xfe, 0x86, 0x77, 0xfc, 0x22, 0xd4, 0x6a, 0xea, 0xd9, 0xe7, 0x78, 0x65, 0xdb, 0xb, 0xef, 0xdd, 0x8, 0xa3, 0xb1, 0x12, 0xb, 0x4d, 0x59, 0x30, 0x54, 0xc3, 0xaa, 0x35, 0xd6, 0x69, 0x73, 0x6e, 0x6a, 0xa1, 0xb6, 0xed, 0x2c, 0x83, 0x92, 0xa3, 0x5b, 0x76, 0xcd, 0x66, 0x5d, 0xf6, 0x87, 0x5c, 0x2a, 0x1d, 0x11, 0x23, 0xaf, 0x96, 0x4b, 0xb6, 0xf8, 0xf6, 0xf, 0x4c, 0xa4, 0xe7, 0xcd, 0xc9, 0x99, 0x5, 0x61, 0xbf, 0xd7, 0x9c, 0x54, 0xef, 0x69, 0xec, 0x33, 0x96, 0x5e, 0xa6, 0x75, 0x43, 0x18, 0xfd, 0x4e, 0x59, 0xde, 0x9e, 0x47, 0xb2, 0xbd, 0xca, 0xad, 0x7b, 0x94, 0xc1, 0xc8, 0x5a, 0x3, 0xfa, 0x51, 0xb0, 0x32, 0xbd, 0x99, 0x2d, 0x99, 0xcc, 0x1c, 0xe3, 0x2e, 0xf6, 0x38, 0xd6, 0xaa, 0xe5, 0x3d, 0xb6, 0xcc, 0x5d, 0xca, 0xce, 0xcc, 0x60, 0x29, 0xd1, 0x40, 0xff, 0x3b, 0xfd, 0x23, 0x99, 0x7e, 0xc8, 0xe1, 0x9, 0x66, 0x18, 0x10, 0x50, 0x1f, 0x30, 0x1f, 0x4c, 0x5f, 0x3d, 0xd5, 0x6a, 0x55, 0x5f, 0x2e, 0x1e, 0xfe, 0xe2, 0x4c, 0x9a, 0xd9, 0xaf, 0x2f, 0xe6, 0x27, 0x18, 0x5e, 0x41, 0xc1, 0x97, 0x2e, 0x99, 0x3a, 0xbf, 0x2e, 0x33, 0xb3, 0x1f, 0xa2, 0xaf, 0xc8, 0xd7, 0x20, 0xd0, 0x34, 0x29, 0x1a, 0xa3, 0x8b, 0x10, 0x67, 0x48, 0x51, 0xb0, 0x73, 0xe5, 0x4f, 0xe4, 0xf9, 0xeb, 0xac, 0xb, 0x44, 0x2f, 0x41, 0x75, 0x7e, 0xfe, 0x3c, 0xf5, 0x7b, 0xfd, 0x70, 0xdb, 0xed, 0xd3, 0x5f, 0xea, 0x47, 0xed, 0xb2, 0x88, 0x77, 0xc6, 0x98, 0x7e, 0x1, 0xe1, 0x4e, 0x22, 0xfa, 0x6b, 0xf, 0x61, 0x57, 0x3d, 0x4, 0x37, 0x59, 0xf6, 0x8c, 0x87, 0x7, 0xaa, 0x94, 0xef, 0x18, 0x60, 0xb3, 0xbf, 0x40, 0xfe, 0xcb, 0x7f, 0x4, 0x35, 0xe2, 0xc0, 0xbc, 0x97, 0xcc, 0x36, 0x5f, 0x49, 0xce, 0x90},
	},
	sample{
		[]byte{},
		[]byte{0x0, 0x0, 0x0, 0x0, 0x0, 0x0},
	},
}
